#pragma once

#include <vector>						// STL vector<>
#include <string>						// STL string
#include <assert.h>					// assert()
using namespace std;				// STL



class CChartData
{
	// -------------------------------------------------------------------------------------

public:

	class CChartDisplayInfo
	{
	public:

		//! constructor
		CChartDisplayInfo( void )
		{
			Init();
		}

		//!
		void Init( void )
		{
			m_iYScale16k=64;
			m_iXScale16=16;
			m_iXLeft16=0;
			m_iRenderMode=0;
		}

		//!
		//! /return 1..
		int GetXScale16( void ) const {	return(m_iXScale16); }

		//!
		//! /return 1..
		int GetYScale16k( void ) const {	return(m_iYScale16k); }

		//!
		//! /param iniPosition +/- 0..0x0fffffff
		//! /return +/- 0..0x0fffffff
		int Position2X( const int iniPosition ) const
		{
			assert(iniPosition<=0x0fffffff);
			int iRet=(m_iXLeft16+iniPosition*m_iXScale16)/16;

			assert(iRet<=0x0fffffff);
			return(iRet);
		}

		//!
		//! /param iniX +/- 0..0x0fffffff in winow
		//! /return +/- 0..0x0fffffff
		int AbsoluteX2Position( const int iniX ) const
		{
			assert(iniX<=0x0fffffff);
			int iRet=(iniX*16)/m_iXScale16;

			assert(iRet<=0x0fffffff);
			return(iRet);
		}

		//!
		//! /return true=value was changed, no changes
		bool SetXScale16( int iniScale16 )
		{
			if(iniScale16<1)iniScale16=1;
			if(iniScale16>0x80)iniScale16=0x80;

			if(iniScale16!=m_iXScale16)
			{
				m_iXScale16=iniScale16;
				return(true);
			}
			return(false);
		}

		//!
		//! /return true=value was changed, no changes
		bool SetYScale16k( int iniScale16k )
		{
			if(iniScale16k<1)iniScale16k=1;
			if(iniScale16k>512)iniScale16k=512;

			if(iniScale16k!=m_iYScale16k)
			{
				m_iYScale16k=iniScale16k;
				return(true);
			}
			return(false);
		}

		//!
		//! /return +/- 0..0x0fffffff
		int GetXLeft16( void ) const { return(m_iXLeft16); }

		//!
		//! /param iniLeft16
		void SetLeft16( const int iniLeft16 ) {	assert(iniLeft16>=0);m_iXLeft16=iniLeft16;	}

	private:
		int m_iXScale16;												//!< in 1/16 parts, default: 16 which means 1:1
		int m_iYScale16k;												//!< in 1/(16*1024) parts, default: 1
		int m_iXLeft16;													//!< in 1/16 parts, default: 0
		int m_iRenderMode;											//!< 0:lines, 1=summed together

		friend class CChartData;
		friend class CChartLayer;
	};

	// -------------------------------------------------------------------------------------

	class CChartElement
	{
	public:

		//! constructor
		CChartElement( const DWORD indwPosition, const DWORD indwValue, const unsigned short inwCount )
		{
			assert(indwPosition<=0x0fffffff);
			assert(indwValue<=0x0fffffff);

			m_dwPosition=indwPosition;
			m_dwValue=indwValue;
			m_wCount=inwCount;
		}

		DWORD						m_dwPosition;						//!< 0..0x0fffffff
		DWORD						m_dwValue;							//!< 0..0x0fffffff in microseconds (1000000=1sec)
		unsigned short	m_wCount;								//!< to distinct between slow functions and often called functions
	};

	// -------------------------------------------------------------------------------------

	class CChartLayer
	{
	public:
		//! constructor
		CChartLayer( const string &insName, const DWORD indColor )
		{
			m_sName=insName;
			m_dwColor=indColor;
			m_dwMin=0;
			m_dwMax=0;
			m_fAvg=0;
			m_bHidden=true;

			if(insName=="__frametime")m_bHidden=false;
		}

		//! generates values in [0..1000000]=[0..1sec] with ms step in [1..10]
//		void GenerateDebugData( void );

		//! inefficient - store this value if you need it often
		void GetMaximum( int &outMaxPosition, int &outMaxValue );

		//!
		//! /param inGraph
		//! /param iNo 0..
		//! /param inInfo
		void Draw( Graphics *inGraph, int iniNo, const CChartDisplayInfo &inInfo, const int iniMaxValue );

		//! load data
		//! /param inoutpPtr pointer is automatically moved behind data
		//! /param indwVersion is usually 0
		//! /return true=success, false otherwise
		bool Parse( char * &inoutpPtr, const DWORD indwLayerSize, const DWORD indwVersion );

		//! used after loading to put the frametime in all layers
		void GeneratePositionFromSumOfGivenValues( CChartLayer &refLayer, const DWORD indwDivideBy );

		// slow
		DWORD GetIndexFromPosition( const int iniPos );

		DWORD GetValueInMicroSecAt( const int iniPos ) const;

		//! name and avg,min,max in ms
		string GetNameAndStats() const;

		string GetNameAndStatsAtIndex( const DWORD indwIndex, const float infSumInMS );

		void CycleColor();


		// attributes -------------------------------------------------
		string GetName( void ) const	{	return(m_sName); }

		DWORD GetColor( void ) const { return(m_dwColor); }

		void SetHidden( const bool iniValue ) { m_bHidden=iniValue; }
		bool GetHidden( void ) const { return(m_bHidden); }


		string									m_sName;					//!<

	private:
		bool										m_bHidden;				//!< hidden flag
		DWORD										m_dwColor;				//!< RGB color
		vector<CChartElement>		m_Elements;				//!< sorted by m_Position from minimum to maximum

		DWORD										m_dwMin;					//!< max y value of this layer, is calculated in Parse()
		DWORD										m_dwMax;					//!< min y value of this layer, is calculated in Parse()
		float										m_fAvg;						//!< average y value of this layer, is calculated in Parse()
	};

	// -------------------------------------------------------------------------------------

	//! constructor
	CChartData( void );

	//! destructor
	virtual ~CChartData( void );

	//!
	//! /param inGraph
	//! /param inInfo
	void Draw( Graphics *inGraph, const CChartDisplayInfo &inInfo );

	//!
	DWORD GetSumInMicroSecAt( const int iniPos ) const;

	//!
//	void GenerateDebugData( void );

	//! /param outdwWidth
	//! /param outdwHeight
	//! /param inInfo
	void GetDimensions( int &outdwWidth, int &outdwHeight, const CChartDisplayInfo &inInfo ) const;

	//!
	void GetDimensions( int &outiWidth16, int &outiHeight16 ) const;

	//! /param inszFilePath filename with path and extension
	//! /return true=success, false otherwise
	bool Load( const char *inszFilePath );

//	bool Save( const char *inszFilePath );

	//! /param iniNo 0..
	//! /return 0 if the index does not exits, otherwise the pointer to the layer (don't store this pointer)
	CChartLayer *GetLayerNo( const int iniNo );

private:

	int												m_iMaxPosition;		//!<
	int												m_iMaxValue;			//!<
	vector<CChartLayer * >		m_Layers;					//!< this class is responsive for deleting the objects in this container

	//!
	void RecalculateDimensions( void );
};
