#include "StdAfx.h"

#include "ColladaExportWriter.h"
#include "ColladaWriter.h"
#include "IExportSource.h"
#include "PathHelpers.h"
#include "IExporter.h"
#include "ResourceCompilerHelper.h"
#include "IExportContext.h"
#include "ProgressRange.h"
#include "XMLWriter.h"
#include "XMLPakFileSink.h"
#include "ISettings.h"
#include "SingleAnimationExportSourceAdapter.h"
#include "GeometryExportSourceAdapater.h"
#include "ModelData.h"
#include "MaterialData.h"
#include "GeometryFileData.h"
#include "FileSystemHelpers.h"
#include "CBAHelpers.h"
#include "ModuleHelpers.h"
#include <ctime>
#include <list>

namespace
{
	class ResourceCompilerLogListener : public IResourceCompilerListener
	{
	public:
		ResourceCompilerLogListener(IExportContext* context): context(context) {}
		virtual void OnRCMessage(MessageSeverity severity, const char* text)
		{
			IExportContext::MessageSeverity logSeverity = (IExportContext::MessageSeverity)severity;

			// Normal RC text should just be debug.
			if (severity == MessageSeverity_Info)
				logSeverity = IExportContext::MessageSeverity_Debug;

			context->Log(logSeverity, text);
		}
	private:
		IExportContext* context;
	};
}

// Include build information - generated in pre-build step.
#include "Generated_BuildInfo.h"

void ColladaExportWriter::Export(IExportSource* source, IExportContext* context)
{
	// Create an object to report on our progress to the export context.
	ProgressRange progressRange(context, &IExportContext::SetProgress);

	// Log build information.
	Log(context, IExportContext::MessageSeverity_Info, "Exporter build created on " BUILD_DATE_STRING);
	Log(context, IExportContext::MessageSeverity_Info, "Build created by username " BUILD_USER_NAME);

#ifdef STLPORT
	Log(context, IExportContext::MessageSeverity_Info, "Using STLport C++ Standard Library implementation");
#else //STLPORT
	Log(context, IExportContext::MessageSeverity_Info, "Using Microsoft (tm) C++ Standard Library implementation");
#endif //STLPORT

#ifndef NDEBUG
	Log(context, IExportContext::MessageSeverity_Info, "******DEBUG BUILD******");
#else //_DEBUG
	Log(context, IExportContext::MessageSeverity_Info, "Release build.");
#endif //_DEBUG

	int wordSize = sizeof(void*);
	int bitCount = wordSize * 8;
	Log(context, IExportContext::MessageSeverity_Debug, "Bit count == %d.", bitCount);

	std::string exePath = ModuleHelpers::GetCurrentModulePath(ModuleHelpers::CurrentModuleSpecifier_Executable);
	Log(context, IExportContext::MessageSeverity_Debug, "Application path: %s", exePath.c_str());

	std::string exporterPath = ModuleHelpers::GetCurrentModulePath(ModuleHelpers::CurrentModuleSpecifier_Library);
	Log(context, IExportContext::MessageSeverity_Debug, "Exporter path: %s", exporterPath.c_str());

	// Log the start time.
	{
		char buf[1024];
		std::time_t t = std::time(0);
		std::strftime(buf, sizeof(buf) / sizeof(buf[0]), "%H:%M:%S on %a, %d/%m/%Y", std::localtime(&t));
		Log(context, IExportContext::MessageSeverity_Info, "Export begun at %s", buf);
	}

	// Select the name of the directory to export to.
	std::string sourceDirectory;
	try
	{
		std::string sourcePath = GetFileName(source);
		sourceDirectory = PathHelpers::GetDirectory(sourcePath);
	}
	catch (IExportSource::NoFileNameError e)
	{
		throw IExporter::NeedSaveError("Scene must be saved before exporting.");
	}

	std::string rootPath = GetRootPath(context);

	GeometryFileData geometryFileData;
	MaterialData materialData;
	std::vector<ModelData> modelData;
	typedef std::vector<std::pair<int, std::pair<std::string, std::string> > > GeometryFileNameList;
	GeometryFileNameList geometryFileNameList;
	typedef std::vector<std::pair<std::pair<int, int>, std::string> > AnimationFileNameList;
	AnimationFileNameList animationFileNameList;
	{
		CurrentTaskScope currentTask(context, "dae");

		// Choose the files to which to export all the animations.
		std::list<SingleAnimationExportSourceAdapter> animationExportSources;
		std::list<GeometryExportSourceAdapater> geometryExportSources;
		typedef std::vector<std::pair<std::string, IExportSource*> > ExportList;
		ExportList exportList;
		{
			ProgressRange readProgressRange(progressRange, 0.2f);

			source->ReadGeometryFiles(context, &geometryFileData);
			source->ReadMaterials(&geometryFileData, &materialData);
			modelData.resize(geometryFileData.GetGeometryFileCount());
			for (int geometryFileIndex = 0, geometryFileCount = geometryFileData.GetGeometryFileCount(); geometryFileIndex < geometryFileCount; ++geometryFileIndex)
				source->ReadModels(&geometryFileData, &materialData, &modelData[geometryFileIndex], geometryFileIndex);
			int animationCount = source->GetAnimationCount();
			for (int geometryFileIndex = 0; geometryFileIndex < geometryFileData.GetGeometryFileCount(); ++geometryFileIndex)
			{
				std::string geometryFileName = geometryFileData.GetGeometryFileName(geometryFileIndex);

				// Choose the output directory - normally the source file directory, but
				// can be overridden by directives in the scene.
				std::string exportDirectory = sourceDirectory;
				std::string exportPathProperty = geometryFileData.GetPropertyValue(geometryFileIndex, "exportPath");
				if (!exportPathProperty.empty())
				{
					exportDirectory = exportPathProperty;
					if (exportDirectory[0] == '/' || exportDirectory[0] == '\\')
						exportDirectory = PathHelpers::Join(PathHelpers::Join(rootPath, "game"), exportDirectory.substr(1));
					Log(context, IExportContext::MessageSeverity_Info, "exportPath property found for model '%s': Setting export directory to '%s'.", geometryFileName.c_str(), exportDirectory.c_str());
				}

				IGeometryFileData::ContentType contentType = geometryFileData.GetContentType(geometryFileIndex);
				bool hasGeometry = contentType != IGeometryFileData::ContentType_CAF && contentType != IGeometryFileData::ContentType_ANM;

				if (hasGeometry && !geometryFileName.empty())
				{
					std::string safeGeometryFileName = geometryFileName;
					std::replace(safeGeometryFileName.begin(), safeGeometryFileName.end(), ' ', '_');
					std::string extension = "missingextension";
					switch (contentType)
					{
						case IGeometryFileData::ContentType_CGF: extension = "cgf"; break;
						case IGeometryFileData::ContentType_CGA: extension = "cga"; break;
						case IGeometryFileData::ContentType_ANM: extension = "anm"; break;
						case IGeometryFileData::ContentType_CGAANM: extension = "cga"; break;
						case IGeometryFileData::ContentType_CHR: extension = "chr"; break;
						case IGeometryFileData::ContentType_CAF: extension = "caf"; break;
						case IGeometryFileData::ContentType_CHRCAF: extension = "chr"; break;
					}

					std::string colladaPath = PathHelpers::Join(exportDirectory, safeGeometryFileName + ".dae");
					std::string assetPath = PathHelpers::Join(exportDirectory, safeGeometryFileName + "." + extension);
					geometryFileNameList.push_back(std::make_pair(geometryFileIndex, std::make_pair(colladaPath, assetPath)));
					geometryExportSources.push_back(GeometryExportSourceAdapater(source, &geometryFileData, geometryFileIndex));
					exportList.push_back(std::make_pair(colladaPath, &geometryExportSources.back()));
				}

				for (int animationIndex = 0; animationIndex < animationCount; ++animationIndex)
				{
					std::string animationName = GetAnimationName(source, &geometryFileData, geometryFileIndex, animationIndex);

					// Animations beginning with an underscore should be ignored.
					bool ignoreAnimation = (animationName[0] == '_');

					if (!ignoreAnimation)
					{
						std::string safeAnimationName = animationName;
						std::replace(safeAnimationName.begin(), safeAnimationName.end(), ' ', '_');
						std::string filename = safeAnimationName + ".dae";

						std::string exportPath = PathHelpers::Join(exportDirectory, filename);
						animationFileNameList.push_back(std::make_pair(std::make_pair(animationIndex, geometryFileIndex), exportPath));
						animationExportSources.push_back(SingleAnimationExportSourceAdapter(source, &geometryFileData, geometryFileIndex, animationIndex));
						exportList.push_back(std::make_pair(exportPath, &animationExportSources.back()));
					}
				}
			}
		}

		// Export the Collada file to the chosen file.
		{
			ProgressRange exportProgressRange(progressRange, 0.6f);

			// Check whether we should export compressed or uncompressed files.
			int exportCompressed = GetSetting<int>(context->GetSettings(), "ExportCompressedCOLLADA", 1);
			Log(context, IExportContext::MessageSeverity_Debug, "Checking ExportCompressedCOLLADA key: %d", exportCompressed);

			int daeCount = int(exportList.size());
			float daeProgressRangeSlice = 1.0f / (daeCount > 0 ? float(daeCount) : 1.0f);
			for (ExportList::iterator itFile = exportList.begin(); itFile != exportList.end(); ++itFile)
			{
				const std::string& colladaFileName = (*itFile).first;
				IExportSource* fileExportSource = (*itFile).second;

				ProgressRange animationExportProgressRange(exportProgressRange, daeProgressRangeSlice);

				try
				{
					Log(context, IExportContext::MessageSeverity_Info, "Exporting to file '%s'", colladaFileName.c_str());

					ColladaWriter writer;
					IPakSystem* pakSystem = (context ? context->GetPakSystem() : 0);
					if (!pakSystem)
						throw IExporter::PakSystemError("No pak system provided.");
					std::string archivePath = colladaFileName + ".zip";
					std::string archiveRelativePath = PathHelpers::GetFilename(colladaFileName);

					// Try to create the directory for the file.
					FileSystemHelpers::EnsureDirectoryExists(PathHelpers::GetDirectory(colladaFileName));

					if (exportCompressed)
					{
						XMLPakFileSink sink(pakSystem, archivePath, archiveRelativePath);
						writer.Write(fileExportSource, context, &sink, animationExportProgressRange);
					}
					else
					{
						XMLFileSink fileSink(colladaFileName);
						writer.Write(fileExportSource, context, &fileSink, animationExportProgressRange);
					}
				}
				catch (FileSystemHelpers::DirectoryCreationFailedError e)
				{
					Log(context, IExportContext::MessageSeverity_Error, "Unable to create directory: %s", e.what());
				}
				catch (IXMLSink::OpenFailedError e)
				{
					Log(context, IExportContext::MessageSeverity_Error, "Unable to open output file: %s", e.what());
				}
			}
		}
	}

	// Run the resource compiler on the Collada file to generate the CAFs.
	{
		ProgressRange compilerProgressRange(progressRange, 0.075f);

		CurrentTaskScope currentTask(context, "rc");

		int daeCount = int(animationFileNameList.size());
		float animationProgressRangeSlice = 1.0f / (daeCount > 0 ? float(daeCount) : 1.0f);
		for (AnimationFileNameList::iterator itFile = animationFileNameList.begin(); itFile != animationFileNameList.end(); ++itFile)
		{
			int animationIndex = (*itFile).first.first;
			std::string colladaFileName = (*itFile).second;

			ProgressRange animationCompileProgressRange(compilerProgressRange, animationProgressRangeSlice);
			ResourceCompilerLogListener listener(context);
			CResourceCompilerHelper rcHelper;
			Log(context, IExportContext::MessageSeverity_Info, "Calling RC to generate uncompressed CAF file: %s", colladaFileName.c_str());
			rcHelper.CallResourceCompiler(colladaFileName.c_str(), 0, &listener);
			Log(context, IExportContext::MessageSeverity_Debug, "RC finished: %s", colladaFileName.c_str());
		}
	}

	// Run the resource compiler on the Collada file to generate the geometry assets.
	{
		ProgressRange compilerProgressRange(progressRange, 0.075f);

		CurrentTaskScope currentTask(context, "rc");

		int daeCount = int(geometryFileNameList.size());
		float assetProgressRangeSlice = 1.0f / (daeCount > 0 ? float(daeCount) : 1.0f);
		for (GeometryFileNameList::iterator itFile = geometryFileNameList.begin(); itFile < geometryFileNameList.end(); ++itFile)
		{
			int geometryFileIndex = (*itFile).first;
			std::string colladaFileName = (*itFile).second.first;
			std::string assetFileName = (*itFile).second.second;

			ProgressRange assetCompileProgressRange(compilerProgressRange, assetProgressRangeSlice);
			ResourceCompilerLogListener listener(context);
			CResourceCompilerHelper rcHelper;
			Log(context, IExportContext::MessageSeverity_Info, "Calling RC to generate raw asset file: %s", colladaFileName.c_str());
			rcHelper.CallResourceCompiler(colladaFileName.c_str(), 0, &listener);
			Log(context, IExportContext::MessageSeverity_Debug, "RC finished: %s", colladaFileName.c_str());
		}
	}

	{
		// Create an RC helper - do it outside the loop, since it queries the registry on construction.
		ResourceCompilerLogListener listener(context);
		CResourceCompilerHelper rcHelper;

		// Check the registry to see whether we should compress the animations or not.
		int processAnimations = GetSetting<int>(context->GetSettings(), "CompressCAFs", 1);

		// Run the resource compiler again on the generated CAF files to compress/process them.
		// TODO: This should not be necessary, the RC should be modified so that CAFs are automatically
		// compressed when exported from Collada.
		if (!processAnimations)
		{
			Log(context, IExportContext::MessageSeverity_Warning, "CompressCAFs registry key set to 0 - not compressing CAFs");
		}
		else
		{
			Log(context, IExportContext::MessageSeverity_Debug, "CompressCAFs not set or set to 1 - compressing CAFs");

			CurrentTaskScope currentTask(context, "compress");
			ProgressRange compressRange(progressRange, 0.025f);

			int cafCount = int(animationFileNameList.size());
			float animationProgressRangeSlice = 1.0f / (cafCount > 0 ? float(cafCount) : 1.0f);
			for (AnimationFileNameList::iterator itFile = animationFileNameList.begin(); itFile != animationFileNameList.end(); ++itFile)
			{
				int animationIndex = (*itFile).first.first;
				std::string colladaFileName = (*itFile).second;
				ProgressRange animationProgressRange(compressRange, animationProgressRangeSlice);

				// Assume the RC generated the CAF file using the take name and adding .CAF.
				std::string cafPath = PathHelpers::ReplaceExtension(colladaFileName, "caf");

				std::string cbaPath = CBAHelpers::FindCBAFileForFile(cafPath, context->GetPakSystem());

				if (cbaPath.empty())
				{
					Log(context, IExportContext::MessageSeverity_Error, "Unable to find CBA file for file \"%s\" (looked for a root game directory that contains a relative path of \"Animations/Animations.cba\"", cafPath.c_str());
				}
				else
				{
					char buffer[2048];
					sprintf(buffer, "/file=\"%s\"", cafPath.c_str());
					Log(context, IExportContext::MessageSeverity_Info, "Calling RC to compress CAF file: (CBA file = %s) %s", cbaPath.c_str(), buffer);
					rcHelper.CallResourceCompiler(cbaPath.c_str(), buffer, &listener);
					Log(context, IExportContext::MessageSeverity_Debug, "RC finished: %s %s", cbaPath.c_str(), buffer);
				}
			}
		}

		// Check the registry to see whether we should optimize the geometry files or not.
		int optimizeGeometry = GetSetting<int>(context->GetSettings(), "OptimizeAssets", 1);

		// Run the resource compiler again on the generated geometry files to compress/process them.
		// TODO: This should not be necessary, the RC should be modified so that assets are automatically
		// compressed when exported from Collada.
		if (!optimizeGeometry)
		{
			Log(context, IExportContext::MessageSeverity_Warning, "OptimizeAssets registry key set to 0 - not compressing CAFs");
		}
		else
		{
			Log(context, IExportContext::MessageSeverity_Debug, "OptimizeAssets not set or set to 1 - optimizing geometry");

			CurrentTaskScope currentTask(context, "compress");
			ProgressRange compressRange(progressRange, 0.025f);

			int assetCount = int(geometryFileNameList.size());
			float assetProgressRangeSlice = 1.0f / (assetCount > 0 ? float(assetCount) : 1.0f);
			for (GeometryFileNameList::iterator itFile = geometryFileNameList.begin(); itFile != geometryFileNameList.end(); ++itFile)
			{
				int geometryFileIndex = (*itFile).first;
				std::string colladaFileName = (*itFile).second.first;
				std::string assetFileName = (*itFile).second.second;
				ProgressRange animationProgressRange(compressRange, assetProgressRangeSlice);

				Log(context, IExportContext::MessageSeverity_Info, "Calling RC to optimize asset \"%s\"", assetFileName.c_str());
				rcHelper.CallResourceCompiler(assetFileName.c_str(), 0, &listener);
				Log(context, IExportContext::MessageSeverity_Debug, "RC finished: %s", assetFileName.c_str());
			}
		}
	}

	// Log the end time.
	{
		char buf[1024];
		std::time_t t = std::time(0);
		std::strftime(buf, sizeof(buf) / sizeof(buf[0]), "%H:%M:%S on %a, %d/%m/%Y", std::localtime(&t));
		Log(context, IExportContext::MessageSeverity_Info, "Export finished at %s", buf);
	}
}
