#ifndef __IEXPORTSOURCE_H__
#define __IEXPORTSOURCE_H__

#include "Exceptions.h"

class ISkeletonData;
class IAnimationData;
class IExportContext;
class IModelData;
class IGeometryFileData;
class IGeometryData;
class IMaterialData;
class ISkinningData;

class IExportSource
{
public:

	// Declare an exception type to report the case where the filename is requested but the scene
	// has no filename.
	struct NoFileNameErrorTag {};
	typedef Exception<NoFileNameErrorTag> NoFileNameError;

	virtual int GetFileName(char* buffer, int bufferSize) const = 0;
	virtual void ReadGeometryFiles(IExportContext* context, IGeometryFileData* geometryFileData) = 0;
	virtual void ReadModels(IGeometryFileData* geometryFileData, IMaterialData* materialData, IModelData* modelData, int geometryFileIndex) = 0;
	virtual void ReadMaterials(IGeometryFileData* geometryFileData, IMaterialData* materialData) = 0;
	virtual bool ReadSkeleton(IGeometryFileData* geometryFileData, int geometryFileIndex, IModelData* modelData, int modelIndex, const IMaterialData* materialData, ISkeletonData* skeletonData) = 0;
	virtual void ReadSkinning(IExportContext* context, ISkinningData* skinningData, IModelData* modelData, int modelIndex, ISkeletonData* skeletonData) = 0;
	virtual int GetAnimationCount() const = 0;
	virtual int GetAnimationName(char* buffer, int bufferSize, IGeometryFileData* geometryFileData, int geometryFileIndex, int animationIndex) const = 0;
	virtual void GetAnimationTimeSpan(float& start, float& stop, int animationIndex) const = 0;
	virtual void ReadAnimationFlags(IExportContext* context, IAnimationData* animationData, IGeometryFileData* geometryFileData, const IModelData* modelData, int modelIndex, const ISkeletonData* skeletonData, int animationIndex) const = 0;
	virtual void ReadAnimation(IExportContext* context, IAnimationData* animationData, IGeometryFileData* geometryFileData, const IModelData* modelData, int modelIndex, const ISkeletonData* skeletonData, int animationIndex, float fps) const = 0;
	virtual bool ReadGeometry(IExportContext* context, IGeometryData* geometry, IModelData* modelData, IMaterialData* materialData, int modelIndex) const = 0;
	virtual bool ReadBoneGeometry(IExportContext* context, IGeometryData* geometry, ISkeletonData* skeletonData, int boneIndex, IMaterialData* materialData) const = 0;
};

inline std::string GetFileName(IExportSource* source)
{
	char buffer[1024];
	source->GetFileName(buffer, sizeof(buffer) / sizeof(buffer[0]));
	return buffer;
}

inline std::string GetAnimationName(IExportSource* source, IGeometryFileData* geometryFileData, int geometryFileIndex, int animationIndex)
{
	char buffer[1024];
	source->GetAnimationName(buffer, sizeof(buffer) / sizeof(buffer[0]), geometryFileData, geometryFileIndex, animationIndex);
	return buffer;
}

#endif //__IEXPORTSOURCE_H__
