////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2008.
// -------------------------------------------------------------------------
//  File name:   PakSystem.cpp
//  Version:     v1.00
//  Created:     8/4/2008 by MichaelS.
//  Compilers:   Visual Studio.NET 2005
//  Description: 
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "PakSystem.h"
#include "PathHelpers.h"

PakSystem::PakSystem()
{
}

PakSystemFile* PakSystem::Open(const char* path, const char* mode)
{
	// Try to open the file.
	FILE* f = fopen(path, mode);

	PakSystemFile* file = (f ? new PakSystemFile() : 0);
	if (file)
	{
		file->type = PakSystemFileType_File;
		file->file = f;
		return file;
	}

	// If the file cannot be opened, look for a zip that contains the file.
	string zipPath = string(path) + ".zip";
	string filename = PathHelpers::GetFilename(path);

	// if it's simple and read-only, it's assumed it's read-only
	unsigned nFactoryFlags = ZipDir::CacheFactory::FLAGS_DONT_COMPACT | ZipDir::CacheFactory::FLAGS_READ_ONLY;

	// Check if file on disk exist.
	bool bFileExists = false;
	FILE *fp = fopen(zipPath.c_str(), "rb");
	if (fp)
	{
		fclose(fp);
		bFileExists = true;
	}
	else
	{
		//no .zip file, so could be within a .pak file
		//find all 'potential' pak files and look within these for a matching file
		std::vector<string> foundFileCountainer; // pak files found

		string dirToSearch = path;
		do 
		{
			dirToSearch = PathHelpers::RemoveSeparator( dirToSearch );
			dirToSearch = PathHelpers::GetParentDirectory( dirToSearch );	//descend path
			if( dirToSearch != "" )
			{
				dirToSearch += "\\";
			}

			WIN32_FIND_DATA FindFileData;
			HANDLE hFind;
			string search = "*.pak";
			search = dirToSearch + search;
	
			hFind = FindFirstFile( search.c_str(),&FindFileData );
			if ( hFind != INVALID_HANDLE_VALUE )
			{
				foundFileCountainer.push_back( dirToSearch + string( FindFileData.cFileName ) );
				while( FindNextFile( hFind,&FindFileData ) != 0 )
				{
					foundFileCountainer.push_back( dirToSearch + string( FindFileData.cFileName ) );

				}
			}

			FindClose(hFind);

		} while( dirToSearch != "" );

		//iterate ('cheaply' without iterators) through found containers and look for relavent files within them
		for( int iFile = 0; iFile < foundFileCountainer.size(); ++iFile )
		{
			zipPath = foundFileCountainer[ iFile ];
			string pathToZip = PathHelpers::GetParentDirectory( zipPath );

			//construct filename by removing path to zip from path to filename
			string pathToFile = PathHelpers::GetParentDirectory( string( path ) );
			string pureFileName = PathHelpers::GetFilename( string( path ) );
			pathToFile = pathToFile.substr( pathToZip.length() + 1 );
			if( pathToFile != "" )
			{
				filename = pathToFile + "\\" + pureFileName;
			}
			else
			{
				filename =  pureFileName;
			}

			ZipDir::CacheFactory factory(ZipDir::ZD_INIT_FAST, nFactoryFlags);
			ZipDir::CachePtr testZip = factory.New( zipPath.c_str() );
			ZipDir::FileEntry* testFileEntry = (testZip ? testZip->FindFile(filename.c_str()) : 0);

			//break out if we have a testFileEntry, as we've found our first (and best) candidate.
			if( testFileEntry )
			{
				bFileExists = true;
				break;
			}
		}
	}

	ZipDir::CacheFactory factory(ZipDir::ZD_INIT_FAST, nFactoryFlags);
	ZipDir::CachePtr zip = (bFileExists ? factory.New(zipPath.c_str()) : 0);
	ZipDir::FileEntry* fileEntry = (zip ? zip->FindFile(filename.c_str()) : 0);

	file = (fileEntry ? new PakSystemFile() : 0);
	if (file)
	{
		file->type = PakSystemFileType_PakFile;
		file->zip = zip;
		file->fileEntry = fileEntry;
		file->data = zip->AllocAndReadFile(file->fileEntry);
		file->dataPosition = 0;
	}


	return file;
}


//Extracts archived file to disk without overwriting any files
//returns true on success, false on failure (due to potential overwrite or no file
//in archive
bool PakSystem::ExtractNoOverwrite(const char* fileToExtract, const char* extractToFile)
{
	if( 0 == extractToFile )
	{
		extractToFile = fileToExtract;
	}

	//open file using pak system
	PakSystemFile* fileZip = Open(fileToExtract, "r");
	if( !fileZip  )
	{
		return false;
	}

	// Try to open a writable file
	FILE* fFileOnDisk = fopen(extractToFile, "wb");
	if(!fFileOnDisk)
	{
		Close( fileZip );
		return false;
	}

	fwrite( fileZip->data, fileZip->fileEntry->desc.lSizeUncompressed, 1, fFileOnDisk );
	fclose( fFileOnDisk );
	return true;
}


void PakSystem::Close(PakSystemFile* file)
{
	if (file)
	{
		switch (file->type)
		{
		case PakSystemFileType_File:
			fclose(file->file);
			break;

		case PakSystemFileType_PakFile:
			file->zip->Free(file->data);
			break;
		}
		delete file;
	}
}

int PakSystem::Read(PakSystemFile* file, void* buffer, int size)
{
	int readBytes = 0;

	if (file)
	{
		switch (file->type)
		{
		case PakSystemFileType_File:
			{
				readBytes = fread(buffer, 1, size, file->file);
			}
			break;

		case PakSystemFileType_PakFile:
			{
				int fileSize = file->fileEntry->desc.lSizeUncompressed;
				readBytes = (fileSize - file->dataPosition > size ? size : fileSize - file->dataPosition);
				memcpy(buffer, static_cast<char*>(file->data) + file->dataPosition, readBytes);
				file->dataPosition += readBytes;
			}
			break;
		}
	}

	return readBytes;
}

bool PakSystem::EoF(PakSystemFile* file)
{
	bool EoF = true;
	if (file)
	{
		switch (file->type)
		{
		case PakSystemFileType_File:
			{
				EoF = (0 != feof(file->file));
			}
			break;

		case PakSystemFileType_PakFile:
			{
				int fileSize = file->fileEntry->desc.lSizeUncompressed;
				EoF = (file->dataPosition >= fileSize);
			}
			break;
		}
	}

	return EoF;
}

PakSystemArchive* PakSystem::OpenArchive(const char* path)
{
	unsigned nFactoryFlags = ZipDir::CacheFactory::FLAGS_DONT_COMPACT | ZipDir::CacheFactory::FLAGS_CREATE_NEW;
	ZipDir::CacheFactory factory(ZipDir::ZD_INIT_FAST, nFactoryFlags);
	ZipDir::CacheRWPtr cache = factory.NewRW(path);
	PakSystemArchive* archive = (cache ? new PakSystemArchive() : 0);
	if (archive)
		archive->zip = cache;
	return archive;
}

void PakSystem::CloseArchive(PakSystemArchive* archive)
{
	if (archive)
	{
		archive->zip->Close();
		delete archive;
	}
}

void PakSystem::AddToArchive(PakSystemArchive* archive, const char* path, void* data, int size)
{
	archive->zip->UpdateFile(path, data, size, ZipFile::METHOD_DEFLATE);
}
