import sys
from crymp import *

LOG_DEBUG = 'DEBUG'
LOG_INFO = 'INFO'
LOG_WARNING = 'WARNING'
LOG_ERROR = 'ERROR'

def log(type, level, message, file_only = False):
  """Log a message.

  :Parameters:
    - `type`: The message type.  This is one of LOG_DEBUG, LOG_INFO,
      LOG_WARNING, LOG_ERROR.
    - `level`: The log level.  This is an integer number.  Higher numbers
      indicate more verbose output.  0 is the standard verbosity.  Use -1 if
      the message _must_ be displayed, even if silent operation is requested.
    - `message`: The message string to be logged.
    - `file_only`: Flag indicating that the message should be logged to the
      log file only and should not be echoed to the standard output or
      standard error.
  """

  log_to_screen = True
  log_to_file = True
  if Options.log_filename is None: log_to_file = False

  if file_only: log_to_screen = False
  if level > Options.verbose: log_to_screen = False
  if level > Options.verbose_file: log_to_file = False
  if type == LOG_DEBUG:
    if level > Options.debug:
      log_to_screen = False
      log_to_file = False

  if message[-1] != '\n': message += '\n'

  if log_to_screen:
    out = sys.stdout
    if type in (LOG_WARNING, LOG_ERROR): out = sys.stderr
    out.write('crymp: ' + message)

  if log_to_file:
    try:
      out = file(Options.log_filename, 'a')
    except:
      filename = Options.log_filename
      Options.log_filename = None
      warning('Can not open log file ' + repr(filename)
	  + ', file logging disabled!')
      out = None
    if out is not None: out.write(message)

def debug(message, level = 0, file_only = True):
  log(LOG_DEBUG, level, message, file_only)

def info(message, level = 0, file_only = False):
  log(LOG_INFO, level, message, file_only)

def warning(message, level = 0, file_only = False):
  log(LOG_WARNING, level, 'Warning: ' + message, file_only)

def error(message, level = -1):
  log(LOG_ERROR, level, 'Error: ' + message)

def loc(filename, line = None):
  """Create a location string for the specified file name and linenumber.

  :Parameters:
    - `filename`: The file name.
    - `line`: The line number or None.

  :Return:
    A string representation of the location.
  """

  s = 'In file ' + repr(filename)
  if line is not None: s += ', line ' + str(line)
  s += ': '
  return s

