import sys, os, os.path, string
from crymp import *

def set_indent(buffer, indent):
  """Set the indent of the specified code block to the specified value.

  :Parameters:
    - `buffer`: The buffer to be indented.
    - `indent`: The requested indent level.  The indent is multplied with
      Options.shift_width. 

  :Return:
    The indented block.
  """

  lines, min_indent = parse.expand(buffer, Options.tab_size, return_lines = True)
  lines = parse.shift_indent(lines, indent * Options.shift_width - min_indent)
  return string.join(lines, '\n')

def output():
  """Generate the tool output files.
  """

  assert Options.basename is not None

  if Options.verbose > 2:
    sys.stdout.write('<!-- XML Dump -->\n')
    output_xml(output_filename = None)

  if Options.debug:
    output_xml(output_filename = Options.basename + '.xml')

  output_commonfile(output_filename = Options.basename + '.cpp')
  output_all_implfiles()

def output_xml(output_filename):
  """Generate a pseudo XML dump of the block structure.

  :Parameters:
    - `output_filename`: The output filename.  None for stdout.
  """

  to_stdout = False
  if output_filename is not None:
    try:
      out = file(output_filename, 'w')
    except IOError, e:
      error('Error writing XML dump: ' + str(e))
      sys.exit(1)
  else:
    out = sys.stdout
    to_stdout = True

  # Dump the block structure.
  out.write(parse.indent() + '<Blocks>\n')
  Options.output_indent += 1
  for name in parse.PBlock.block_map:
    pblock = parse.PBlock.block_map[name]
    out.write(str(pblock))
  Options.output_indent -= 1
  out.write(parse.indent() + '</Blocks>\n')

  if not to_stdout: out.close()


def output_commonfile(output_filename):
  """Generate the implementation file for the parallelized code.

  The implementation file contains common implementation code for all parallel
  blocks.  Note that no implementation code from CM blocks goes into the C++
  file.

  :Parameters:
    - `output_filename`: The output filename.  None for stdout.
  """

  to_stdout = False
  if output_filename is not None:
    try:
      out = file(output_filename, 'w')
    except IOError, e:
      error('Error writing C++ file: ' + str(e))
      sys.exit(1)
  else:
    out = sys.stdout
    to_stdout = True

  skel = Options.skel
  cppfile_filename = Options.basename + '.cpp'
  context_dict = {
    'basename': Options.basename,
    'prefix': Options.prefix,
    'verbose': Options.verbose,
    'debug': Options.debug,
    'filename': cppfile_filename
  }
  context = Context(context_dict)
  commonfile_head = skel.expand('commonfile_head', context)
  commonfile_tail = skel.expand('commonfile_tail', context)
  out.write(set_indent(commonfile_head, 0))
  out.write(set_indent(commonfile_tail, 0))

  if not to_stdout: out.close()

def output_all_implfiles():
  """Output all generated output implementation files.
  """

  # Create an input file name to block list mapping.
  file_map = { }
  for name in parse.PBlock.block_map:
    pblock = parse.PBlock.block_map[name]
    filename = pblock.filename
    if filename in file_map:
      file_map[filename].append(pblock)
    else:
      file_map[filename] = [ pblock ]

  for filename in file_map:
    include_blocks = [ ]
    basename = os.path.basename(filename)
    if basename.find('.') != -1:
      output_filename = (
	filename[:filename.rfind('.')]
	+ '_out'
	+ filename[filename.rfind('.'):])
    else:
      output_filename = filename + '_out.cpp'
    output_implfile(output_filename, file_map[filename])

def output_implfile(output_filename, pblock_list):
  """Output the implementation code for a list of parallel blocks to the
  specified output file.

  :Parameters:
    - `output_filename`: The output filename.  None for stdout.
    - `pblock_list`: List of parallel blocks.
  """

  to_stdout = False
  if output_filename is not None:
    try:
      out = file(output_filename, 'w')
    except IOError, e:
      error('Error writing implementation file: ' + str(e))
      sys.exit(1)
  else:
    out = sys.stdout
    to_stdout = True

  include_blocks = [ ]
  skel = Options.skel
  context_dict = {
    'basename': Options.basename,
    'prefix': Options.prefix,
    'verbose': Options.verbose,
    'debug': Options.debug,
    'filename': os.path.basename(output_filename)
  }
  context = Context(context_dict)
  implfile_head = skel.expand('implfile_head', context)
  implfile_tail = skel.expand('implfile_tail', context)
  out.write(set_indent(implfile_head, 0))

  for pblock in pblock_list:

    pblock_context = pblock.get_context(context)
    pblock_class_head = skel.expand('pblock_class_head', pblock_context)
    pblock_class_tail = skel.expand('pblock_class_tail', pblock_context)

    out.write(set_indent(pblock_class_head, 0))

    for pdecl in pblock.decls:
      pdecl_context = pdecl.get_context(pblock_context)
      pdecl_decl_shared = skel.expand('pdecl_decl_shared', pdecl_context)
      out.write(set_indent(pdecl_decl_shared, 1))

    for job in pblock.jobs:
      job_decl = job.get_decl(skel, pblock_context, 'shared')
      out.write(set_indent(job_decl, 1))

    run_decl = skel.expand('pblock_run_decl', pblock_context)
    out.write(set_indent(run_decl, 1))

    out.write(set_indent(pblock_class_tail, 0))

    pblock_local_class_head = skel.expand(
	'pblock_local_class_head', pblock_context)
    pblock_local_class_tail = skel.expand(
	'pblock_local_class_tail', pblock_context)

    out.write(set_indent(pblock_local_class_head, 0))

    for pdecl in pblock.decls:
      pdecl_context = pdecl.get_context(pblock_context)
      pdecl_decl = skel.expand('pdecl_decl', pdecl_context)
      out.write(set_indent(pdecl_decl, 1))

    for job in pblock.jobs:
      job_decl = job.get_decl(skel, pblock_context, 'local')
      out.write(set_indent(job_decl, 1))

    out.write(set_indent(pblock_local_class_tail, 0))

    for job in pblock.jobs:
      job_shared_def = job.get_def(skel, pblock_context, 'shared')
      job_local_def = job.get_def(skel, pblock_context, 'local')
      out.write(set_indent(job_shared_def, 0))
      out.write(set_indent(job_local_def, 0))

    run_head = skel.expand('pblock_run_head', pblock_context)
    run_tail = skel.expand('pblock_run_tail', pblock_context)

    run_body = None
    if Options.driver == Options.DRIVER_SINGLECORE:
      run_body = singlecore.run_body(pblock, skel, pblock_context)
    elif Options.driver == Options.DRIVER_MULTICORE:
      run_body = multicore.run_body(pblock, skel, pblock_context)
    elif Options.driver == Options.DRIVER_PS3:
      run_body = ps3.run_body(pblock, skel, pblock_context)
    else:
      raise ValueError

    out.write(set_indent(run_head, 0))
    out.write(set_indent(run_body, 1))
    out.write(set_indent(run_tail, 0))

  out.write(set_indent(implfile_tail, 0))

  if not to_stdout: out.close()

