import sys, re, string
from crymp import *

def line_offset(buffer, line_number):
  """Get the offset of the specified line number.

  :Parameters:
    - `buffer`: The buffer.
    - `line_number`: The line number.

  :Return:
    The offset of the requested line, or -1 if the line number is not valid.
  """

  offset = 0
  for i in range(1, line_number):
    offset = string.find(buffer, '\n', offset) + 1
    if offset == 0: return -1
  return offset

class SBlock:
  """Class representing a skeleton block.
  """

  __rx_conditional = re.compile(
      r'^\s*#\s*(IF|ELIF|ELSE|ENDIF)[ \t]*(.*)$', re.MULTILINE)
  __rx_comment_start = re.compile(
      r'\s*/(/|\*).*$', re.MULTILINE)
  __rx_vref_start = re.compile(r'\$(\(|{|\$)')

  def __init__(self, name, filename, line_number, code):
    """Constructor.

    :Parameters:
      - `name`: The name of the skeleton block.
      - `line_number`: The (1-based) line number of the code block.
      - `code`: The code block.
    """

    self.name = name
    self.filename = filename
    self.line_number = line_number
    self.code = code

  def expand(self, context = None):
    """Expand the code block in the specified context.

    :Parameters:
      - `context`: The expansion context.  This context must be a mapping
	suitable as a 'globals' argument of the Python built-in 'eval()'
	function.

    :Return:
      The expanded code block.
    """

    # Pass 1: Process all conditionals.
    condition_stack = [ ]
    code_offset = 0
    code = self.code
    output = ''
    while True:
      match = self.__rx_conditional.search(code, code_offset)
      if match is None: break
      type = match.group(1)
      condition = match.group(2)
      line_number = self.line_number + string.count(
	  code, '\n', 0, match.start())
      if type in ('ELSE', 'ENDIF'):
	if (len(condition) > 0
	    and not self.__rx_comment_start.match(condition)):
	  warning(loc(self.filename, line_number)
	      + 'Garbage at end of #' + type + ' directive (ignored)')
	condition = None

      # Compute the current condition value from the condition stack.
      condition_value = True
      for x_type, x_value, x_line in condition_stack:
	if x_value != True:
	  condition_value = False
	  break
      if condition_value:
	output += code[code_offset:match.start()]

      # Check if the condition evaluation should be skipped.  This is the
      # case, if the result of the evaluation is not relevant for the
      # generated output (short-circuit semantics).
      skip_eval = False
      if type == 'IF' and not condition_value:
	skip_eval = True
      if (type == 'ELIF'
	  and len(condition_stack) > 0 # Misplaced #ELIF is checked below.
	  and condition_stack[-1][1] != False):
	skip_eval = True
      code_offset = match.end()
      if type in ('IF', 'ELIF') and len(condition) == 0:
	error(loc(self.filename, line_number)
	    + 'Missing condition in #' + type + ' directive')
	sys.exit(1)
      if condition is not None and not skip_eval:
	try:
	  condition = eval(condition, {}, context)
	except SyntaxError, e:
	  error(loc(self.filename, line_number)
	      + 'Syntax error in #' + type + ' directive: ' + str(e))
	  sys.exit(1)
	except Exception, e:
	  error(loc(self.filename, line_number)
	      + 'Evaluation error in #' + type + ' directive: ' + str(e))
	  sys.exit(1)
	condition = bool(condition)
      else:
	condition = None

      if type == 'IF':
	condition_stack.append(('IF', condition, line_number))
      else:
	if len(condition_stack) == 0:
	  error(loc(self.filename, line_number) + 'Unbalanced #' + type)
	  sys.exit(1)
	if type == 'ELIF':
	  if condition_stack[-1][0] not in ('IF', 'ELIF'):
	    error(loc(self.filename, line_number) + 'Misplaced #ELIF')
	    sys.exit(1)
	  if condition_stack[-1][1] == False:
	    condition_stack[-1] = 'ELIF', condition, line_number
	  else:
	    condition_stack[-1] = 'ELIF', None, line_number
	elif type == 'ELSE':
	  if condition_stack[-1][0] not in ('IF', 'ELIF'):
	    error(loc(self.filename, line_number) + 'Misplaced #ELSE')
	    sys.exit(1)
	  if condition_stack[-1][1] == False:
	    condition_stack[-1] = 'ELSE', True, line_number
	  else:
	    condition_stack[-1] = 'ELSE', None, line_number
	else:
	  assert type == 'ENDIF'
	  condition_stack.pop()
    if len(condition_stack) > 0:
      error(loc(self.filename, line_number) + 'Unexpected EOF')
      for x_type, x_value, x_line in condition_stack:
	warning(
	    loc(self.filename, x_line) + 'Unterminated #' + x_type,
	    level = -1)
    output += code[code_offset:]

    # Pass 2: Expand all variable references.
    code_offset = 0
    code = output
    output = ''
    while True:
      match = self.__rx_vref_start.search(code, code_offset)
      if match is None: break
      output += code[code_offset:match.start()]
      vref_start = match.start()
      line_number = self.line_number + string.count(
	  code, '\n', 0, vref_start)
      delim = match.group(1)
      expr_start = match.end(1)
      if delim == '$':
	output += '$'
	code_offset = match.end(1)
	continue
      paren_stack = [ ]
      index = -1
      paren_error = False
      for c in code[expr_start:]:
	index += 1
	if c in ('(', '{'):
	  paren_stack.append(c)
	elif c == ')':
	  if len(paren_stack) == 0:
	    if delim != '(': paren_error = True
	    break
	  else:
	    if paren_stack.pop() != '(':
	      paren_error = True
	      break
	elif c == '}':
	  if len(paren_stack) == 0:
	    if delim != '{': paren_error = True
	    break
	  else:
	    if paren_stack.pop() != '{':
	      paren_error = True
	      break
      if paren_error:
	error(loc(self.filename, line_number)
	    + 'Syntax error in expression')
	sys.exit(1)
      expr_end = expr_start + index
      expression = code[expr_start:expr_end]
      code_offset = expr_end + 1
      try:
        value = eval(expression, {}, context)
      except SyntaxError, e:
	error(loc(self.filename, line_number)
	    + 'Syntax error in expression: ' + str(e))
	sys.exit(1)
      except Exception, e:
	error(loc(self.filename, line_number)
	    + 'Evaluation error in expression: ' + str(e))
	sys.exit(1)
      if value is not None:
	value = str(value)
      else:
	value = ''

      # If the vref is the only code on the line and if the expanded value
      # spans multple lines, then the expanded value will be indent-fixed to
      # the indentation level of the vref.
      if value.find('\n') != -1:
	line_start = None
	line_end = None
	offset = vref_start - 1
	while offset >= 0:
	  c = code[offset]
	  if c == '\n':
	    line_start = offset + 1
	    break
	  if not c.isspace():
	    break
	  offset -= 1
	if offset == -1:
	  line_start = 0
	offset = code_offset
	while offset < len(code):
	  c = code[offset]
	  if c == '\n':
	    line_end = offset
	    break
	  if not c.isspace():
	    break
	  offset += 1
	if line_start is not None and line_end is not None:
	  indent_prefix = code[line_start:vref_start]
	  indent_prefix, indent_level = parse.expand(
	      indent_prefix + 'X', Options.tab_size)
	  indent_prefix = indent_prefix[:-1]
	  value_lines, value_indent = parse.expand(
	      value, Options.tab_size, return_lines = True)
	  value_lines = parse.shift_indent(
	      value_lines, indent_level - value_indent)
	  value = string.join(value_lines, '\n')

      output += value
    output += code[code_offset:]

    return output

class Skel:
  """Class representing a skeleton.

  A skeleton is a mapping from block names to code fragments.  The code
  fragments may contain embedded Python code in the form of variable
  references and conditional code blocks.

  References to Python expressions are of the form ${expr} or $(expr) (both
  forms are equivalent).

  Conditions are of the form

    #IF expr
    #ELIF expr
    #ELSE
    #ENDIF

  Both the #ELIF part(s) and the #ELSE part are optional.
  """

  __rx_cplusplus_comment = re.compile(
      r'//.*$', re.MULTILINE)
  __rx_c_comment = re.compile(
      r'/\*[^*]*\*/')
  __rx_block_start = re.compile(
      r'(^|\n)\s*([A-Za-z_.-][A-Za-z0-9_.-]*)\s*\n(%{)')
  __rx_block_end = re.compile(
      r'^%}\s*$', re.MULTILINE)
  __rx_unmatched = re.compile(
      r'^.*[^\s\n].*$', re.MULTILINE)

  def __init__(self, skel_filename):
    """Constructor.

    :Parameters:
    - `skel_filename`: The filename of the skeleton file.
    """

    self.map = { }
    self.__parse_file(skel_filename)

  def __parse_file(self, skel_filename):
    try:
      f = file(skel_filename, 'r')
      buffer = f.read()
      f.close()
    except IOError, e:
      error('Error reading skeleton file: ' + str(e))
      sys.exit(1)

    self.__parse(buffer, skel_filename)

  def __parse(self, buffer, filename):
    """Parse the skeleton from the specified buffer.
    """

    self.skel_filename = filename

    # We'll create a comment-stripped copy of the buffer for locating the
    # named blocks.
    buffer_stripped = self.__strip_comments(buffer)
    block_list = [ ]
    block_start = 0
    block_end = 0
    while True:
      match = Skel.__rx_block_start.search(buffer_stripped, block_end)
      if match is None: break
      block_start = match.start()
      line_number = 1 + string.count(buffer_stripped, '\n', 0, block_start)
      match_end = self.__rx_block_end.search(buffer_stripped, match.end())
      if match_end is None:
	error(loc(self.skel_filename, line_number)
	    + 'Unterminated skeleton block')
	sys.exit(1)
      name = match.group(2)
      if name in self.map:
	error(loc(self.skel_filename, line_number)
	    + 'Redefinition of skeleton block ' + repr(name))
	sys.exit(1)
      block_startline = 1 + string.count(
	  buffer_stripped, '\n', 0, match.start(3))
      block_endline = block_startline + string.count(
	  buffer_stripped, '\n', match.start(3), match_end.start())
      block_start_unstripped = line_offset(buffer, block_startline) + 2
      block_end_unstripped = line_offset(buffer, block_endline)
      block_code = buffer[block_start_unstripped:block_end_unstripped]
      block = SBlock(name, self.skel_filename, block_startline, block_code)
      self.map[name] = block
      block_end = match_end.end()
      block_list.append((block_start, block_end))

    # Strip all matched blocks and warn about any unmatched input.
    block_list.reverse()
    for block_start, block_end in block_list:
      num_lines = string.count(buffer_stripped, '\n', block_start, block_end)
      repl = ''
      for i in range(0, num_lines): repl += '\n'
      buffer_stripped = (
	  buffer_stripped[:block_start] + repl + buffer_stripped[block_end:])
    for match in self.__rx_unmatched.finditer(buffer_stripped):
      line_number = 1 + string.count(
	  buffer_stripped, '\n', 0, match.start())
      warning(loc(self.skel_filename, line_number)
	  + 'Unrecognized input in skeleton file: ' + repr(match.group(0)))

  def __strip_comments(self, buffer):
    """Strip all comments.

    C and C++ style comments are replaced by the appropriate number of blank
    lines.
    
    :Parameters:
      - `buffer`: The input buffer.

    :Return:
      The buffer with all comments stripped.
    """

    match_list = [ ]
    for match in self.__rx_c_comment.finditer(buffer):
      match_list.append(match)
    match_list.reverse()
    for match in match_list:
      num_lines = match.group(0).count('\n')
      repl = ''
      for i in range(0, num_lines): repl += '\n'
      if repl == '': repl = ' '
      buffer = buffer[:match.start()] + repl + buffer[match.end():]
    match_list = [ ]
    for match in self.__rx_cplusplus_comment.finditer(buffer):
      match_list.append(match)
    match_list.reverse()
    for match in match_list:
      buffer = buffer[:match.start()] + ' ' + buffer[match.end():]
    return buffer

  def expand(self, name, context = None):
    """Expand a named skeleton block within the specified context.

    :Parameters:
      - `name`: The name of the skeleton block.
      - `context`: The expansion context.  This context must be a mapping
	suitable as a 'globals' argument of the Python built-in 'eval()'
	function.  None is equivalent to an empty context.

    :Return:
      The expanded code block.  If the specified block name is not defined,
      then the method will raise a KeyError.

    :Exceptions:
      - `KeyError`: The specified block name is not defined.
    """

    return self.map[name].expand(context)

