#include "stdafx.h"
#include "..\Headers.h"

#define COS_EPSILON		0.000001f
#define HALFPI			(float)1.570796326794895

#pragma warning( disable : 4244)

//  ______________________________________________________________________________________
CryPoint3 CryPoint3::operator*(CryMatrix &m)
{
	CryPoint3 res;
	res.x= x*m.data[0] + y*m.data[4] + z*m.data[ 8] + m.data[12];
	res.y= x*m.data[1] + y*m.data[5] + z*m.data[ 9] + m.data[13];
	res.z= x*m.data[2] + y*m.data[6] + z*m.data[10] + m.data[14];
	return res;
};

//  ______________________________________________________________________________________
CryPoint3 CryPoint3::operator+(CryPoint3 b)
{
	CryPoint3 c;
	c.x=x+b.x;
	c.y=y+b.y;
	c.z=z+b.z;
	return c;
}

//  ______________________________________________________________________________________
CryPoint3 CryPoint3::operator-(CryPoint3 b)
{
	CryPoint3 c;
	c.x=x-b.x;
	c.y=y-b.y;
	c.z=z-b.z;
	return c;
}

//  ______________________________________________________________________________________
CryPoint3 CryPoint3::operator+=(CryPoint3 b)
{
	x+=b.x;
	y+=b.y;
	z+=b.z;
	return *this;
}

//  ______________________________________________________________________________________
CryPoint3 CryPoint3::operator*(float f)
{
	CryPoint3 c;
	c.x=f*x;
	c.y=f*y;
	c.z=f*z;
	return c;
}

//  ______________________________________________________________________________________
CryPoint3 CryPoint3::operator^(CryPoint3 p) //cross product
{
	CryPoint3 res;
	res.x= y*p.z - z*p.y;
	res.y= z*p.x - x*p.z;
	res.z= x*p.y - y*p.x;

	return res;
}
//  ______________________________________________________________________________________
float CryPoint3::operator*(CryPoint3 p) //dot product
{
	return x*p.x+y*p.y+z*p.z;
}

// Init ______________________________________________________________________________________
void CryMatrix::Init()
{
	memset(data,0,sizeof(data));
	data[0]=data[5]=data[10]=data[15]=1.0f;
}

// Build ______________________________________________________________________________________
void CryMatrix::Build(const CryQuat &q, const CryPoint3 &p)
{
	float qx2=2*q.x*q.x;
	float qy2=2*q.y*q.y;
	float qz2=2*q.z*q.z;

	float qxy=2*q.x*q.y;
	float qxz=2*q.x*q.z;
	float qyz=2*q.y*q.z;

	float qwx=2*q.w*q.x;
	float qwy=2*q.w*q.y;
	float qwz=2*q.w*q.z;

	data[ 0]= 1.0f - (qy2+qz2);		data[ 1]= qxy - qwz;			data[ 2]= qxz + qwy;		data[ 3] = 0.0f;
	data[ 4]= qxy + qwz;			data[ 5]= 1.0f - (qx2+qz2);		data[ 6]= qyz-qwx;			data[ 7] = 0.0f;
	data[ 8]= qxz-qwy;				data[ 9]= qyz+qwx;				data[10]= 1.0f -(qx2+qy2);	data[11] = 0.0f;
	data[12]= p.x;					data[13]= p.y;					data[14]= p.z;				data[15] = 1.0f;
}	

// Build ______________________________________________________________________________________
void CryMatrix::Build(const CryQuat &q, const CryPoint3 &p, const CryPoint3 &s)
{
	Build(q,p);
	data[ 0]*=s.x;	data[ 1]*=s.y;	data[ 2]*=s.z;
	data[ 4]*=s.x;	data[ 5]*=s.y;	data[ 6]*=s.z;
	data[ 8]*=s.x;	data[ 9]*=s.y;	data[10]*=s.z;
	//data[12]*=s.x;	data[13]*=s.y;	data[14]*=s.z; // applied before translate: so do not scale translation.
}

// Build ______________________________________________________________________________________
void CryMatrix::Multiply(const CryMatrix &m1, const CryMatrix &m2)
{
	data[0] = m1.data[0]*m2.data[0] + m1.data[4]*m2.data[1] + m1.data[ 8]*m2.data[2];
	data[1] = m1.data[1]*m2.data[0] + m1.data[5]*m2.data[1] + m1.data[ 9]*m2.data[2];
	data[2] = m1.data[2]*m2.data[0] + m1.data[6]*m2.data[1] + m1.data[10]*m2.data[2];

	data[4] = m1.data[0]*m2.data[4] + m1.data[4]*m2.data[5] + m1.data[ 8]*m2.data[6];
	data[5] = m1.data[1]*m2.data[4] + m1.data[5]*m2.data[5] + m1.data[ 9]*m2.data[6];
	data[6] = m1.data[2]*m2.data[4] + m1.data[6]*m2.data[5] + m1.data[10]*m2.data[6];

	data[8] = m1.data[0]*m2.data[8] + m1.data[4]*m2.data[9] + m1.data[ 8]*m2.data[10];
	data[9] = m1.data[1]*m2.data[8] + m1.data[5]*m2.data[9] + m1.data[ 9]*m2.data[10];
	data[10]= m1.data[2]*m2.data[8] + m1.data[6]*m2.data[9] + m1.data[10]*m2.data[10];

	data[12]= m1.data[0]*m2.data[12]+ m1.data[4]*m2.data[13]+ m1.data[ 8]*m2.data[14]+ m1.data[12];
	data[13]= m1.data[1]*m2.data[12]+ m1.data[5]*m2.data[13]+ m1.data[ 9]*m2.data[14]+ m1.data[13];
	data[14]= m1.data[2]*m2.data[12]+ m1.data[6]*m2.data[13]+ m1.data[10]*m2.data[14]+ m1.data[14];
	
	data[3]	= data[7] = data[11] = 0.0f;
	data[15] = 1.0f;
};

// Scale  ______________________________________________________________________________________
void CryMatrix::Scale(float k)
{
	data[ 0]*=k;
	data[ 5]*=k;
	data[10]*=k;
}

// Translate  ______________________________________________________________________________________
void CryMatrix::Translate(const CryPoint3 &off)
{
	data[12]+=off.x;
	data[13]+=off.y;
	data[14]+=off.z;
}

void CryMatrix::RotateX(float aa)
{
	float a=aa/90*HALFPI;

	CryMatrix r,tmp;
	r.Init();

	float ca=cos(a);
	float sa=sin(a);

	r.data[5]= ca;		r.data[ 6]=sa;
	r.data[9]=-sa;		r.data[10]=ca;

	memcpy(tmp.data,data,sizeof(data));
	Multiply(tmp,r);
}

void CryMatrix::RotateY(float aa)
{
	float a=aa/90*HALFPI;
	CryMatrix r,tmp;
	r.Init();
	float ca=cos(a);
	float sa=sin(a);
	r.data[0]= ca;		r.data[ 2]=-sa;
	r.data[8]= sa;		r.data[10]= ca;

	memcpy(tmp.data,data,sizeof(data));
	Multiply(tmp,r);
}

void CryMatrix::RotateZ(float aa)
{
	float a=aa/90*HALFPI;
	CryMatrix r,tmp;
	r.Init();
	float ca=cos(a);
	float sa=sin(a);

	r.data[0]= ca;		r.data[1]=sa;
	r.data[4]=-sa;		r.data[5]=ca;
	memcpy(tmp.data,data,sizeof(data));
	Multiply(tmp,r);
}


// Slerp ______________________________________________________________________________________
void CryQuat::Slerp(const CryQuat& p, const CryQuat& q, float t)
{
	CryQuat to1;
	double omega, cosom, sinom, scale0, scale1;

	// calc cosine
	cosom = p.x * q.x + p.y * q.y + p.z * q.z + p.w * q.w;

	// adjust signs (if necessary)
	if(cosom < 0.0)
	{
		cosom = -cosom;
		to1.x = - q.x;
		to1.y = - q.y;
		to1.z = - q.z;
		to1.w = - q.w;
	}
	else
	{
		to1.x = q.x;
		to1.y = q.y;
		to1.z = q.z;
		to1.w = q.w;
	}

	// calculate coefficients
	if((1.0 - cosom) > COS_EPSILON)
	{
		// standard case (slerp)
		omega = acos(cosom);
		sinom = sin(omega);
		scale0 = sin((1.0 - t) * omega) / sinom;
		scale1 = sin(t * omega) / sinom;
	}
	else
	{        
		// "from" and "to" quaternions are very close 
		//  ... so we can do a linear interpolation
		scale0 = 1.0 - t;
		scale1 = t;
	}

	// calculate final values
	x = scale0 * p.x + scale1 * to1.x;
	y = scale0 * p.y + scale1 * to1.y;
	z = scale0 * p.z + scale1 * to1.z;
	w = scale0 * p.w + scale1 * to1.w;
}