// Geom.cpp: implementation of the Geom class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "CryViewer.h"
#include "Geom.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

#pragma warning( disable : 4244)

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Geom::Geom() : BaseObj()
{
	memset(&chunk,0,sizeof(chunk));
	vertices	=NULL;
	faces		=NULL;
	links		=NULL;
	uvs			=NULL;
	propstr		=NULL;
	bones		=NULL;
	texFaces	=NULL;
	nLinks		=NULL;
	links		=NULL;
	vcols		=NULL;
	BoneNames	=NULL;
}

Geom::~Geom()
{
	if(links)		
	{
		for(int i=0;i<chunk.nVerts;i++) if(links[i]) free(links[i]);
		free(links);
	}

	if(bones)		free(bones);
	if(faces)		free(faces);
	if(vertices)	free(vertices);
	if(uvs)			free(uvs);
	if(propstr)		free(propstr);
	if(nLinks)		free(nLinks);
	if(vcols)		free(vcols);
}

bool Geom::Load(FILE *f, long pos)
{
	if(fseek(f,pos,SEEK_SET)) return true;

	int res=fread(&chunk,sizeof(chunk),1,f);
	if(res!=1) return true;

	hdr=chunk.chdr;

	if(hdr.ChunkType != ChunkType_Mesh || hdr.ChunkVersion != MESH_CHUNK_DESC_VERSION)
	{
		memset(&chunk,0,sizeof(chunk));
		return true;
	}

	//read verts
	vertices=(CryVertex*)calloc(sizeof(CryVertex),chunk.nVerts);
	assert(vertices);
	res=fread(vertices,sizeof(CryVertex),chunk.nVerts,f);
	if(res!=chunk.nVerts) return true;

	//read faces
	faces=(CryFace*)calloc(sizeof(CryFace),chunk.nFaces);
	assert(faces);
	res=fread(faces,sizeof(CryFace),chunk.nFaces,f);
	if(res!=chunk.nFaces) return true;

	//read tverts
	if(chunk.nTVerts)
	{
		uvs=(CryUV*)calloc(sizeof(CryUV),chunk.nTVerts);
		assert(uvs);
		res=fread(uvs,sizeof(CryUV),chunk.nTVerts,f);
		if(res!=chunk.nTVerts) return true;

		//read Tfaces
		if(chunk.nVerts != chunk.nTVerts)
		{
			texFaces=(CryTexFace*)calloc(sizeof(CryTexFace),chunk.nFaces);
			assert(texFaces);

			res=fread(texFaces,sizeof(CryTexFace),chunk.nFaces,f);
			if(res!=chunk.nFaces) return true;
		}
	}

	//read Links
	if(chunk.HasBoneInfo)
	{
		nLinks=(int *)calloc(sizeof(int),chunk.nVerts);
		assert(nLinks);

		links=(CryLink **)calloc(sizeof(CryLink *),chunk.nVerts);
		assert(links);

		for(int i=0;i<chunk.nVerts;i++)
		{
			int res=fread(nLinks+i,sizeof(int),1,f);
			if(res!=1) return true;

			char s[256];
			sprintf(s,"\nnLinks for vertex %4d = %4d",i,nLinks[i]);
			OutputDebugString(s);

			links[i]=(CryLink *)calloc(sizeof(CryLink),nLinks[i]);
			assert(links[i]);

			res=fread(links[i],sizeof(CryLink),nLinks[i],f);
			if(res!=nLinks[i]) return true;

			for(int j=0;j<nLinks[i];j++)
			{
				sprintf(s,"(Blend=%f) ",links[i][j].Blending);
				OutputDebugString(s);
			}
		}
	}

	//read Vertex Colors
	if(chunk.HasVertexCol)
	{
		vcols=(CryIRGB *)calloc(chunk.nVerts,sizeof(CryIRGB));
		assert(vcols);

		int res=fread(vcols, sizeof(CryIRGB), chunk.nVerts ,f);
		if(res!=chunk.nVerts) return true;		
	}

	return false;
}

RECT Geom::Paint(HDC hdc, DispOptions &opt)
{
	CryMatrix &tm=(bones && nLinks && links) ? opt.vtm : opt.tm;

	if(opt.ShowFaces)
	{
		for(int i=0;i<chunk.nFaces;i++)
		{
			CryFace *f=faces+i;
			CryVertex a,b,c;
			a=vertices[f->v0];
			b=vertices[f->v1];
			c=vertices[f->v2];

			CryPoint3 pa=a.p*tm;
			CryPoint3 pb=b.p*tm;
			CryPoint3 pc=c.p*tm;

			MoveToEx(hdc,pa.x, pa.y, NULL);
			LineTo(hdc, pb.x, pb.y);
			LineTo(hdc, pc.x, pc.y);
			LineTo(hdc, pa.x, pa.y);
		}
	}

	CRect rec;
	rec.SetRectEmpty();
	rec.top=1000000;
	rec.bottom = -1000000;
	rec.left = 1000000;
	rec.right=-1000000;

	char s[16];

	for(int i=0;i<chunk.nVerts;i++)
	{
		CryPoint3 pp=vertices[i].p*tm;
		CryPoint3 pn=vertices[i].n*tm;
		pn.x -= tm.data[12];
		pn.y -= tm.data[13];
		pn.z -= tm.data[14];

		if(opt.ShowVerts)
		{
			//point
			SetPixel(hdc,pp.x, pp.y, vcols ? RGB(vcols[i].r, vcols[i].g, vcols[i].b) : RGB(255,255,0));

			//normal
			if(opt.ShowNormals)
			{
				HGDIOBJ  old_pen=SelectObject(hdc,GetStockObject(WHITE_PEN));
				MoveToEx(hdc,pp.x, pp.y, NULL);
				LineTo(hdc, pp.x+opt.NormalSize*pn.x, pp.y+opt.NormalSize*pn.y);
				SelectObject(hdc,old_pen);
			}

			//vertex #
			if(opt.ShowVertNo)
			{
				itoa(i,s,10);
				TextOut(hdc,pp.x,pp.y, s, strlen(s));
			}

		}

		rec.top		=min(rec.top, pp.y);
		rec.bottom	=max(rec.bottom, pp.y);
		rec.left	=min(rec.left,pp.x);
		rec.right	=max(rec.right,pp.x);
	}

	return rec;

}

void Geom::Bind(BaseObj **all_objects, int n_obj)
{
	for(int i=0;i<n_obj;i++)
	{
		BaseObj *o=all_objects[i];
		if(!o) continue;

		if(o->hdr.ChunkType==ChunkType_BoneNameList)
		{
			BoneNames=(BoneNameList*)o;
		}

		if(o->hdr.ChunkType==ChunkType_BoneAnim)
		{
			bones=(Bone **)realloc(bones, sizeof(Bone *)*BoneNames->nBones);
			assert(bones);
			ConstructBoneList((Bone*)o);
		}
	}
}

//deform according to bones' cache matrices
void Geom::Deform()
{
	if(!bones) return;
	if(!nLinks) return;
	if(!links) return;

	for(int i=0;i<chunk.nVerts;i++)
	{
		CryPoint3 p;
		p.x=p.y=p.z=0.0f;

		for(int j=0;j<nLinks[i];j++)
		{
			CryLink *link	= links[i]+j;
			Bone	*bone	= bones[link->BoneID];	

			p+=(link->offset * bone->cache_mat) * link->Blending;
		}

		vertices[i].p=p;
	}
}

void Geom::ConstructBoneList(Bone *rootbone)
{
	if(!rootbone)			return;
	if(!BoneNames)			return;
	if(!BoneNames->nBones)	return;

	for(int i=0;i<BoneNames->nBones;i++)
	{
		NAME_ENTITY *gent=BoneNames->names+i;

		Bone *bone=rootbone->FindBone(gent->name);
		if(bone || !bones[i]) bones[i]=bone;
	}
}

