#include "StdAfx.h"
#include "chunkfilewriter.h"


//////////////////////////////////////////////////////////////////////////
// Opens the given file and prepares it for writing chunk data
CChunkFileWriter::CChunkFileWriter(const char* szFileName, int nFileType, int nVersion)
{
	m_fStream = fopen (szFileName, "wb");
	if (!m_fStream)
		throw Error("Cannot open file stream \"%s\", error %d", szFileName, errno);

	strcpy (m_FileHeader.Signature, FILE_SIGNATURE);
	m_FileHeader.FileType = nFileType;
	m_FileHeader.Version = nVersion;

	writeFileHeader ();
}

CChunkFileWriter::~CChunkFileWriter(void)
{
	// write the offset to the chunk table to the file header
	writeFileHeader();

	write ((int)m_arrChunks.size());

	// write the chunk table
	writeArray (&m_arrChunks[0], m_arrChunks.size());

	fclose (m_fStream);
}


//////////////////////////////////////////////////////////////////////////
// returns the current position in the file, in bytes, from the beginning of the file
int CChunkFileWriter::getFilePos ()
{
	if (!m_fStream)
		throw Error ("getFilePos: stream not opened");
	long nFilePos = ftell (m_fStream);
	if (nFilePos == -1l)
		throw Error ("Cannot retrieve stream position, error %d", errno);
	return nFilePos;
}


//////////////////////////////////////////////////////////////////////////
// sets the file position
void CChunkFileWriter::setFilePos (int nNewPos)
{
	int nError = fseek (m_fStream, nNewPos, SEEK_SET);
	if (nError)
	{
		int nLibErr = errno;
		throw Error ("Cannot seek to position %d within the file, error #%d: %s", nNewPos, nLibErr, strerror(nLibErr));
	}
}


//////////////////////////////////////////////////////////////////////////
// constructs an error (exception) object out of a printf-like string
CChunkFileWriter::Error::Error(const char* szFormat, ...)
{
  static char szMessage[0x800];
	va_list args;
	va_start(args, szFormat);
	_vsnprintf (szMessage, sizeof(szMessage), szFormat, args);
	va_end(args);
	m_sMessage = szMessage;
}



//////////////////////////////////////////////////////////////////////////
// returns the error reason, in C string format
const char* CChunkFileWriter::Error::c_str() const
{
	return m_sMessage.c_str();
}


//////////////////////////////////////////////////////////////////////////
// writes the file header at the beginning of the file, given the current position
// of the file and then returns to the current position
void CChunkFileWriter::writeFileHeader ()
{
	// memorize the original file position
	int nFilePosOrig = getFilePos ();

	m_FileHeader.ChunkTableOffset = nFilePosOrig;

	setFilePos (0);
	write (m_FileHeader);

	// if the file wasn't empty, set the pointer to the previous position
	if (nFilePosOrig > sizeof(FileHeader))
		setFilePos (nFilePosOrig);
}


//////////////////////////////////////////////////////////////////////////
// writes the given number of bytes into the file
void CChunkFileWriter::write (const void* pData, unsigned int nDataSize)
{
#ifdef _DEBUG
		fflush (m_fStream);
#endif
	if (nDataSize > 0)
	{
		if (1 != fwrite (pData, nDataSize, 1, m_fStream))
			throw Error ("Cannot write to the stream: error %d, data at 0x%08x size 0x%x == %d", errno, pData, nDataSize, nDataSize);
	}
#ifdef _DEBUG
		fflush (m_fStream);
#endif
}

// returns the number of chunk headers present
unsigned CChunkFileWriter::numChunks ()const
{
	return m_arrChunks.size();
}

// returns the reference to the chunk by index.
// it's acceptable to modify it
CChunkFileWriter::ChunkHeader& CChunkFileWriter::getChunk(int nChunkIdx)
{
	return m_arrChunks[nChunkIdx];
}

// returns the chunk currently being written to
CChunkFileWriter::ChunkHeader& CChunkFileWriter::getChunk()
{
	return m_arrChunks.back();
}

// adds a new chunk to the sequence (thus closing the previous)
// the FileOffset in the chunk header structure is ignored
void CChunkFileWriter::addChunk (const ChunkHeader& chunk)
{
	addChunk (chunk.ChunkType, chunk.ChunkVersion, chunk.ChunkID);
}

// adds a new chunk to the sequence (thus closing the previous)
void CChunkFileWriter::addChunk (ChunkTypes	nType, int nVersion, int nID)
{
#ifdef _DEBUG
	for (unsigned i = 0; i < m_arrChunks.size(); ++i)
	{
		ChunkHeader& chunk = m_arrChunks[i];
		// if this assertion worked, it means a duplicate chunk id has been written
		assert(chunk.ChunkID != nID);
	}
#endif
	m_arrChunks.resize (m_arrChunks.size() + 1);
	m_arrChunks.back().ChunkType    = nType;
	m_arrChunks.back().ChunkVersion = nVersion;
	m_arrChunks.back().FileOffset   = getFilePos();
	m_arrChunks.back().ChunkID      = nID;
}
