#include "StdAfx.h"
#include "BezierVec3dApproximation.h"
#include "b3papproximation.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#endif

CB3PApproximation::CB3PApproximation(void)
{
}

CB3PApproximation::~CB3PApproximation(void)
{
	clear();
}

void CB3PApproximation::clear()
{
	for (PieceArray::iterator it = m_arrPieces.begin(); it != m_arrPieces.end(); ++it)
		delete *it;
	m_arrPieces.clear();
}

// number of pieces
int CB3PApproximation::numPieces ()const
{
	return m_arrPieces.size();
}

// returns one of the approximated pieces
const CBezierVec3dApproximation& CB3PApproximation::getPiece(int nPiece)
{
	assert (nPiece >= 0 && nPiece < (int)m_arrPieces.size());
	return *m_arrPieces[nPiece];
}

// returns the boundaries of the piece in the global enumeration
std::pair<int,int> CB3PApproximation::getPieceInterval (int nPiece) const
{
	assert (nPiece >= 0 && nPiece < (int)m_arrPieces.size());
	int nStart = 0;
	for (int i = 0; i < nPiece; ++i)
		nStart += m_arrPieces[i]->numSamples();
	return std::pair<int,int> (nStart, nStart + m_arrPieces[nPiece]->numSamples());
}

CB3PApproximation::CB3PApproximation(Vec3d* pData, int nSamples, int nDegree, float fMaxAveError)
{
	init (pData, nSamples, nDegree, fMaxAveError);
}

// approximate piecewise, with maximum total error
// This is a desirable error, modulus (not square)
float CB3PApproximation::init (Vec3d* pData, int nSamples, int nDegree, float fError)
{
	float fTotalError = 0; // total error
	clear();
	double fSqrErrorPerElement = fError / nSamples;
	
	// this will increase "eating" the data points
	int nDataStart = 0; 

	// the last successful sample set length, successful set is the set that was approximated
	// within the given error boundaries
	// be optimistic and first try the whole dataset
	int nSetLength = nSamples; 

	// this is the working piece, it will be swapped with the best piece
	// during successful refinement iterations
	CBezierVec3dApproximation* pPiece = new CBezierVec3dApproximation();

	while (nDataStart < nSamples)
	{
		Vec3d* pDataStart = pData + nDataStart;
		int nDataSize = nSamples - nDataStart;

		CBezierVec3dApproximation* pBestPiece = new CBezierVec3dApproximation();
		float fBestError = 0;

		if (nDataSize <= nDegree + 1)
		{
			// we can interpolate the data
			pBestPiece->init (pDataStart, nDataSize, nDegree);
			m_arrPieces.push_back(pBestPiece);
			break;
		}

		// adjust the initial set length, and it's boundaries
		if (nSetLength > nDataSize)
			nSetLength = nDataSize;
		
		// the boundaries are inclusive for the length
		int nSetLengthMin = nDegree+1, nSetLengthMax = nDataSize;
		assert (nSetLengthMin < nSetLengthMax);

		// each time, make the interval shorter and move towards more successful end
		do
		{
			pPiece->init (pDataStart, nSetLength, nDegree);
			double fError = pPiece->getError(pDataStart);

			if (fError <= fSqrErrorPerElement*nSetLength)
			{
				// the error is small enough, but we've tried and recorded it. exclude it from the interval
				nSetLengthMin = nSetLength + 1;
				tswap (pPiece, pBestPiece);
				fBestError = (float)fError;
			}
			else
			{
				// the error is too big; since we've tried the nSetLength, exclude it from inclusive boundaries
				nSetLengthMax = nSetLength - 1;
			}
			// this is the next length to try
			nSetLength = (nSetLengthMin + nSetLengthMax + 1) / 2;
		}
		// ... if there is room for trying
		while (nSetLengthMin <= nSetLengthMax);

		if (!pBestPiece->numSamples())
		{
			assert (nSetLength == nDegree+1);
			assert (nSetLength == pPiece->numSamples());
			// we didn't find such sample
			tswap (pPiece, pBestPiece);
			fBestError = fError;
		}

		m_arrPieces.push_back(pBestPiece);
		fTotalError += fBestError;
		nDataStart += pBestPiece->numSamples();
	}
	delete pPiece;
	return fTotalError;
}

// returns the length, in samples, of the nPiece'th piece
int CB3PApproximation::getPieceSize(int nPiece)const
{
	assert (nPiece >= 0 && nPiece < (int)m_arrPieces.size());
	return m_arrPieces[nPiece]->numSamples();
}
