//////////////////////////////////////////////////////////////////////////////
// Implementation of CBezierApproximation,
// the class that is used to approximate a set of sample points with a linear
// function y = a[0] + a[1]*x
//////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "GLUtils.h"
#include "BigVector.h"
#include "BigMatrix.h"
#include "BezierApproximation.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#endif

typedef std::vector<CBigMatrix*>CBigMatrixArray;
typedef std::vector<CBigMatrixArray> CBigMatrixArray2D;

static CBigMatrixArray2D* g_arrAJ = NULL;

double BezierBase (int nBase, int nDegree, double x)
{
	double b = 1;
	int i;
	// find x^nBase*(1-x)^(nDegree-nBase)
	if (nBase < nDegree-nBase)
	{
		for (i = 0; i < nBase; ++i)
			b *= (1-x)*x;
		for (/*i = nBase*/; i < nDegree-nBase; ++i)
			b *= (1-x);
	}
	else
	{
		for (i = 0; i < nDegree-nBase; ++i)
			b *= (1-x)*x;
		for (/*i = nDegree-nBase*/; i < nBase; ++i)
			b *= x;
	}
	return b;
}

// prepares the matrix for the given dimension (nSamples >= nDegree+1)
// and returns the reference to it
CBigMatrix* CBezierApproximation::getAntiJacobian (int nSamples, int nDegree)
{
	assert (nDegree >= 1);
	if (!g_arrAJ)
		g_arrAJ = new CBigMatrixArray2D;

	if (g_arrAJ->size() < (unsigned)nDegree + 1)
		g_arrAJ->resize(nDegree+1);

	CBigMatrixArray& arr1D = (*g_arrAJ)[nDegree];
	if (arr1D.size() < (unsigned)nSamples+1)
		arr1D.resize (nSamples+1, NULL);

	CBigMatrix* &pAJ = arr1D[nSamples];

	if (!pAJ)
	{
		pAJ = new CBigMatrix();

		CBigMatrix mxJ(nSamples, nDegree+1);
		double fSampleStep = 1.0 / (nSamples-1);
		for (int nSample = 0; nSample < nSamples; ++nSample)
		{
			for (int nBase = 0; nBase <= nDegree; ++nBase)
				mxJ[nSample][nBase] = BezierBase (nBase, nDegree, nSample*fSampleStep);
		}

		mxJ.ComposePseudoInverse(*pAJ, 1e-5);
	}

	return pAJ;
}

// collects all temporary precalculated matrices
void CBezierApproximation::garbageCollect()
{
	if (g_arrAJ)
	{
		for (CBigMatrixArray2D::iterator it2D = g_arrAJ->begin(); it2D != g_arrAJ->end(); ++it2D)
		{
			for (CBigMatrixArray::iterator it = it2D->begin(); it != it2D->end(); ++it)
			{
				if (*it)
					delete *it;
			}
		}
		delete g_arrAJ;
		g_arrAJ = NULL;
	}
}

CBezierApproximation::CBezierApproximation():
	m_nDegree(0),
	m_nSamples(0),
	m_fTerm(0),
	m_fStep(0)
{
}

void CBezierApproximation::init (IData* pData, int nDegree)
{
	if (nDegree != m_nDegree)
	{
		clear();
		m_fTerm = new double[nDegree+1];
		m_nDegree = nDegree;
	}

	m_nSamples = pData->numPoints();
	m_fStep    = 1.0 / (m_nSamples-1);

	int nPointCount = pData->numPoints();
	int d;
	for (d = 0; d <= nDegree; ++d)
		m_fTerm[d] = 0;
	
	if (nPointCount<=0)
	{
		// leave it as is
	}
	else
	if (nPointCount == 1)
	{
		// stationary equation for one-point
		m_fTerm[0] = pData->getPoint(0);
	}
	//else
	//if (nPointCount <= nDegree+1)
//	{
		// we can interpolate, rather than approximate, with a polynom of degree nPointCount-1
		
//	}
	else
	{
		// inverse jacobian
		CBigMatrix* pAJ = getAntiJacobian (nPointCount, nDegree);

		for (int nBase = 0; nBase <= nDegree; ++nBase)
		{
			for (int nSample = 0; nSample < nPointCount; ++nSample)
				m_fTerm[nBase] += (*pAJ)[nBase][nSample] * pData->getPoint(nSample);
		}

#ifdef _DEBUG
		if (nDegree+1 == nPointCount)
		{
			for (int nPoint = 0; nPoint < nPointCount; ++nPoint)
				assert (tabs(getValueAtFrame (nPoint) -pData->getPoint(nPoint)) < 1e-5);
		}
#endif
	}
}

//////////////////////////////////////////////////////////////////////
// constructs the approximation from the given set of points
// using least-square method
//////////////////////////////////////////////////////////////////////
CBezierApproximation::CBezierApproximation (IData* pData, int nDegree):
	m_nDegree(0),
	m_nSamples(0),
	m_fTerm(0),
	m_fStep(0)
{
	init (pData, nDegree);
}

CBezierApproximation::CBezierApproximation (const CBezierApproximation& smp):
	m_fStep (smp.m_fStep),
	m_nDegree (smp.m_nDegree),
	m_nSamples (smp.m_nSamples)
{
	m_fTerm = new double [m_nDegree+1];
	memcpy (m_fTerm, smp.m_fTerm, (m_nDegree+1) * sizeof(m_fTerm[0]));
}

CBezierApproximation::~CBezierApproximation ()
{
	delete []m_fTerm;
}

// returns the sum of deviations of the approximation from the actual data set
double CBezierApproximation::getError (IData* pData)const
{
	int nCount = pData->numPoints();
	double fError = 0;
	for (int nFrame = 0; nFrame < nCount; ++nFrame)
	{
		fError += tsqr(pData->getPoint(nFrame)-getValueAtFrame (nFrame));
	}
	return fError;
}

// returns the approximation line value at the given argument
double CBezierApproximation::getValueAtFrame(int nFrame)const
{
	return getValueAtTime(nFrame*m_fStep);
}

// gets the value, time runs 0..1
double CBezierApproximation::getValueAtTime(double x)const
{
	double fSum = 0;
	for (int nBase = 0; nBase <= m_nDegree; ++nBase)
		fSum += m_fTerm[nBase]*BezierBase(nBase, m_nDegree, x);
	return fSum;
}

void CBezierApproximation::clear()
{
	m_fStep = 0;
	delete [] m_fTerm;
	m_fTerm = NULL;
	m_nDegree = 0;
	m_nSamples = 0;
}