///////////////////////////////////////////////////////////////////////////
// This header contains declarations of the linear approximation utility
// class that is used to approximate a set of sample points with a bezier polynom
///////////////////////////////////////////////////////////////////////////

#ifndef _MIGDALSKIY_BEZIER_APPROXIMATION_UTILS_HDR_
#define _MIGDALSKIY_BEZIER_APPROXIMATION_UTILS_HDR_

class CBigMatrix;

/////////////////////////////////////////////////////////
// Approximates a sequence of samples with a bezier curve
/////////////////////////////////////////////////////////
class CBezierApproximation
{
public:
	struct IData
	{
		virtual double getPoint (int nSample) = 0;
		virtual int numPoints() = 0;
	};

	// constructs the approximation from the given set of points
	CBezierApproximation ();
	CBezierApproximation (IData* pData, int nDegree);
	CBezierApproximation (const CBezierApproximation& smp);
	~CBezierApproximation(void);

	//returns the term near the nTerm's power of x
	double getTerm (int nTerm) const {return m_fTerm[nTerm];}

	// calculates the error - since this is the least-square method,
	// the error is the sum of squares of differencies
	double getError (IData*) const;

	// gets the value, time runs 0..1
	double getValueAtTime(double x)const;

	// gets the value
	double getValueAtFrame(int nFrame)const;

	// returns the coefficient of the nTerm-th power of the parameter variable
	double operator [] (int nTerm)const {return m_fTerm[nTerm];}

	// collects all temporary precalculated matrices
	static void garbageCollect();

	// prepares the matrix for the given dimension (nSamples >= nDegree+1)
	// and returns the reference to it
	static CBigMatrix* getAntiJacobian (int nSamples, int nDegree);

	double getFrameStep ()const {return m_fStep;}
	int numSamples()const {return m_nSamples;}

	void init (IData* pData, int nDegree);
	
	void clear();
protected:
	double m_fStep; // step of one sample in the original data
	// the equation is f[x] = sum(term[i]*b(i)), i=0..nDegree, b(i)=x^i
	double*m_fTerm;
	int m_nDegree;
	int m_nSamples;
};

#endif
