/*
ScPl - A plotting library for .NET

Bitmap.PlotSurface2D.cs
Copyright (C) 2003 
Matt Howlett

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
2. Redistributions in binary form must reproduce the following text in 
   the documentation and / or other materials provided with the 
   distribution: 
   
   "This product includes software developed as part of 
   the ScPl plotting library project available from: 
   http://www.netcontrols.org/scpl/" 

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

$Id: Bitmap.PlotSurface2D.cs,v 1.15 2004/05/13 12:19:07 mhowlett Exp $

*/

using System;
using System.Drawing;
using System.Drawing.Drawing2D;

namespace scpl
{
	namespace Bitmap
	{
		/// <summary>
		/// The PlotSurface2D object that has the ability to be rendered in a bitmap.
		/// </summary>
		public class PlotSurface2D: IPlotSurface2D 
		{
			#region Constructors
			/// <summary>
			/// Bitmap.PlotSurface2D constructor.
			/// </summary>
			/// <param name="xSize">width of the bitmap.</param>
			/// <param name="ySize">height of the bitmap.</param>
			public PlotSurface2D( int xSize, int ySize )
			{
				b_ = new System.Drawing.Bitmap( xSize, ySize );
				ps_ = new scpl.PlotSurface2D();	
			}

			/// <summary>
			/// Bitmap.PlotSurface2D constructor.
			/// </summary>
			/// <param name="b">The Bitmap where the control is to be rendered.</param>
			public PlotSurface2D( System.Drawing.Bitmap b )
			{
				b_ = b;
				ps_ = new scpl.PlotSurface2D();	
			}
			#endregion

			#region Draw
			/// <summary>
			/// Renders the plot.
			/// </summary>
			/// <param name="g">The graphics surface.</param>
			/// <param name="bounds">The rectangle storing the bounds for rendering.</param>
			public void Draw( Graphics g, Rectangle bounds )
			{
				ps_.Draw( g, bounds );
			}
			#endregion
			#region Clear
			/// <summary>
			/// Clears the plot.
			/// </summary>
			public void Clear()
			{
				ps_.Clear();
			}
			#endregion
			#region Add(IDrawable p)
			/// <summary>
			/// Adds a trace plot to the plot surface.
			/// </summary>
			/// <param name="p">The trace plot to add.</param>
			public void Add( IDrawable p )
			{
				ps_.Add( p );
			}
			#endregion
			#region Add( IDrawable, XAxisPosition, YAxisPosition )
			/// <summary>
			/// Adds a trace plot to the plot surface.
			/// </summary>
			/// <param name="p">The trace plot to add.</param>
			/// <param name="xp">The position of the X axis.</param>
			/// <param name="yp">The position of the Y axis.</param>
			public void Add( IDrawable p, scpl.PlotSurface2D.XAxisPosition xp, scpl.PlotSurface2D.YAxisPosition yp )
			{
				ps_.Add( p, xp, yp );
			}
			#endregion

			#region get/set Title
			/// <summary>
			/// The plot title.
			/// </summary>
			public string Title
			{
				get 
				{
					return ps_.Title;
				}
				set 
				{
					ps_.Title = value;
				}
			}
			#endregion
			#region get/set TitleFont
			/// <summary>
			/// The plot title font.
			/// </summary>
			public Font TitleFont 
			{
				get 
				{
					return ps_.TitleFont;
				}
				set 
				{
					ps_.TitleFont = value;
				}
			}
			#endregion
			#region get/set Padding
			/// <summary>
			/// The padding space (in graphics units) for the plot.
			/// </summary>
			public int Padding
			{
				get
				{
					return ps_.Padding;
				}
				set
				{
					ps_.Padding = value;
				}
			}
			#endregion
			#region get/set XAxis1
			/// <summary>
			/// The bottom abscissa axis.
			/// </summary>
			public Axis XAxis1
			{
				get
				{
					return ps_.XAxis1;
				}
				set
				{
					ps_.XAxis1 = value;
				}
			}
			#endregion
			#region get/set YAxis1
			/// <summary>
			/// The left ordinate axis.
			/// </summary>
			public Axis YAxis1
			{
				get
				{
					return ps_.YAxis1;
				}
				set
				{
					ps_.YAxis1 = value;
				}
			}
			#endregion
			#region get/set XAxis2
			/// <summary>
			/// The top abscissa axis.
			/// </summary>
			public Axis XAxis2
			{
				get
				{
					return ps_.XAxis2;
				}
				set
				{
					ps_.XAxis2 = value;
				}
			}
			#endregion
			#region get/set YAxis2
			/// <summary>
			/// The right ordinate axis.
			/// </summary>
			public Axis YAxis2
			{
				get
				{
					return ps_.YAxis2;
				}
				set
				{
					ps_.YAxis2 = value;
				}
			}
			#endregion
			#region get/set Legend
			/// <summary>
			/// 
			/// </summary>
			public scpl.Legend Legend
			{
				get
				{
					return ps_.Legend;
				}
				set
				{
					ps_.Legend = value;
				}
			}
			#endregion
			#region get/set PlotBackColor
			/// <summary>
			/// The plot background color.
			/// </summary>
			public System.Drawing.Color PlotBackColor
			{
				get
				{
					return ps_.PlotBackColor;
				}
				set
				{
					ps_.PlotBackColor = value;
				}
			}
			#endregion
			#region get/set SmoothingMode
			public System.Drawing.Drawing2D.SmoothingMode SmoothingMode 
			{ 
				get
				{
					return ps_.SmoothingMode;
				}
				set
				{
					ps_.SmoothingMode = value;
				}
			}
			#endregion

			#region get Width
			/// <summary>
			/// The bitmap width (read only).
			/// </summary>
			public int Width 
			{
				get
				{
					return b_.Width;
				}
			}
			#endregion
			#region get Height
			/// <summary>
			/// The bitmap height (read only).
			/// </summary>
			public int Height 
			{
				get 
				{
					return b_.Height;
				}
			}
			#endregion

			#region ToBrowser
			/// <summary>
			/// Renders the bitmap to a MemoryStream.
			/// </summary>
			/// <returns>The MemoryStream object.</returns>
			public System.IO.MemoryStream ToBrowser() 
			{
				System.IO.MemoryStream stream = new System.IO.MemoryStream();
				ps_.Draw(Graphics.FromImage(this.Bitmap),new System.Drawing.Rectangle(0,0,b_.Width,b_.Height));
				this.Bitmap.Save(stream, System.Drawing.Imaging.ImageFormat.Png);
				return stream;
			}
			#endregion

			#region get/set Bitmap
			/// <summary>
			/// The bitmap where to render the control.
			/// </summary>
			public System.Drawing.Bitmap Bitmap 
			{
				get 
				{
					return b_;
				}
				set
				{
					b_ = value;
				}
			}
			#endregion

			#region set BackColor
			public Color BackColor
			{
				set
				{
					backColor_ = value;
				}
			}
			object backColor_ = null;
			#endregion

			#region Refresh
			/// <summary>
			/// Refreshes the plot.
			/// </summary>
			public void Refresh()
			{
				if (this.backColor_!=null)
				{
					Graphics g = Graphics.FromImage( b_ );
					g.FillRectangle( (new Pen( (Color)this.backColor_)).Brush,0,0,b_.Width,b_.Height );
				}
				ps_.Draw( Graphics.FromImage(b_), new System.Drawing.Rectangle(0,0,b_.Width,b_.Height) );
			}
			#endregion

			/*
			 
			// May put these back in in the future.
			
			#region DataSource
			public object DataSource
			{
				get
				{
					return ps_.DataSource;
				}
				set
				{
					ps_.DataSource = value;
				}
			}
			#endregion
			#region ValueData
			public object ValueData
			{
				get
				{
					return ps_.ValueData;
				}
				set
				{
					ps_.ValueData = value;
				}
			}
			#endregion		
			#region AbscissaData
			public object AbscissaData
			{
				get
				{
					return ps_.AbscissaData;
				}
				set
				{
					ps_.AbscissaData = value;
				}
			}
			#endregion
			*/

			private scpl.PlotSurface2D ps_;
			private System.Drawing.Bitmap b_;

			#region AddAxesConstraint
			public void AddAxesConstraint( AxesConstraint c )
			{
				ps_.AddAxesConstraint( c );
			}
			#endregion
		}
	}
}
