/*
ScPl - A plotting library for .NET

LinePlot.cs
Copyright (C) 2003
Matt Howlett

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
2. Redistributions in binary form must reproduce the following text in 
   the documentation and / or other materials provided with the 
   distribution: 
   
   "This product includes software developed as part of 
   the ScPl plotting library project available from: 
   http://www.netcontrols.org/scpl/" 

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

$Id: LinePlot.cs,v 1.28 2004/05/07 21:19:38 mhowlett Exp $

*/

using System;
using System.Drawing;
using System.Diagnostics;

namespace scpl
{
	/// <summary>
	/// Provides ability to trace line plots.
	/// </summary>
	public class LinePlot : BasePlot, ISequencePlot
	{
		#region Constructor
		public LinePlot()
		{
		}
		#endregion

		#region Draw
		/// <summary>
		/// Renders the line plot.
		/// </summary>
		/// <param name="g">The Graphics surface.</param>
		/// <param name="xAxis">The X axis where the trace plot is attached to.</param>
		/// <param name="yAxis">The Y axis where the trace plot is attached to.</param>
		public void Draw( Graphics g, PhysicalAxis xAxis, PhysicalAxis yAxis )
		{
			SequenceAdapter data_ = 
				new SequenceAdapter( this.DataSource, this.DataMember, this.ValueData, this.AbscissaData );

			int numberPoints = data_.Count;
			
			if (data_.Count == 0)
			{
				return;
			}

			Pen	p;
		
			if (this.pen_ != null)
			{
				p = (System.Drawing.Pen)this.pen_;
			}
			else
			{
				p =	new	Pen(this.color_);
			}

			// clipping is now handled assigning a clip region in the
			// graphic object before this call
			if ( numberPoints == 1 )
			{
				PointD point = data_[0];
				PointF xPos = xAxis.WorldToPhysical(point.X, false);
				PointF yPos = yAxis.WorldToPhysical(point.Y, false);
				g.DrawLine( p ,(float)(xPos.X-0.5f), (float)yPos.Y, (float)xPos.X+0.5f, (float)yPos.Y);
			}
			else
			{
				for ( int i = 1; i < numberPoints; ++i )
				{
					// if any required points aren't defined, then skip them. 
					if ( Double.IsNaN(data_[i-1].X) || Double.IsNaN(data_[i-1].Y) && 
						Double.IsNaN(data_[i].X) || Double.IsNaN(data_[i].Y) )
					{
						continue;
					}

					// else draw line.
					PointD point = data_[i-1];
					PointF x1 = xAxis.WorldToPhysical(point.X, false);
					PointF y1 = yAxis.WorldToPhysical(point.Y, false);

					point = data_[i];
					PointF x2 = xAxis.WorldToPhysical(point.X, false);
					PointF y2 = yAxis.WorldToPhysical(point.Y, false);
					
					g.DrawLine(p, (float)x1.X, (float)y1.Y, (float)x2.X, (float)y2.Y);

				}
			}

		}
		#endregion

		#region Axis suggestion
		/// <summary>
		/// Provides a hint for the X axis.
		/// </summary>
		/// <returns>The X axis.</returns>
		public Axis SuggestXAxis()
		{
			SequenceAdapter data_ = 
				new SequenceAdapter( this.DataSource, this.DataMember, this.ValueData, this.AbscissaData );

			return data_.SuggestXAxis();
		}
		/// <summary>
		/// Provides a hint for the Y axis.
		/// </summary>
		/// <returns>The Y axis.</returns>
		public Axis SuggestYAxis()
		{
			SequenceAdapter data_ = 
				new SequenceAdapter( this.DataSource, this.DataMember, this.ValueData, this.AbscissaData );

			return data_.SuggestYAxis();
		}
		#endregion

	}
}
