/* 
	definitions for SPU memory manager
	basic concept:
		- each SPU gets some PPU allocated memory
		- memory is divided into (currently 8) buckets
				available buckets are stored in a directory and the linked list just contains indices
				directory is using a free list to insert and delete entries easily
				bucket pointers with its sizes are read-only and transferred to SPU initially
		- if SPU job allocates memory, it is first checked if it is below the largest available bucket size
				if not, the usual interrupt method is taken and waited for PPU response
				if available, the SPU determines the next best bucket size and updates the linked list and 
					available buckets accordingly (using the software cache)
				linked list just contains indices (2 bytes)
		- if memory is freed, it is added to a free list and at least once a frame the PPU fills the missing buckets 
				and frees memory specified by that free list
		- the directory and bucket indices are kept together to provide traversal with the least possible cache lines
		- the directory is written back asynchronously (and therefore 128 byte aligned)
		- bucket sizes are in power of 2 for simplicity, bucket split is not performed since this can cause index
				overflow if we would add it to the previous one
		- all memory blocks are aligned according to size (max 128)
		- allocation history table saves the last 8 allocations to delete them quickly 
				keep 2x4 entries, rotate each time a new one is added and move first one of 1 into first one of second
*/

#ifndef __SPU_MEM_MANAGER_BASE_H
#define __SPU_MEM_MANAGER_BASE_H
#pragma once

#if defined(PS3)

namespace NSPU
{
	//bucket header for each bucket set
	//2 linked lists are managed: 
	//	- one for the available directory entries (map 1:1 to linked list entry)
	//	- one for the available buckets (index into directory)
	#define BUCKET_NULL (unsigned char)255
	struct SBucketHeader
	{
		static const unsigned int scBucketHeaderSize = 4;
		unsigned char listIndex;		//start index into bucket list (first available bucket), BUCKET_NULL if empty
		unsigned char listIndexEnd;	//current last index of the bucket list (last available bucket)
		unsigned char dirIndex;			//start index for directory free list, BUCKET_NULL if empty
		unsigned char dirIndexEnd;	//current end index for directory free list, BUCKET_NULL if empty (used to append entries)
		//linked list	follows here
		//directory list linked list
	} _ALIGN(128);

	//bucket directory, single linked list for the used ones, usual memory block address otherwise
	struct SBucketDir
	{
		static const unsigned int scUnusedFlag = (1 << 31);//required when freeing it
		const bool IsUnused() const {return ((address & scUnusedFlag) != 0);}//true when part of linked list
		void SetLinkIndex(const unsigned char cLinkVal){MarkAsUnsed(); address |= cLinkVal;}
		void MarkAsUnsed(){address = scUnusedFlag;}
		const unsigned char GetLinkIndex() const{assert(IsUnused()); return (address & ~scUnusedFlag);}

		unsigned int address;
	};

	//represents a bucket, data is readonly and transferred once to SPU
	struct SBucket
	{
		unsigned int size;									//size of bucket
		unsigned short available;							//number of available blocks
		uint16 numTotal;							//total number of blocks in this bucket
		SBucketHeader *pBucketHeader;	//pointer to header
	};

	//contains bucket headers and free list, 8*16+16=112 bytes
	struct SBucketInfo
	{
		static const unsigned int scBucketCount				= 8;		//numbers of different buckets, must be an even number
		static const unsigned int scBucketSizeMin			= 32;		//smallest bucket size
		static const unsigned int scBucketSizeMinLog2 = 5;		//smallest bucket size exponent (2 base)
		static const unsigned int scFreedMaxCount			= 512;	//max number of registerable memory releases

#if defined(__SPU__)
		unsigned int pad0[4];																	//padding to align it to 128 byte for max performance
#endif

		NSPU::SBucket bucketHeaders[scBucketCount];			//bucket headers containing size and location
		unsigned int freedCount;															//number of already registered freed allocations
		unsigned int *pFreedList;															//pointer to list with freed memory addresses
		unsigned int pad1;
		unsigned int pad2;
	};
}//NSPU

#endif //PS3
#endif //__SPU_MEM_MANAGER_BASE_H
