/* 
	definitions for SPU memory manager and cache management
	
	embeds the bucket allocator and serves as static interface to the Cache
	also contains a 4 element allocation history table to cope with memory allocations 
		and corresponding releases here on SPU and not later on PPU
*/

#ifndef __SPU_MEM_MANAGER_H
#define __SPU_MEM_MANAGER_H
#pragma once

#if defined(PS3)

#if defined(__SPU__)

#include "SPU.h"
#include "../PPU/SPUMemAreaMan.h"
#include "Memory.h"

namespace NSPU
{
	namespace NCache
	{
		extern vec_uint4 g_AsyncRangesDirFrom;
		extern vec_uint4 g_AsyncRangesDirTo;
	}

	class CSPUMemMan
	{
	public:
		//---------------------------------------------software cache---------------------------------------------

#if !defined(SPU_CACHE_MISS_USE_ASM)
		static void FlushCacheComplete(const int cDoSync = 0, const bool cTransProfdata = false);//flushes cache area
#endif
		//---------------------------------------------memory management--------------------------------------------

		//return local store bucket address
		const uint32 GetBucketSPUAddr() const{return (uint32)&m_BucketInfo;}

		const uint32 GetBucketTransferSize() const{return sizeof(SBucketInfo);}

		SBucketInfo& GetBucketInfo()
		{
			return m_BucketInfo;
		}

		SBucket& GetBucketHeader(const uint32 cIndex)
		{
			return m_BucketInfo.bucketHeaders[cIndex];
		}

		void Init(const uint32 cMemAreaEA)
		{
			//copy from PC to get pointers and range for the async cache mapping
			MemcpyLS
			(
				(void*)GetBucketSPUAddr(),
				cMemAreaEA,
				128, 
				g_scDMAPPUMemTag
			);
			SyncMemory(g_scDMAPPUMemTag);
			m_AsyncFrom	=	(uint32)m_BucketInfo.bucketHeaders[0].pBucketHeader;
			m_AsyncTo   = (uint32)&m_BucketInfo.pFreedList[SBucketInfo::scFreedMaxCount];
		}

		__attribute__((always_inline))
		inline void Reset()
		{
#if defined(SUPP_ASYNC)
			//resets the asynchronous range for the cache
			const uint32 cEAFrom	=	(uint32)m_AsyncFrom;
			const uint32 cEATo		= (uint32)m_AsyncTo;
			//start at next cache line boundary (unsafe to start within a cacheline)
			const uint32 cEAAlignedFrom = (cEAFrom + scSPUCacheLineSizeMask) & ~scSPUCacheLineSizeMask;
			const uint32 cEAAlignedTo		= (cEATo + scSPUCacheLineSizeMask) & ~scSPUCacheLineSizeMask;
			//rotate existing one 4 bytes and insert into slot 0
			NSPU::NCache::g_AsyncRangesDirFrom	= spu_rlqwbyte(NSPU::NCache::g_AsyncRangesDirFrom, 4);
			NSPU::NCache::g_AsyncRangesDirTo		= spu_rlqwbyte(NSPU::NCache::g_AsyncRangesDirTo, 4);
			NSPU::NCache::g_AsyncRangesDirFrom	= spu_insert(cEAAlignedFrom, NSPU::NCache::g_AsyncRangesDirFrom, 0);
			NSPU::NCache::g_AsyncRangesDirTo		= spu_insert(cEAAlignedTo, NSPU::NCache::g_AsyncRangesDirTo, 0);
#endif
			m_HistoryTableEA					  = spu_splats((uint32)0xFFFFFFFF);
			m_HistoryTableBucketIndices = spu_splats((uint32)0xFFFFFFFF);
		}

		//inserts a new entry to the allocation history table
		__attribute__((always_inline))
		inline void AddHistoryEntry(const uint32 cEA, const uint8 cBucketIndex)
		{
			m_HistoryTableEA						= spu_rlqwbyte(m_HistoryTableEA, 4);//rotate 4 bytes to the left
			//rotate corresponding bucket indices
			m_HistoryTableBucketIndices = spu_rlqwbyte(m_HistoryTableBucketIndices, 4);//rotate 4 bytes to the left
			//insert new entry
			m_HistoryTableEA						= spu_insert(cEA, m_HistoryTableEA, 0);
			m_HistoryTableBucketIndices = spu_insert(cBucketIndex, m_HistoryTableBucketIndices, 0);
		}

		const vec_uint4 GetHistoryAllocTable() const
		{
			return m_HistoryTableEA;
		}

		const vec_uint4 GetHistoryAllocBucketIndices() const
		{
			return m_HistoryTableBucketIndices;
		}

	private:
		SBucketInfo m_BucketInfo _ALIGN(128);//bucket headers
		uint32 m_AsyncFrom;		//lower asynchronous cache mapping boundary
		uint32 m_AsyncTo;			//upper asynchronous cache mapping boundary

		vec_uint4 m_HistoryTableEA;	//4 Entries of history table
		vec_uint4 m_HistoryTableBucketIndices;//bucket indices to the 4 entries
	} _ALIGN(128);
}//NSPU

#endif //__SPU__
#endif //PS3
#endif //__SPU_MEM_MANAGER_H
