#pragma once

#include <assert.h>
#include <vector>														// STL vector
#include "types.h"													// uint32


//! memory block used as bitmap
//! if you might need mipmaps use CMippedBitmap from beginning on as saves you time dealing with on type only
template <class RasterElement>
class CSimpleBitmap
{
public:

	//! default constructor
	CSimpleBitmap()
	{
		m_pData=0;
		m_dwWidth=0;
		m_dwHeight=0;
	}

	//! destructor
	~CSimpleBitmap()
	{
		FreeData();
	}

	//! free all the memory resources
	void FreeData()
	{
		delete [] m_pData;
		m_pData=0;
		m_dwWidth=0;
		m_dwHeight=0;
	}

	// used to save the whole memory
	// \return in bytes
	uint32 CalcBitmapSize() const
	{
		return m_dwWidth*m_dwHeight*sizeof(RasterElement);
	}

	//!
	//! /param inWidth 1..
	//! /param inHeight 1..
	//! /return true=operator was successful, false otherwise (low memory)
	bool Alloc( const uint32 indwWidth, const uint32 indwHeight, const RasterElement *inpInitial=0 )
	{
		assert(indwWidth>0);
		assert(indwHeight>0);

		FreeData();

		m_pData=new RasterElement[indwWidth*indwHeight];

		if(!m_pData)
			return false;

		m_dwWidth=indwWidth;
		m_dwHeight=indwHeight;

		if(inpInitial)
			Clear(*inpInitial);		// could be done better (copy constructor calls)

		return true;
	}

	//! works only within the Bitmap
	//! /param inX 0..m_dwWidth-1 or the method returns false
	//! /param inY 0..m_dwHeight-1 or the method returns false
	//! /param outValue
	//! /return true=position was in the bitmap, false otherwise
	bool Get( const uint32 inX, const uint32 inY, RasterElement &outValue ) const
	{
		if(!m_pData)
			return false;

		if(inX>=m_dwWidth || inY>=m_dwHeight)
			return false;

		outValue=m_pData[inY*m_dwWidth+inX];

		return true;
	}

		
	//! bilinear, workes only well within 0..1
	bool GetFiltered( const float infX, const float infY, RasterElement &outValue ) const
	{
		float fIX=floorf(infX),	fIY=floorf(infY);
		float fFX=infX-fIX,			fFY=infY-fIY;
		int		iXa=(int)fIX,			iYa=(int)fIY;
		int		iXb=iXa+1,			iYb=iYa+1;

		if(iXb==m_dwWidth)
			iXb=0;

		if(iYb==m_dwHeight)
			iYb=0;

		RasterElement p[4];

		if(Get(iXa,iYa,p[0]) && Get(iXb,iYa,p[1]) && Get(iXa,iYb,p[2]) && Get(iXb,iYb,p[3]))
		{
			outValue = p[0]	* ((1.0f-fFX)*(1.0f-fFY))		// left top
								+p[1] * ((     fFX)*(1.0f-fFY))		// right top
								+p[2] * ((1.0f-fFX)*(     fFY))		// left bottom
								+p[3] * ((     fFX)*(     fFY));	// right bottom

			return true;
		}

		return false;
	}


	//! works only within the Bitmap
	//! /param inX 0..m_dwWidth-1 or the method returns false
	//! /param inY 0..m_dwHeight-1 or the method returns false
	RasterElement &GetRef( const uint32 inX, const uint32 inY )
	{
		assert(m_pData);
		assert(inX<m_dwWidth && inY<m_dwHeight);

		return m_pData[inY*m_dwWidth+inX];
	}

	//! works only within the Bitmap
	//! /param inX 0..m_dwWidth-1 or the method returns false
	//! /param inY 0..m_dwHeight-1 or the method returns false
	const RasterElement &GetRef( const uint32 inX, const uint32 inY ) const
	{
		assert(m_pData);
		assert(inX<m_dwWidth && inY<m_dwHeight);

		return m_pData[inY*m_dwWidth+inX];
	}

	//! works only within the Bitmap
	//! /param inX 0..m_dwWidth-1 or the method returns false
	//! /param inY 0..m_dwHeight-1 or the method returns false
	//! /param inValue
	bool Set( const uint32 inX, const uint32 inY, const RasterElement &inValue )
	{
		if(!m_pData)
			return false;

		if(inX>=m_dwWidth || inY>=m_dwHeight)
			return false;

		assert(m_pData);

		m_pData[inY*m_dwWidth+inX]=inValue;

		return true;
	}

	uint32 GetWidth() const
	{
		return m_dwWidth;
	}

	uint32 GetHeight() const
	{
		return m_dwHeight;
	}

	//! /return could be 0 if point is outside the bitmap
	RasterElement *GetPointer( const uint32 inX=0, const uint32 inY=0 ) const
	{
		if(inX>=m_dwWidth || inY>=m_dwHeight)
			return 0;

		return &m_pData[inY*m_dwWidth+inX];
	}

	void Clear( const RasterElement &inValue )
	{
		RasterElement *pPtr=m_pData;

		for(uint32 y=0;y<m_dwHeight;y++)
			for(uint32 x=0;x<m_dwWidth;x++)
				*pPtr++=inValue;
	}


	//! all elements that are equal to the given mask and have a neighbor become the blended value of the neighbors
	//! not too fast but convenient - call several times to expand more than one pixel
	//! RasterElement need to support: operator+=(const RasterElement )    operator*( const float )
	//! RasterElement2 need to support: operator==
	template <class RasterElement2>
		void ExpandBorder1Pixel( CSimpleBitmap<RasterElement2> &rMaskBitmap, const RasterElement2 &inMaskElement, const uint32 dwElementWidth=1 )
	{
		if(!m_pData)
			return;

		assert(dwElementWidth);

		CSimpleBitmap<RasterElement> tempBimap;

		tempBimap.Alloc(m_dwWidth,m_dwHeight);

		RasterElement *pSrc=m_pData;
		RasterElement *pDest=tempBimap.GetPointer();

		uint32 dwWidthDiv = m_dwWidth/dwElementWidth;

		for(uint32 y=0;y<m_dwHeight;y++)
			for(uint32 x=0;x<dwWidthDiv;x++)
			{
				if(rMaskBitmap.GetRef(x,y)==inMaskElement)
				{
					uint32 dwI=0;

					if(x!=0							&& !(rMaskBitmap.GetRef(x-1,y)==inMaskElement))	dwI++;
					if(x!=dwWidthDiv-1	&& !(rMaskBitmap.GetRef(x+1,y)==inMaskElement))	dwI++;
					if(y!=0							&& !(rMaskBitmap.GetRef(x,y-1)==inMaskElement))	dwI++;
					if(y!=m_dwHeight-1	&& !(rMaskBitmap.GetRef(x,y+1)==inMaskElement))	dwI++;

					if(dwI)
					{
						float fInvI = 1.0f/dwI;

						for(uint32 dwE=0;dwE<dwElementWidth;++dwE)
						{
							RasterElement Sum = pSrc[0] * 0;		// set to 0

							if(x!=0							&& !(rMaskBitmap.GetRef(x-1,y)==inMaskElement))	Sum += static_cast<RasterElement>(pSrc[-(int)dwElementWidth]*fInvI);
							if(x!=dwWidthDiv-1	&& !(rMaskBitmap.GetRef(x+1,y)==inMaskElement))	Sum += static_cast<RasterElement>(pSrc[dwElementWidth]*fInvI);
							if(y!=0							&& !(rMaskBitmap.GetRef(x,y-1)==inMaskElement))	Sum += static_cast<RasterElement>(pSrc[-(int)m_dwWidth]*fInvI);
							if(y!=m_dwHeight-1	&& !(rMaskBitmap.GetRef(x,y+1)==inMaskElement))	Sum += static_cast<RasterElement>(pSrc[m_dwWidth]*fInvI);

							pSrc++;
							*pDest++ = Sum;		// set pixel based on neighbors
						}
						continue;
					}
				}

				// copy pixels
				for(uint32 dwE=0;dwE<dwElementWidth;++dwE)
					*pDest++=*pSrc++;
			}

		memcpy(m_pData,tempBimap.GetPointer(),sizeof(RasterElement)*m_dwWidth*m_dwHeight);
	}

protected: // ------------------------------------------------------

	RasterElement *						m_pData;			//!< [m_dwWidth*m_dwHeight]
	uint32										m_dwWidth;		//!< width of m_pData
	uint32										m_dwHeight;		//!< height of m_pData
};




