/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2008.
-------------------------------------------------------------------------
$Id: TextureSplitter.cpp,v 1.0 2008/01/17 15:14:13 AntonKaplanyan Exp wwwrun $
$DateTime$
Description:  Routine for creation of streaming pak
							file from file with list of resources
-------------------------------------------------------------------------
History:
- 17:1:2008   10:31 : Created by Anton Kaplanyan

*************************************************************************/

#include "StdAfx.h"
#include <IConfig.h>
#include <ICryCompressorRC.h>
#include "TextureSplitter.h"
#include "..\..\ResourceCompiler\IResCompiler.h"
#include <BitFiddling.h>
#include <ImageExtensionHelper.h>
#include <IConsole.h>
#include "DbgHelp.h"
#include "FileUtil.h"
#include "IRCLog.h"
#include <omp.h>

//#define DEBUG_DISABLE_TEXTURE_SPLITTING
//#define DEBUG_FILE_SIZE_LIMIT 140000

#ifndef COMPILE_TIME_ASSERT
	#define COMPILE_TIME_ASSERT(x) { switch(false) { case(false):case(x):; } }
#endif

#define MAKEFOURCC(ch0, ch1, ch2, ch3)                              \
				((uint32)(uint8)(ch0) | ((uint32)(uint8)(ch1) << 8) |       \
				((uint32)(uint8)(ch2) << 16) | ((uint32)(uint8)(ch3) << 24 ))

const string CTextureSplitter::s_sInputExtensions[] = { "stm", "dds" };

// global convention for naming files and directories inside pak. 
namespace TextureHelper
{
	static ETEX_Format GetTextureFormat( const CImageExtensionHelper::DDS_HEADER *ddsh )
	{
		assert(ddsh);

		if (ddsh->ddspf.dwFourCC == DDSFormats::DDSPF_DXT1.dwFourCC)
			return eTF_DXT1;
		else if (ddsh->ddspf.dwFourCC == DDSFormats::DDSPF_DXT3.dwFourCC)
			return eTF_DXT3;
		else if (ddsh->ddspf.dwFourCC == DDSFormats::DDSPF_DXT5.dwFourCC)
			return eTF_DXT5;
		else if (ddsh->ddspf.dwFourCC == DDSFormats::DDSPF_3DC.dwFourCC)
			return eTF_3DC;
		else if( ddsh->ddspf.dwFourCC == DDSFormats::DDSPF_R32F.dwFourCC)
			return eTF_R32F;
		else if( ddsh->ddspf.dwFourCC == DDSFormats::DDSPF_G16R16F.dwFourCC)
			return eTF_G16R16F;  
		else if (ddsh->ddspf.dwFlags == DDS_RGBA && ddsh->ddspf.dwRGBBitCount == 32 && ddsh->ddspf.dwABitMask == 0xff000000)
			return eTF_A8R8G8B8;
		else if (ddsh->ddspf.dwFourCC == DDSFormats::DDSPF_A16B16G16R16F.dwFourCC)
			return eTF_A16B16G16R16F;
		else if (ddsh->ddspf.dwFlags == DDS_RGBA && ddsh->ddspf.dwRGBBitCount == 16)
			return eTF_A4R4G4B4;
		else if (ddsh->ddspf.dwFlags == DDS_RGB  && ddsh->ddspf.dwRGBBitCount == 24)
			return eTF_R8G8B8;
		else if (ddsh->ddspf.dwFlags == DDS_RGB  && ddsh->ddspf.dwRGBBitCount == 32)
			return eTF_X8R8G8B8;
		else if (ddsh->ddspf.dwFlags == DDS_LUMINANCE  && ddsh->ddspf.dwRGBBitCount == 8)
			return eTF_L8;
		else if ((ddsh->ddspf.dwFlags == DDS_A || ddsh->ddspf.dwFlags == DDS_A_ONLY) && ddsh->ddspf.dwRGBBitCount == 8)
			return eTF_A8;
		else if (ddsh->ddspf.dwRGBBitCount == 8)
			return eTF_A8;
		else
		{
			assert(0);
			return eTF_Unknown;
		}
	}

	static int BytesPerBlock(const ETEX_Format eTF)
	{
		switch (eTF)
		{
			case eTF_A8R8G8B8:
			case eTF_X8R8G8B8:
			case eTF_X8L8V8U8:
				return 4;
			case eTF_A4R4G4B4:
				return 2;
			case eTF_R8G8B8:
			case eTF_L8V8U8:
				return 3;
			case eTF_V8U8:
				return 2;
			case eTF_R5G6B5:
			case eTF_R5G5B5:
				return 2;
			case eTF_A16B16G16R16:
			case eTF_A16B16G16R16F:
				return 8;
			case eTF_A32B32G32R32F:
				return 16;
			case eTF_G16R16:
			case eTF_G16R16F:
			case eTF_V16U16:
				return 4;
			case eTF_A8:
	#if defined(XENON)
			case eTF_A8_LIN:
	#endif
			case eTF_L8:
				return 1;
			case eTF_A8L8:
				return 2;
			case eTF_R32F:
				return 4;
			case eTF_DXT3:
			case eTF_DXT5:
			case eTF_3DC:
				return 16;
			case eTF_DXT1:
				return 8;
			case eTF_R16F:
			case eTF_DF16:
				return 2;
			case eTF_DF24:
				return 3;
			case eTF_D16:
				return 2;
			case eTF_D24S8:
			case eTF_D32F:
				return 4;
			case eTF_NULL:
			case eTF_Unknown:
			default:
				assert(0);
				return 0;
		}
		return 0;
	}

	static bool IsDXTCompressed(const ETEX_Format eTF)
	{
		if (eTF == eTF_DXT1 || eTF == eTF_DXT3 || eTF == eTF_DXT5 || eTF == eTF_3DC)
			return true;
		return false;
	}
	static int TextureDataSize(int nWidth, int nHeight, int nDepth, int nMips, const ETEX_Format eTF)
	{
		if (eTF == eTF_Unknown)
			return 0;

		if (nMips <= 0)
			nMips = 1;
		int nSize = 0;
		int nM = 0;
		while (nWidth || nHeight || nDepth)
		{
			if (!nWidth)
				nWidth = 1;
			if (!nHeight)
				nHeight = 1;
			if (!nDepth)
				nDepth = 1;
			nM++;

			int nSingleMipSize;
			if (IsDXTCompressed(eTF))
			{
				int blockSize = (eTF == eTF_DXT1) ? 8 : 16;
				nSingleMipSize = ((nWidth+3)/4)*((nHeight+3)/4)*nDepth*blockSize;
			}
			else
				nSingleMipSize = nWidth * nHeight * nDepth * BytesPerBlock(eTF);
			nSize += nSingleMipSize;

			nWidth >>= 1;
			nHeight >>= 1;
			nDepth >>= 1;
			if (nMips == nM)
				break;
		}

		return nSize;
	}
};

static string MakeFileName( const string& filePath, const uint32 nChunk, const uint32 nFlags )
{
	string suffix = "";

	if(nFlags & CTextureSplitter::esrfDDSIsAttachedAlpha)
		suffix = "a";

	string sFullDestFileName;
	sFullDestFileName.Format("%s.%d%s", filePath, nChunk, suffix.c_str());

	return sFullDestFileName;
}

bool CTextureSplitter::SaveFile( const string& sFileName, const void* pBuffer, const size_t nSize )
{
	// create all subdirs
	string sDir = PathHelpers::AddSeparator(PathHelpers::GetDirectory(sFileName));

	#pragma omp critical
	{
		if(m_pResourceCompiler)
			m_pResourceCompiler->EnsureDirectoriesPresent(sDir);
	}

	// create file
	FILE* file = fopen(sFileName, "wb");
	if(!file)
	{
		CCLOG->LogError("Error '%s': Failed to create file '%s'\n", strerror(errno), sFileName.c_str());
		return false;
	}

	// save content
	bool bResult = true;
	if(fwrite(pBuffer, nSize, 1, file) != 1)
	{
		bResult = false;
		CCLOG->LogError("Error '%s': Failed to write content of file '%s'\n", strerror(errno), sFileName.c_str());
	}

	fclose(file);

	return bResult;
}

CTextureSplitter::CTextureSplitter( ) : 
		m_refCount(1),
		m_currentEndian(eLittleEndian),
		m_pCC(NULL)
{
}

CTextureSplitter::~CTextureSplitter( )
{
}

bool CTextureSplitter::BuildDDSChunks(const CImageExtensionHelper::DDS_HEADER* const pDDSHeader, SStreamableResource* pResource )
{
	if (pDDSHeader->dwSize != sizeof(CImageExtensionHelper::DDS_HEADER))
	{
		CCLOG->LogError("Error: Unknown DDS file header: '%s'\n", pResource->m_sFileName.c_str());
		return false;
	}

	// get number of real MIPs
	const uint32 nNumMips = pDDSHeader->GetMipCount();

	int dataHeaderOffset = sizeof(DWORD) + sizeof(CImageExtensionHelper::DDS_HEADER);

	// if we load alpha channel we need take into account main mips offset
	if((pResource->m_nFileFlags & esrfDDSIsAttachedAlpha) != 0)
		dataHeaderOffset = sizeof(CImageExtensionHelper::DDS_HEADER);		// we don't have DWORD header for attached alpha channel

	// parse chunks
	int nEstimatedMips = (int)nNumMips;

	// header file
	for(int currentPhase = 0;nEstimatedMips > 0;++currentPhase)
	{
		SStreamableResource::SChunkDesc chunkDesc;
		chunkDesc.m_nChunkNumber = currentPhase;
		assert(chunkDesc.m_nChunkNumber < 256);

		// calculate mips to save
		uint32 nStartMip;
		uint32 nEndMip;
		if(currentPhase == 0)
		{
			nStartMip = max(0, (int)nNumMips - ehiNumLastMips);
			nEndMip = nNumMips;
		}
		else
		{
			nStartMip = max(0, ((int)nNumMips - ehiNumLastMips) - currentPhase);
			nEndMip = min(nNumMips, nStartMip + 1);
		}

		const uint32 nNumMipsToStore = nEndMip - nStartMip;

		assert(nNumMipsToStore > 0);

		// whole texture mips to store
		chunkDesc.m_nOriginalOffset = nStartMip == 0 ? 0 : TextureHelper::TextureDataSize(pDDSHeader->dwWidth, 
																																								pDDSHeader->dwHeight, pDDSHeader->dwDepth, nStartMip, 
																																								TextureHelper::GetTextureFormat(pDDSHeader));

		// size is a difference
		chunkDesc.m_nChunkSize = TextureHelper::TextureDataSize(pDDSHeader->dwWidth, 
																											pDDSHeader->dwHeight, pDDSHeader->dwDepth, nEndMip,
																											TextureHelper::GetTextureFormat(pDDSHeader))
																											- chunkDesc.m_nOriginalOffset;

		chunkDesc.m_nOriginalOffset += dataHeaderOffset;	// need for direct reading

		assert(chunkDesc.m_nOriginalOffset + chunkDesc.m_nChunkSize >= chunkDesc.m_nOriginalOffset);

		// adding header size
		if(currentPhase == 0)
			chunkDesc.m_nChunkSize += dataHeaderOffset;

		// add chunk
		pResource->m_chunks.push_back(chunkDesc);

		nEstimatedMips -= nNumMipsToStore;
		assert(nNumMips >= 0);

		// calculate file size
		pResource->m_nFileSize += chunkDesc.m_nChunkSize;
	}

	// sort all chunks in offset ascending order
	pResource->m_chunks.sort();

#ifdef DEBUG_DISABLE_TEXTURE_SPLITTING
	pResource->m_chunks.clear();
#endif

	return true;
}


void CTextureSplitter::PostLoadProcessTexture( SStreamableResource& resource, std::vector<uint8>& fileContent )
{
	CImageExtensionHelper::DDS_FILE_DESC* pFileDesc = (CImageExtensionHelper::DDS_FILE_DESC*)&fileContent[0];
	const ETEX_Format format = DDSFormats::GetFormatByDesc(pFileDesc->header.ddspf);

	const int nOldMips = pFileDesc->header.GetMipCount();
	int nFaces = 1;
	if ((pFileDesc->header.dwSurfaceFlags & DDS_SURFACE_FLAGS_CUBEMAP) != 0 && (pFileDesc->header.dwCubemapFlags & DDS_CUBEMAP_ALLFACES) != 0)
		nFaces = 6;

	// if we have already texture with proper mips, just return
	if( (pFileDesc->header.dwWidth >> nOldMips) >= 4 && (pFileDesc->header.dwHeight >> nOldMips) >= 4
		|| pFileDesc->header.GetMipCount() == 1)
		return;

	// else calculate # mips to drop and drop it
	int nNumMipsToDrop = 0;
	while(pFileDesc->header.GetMipCount() - nNumMipsToDrop > 1)
	{
		int nNewMips = pFileDesc->header.GetMipCount() - nNumMipsToDrop - 1;
		if((pFileDesc->header.dwWidth >> nNewMips) >= 4 && 
			(pFileDesc->header.dwHeight >> nNewMips) >= 4)
			break;
		nNumMipsToDrop++;
	}

	if(!nNumMipsToDrop)
		return;

	const int nTexDataSize = TextureHelper::TextureDataSize(pFileDesc->header.dwWidth, pFileDesc->header.dwHeight, 
																													max(1ul, pFileDesc->header.dwDepth), 
																													pFileDesc->header.GetMipCount(), format);
	const int nNewTexDataSize = TextureHelper::TextureDataSize(pFileDesc->header.dwWidth, pFileDesc->header.dwHeight, 
																															max(1ul, pFileDesc->header.dwDepth), 
																															pFileDesc->header.GetMipCount() - nNumMipsToDrop, format);
	const int nDiffOffset = nTexDataSize - nNewTexDataSize;
	uint8* pData = &fileContent[sizeof(CImageExtensionHelper::DDS_FILE_DESC)];
	int nEndOffset = sizeof(CImageExtensionHelper::DDS_FILE_DESC) + nTexDataSize;
	int nNewEndOffset = sizeof(CImageExtensionHelper::DDS_FILE_DESC) + nNewTexDataSize;
	for(int iSide = 1;iSide < nFaces;++iSide)
	{
		uint8* pOldSideData = pData + nTexDataSize * iSide;
		uint8* pNewSideData = pData + nNewTexDataSize * iSide;
		assert(sizeof(CImageExtensionHelper::DDS_FILE_DESC) + nTexDataSize + nDiffOffset + nNewTexDataSize <= fileContent.size());
		memmove(pNewSideData, pOldSideData, nNewTexDataSize);
		nEndOffset += nTexDataSize;
		nNewEndOffset += nNewTexDataSize;
	}

	// copy the rest of the file
	int nRest = fileContent.size() - nEndOffset;
	if(nRest)
	{
		assert(nRest > 0);
		memmove(&fileContent[nNewEndOffset], &fileContent[nEndOffset], nRest);
	}

	fileContent.resize(fileContent.size() - nDiffOffset * nFaces);

	// write to the header
	pFileDesc->header.dwMipMapCount -= nNumMipsToDrop;
}

bool CTextureSplitter::LoadTexture( SStreamableResource& resource, std::vector<uint8>& fileContent )
{
	// init file size
	resource.m_nFileSize = 0;

	// try to open file
	FILE* file = fopen(resource.m_sFileName.c_str(), "rb");
	if(file == NULL)
	{
		CCLOG->LogError("Error: Cannot open texture file: '%s'\n", resource.m_sFileName.c_str());
		return false;
	}

	// get file size
	fseek( file,0,SEEK_END );
	const uint32 nOriginalSize = ftell(file);
	fseek( file,0,SEEK_SET );

	// debug helper
#ifdef DEBUG_FILE_SIZE_LIMIT
	if(nOriginalSize > DEBUG_FILE_SIZE_LIMIT)
	{
		fclose(file);
		return false;
	}
#endif

	if(nOriginalSize < sizeof(CImageExtensionHelper::DDS_FILE_DESC))
	{
		CCLOG->LogError("Error: Cannot load texture, file too small: '%s'\n", resource.m_sFileName.c_str());
		fclose(file);
		return false;
	}

	// allocate memory for file content
	fileContent.resize(nOriginalSize);

	if(fread(&fileContent[0], nOriginalSize, 1, file) != 1)
	{
		fclose(file);
		return false;
	}

	CImageExtensionHelper::DDS_FILE_DESC* pFileDesc = (CImageExtensionHelper::DDS_FILE_DESC*)&fileContent[0];

	// check for valid CryEngine DDS file
	if (pFileDesc->dwMagic != MAKEFOURCC('D','D','S',' ') || 
		pFileDesc->header.dwSize != sizeof(pFileDesc->header) ||
		pFileDesc->header.ddspf.dwSize != sizeof(pFileDesc->header.ddspf) || 
		pFileDesc->header.dwDepth > 8192 || 
		pFileDesc->header.dwHeight > 8192 ||
		pFileDesc->header.dwWidth > 8192)
	{
		CCLOG->LogError("Error: Cannot load texture header: '%s'\n", resource.m_sFileName.c_str());
		fclose(file);
		return false;
	}

	const ETEX_Format format = DDSFormats::GetFormatByDesc(pFileDesc->header.ddspf);
	if(format == eTF_Unknown)
	{
		CCLOG->LogError("Error: Cannot load texture(unknown format): '%s'\n", resource.m_sFileName.c_str());
		fclose(file);
		return false;
	}
	int nFaces = 1;
	if ((pFileDesc->header.dwSurfaceFlags & DDS_SURFACE_FLAGS_CUBEMAP) != 0 && (pFileDesc->header.dwCubemapFlags & DDS_CUBEMAP_ALLFACES) != 0)
		nFaces = 6;
	int nTexDataSize = TextureHelper::TextureDataSize(pFileDesc->header.dwWidth, pFileDesc->header.dwHeight, max(1ul, pFileDesc->header.dwDepth), 
																											pFileDesc->header.GetMipCount(), format) * nFaces;
	if(sizeof(CImageExtensionHelper::DDS_FILE_DESC) + nTexDataSize > nOriginalSize)
	{
		CCLOG->LogError("Error: Cannot load texture, file content too small: '%s'\n", resource.m_sFileName.c_str());
		fclose(file);
		return false;
	}

	fclose(file);

	PostLoadProcessTexture(resource, fileContent);

	return true;
}

void CTextureSplitter::ParseDDSTexture( SStreamableResource& resource, std::vector<uint8>& fileContent )
{
	if(!LoadTexture(resource, fileContent))
	{
		CCLOG->LogError("Error: Cannot load texture: '%s'\n", resource.m_sFileName.c_str());
		fileContent.clear();
		return;
	}

	// process texture platform-specific conversions
	ProcessPlatformSpecificConversions(&fileContent[sizeof(DWORD)], fileContent.size() - sizeof(DWORD));

	static const int dataHeaderOffset = sizeof(DWORD) + sizeof(CImageExtensionHelper::DDS_HEADER);

	// load header
	CImageExtensionHelper::DDS_HEADER* pDDSHeader = (CImageExtensionHelper::DDS_HEADER*)(&fileContent[0] + sizeof(DWORD));

	// cubemap streaming is not supported
	if ((pDDSHeader->dwSurfaceFlags & DDS_SURFACE_FLAGS_CUBEMAP) != 0 && (pDDSHeader->dwCubemapFlags & DDS_CUBEMAP_ALLFACES) != 0)
		return;

	//COMPILE_TIME_ASSERT(sizeof(CImageExtensionHelper::CImageExtensionHelper::DDS_HEADER) == sizeof(CImageExtensionHelper::DDS_HEADER));
	const uint32 imageFlags = CImageExtensionHelper::GetImageFlags((CImageExtensionHelper::DDS_HEADER *)pDDSHeader);

	// we need to skip textures with DontStream flag
	if((imageFlags & CImageExtensionHelper::EIF_DontStream))
		return;

	// skip splitting if we have an explicit hint for that
	if(m_pCC->config->HasKey("dont_split"))
		return;

	// build main mips
	if(!BuildDDSChunks(pDDSHeader, &resource))
		return;

	// load attached alpha chanel
	if(resource.m_nFileSize < fileContent.size())
	{
		// load last portion of file
		// gets file size
		uint8* pAttachedData = &fileContent[resource.m_nFileSize];

		// try to read header of attached alpha channel
		CImageExtensionHelper::DDS_HEADER* pAlphaDDSHeader = (CImageExtensionHelper::DDS_HEADER*)CImageExtensionHelper::GetAttachedImage(&pAttachedData[0]);
		if (pAlphaDDSHeader)
		{
			// force set A8 FORMAT
			pAlphaDDSHeader->ddspf = DDSFormats::DDSPF_A8;

			// set flag
			resource.m_nFileFlags |= esrfDDSHasAdditionalAlpha;

			assert(!resource.m_sFileName.empty());
			SStreamableResource* attachedResource;
			#pragma omp critical(additional_resource)
			{
				assert(m_AttachedResources.find(resource.m_sFileName) == m_AttachedResources.end());
				attachedResource = &m_AttachedResources[resource.m_sFileName];
			}

			attachedResource->m_sFileName = resource.m_sFileName;
			attachedResource->m_nFileSize = 0;
			attachedResource->m_nFileFlags |= esrfDDSIsAttachedAlpha;

			// build alpha mips
			if(!BuildDDSChunks(pAlphaDDSHeader, attachedResource))
			{
				CCLOG->LogError("Error: Cannot load attached alpha channel of texture header: '%s'\n", resource.m_sFileName.c_str());
				return;
			}
		}
	}

	// validate file size
	//assert(fabsf(nOriginalSize - resource.m_nFileSize) <= 512);
}

void CTextureSplitter::AssembleDDSTextureChunk( const std::vector<uint8>& fileContent, SStreamableResource& resource, 
																							 const uint32 nChunkNumber, std::vector<uint8>* pOutChunk )
{
	int dataHeaderOffset = sizeof(CImageExtensionHelper::DDS_FILE_DESC);

	assert(resource.m_chunks.size() > nChunkNumber);
	assert(pOutChunk);

	// offset and size of chunk
	std::list<SStreamableResource::SChunkDesc>::iterator it = resource.m_chunks.begin();
	for(uint32 i=0;i<resource.m_chunks.size();++i)
	{
		if(it->m_nChunkNumber == nChunkNumber)
		{
			break;
		}
		++it;
	}
	SStreamableResource::SChunkDesc& currentChunk = *it;

	// allocate space for it
	pOutChunk->resize(currentChunk.m_nChunkSize);

	// check for attached resource
	SStreamableResource::SChunkDesc const* attachedChunk = NULL;
	SStreamableResource* attachedAlpha = NULL;
	if( (resource.m_nFileFlags & esrfDDSHasAdditionalAlpha) != 0)
	{
		std::map<string, SStreamableResource>::iterator it;
		#pragma omp critical
		{
			it = m_AttachedResources.find(resource.m_sFileName);
			assert(it != m_AttachedResources.end());
		}

		attachedAlpha = &it->second;
		if(attachedAlpha->m_chunks.size() > nChunkNumber)
		{
			std::list<SStreamableResource::SChunkDesc>::const_iterator it = attachedAlpha->m_chunks.begin();
			std::advance(it, nChunkNumber);
			attachedChunk = &(*it);
		}
	}

	// get DDS header for compression
	CImageExtensionHelper::DDS_HEADER* pDDSHeader;
	if(!(resource.m_nFileFlags & esrfDDSIsAttachedAlpha))
		pDDSHeader = (CImageExtensionHelper::DDS_HEADER*)(&fileContent[0] + sizeof(DWORD));
	else
	{
		pDDSHeader = CImageExtensionHelper::GetAttachedImage(&fileContent[0]);
		dataHeaderOffset = sizeof(CImageExtensionHelper::DDS_HEADER);
	}

	// header file
	if(nChunkNumber == 0)
	{
		// copy header to head of array
		if(!(resource.m_nFileFlags & esrfDDSIsAttachedAlpha))
		{
			memcpy(&(*pOutChunk)[0], &fileContent[0], dataHeaderOffset);
			pDDSHeader = (CImageExtensionHelper::DDS_HEADER*)(&(*pOutChunk)[0] + sizeof(DWORD));
		}
		else
		{
			memcpy(&(*pOutChunk)[0], pDDSHeader, dataHeaderOffset);
			pDDSHeader = (CImageExtensionHelper::DDS_HEADER*)(&(*pOutChunk)[0]);
		}
		// copy mips and to array
		memcpy(&(*pOutChunk)[dataHeaderOffset], &fileContent[currentChunk.m_nOriginalOffset], 
							currentChunk.m_nChunkSize - dataHeaderOffset);

		// compress chunk
		if(0)//m_bSupportsXenonCompression && (resource.m_nFileFlags & (esrfDDSHasAdditionalAlpha|esrfDDSIsAttachedAlpha)) == 0)
		{
			// create storage
			uint32 nStartMip = max(0, (int32)pDDSHeader->GetMipCount() - ehiNumLastMips);
			uint32 nNumMips = min(pDDSHeader->GetMipCount(), (uint32)ehiNumLastMips);
			assert(m_pCC && m_pCC->pRC);
			assert(m_pCC->pRC->GetCompressor());
			uint32 nSize = currentChunk.m_nChunkSize - dataHeaderOffset;
			std::vector<uint8> vecContent(nSize * 2 + 2048);
			if(m_pCC->pRC->GetCompressor()->CompressMips(pDDSHeader, nStartMip, 
																															nNumMips, &(*pOutChunk)[dataHeaderOffset], nSize,
																															vecContent)
																															&& vecContent.size() + dataHeaderOffset < pOutChunk->size())
			{
				pOutChunk->resize(vecContent.size() + dataHeaderOffset);
				memcpy(&((*pOutChunk)[dataHeaderOffset]), &vecContent[0], vecContent.size());
			}
		}

		// check for alpha
		//if( (resource.m_nFileFlags & esrfDDSHasAdditionalAlpha) != 0)
		//{
		//	assert(attachedChunk);
		//	const uint32 alphaHeaderOffset = sizeof(CImageExtensionHelper::DDS_HEADER);
		//	// allocate space for it
		//	const uint32 oldOffset = pOutChunk->size();
		//	pOutChunk->resize(pOutChunk->size() + attachedChunk->m_nChunkSize);

		//	const CImageExtensionHelper::DDS_HEADER* pAlphaHeader = (const CImageExtensionHelper::DDS_HEADER*)CImageExtensionHelper::GetAttachedImage(&fileContent[0] + resource.m_nFileSize);
		//	assert(pAlphaHeader);
		//	const char* AttachedFileStart = (const char*)pAlphaHeader + alphaHeaderOffset;

		//	// copy header to the end of array
		//	memcpy(&(*pOutChunk)[oldOffset], pAlphaHeader, alphaHeaderOffset);
		//	// copy mips and to array
		//	memcpy(&(*pOutChunk)[oldOffset + alphaHeaderOffset], 
		//		AttachedFileStart + attachedChunk->m_nOriginalOffset - alphaHeaderOffset, 
		//		attachedChunk->m_nChunkSize - alphaHeaderOffset);

		//	currentChunk.m_nChunkSize += attachedChunk->m_nChunkSize;

		//	// correct file size once!
		//	resource.m_nFileSize += attachedAlpha->m_nFileSize;
		//}
	}
	else	// bigger mips
	{
		// copy mips to head of array
		memcpy(&(*pOutChunk)[0], &fileContent[currentChunk.m_nOriginalOffset], currentChunk.m_nChunkSize);

		// compress chunk
		if(!(resource.m_nFileFlags & (esrfDDSHasAdditionalAlpha|esrfDDSIsAttachedAlpha)))
		{
			// create storage
			assert(pDDSHeader->GetMipCount() >= ehiNumLastMips + currentChunk.m_nChunkNumber); 
			uint32 nStartMip = pDDSHeader->GetMipCount() - ehiNumLastMips - currentChunk.m_nChunkNumber;
			uint32 nNumMips = 1;
			assert(m_pCC &&m_pCC->pRC);
			assert(m_pCC->pRC->GetCompressor());
			uint32 nSize = currentChunk.m_nChunkSize;
			std::vector<uint8> vecContent(nSize * 2 + 512);	// just in case
			vecContent.resize(nSize * 2 + 512);

			if(m_pCC->pRC->GetCompressor()->CompressMips(pDDSHeader, nStartMip, 
																																nNumMips, &(*pOutChunk)[0], nSize,
																																vecContent) && vecContent.size() < pOutChunk->size())
			{
				pOutChunk->resize(vecContent.size());
				memcpy(&(*pOutChunk)[0], &vecContent[0], vecContent.size());
			}
		}
	}

	// check for alpha
	if(attachedChunk)
	{
		std::vector<byte> vecAttachedAlpha(attachedAlpha->m_nFileSize);
		memcpy(&vecAttachedAlpha[0], &fileContent[resource.m_nFileSize], attachedAlpha->m_nFileSize);
		ProcessResource(*attachedAlpha, vecAttachedAlpha, nChunkNumber);
	}

	assert(currentChunk.m_nChunkSize > 1);
}

void CTextureSplitter::ProcessResource( SStreamableResource& resource, const std::vector<uint8>& fileContent, const uint32 nChunk )
{
	// complete relative file path
	string sFullDestFileName = PathHelpers::Join(m_sCurrentDirectory, MakeFileName(resource.m_sFileName, nChunk, resource.m_nFileFlags));

	// Compare time stamp of output file.
	if (!m_pCC->config->HasKey("refresh"))
	{
		unsigned nTimeSrc = FileUtil::GetFileUnixTimeMax( resource.m_sFileName );
		unsigned nTimeTgt = FileUtil::GetFileUnixTimeMax( sFullDestFileName );

		if (nTimeSrc < nTimeTgt)
		{
			// both Source and Filter code are older than target,
			// thus the target is up to date
			CCLOG->Log("Skipping %s: File is up to date\n", resource.m_sFileName.c_str());
			return;
		}
	}

	// get file chunk
	std::vector<uint8> vecChunkToWrite;
	AssembleDDSTextureChunk(fileContent, resource, nChunk, &vecChunkToWrite);

	assert(!vecChunkToWrite.empty());

	// adds file to pak
	if(!SaveFile(sFullDestFileName, &vecChunkToWrite[0], vecChunkToWrite.size() ) )
		CCLOG->LogError("Error: Cannot save file '%s'\n", resource.m_sFileName.c_str());
}

void CTextureSplitter::PrintStatistics()
{
	if(m_pResourceCompiler && m_pResourceCompiler->GetIRCLog())
	{
		m_pResourceCompiler->GetIRCLog()->Log("Compression statistics:\n");
		const float MCTRatio = m_pResourceCompiler->GetCompressor()->GetStatistics().GetAvgRatio<ectMCT>();
		const float PTCRatio = m_pResourceCompiler->GetCompressor()->GetStatistics().GetAvgRatio<ectPTC>();
		const float JPGRatio = m_pResourceCompiler->GetCompressor()->GetStatistics().GetAvgRatio<ectJPEG>();
		m_pResourceCompiler->GetIRCLog()->Log("    Average PTC compression ratio: %.2f\n", PTCRatio);
		m_pResourceCompiler->GetIRCLog()->Log("    Average MCT compression ratio: %.2f\n", MCTRatio);
		m_pResourceCompiler->GetIRCLog()->Log("    Average JPG compression ratio: %.2f\n", JPGRatio);
	}
}

void CTextureSplitter::AddResourceToAdditionalList( SStreamableResource& resource, const std::vector<uint8>& fileContent )
{
	string sNewFile = PathHelpers::Join(m_sCurrentDirectory, resource.m_sFileName);
	// Compare time stamp of output file.
	//#pragma omp critical(external_routines)
	if (!m_pCC->config->HasKey("refresh"))
	{
		unsigned nTimeSrc = FileUtil::GetFileUnixTimeMax( resource.m_sFileName );
		unsigned nTimeTgt = FileUtil::GetFileUnixTimeMax( sNewFile );

		if (nTimeSrc < nTimeTgt)
		{
			// both Source and Filter code are older than target,
			// thus the target is up to date
			CCLOG->Log("Skipping %s: File is up to date\n", resource.m_sFileName.c_str());
			return;
		}
	}

	const uint32 nDDSHeaderSize = sizeof(DWORD) + sizeof(CImageExtensionHelper::DDS_HEADER);

	byte const* pFileHeader = &fileContent[0];
	CImageExtensionHelper::DDS_HEADER* pDDSHeader = (CImageExtensionHelper::DDS_HEADER*)(&fileContent[0] + sizeof(DWORD));
	byte const* pData = &fileContent[0] + nDDSHeaderSize;
	const uint32 nTextureSize = TextureHelper::TextureDataSize(pDDSHeader->dwWidth, 
																											pDDSHeader->dwHeight, pDDSHeader->dwDepth, pDDSHeader->GetMipCount(),
																											TextureHelper::GetTextureFormat(pDDSHeader));

	assert(nTextureSize <= fileContent.size() - nDDSHeaderSize);

	// add resource to list of non-streamable resources
	//#pragma omp critical(additional_pak)
	std::vector<byte> finalContent;
	{
		finalContent.resize(nTextureSize * 2 + nDDSHeaderSize);

		bool ableToBeCompressed = false;//true;
		if ((pDDSHeader->dwSurfaceFlags & DDS_SURFACE_FLAGS_CUBEMAP) && (pDDSHeader->dwCubemapFlags & DDS_CUBEMAP_ALLFACES))
			ableToBeCompressed = false;
		if (pDDSHeader->dwDepth > 1)
			ableToBeCompressed = false;

		// check if we have attached alpha channel
		CImageExtensionHelper::DDS_HEADER* pAlphaDDSHeader = NULL;
		if(fileContent.size() > nDDSHeaderSize+nTextureSize)
			pAlphaDDSHeader = (CImageExtensionHelper::DDS_HEADER*)CImageExtensionHelper::GetAttachedImage(&fileContent[nDDSHeaderSize+nTextureSize]);
		if(pAlphaDDSHeader)
			ableToBeCompressed = false;

		if(ableToBeCompressed && m_pCC->pRC->GetCompressor()->CompressMips(pDDSHeader, 0, 
																																										pDDSHeader->GetMipCount(), 
																																										pData, nTextureSize, finalContent)
					&& finalContent.size() + nDDSHeaderSize < fileContent.size())
		{
			pDDSHeader->dwReserved1[0] |= CImageExtensionHelper::EIF_Compressed;
			size_t oldSize = finalContent.size();
			finalContent.resize(oldSize + nDDSHeaderSize);
			memmove(&finalContent[nDDSHeaderSize], &finalContent[0], oldSize);
			memcpy(&finalContent[0], pFileHeader, nDDSHeaderSize);
		}
		else
		{
			finalContent.resize(fileContent.size());
			memcpy(&finalContent[0], &fileContent[0], fileContent.size());
		}
		bool res = SaveFile(sNewFile, &finalContent[0], finalContent.size());
		assert(res);
		CCLOG->Log("Resource added to additional pak: '%s'\n", resource.m_sFileName.c_str());
	}
}

void CTextureSplitter::Init( IConfig* config, IRCLog* log, const char* exePath )
{
}

void CTextureSplitter::DeInit( )
{
	// output statistics
	PrintStatistics();

	m_pResourceCompiler = NULL;
}

void CTextureSplitter::Release( )
{
	if (InterlockedDecrement(&m_refCount) == 0)
		delete this;
}

bool CTextureSplitter::Process( ConvertContext &cc )
{
	Cleanup();	// cleanup containers

	// default values
	m_sCurrentDirectory = "Streaming";

	cc.pRC->GetCompressor()->SetPlatform(cc.platform);

	m_pCC = &cc;

	m_pResourceCompiler = cc.pRC;

	if(cc.platform == PLATFORM_X360 || cc.platform == PLATFORM_PS3)
		m_currentEndian = eBigEndian;
	else
		m_currentEndian = eLittleEndian;


	// Get the files to process.
	string sOutputFile = cc.getOutputPath();
	string sInputFile = cc.getSourcePath();

	// parse input file and create ordered list of resources to process
	if(!LoadResourceList(sInputFile))
	{
		CCLOG->LogError("E: Failed to load config file '%s'\n", sInputFile.c_str());
		return false;
	}

	const string ext = PathHelpers::FindExtension(sInputFile);
	// if it's an *.stm file, then we need to branch a butch of threads
	// but if it's *.dds files, then it's already done by high-level RC function
	const bool isResourceList = _stricmp(ext, "stm") == 0;				

	// process this list
	int32 nThreadsToUse = min(m_ResourcesList.size(), m_pCC->pRC->GetNumThreads());
	if(!isResourceList)
		nThreadsToUse = 1;
	{
		//#pragma omp  num_threads(nThreads)
		{
			#pragma omp parallel for num_threads(nThreadsToUse) schedule(dynamic, 1)
			for(int32 iResource = 0;iResource < m_ResourcesList.size();++iResource)
			{
				std::vector<uint8> fileContent;

				SStreamableResource resource;
				#pragma omp critical
				{
					std::set<string>::const_iterator it = m_ResourcesList.begin();
					std::advance(it, iResource);
					resource.m_sFileName = *it;
				}

				// load texture parse dds chunks
				ParseDDSTexture(resource, fileContent);

				if(fileContent.empty())
					continue;

				if(resource.m_chunks.size())
				{
					// write out dds chunks
					for(int32 nChunk = 0;nChunk < resource.m_chunks.size();++nChunk)
					{
						// split texture's MIPs regarding to current chunk
						// save it to separate file
						ProcessResource(resource, fileContent, nChunk);
					}

					CCLOG->Log("Added file '%s'\n", resource.m_sFileName.c_str());
				}
				else
				{
					// if failed then this resource is not for streaming
					// add it to additional pak
					AddResourceToAdditionalList(resource, fileContent);
				}
				if(omp_get_thread_num() == 0 && isResourceList)
				{
					char str[0x400];

					_snprintf(str, sizeof(str),"Progress: %.1f%%, avg MCT ratio: %.2f, avg PTC ratio: %.2f, avg JPEG ratio: %.2f", (float)iResource * 100 / m_ResourcesList.size(),
					m_pResourceCompiler->GetCompressor()->GetStatistics().GetAvgRatio<ectMCT>(), 
					m_pResourceCompiler->GetCompressor()->GetStatistics().GetAvgRatio<ectPTC>(),
					m_pResourceCompiler->GetCompressor()->GetStatistics().GetAvgRatio<ectJPEG>());

					SetConsoleTitle(str);
				}
			}
		}
	}

	CCLOG->Log("Processing streaming config file '%s' finished.\n", sInputFile.c_str());

	m_pCC = NULL;

	return true;
}

bool CTextureSplitter::GetOutputFile( ConvertContext &cc )
{
	string sExtension = PathHelpers::FindExtension(cc.sourceFileFinal);
	sExtension = "bin" + sExtension;
	cc.outputFile = PathHelpers::ReplaceExtension(cc.sourceFileFinal, sExtension);
	return true;
}

int CTextureSplitter::GetNumPlatforms() const
{
	return 3;
}

Platform CTextureSplitter::GetPlatform( int index ) const
{
	switch (index)
	{
	case 0:
		return PLATFORM_PC;
	case 1:
		return PLATFORM_X360;
	case 2:
		return PLATFORM_PS3;
	default:
		assert(0);
		return PLATFORM_UNKNOWN;
	}
}

int CTextureSplitter::GetNumExt( ) const
{
	return sizeof(s_sInputExtensions) / sizeof(s_sInputExtensions[0]);
}

const char* CTextureSplitter::GetExt( int index ) const
{
	return s_sInputExtensions[index].c_str();
}

DWORD CTextureSplitter::GetTimestamp( ) const
{
	return GetTimestampForLoadedLibrary(g_hInst);
}

ICompiler* CTextureSplitter::CreateCompiler( )
{
	if(m_refCount == 1)
	{
		++m_refCount;
		return this;
	}
	CTextureSplitter* pNewSplitter = new CTextureSplitter();
	return pNewSplitter;
}

bool CTextureSplitter::SupportsMultithreading( ) const
{
	return true;
}

bool CTextureSplitter::AddResourceToProcessing(const string& sResourceFile)
{
	const string ext = PathHelpers::FindExtension(sResourceFile);
	if(_stricmp(ext.c_str(), "dds") != 0)
	{
		assert(0);
		CCLOG->LogError("Extension %s is not supported by texture splitter\n", ext.c_str());
		return false;
	}
	FILE *hFile = fopen(sResourceFile,"rb");
	const bool isDDSExists = hFile != NULL;
	if(hFile)
		fclose(hFile);
	if(!isDDSExists)
	{
		CCLOG->LogError("File %s is not exists\n", sResourceFile.c_str());
		return false;
	}
	m_ResourcesList.insert(sResourceFile);		// add resource to pak
	return true;
}

bool CTextureSplitter::LoadResourceList( const string& sConfigFile )
{
	const string ext = PathHelpers::FindExtension(sConfigFile);

	// in case of direct *.dds mask add it directly
	if(_stricmp(ext, "stm") != 0)
	{
		m_sCurrentDirectory = "";
		string destFolder;	// getting optional dest folder
		if(m_pCC->config->Get("dest", destFolder))
		{
			m_sCurrentDirectory = destFolder;

			// check if the file is from the current output directory
			string sPartenDir;
			string sDir = sConfigFile;
			while(!(sDir = PathHelpers::GetParentDirectory(sDir)).empty())
				sPartenDir = sDir;
			if(sPartenDir.compareNoCase(m_sCurrentDirectory) == 0)
				return true;		// just skip it
		}
		return AddResourceToProcessing(sConfigFile);
	}

	CCLOG->Log("Processing streaming config file '%s'...\n", sConfigFile.c_str());

	// opens file with list of resources
	FILE *hFile = fopen(sConfigFile,"rb");

	if(hFile)
	{
		std::vector<char> vBuffer;		

		// reads file size
		// gets file size
		fseek( hFile,0,SEEK_END );
		const uint32 len = ftell(hFile);
		fseek( hFile,0,SEEK_SET );
		vBuffer.resize(len+1);

		if(len)
		{
			// reads file's content
			if(fread(&vBuffer[0],len,1,hFile)==1)
			{
				vBuffer[len]=0;															// end terminator

				char *p = &vBuffer[0];

				// parses the file
				while(*p)
				{
					while(*p!=0 && *p<=' ')										// jump over whitespace
						++p;

					char *pLineStart = p;

					while(*p!=0 && *p!=10 && *p!=13)					// goto end of line
						++p;

					char *pLineEnd = p;

					while(*p!=0 && (*p==10 || *p==13))				// goto next line with data
						++p;

					if(*pLineStart!=';')											// if it's not a commented line
					{
						*pLineEnd=0;

						// find paths
						string sLine = pLineStart;
						if(!sLine.empty() && sLine[0]=='%')
						{
							sLine = sLine.Mid(1);
							sLine.MakeLower();
							int l = -1;
							if( (l = sLine.find("dest_dir=")) >= 0 )
							{
								sLine = sLine.Mid(l+sizeof("dest_dir=")-1);
								m_sCurrentDirectory = sLine;
							}
							else if( (l = sLine.find("src_dir=")) >= 0 )
							{
								sLine = sLine.Mid(l+sizeof("src_dir=")-1);
								CreateCompleteResourceList(PathHelpers::AddSeparator(sLine));
							}
						}
						else	//	add resource
						{
							std::set<string>::const_iterator it = m_ResourcesList.find(sLine);
							if( it == m_ResourcesList.end() )
							{
								AddResourceToProcessing(sLine);	// add resource to pak
							}
							else
							{
								CCLOG->LogWarning("Warning: resource file '%s' is duplicated!\n", sLine.c_str());
							}
						}

					}
				}
			}
			else
			{
				CCLOG->LogError("Error: Config file read '%s' failed\n", sConfigFile.c_str());
				return false;
			}
		}
		else
		{
			CCLOG->LogError("Error: Config file '%s' is empty\n", sConfigFile.c_str());
			return false;
		}
		fclose(hFile);
	}
	else
	{
		CCLOG->LogError("Error: Open resource list file failed: '%s' \n",sConfigFile.c_str());
		return false;
	}

	CCLOG->Log("Resource list assembling done: %d resources to process\n", m_ResourcesList.size());

	return true;
}

void CTextureSplitter::CreateCompleteResourceList( const string& sPath )
{
	_finddata_t fd;

	string sPathPattern = PathHelpers::Join(sPath,"*.*");

	intptr_t handle = _findfirst(sPathPattern.c_str(), &fd );

	if(handle<0)
	{
		CCLOG->LogError("E: BuildResourcesListRecursive failed '%s'\n",sPathPattern.c_str());
		return;
	}

	do 
	{
		if(fd.attrib&_A_SUBDIR)
		{
			if(strcmp(fd.name,".")!=0 && strcmp(fd.name,"..")!=0)
				CreateCompleteResourceList(PathHelpers::Join(sPath, fd.name));
			continue;
		}

		string sFilePath = PathHelpers::Join(sPath, fd.name);
		if(_stricmp(PathHelpers::FindExtension(fd.name).c_str(), "dds") == 0)
		{
			// add resource
			m_ResourcesList.insert(sFilePath);
		}
	} while (_findnext( handle, &fd ) >= 0);

	_findclose(handle);
}

void CTextureSplitter::ProcessPlatformSpecificConversions( byte* fileContent, const size_t fileSize )
{
	assert(fileContent);

	CImageExtensionHelper::DDS_HEADER& header = (CImageExtensionHelper::DDS_HEADER&)*fileContent;
	if(header.dwTextureStage != 'CRYF')
	{
		header.dwReserved1[0] = 0;
		header.dwTextureStage = 'CRYF';
	}

	byte* pDataHead = fileContent + sizeof(CImageExtensionHelper::DDS_HEADER);

	DWORD& imageFlags = header.dwReserved1[0];

	if(m_pCC->platform == PLATFORM_PC)		// there is no conversions for PC platform
		return;

	// check if this texture already native-converted
	switch (m_pCC->platform)
	{
	case PLATFORM_X360:
		if(imageFlags & CImageExtensionHelper::EIF_XBox360Native) return;
		break;
	case PLATFORM_PS3:
		if(imageFlags & CImageExtensionHelper::EIF_PS3Native) return;
		break;
	}

	// we need to process endians only if it's necessary
	bool currentEndian = eLittleEndian;
	if(m_pCC->platform == PLATFORM_X360/* || m_pCC->platform == PLATFORM_PS3*/)
		currentEndian = eBigEndian;

	// set "big_endianness" texture flag
	if(currentEndian == eBigEndian)
		imageFlags |= CImageExtensionHelper::EIF_BigEndianness;

	// mark this texture as native for current platform
	switch (m_pCC->platform)
	{
	case PLATFORM_X360:
		imageFlags |= CImageExtensionHelper::EIF_XBox360Native;
		break;
	case PLATFORM_PS3:
		imageFlags |= CImageExtensionHelper::EIF_PS3Native;
		break;
	}
	
	// get pixel format of texture
	const CImageExtensionHelper::DDS_PIXELFORMAT& pixelFormat = header.ddspf;

	// get dimensions
	uint32 dwWidth,dwHeight,dwDepth,dwSides,dwMips;
	dwWidth = header.dwWidth;
	dwHeight = header.dwHeight;
	dwDepth = max(1ul, header.dwDepth);
	dwMips = header.GetMipCount();
	const bool cubemap = (header.dwSurfaceFlags & DDS_SURFACE_FLAGS_CUBEMAP) != 0 && (header.dwCubemapFlags & DDS_CUBEMAP_ALLFACES) != 0;
	dwSides = cubemap ? 6 : 1;
	const ETEX_Format format = DDSFormats::GetFormatByDesc(header.ddspf);
	const bool bDXTCompressed = TextureHelper::IsDXTCompressed(format);
	const uint32 nBitsPerPixel = bDXTCompressed ? (TextureHelper::BytesPerBlock(format) / 2) : (TextureHelper::BytesPerBlock(format) * 8);
	if(nBitsPerPixel == 8 && !bDXTCompressed)		// ignore one-bit textures
		return;

	// process mips
	uint32 nDataOffset = 0;
	for(uint32 dwSide=0;dwSide<dwSides;++dwSide)
	{
		for(uint32 dwMip=0;dwMip<dwMips;++dwMip)
		{
			// calc mip's size
			const uint32 dwLocalWidth = max(1u, dwWidth >> dwMip);
			const uint32 dwLocalHeight = max(1u, dwHeight >> dwMip);
			const uint32 dwLocalDepth = max(1u, dwDepth >> dwMip);
			const uint32 dwBlockWidth = bDXTCompressed ? (dwLocalWidth+3)/4 : dwLocalWidth;
			const uint32 dwBlockHeight = bDXTCompressed ? (dwLocalHeight+3)/4 : dwLocalHeight;
			const uint32 dwPitch = TextureHelper::TextureDataSize(dwLocalWidth, 1, 1, 1, format);
			const uint32 dwMipSize = TextureHelper::TextureDataSize(dwLocalWidth, dwLocalHeight, dwLocalDepth, 1, format);

			byte* pMem=0;

			// get offsetted mip chunk
			byte* pStartMem = pDataHead + nDataOffset;

			pMem = pStartMem;

			for(uint32 d=0;d<dwDepth;++d)
			{
				for(uint32 v=0;v<dwBlockHeight;++v)	// for each line of blocks
				{
					// swizzling and endians swapping for specific platforms
					if(bDXTCompressed)
					{
						SwapEndian((uint16*)pMem, dwPitch / sizeof(uint16), currentEndian);
					}
					else if(format == eTF_G16R16F)
					{
						if(m_pCC->platform == PLATFORM_X360)
						{
							for(uint32 i=0;i<dwLocalWidth;++i)
							{
								std::swap(pMem[i*4+0], pMem[i*4+3]);
								std::swap(pMem[i*4+1], pMem[i*4+2]);
							}
						}
						else
						{
							SwapEndian((uint32*)pMem, dwPitch / sizeof(uint32), currentEndian);
						}
					}
					else if(format == eTF_A8R8G8B8)
					{
						//if(m_pCC->platform == PLATFORM_PS3)
						//{
						//	for(uint32 i=0;i<dwLocalWidth;++i)
						//		std::swap(pMem[i*4], pMem[i*4+2]);
						//}
						SwapEndian((uint32*)pMem, dwPitch / sizeof(uint32), currentEndian);
					}
					else if(format == eTF_A4R4G4B4)
					{
						SwapEndian((uint32*)pMem, dwPitch / sizeof(uint32), currentEndian);
					}
					else if(format == eTF_X8R8G8B8)
					{
						SwapEndian((uint32*)pMem, dwPitch / sizeof(uint32), currentEndian);
					}
					else if(format == eTF_R8G8B8)
					{
						assert(nBitsPerPixel == 24);
						if((imageFlags & DDS_RESF1_DSDT) == 0)
						{
							if(m_pCC->platform == PLATFORM_X360 || m_pCC->platform == PLATFORM_PS3)
							{
								for(uint32 i=0;i<dwLocalWidth;++i)
									std::swap(pMem[i*3], pMem[i*3+2]);
							}
							SwapEndian((uint32*)pMem, dwPitch / sizeof(uint32), currentEndian);
						}
						else	// DUDV(L8V8U8)
						{
							for(uint32 i=0;i<dwLocalWidth;++i)
								std::swap(pMem[i*3], pMem[i*3+2]);
						}
					}
					else
					{
						if(format == eTF_DXT1)
							SwapEndian((uint16*)pMem, dwPitch / sizeof(uint16), currentEndian);
						else 
							SwapEndian((uint32*)pMem, dwPitch / sizeof(uint32), currentEndian);
					}

					pMem += dwPitch;
				}

				// tile texture level for XBox360
				if(m_pCC->platform == PLATFORM_X360)
				{
					const uint32 nBlockSize = bDXTCompressed ? (4*4*nBitsPerPixel/8) : nBitsPerPixel / 8;
					const bool isBlockSizePowOfTwo = (nBlockSize & (nBlockSize - 1)) == 0;
					const bool isPitchGPUFriendly = dwPitch%32 == 0;
					// tile only levels that more that 64 in both width and height
					// otherwise we will have huge disk space waste due to GPU alignment
					if(dwLocalWidth > 64 && dwLocalHeight > 64 && isBlockSizePowOfTwo && isPitchGPUFriendly)	
					{
						const uint32 nBlockWidth = bDXTCompressed ? (dwLocalWidth + 3) / 4 : dwLocalWidth;
						const uint32 nBlockHeight = bDXTCompressed ? (dwLocalHeight + 3) / 4 : dwLocalHeight;

						m_pCC->pRC->GetCompressor()->TileTextureLevelForXBox((byte*)pStartMem, (byte*)pStartMem, 
							nBlockWidth, nBlockHeight, format, dwPitch, nBlockSize );
					}
				}
			}
			nDataOffset += dwMipSize;
		}
	}

	// apply to attached images recursively
	if(nDataOffset + sizeof(CImageExtensionHelper::DDS_HEADER) < fileSize)
	{
		uint32 size;
		byte* pAttachedImage = (byte*)CImageExtensionHelper::GetAttachedImage(pDataHead + nDataOffset, &size);
		if(pAttachedImage)
			ProcessPlatformSpecificConversions(pAttachedImage, size);
	}
}

void CTextureSplitter::Cleanup()
{
	m_AttachedResources.clear();
	m_ResourcesList.clear();
}
