#include "StdAfx.h"

#include "AnimationCompiler.h"
#include "AnimationInfoLoader.h"
#include "CGF\CGFLoader.h"
#include "CGF\CGFSaver.h"
#include "SkeletonInfo.h"
#include "AnimationLoader.h"
#include "IConfig.h"
#include "TrackStorage.h"

#include "IPakSystem.h"
#include "TempFilePakExtraction.h"


CAnimationCompiler * g_pAnimationCompiler;

static bool ScanDirectoryRecursive( const string &root,const string &path,const string &file,std::vector<string> &files, bool recursive , std::vector<uint32>& sizes);


class CAnimationLoaderListener : public IAnimationLoaderListener
{
public:
  CAnimationLoaderListener(ConvertContext &cc)
    :	cc(cc)
  {
  }

  virtual void OnAnimationLoaderMessage(MessageType type, const char* szMessage)
  {
    switch (type)
    {
    case MESSAGE_INFO:
      cc.pLog->Log("%s", szMessage);
      break;
    case MESSAGE_WARNING:
      cc.pLog->LogWarning("%s", szMessage);
      break;
    case MESSAGE_ERROR:
      cc.pLog->LogError("%s", szMessage);
      break;
    }
  }

private:
  ConvertContext &cc;
};


CAnimationCompiler::CAnimationCompiler(ICryXML * pXML) : m_pXMLParser(pXML)
{
  m_TotalShared = 0;
  m_TotalMemoryShared = 0;
  m_TotalMemory = 0;
  m_TotalTracks = 0;
  m_pTrackStorage.reset(0);
  m_refCount = 1;
  m_iCount = 0;
  m_fAverage = 0;
  m_fTotal = 0;
  //	m_iThreads = 0;
}

CAnimationCompiler::~CAnimationCompiler(void)
{
}

void CAnimationCompiler::Release()
{
  if (--m_refCount <= 0)
    delete this;
}

bool /*CAnimationCompiler::*/GetFileTime( const char *filename,FILETIME *ftimeModify, FILETIME*ftimeCreate )
{
  WIN32_FIND_DATA FindFileData;
  HANDLE hFind;
  hFind = FindFirstFile( filename,&FindFileData );
  if (hFind == INVALID_HANDLE_VALUE)
  {
    return false;
  }
  FindClose(hFind);
  if (ftimeCreate)
  {
    ftimeCreate->dwLowDateTime = FindFileData.ftCreationTime.dwLowDateTime;
    ftimeCreate->dwHighDateTime = FindFileData.ftCreationTime.dwHighDateTime;
  }
  if (ftimeModify)
  {
    ftimeModify->dwLowDateTime = FindFileData.ftLastWriteTime.dwLowDateTime;
    ftimeModify->dwHighDateTime = FindFileData.ftLastWriteTime.dwHighDateTime;
  }
  return true;
}


// special mode for counting shared tracks information
void CAnimationCompiler::ReportMode(ConvertContext& cc, int iReportMode, const char * singleFile, ILoaderCGFListener * listener)
{

  std::vector<string> fileList;
  string dirPath(cc.sourceFolder);
  string destPath;
  std::vector<uint32> sizes;
  ScanDirectoryRecursive(dirPath.c_str(),string(""), string("*.caf"), fileList, true, sizes);

  bool bSingleFile(false);

  for (uint32  anim = 0; anim < fileList.size(); ++anim)
  {

    string animFile(dirPath);
	animFile = PathHelpers::Join(animFile, fileList[anim]);

    if ( (!bSingleFile) || (bSingleFile && (strstr(fileList[anim], singleFile) || strstr(singleFile, fileList[anim]))))
    {

      if (bSingleFile && strlen(singleFile) == 0)
        continue;
      if (bSingleFile && strlen(fileList[anim]) == 0)
        continue;

      CAnimation currentAnimation(0);

      SAnimationDesc desc;
      if (currentAnimation.LoadAnimationFromFile(animFile.c_str(), animFile.c_str(), listener, desc, false,m_Behavior.m_bNeedCalculate))
      {

        //if ( m_Behavior.m_bCheckLocomotion)
        {
          string res;
          if (currentAnimation.CheckLoco(cc,res))
            cc.pLog->Log("File %s. %s", animFile.c_str(), res.c_str());
        }

        if (iReportMode == 1)
        {
          if (currentAnimation.HasOldFormat() && !currentAnimation.HasNewFormat())
          {
            cc.pLog->Log("File %s has only uncompressed chunks", animFile.c_str());
          }
        }
        else
          if (iReportMode == 2)
          {
            if (currentAnimation.HasOldFormat() && currentAnimation.HasNewFormat())
            {
              cc.pLog->Log("File %s has both uncompressed and compressed data", animFile.c_str());
            }
          }

          if (bSingleFile)
            return;
      }
      else
      {
        cc.pLog->Log("Failed processed file %s", animFile.c_str());
      }
    }
  }

}

void RunThread(void * pData) 
{
  ThreadCompiler * pCompiler = (ThreadCompiler *)pData;
  pCompiler->Run();

}


int  CAnimationCompiler::ProcessSingleAnimation(CSkeletonInfo* currentSkeleton, const string& animFile, const string& destFile, 
                                                ILoaderCGFListener * listener, ConvertContext &cc, SAnimationDesc& pDesc, int oldsize)
{
	ThreadCompiler * pCurrThread = AcquireThread();

  pCurrThread->currentSkeleton = currentSkeleton;
  pCurrThread->animFile = animFile;
  pCurrThread->destFile = destFile; 
  pCurrThread->listener = listener;
  pCurrThread->cc = cc;
  pCurrThread->pDesc = pDesc;
  pCurrThread->oldsize = oldsize;
  pCurrThread->pCompiler = this;

  if (!m_Behavior.m_bUseMultiThreading)
    //(pCurrThread->Run();
    RunThread((void *)pCurrThread);
  else {
    cc.pLog->Log("Threads available %i. Started %i %s", m_vThreads.size(), pCurrThread->id, pCurrThread->destFile.c_str() );
    _beginthread(RunThread, 0, (void *)pCurrThread);
  }

  return pCurrThread->resSize;
}

void ThreadCompiler::Run()
{
  //	int resSize = -1;
  // check for update
  FILETIME  updated, created;
  GetFileTime(animFile.c_str(), &updated, &created);
  DWORD latest = max(FileUtil::FiletimeToUnixTime(updated), FileUtil::FiletimeToUnixTime(created));
  static volatile int g_lockMemMan;
  if (pCompiler->m_Behavior.m_bNeedUpdateOnlyNew)
  {


		bool bFoundDesc = false;

    if (pCompiler->m_pTrackStorage.get())
    {
      //m_pTrackStorage
      DWORD stored;
      if (pCompiler->m_pTrackStorage->GetAnimFileTime(animFile, &stored))
      {
				bFoundDesc = true;
        // dont need update
        if (stored >= latest)
        {
          WriteLock lock(g_lockMemMan);

          pCompiler->m_iCount++;
          pCompiler->m_fAverage = pCompiler->m_fTotal/ (float)pCompiler->m_iCount;
					pCompiler->ReleaseThread(this);
          return;
        }
      }
    }

		if (!bFoundDesc) {
			// try to load a single file



			string name = destFile;
			FixString(name);

			FILETIME newUpdated, newCreated;
			GetFileTime(name.c_str(), &newUpdated, &newCreated);
			DWORD newLatest = max(FileUtil::FiletimeToUnixTime(newUpdated), FileUtil::FiletimeToUnixTime(newCreated));

			if (newLatest  >= latest) {
				pCompiler->m_iCount++;
				pCompiler->m_fAverage = pCompiler->m_fTotal/ (float)pCompiler->m_iCount;
				pCompiler->ReleaseThread(this);
				return;

			}



		}
  }

  //	cc.pLog->Log("Started %s", destFile.c_str());
  std::auto_ptr<CAnimation> currentAnimation;
  currentAnimation.reset(new CAnimation(currentSkeleton));
  bool bNeedCalculate(pCompiler->m_Behavior.m_bNeedCalculate);
  if (currentAnimation->LoadAnimationFromFile(animFile.c_str(), destFile.c_str(), listener, pDesc, true, pCompiler->m_Behavior.m_bNeedCalculate))
  {

    if (pCompiler->m_Behavior.m_bNeedCalculate && !currentAnimation->HasOldFormat())
    {
      // we don't have an uncompressed data. File not processed
      //			cc.pLog->LogError("File %s has only compressed data. File deleted", animFile.c_str());
      cc.pLog->Log("File %s has only compressed data. Stored", animFile.c_str());
      // free chunk file
      bNeedCalculate = false;
      //			currentAnimation.reset(0);
      //			DeleteFile(animFile.c_str());
      //			return 0;
    }

    if (!pCompiler->m_Behavior.m_bNeedCalculate && !currentAnimation->HasNewFormat())
    {
      // we don't have a compressed data. need calculate this
      cc.pLog->Log("File %s has only uncompressed data. Compiled", animFile.c_str());
      bNeedCalculate = true;
    }

    CAnimation::EDeleteMethod delFlag;

    bool bNeedProcessing(true);

    if (bNeedCalculate)
    {
      delFlag = CAnimation::eNew;

      if (pCompiler->m_Behavior.m_bNeedDeleteUncompressedData || pCompiler->m_Behavior.m_bNeedDeleteAllData)
        delFlag = CAnimation::eAll;
    }
    else
    {
			if (!pCompiler->m_Behavior.m_bNeedSwapEndian) 
			{
        delFlag = CAnimation::eOld;

        if (currentAnimation->HasNewFormat())
        {
          bNeedProcessing = false;
        }
			} 
			else 
			{
        delFlag = CAnimation::eAll;
      }
    }


    currentAnimation->DeleteOldChunk(delFlag, bNeedProcessing);

    if(currentAnimation->ProcessAnimation(cc,listener, pCompiler->m_pTrackStorage.get(), bNeedCalculate, pCompiler->m_Behavior.m_bNeedUpdateOnlyNew))
    {
      resSize = currentAnimation->SaveAnimationToFile(destFile.c_str(), listener, cc, bNeedProcessing, updated);

      GetFileTime(animFile.c_str(), &updated, &created);
      latest = max(FileUtil::FiletimeToUnixTime(updated), FileUtil::FiletimeToUnixTime(created));
      currentAnimation->m_dwTimestamp = latest;
      if (!pDesc.m_SkipSaveToDatabase)
        currentAnimation->SaveToDB(pCompiler->m_pTrackStorage.get(), pCompiler->m_Behavior.m_bNeedUpdateOnlyNew);
    }
    else
    {
      cc.pLog->LogError("Error in processing %s", animFile.c_str());
    }

    if (pCompiler->m_Behavior.m_bNeedSaveDBA && !pDesc.m_SkipSaveToDatabase)
    {
      currentAnimation.reset(0);
      DeleteFile(destFile.c_str());
    }
  }

  if (oldsize > 0)
  {
    float currentRatio = 100.0f - (float)resSize * 100.0f / (float)oldsize;
    pCompiler->m_fTotal += currentRatio;
    cc.pLog->Log("Processed %s. Old size %d. New size %d. Compress ratio %.1f%%", destFile.c_str(), oldsize, resSize, currentRatio );
  }

  WriteLock lock(g_lockMemMan);

  pCompiler->m_iCount++;
  pCompiler->m_fAverage = pCompiler->m_fTotal/ (float)pCompiler->m_iCount;
	pCompiler->ReleaseThread(this);
  return ;
}


void CAnimationCompiler::PrepareThreads()
{
	CryAutoLock lock(m_threadsLock);
  int count = 1;

  if (m_Behavior.m_bUseMultiThreading)
    count = std::max(1, m_Behavior.m_iSupportedThreads);

	for (int i = 0; i < count; ++i)
	{
    m_vThreads.push_back(new ThreadCompiler);
    m_vThreads.back()->id = i;
	}
}

ThreadCompiler* CAnimationCompiler::AcquireThread()
{
	if (m_Behavior.m_bUseMultiThreading)
	{
		while(true)
		{
			{
				CryAutoLock lock(m_threadsLock);
				if(!m_vThreads.empty())
				{
					ThreadCompiler * pCurrThread= m_vThreads.back();
					m_vThreads.pop_back();
					return pCurrThread;
  }
}
			Sleep(0);
		}
	}
	ThreadCompiler * pCurrThread= m_vThreads.back();
	m_vThreads.pop_back();
	return pCurrThread;
}

void CAnimationCompiler::ReleaseThread(ThreadCompiler* thread)
{
	if (m_Behavior.m_bUseMultiThreading)
	{
		CryAutoLock lock(m_threadsLock);
		m_vThreads.push_back(thread);
	}
	else
		m_vThreads.push_back(thread);	
}

void CAnimationCompiler::WaitAllThreads()
{
	if (m_Behavior.m_bUseMultiThreading)
	{
		while(true)
		{
			{
				CryAutoLock lock(m_threadsLock);
				if(m_vThreads.size() == m_Behavior.m_iSupportedThreads)
				{
					return;
				}
			}
			Sleep(0);
		}
	}
}

//bool 


bool CAnimationCompiler::Process( ConvertContext &cc ) 
{

  string descfile;

  if (stricmp(cc.sourceFileFinalExtension,"cba") == 0)
  {
    descfile = cc.getSourcePath();
  }

  if (cc.platform == PLATFORM_X360 || cc.platform == PLATFORM_PS3)
  {
    m_Behavior.m_bNeedSwapEndian = true;
    cc.pLog->Log("Endian conversion specified");

  }

  string singleFile;
  bool bSingleFile = cc.config->Get("file", singleFile);
  string fix(singleFile);
  FixString(fix);
  singleFile = fix.c_str();

  string destFolder;
  bool bDestFolder = cc.config->Get("dest", destFolder);

  int iReportMode(false);
  bool bReport = cc.config->Get("report", iReportMode);

  int iCheckloco(false);
  bool bCheckLoco = cc.config->Get("checkloco", iCheckloco);

  m_Behavior.m_bCheckLocomotion = bCheckLoco && iCheckloco > 0;


  {

    bool bCleanup(false);
    bool bF =	cc.config->Get("cleanupfast", bCleanup);

    bCleanup = bCleanup && bF;

    bool bOnlynew(false);
    bF =	cc.config->Get("cleanup", bOnlynew);

    bOnlynew = bOnlynew && bF;

    //int iCountTracks(0);
    //bool bCountTracks = cc.config->Get("statistics", iCountTracks);

    int iBuildTracks(0);
    bool bBuildTracks = cc.config->Get("build", iBuildTracks);

    int iUpdateTracks(0);

    bool bUpdateTracks = cc.config->Get("update", iUpdateTracks);

    //int iThreadsNumber(0);
    //bool bUseMultiThreading = cc.config->Get("threads", iThreadsNumber);

    m_Behavior.m_bUseMultiThreading = !bSingleFile && cc.threads > 1;//bUseMultiThreading;
    m_Behavior.m_iSupportedThreads = cc.threads;

    if (m_Behavior.m_bNeedSwapEndian)
    {
      bUpdateTracks = false;
      bBuildTracks = true;
    }



    // set vars for behavior
    if (!bCleanup)//bCleanup || iCompress)
      m_Behavior.m_bNeedCalculate = true;

    if (iUpdateTracks)	
    {
      m_Behavior.m_bNeedLoadDBA = true; 
      m_Behavior.m_bNeedUpdateOnlyNew = true;
    }

    if (iUpdateTracks || iBuildTracks)
      m_Behavior.m_bNeedSaveDBA = true; 

    if (bCleanup)
      m_Behavior.m_bNeedDeleteUncompressedData = true;

    if (bOnlynew)
      m_Behavior.m_bNeedDeleteAllData = true;
  }

  if (m_Behavior.m_bNeedSaveDBA)
  {
    m_pTrackStorage.reset(new CTrackStorage);
  }

  class Listener : public ILoaderCGFListener
  {
  public:
    Listener(ConvertContext& cc): cc(cc) {}
    virtual void Warning( const char *format ) {cc.pLog->LogWarning("%s", format);}
    virtual void Error( const char *format ) {cc.pLog->LogError("%s", format);}

  private:
    ConvertContext& cc;
  };
  Listener listener(cc);


  if (bReport && iReportMode > 0)
  {
    ReportMode(cc, iReportMode, singleFile, &listener);
    return true;
  }

  PrepareThreads();

  if (bSingleFile)
  {
    m_pTrackStorage.reset(0);
  }

  CAnimationInfoLoader loader(m_pXMLParser);

  CAnimationLoaderListener listenerAnim(cc);

  IPakSystem* pPakSystem = (cc.pRC ? cc.pRC->GetPakSystem() : 0);
  if (!loader.LoadDescription(descfile, &listenerAnim, pPakSystem))
    return false;

	enum eErrorState 
	{
    eUnknown,
    eMissedModel
  };

  eErrorState errorState = eUnknown;
  string errorModel;

  bool bProcessed(false);

	//------------------------------------------------------------------------------------
	//------------------------------------------------------------------------------------
	//------------------------------------------------------------------------------------
	cc.pLog->Log("SourceFolder: %s",cc.sourceFolder);

	uint32 numDefinitions = loader.m_Definitions.size();
	for (uint32 i=0; i<numDefinitions; ++i)
  {
    string filename(cc.sourceFolder);
    //		if (bDestFolder)
    //			filename =  destFolder;
	filename = PathHelpers::Join(filename,loader.m_Definitions[i].m_Model);


    //file may be in a pakfile, so extract it if required

    IPakSystem* pPakSystem = (cc.pRC ? cc.pRC->GetPakSystem() : 0);
    TempFilePakExtraction fileProxy( filename.c_str(), pPakSystem );

    CSkeletonInfo currentSkeleton;
		const char* strReferenceSkeleton = fileProxy.GetTempName();
		bool res = currentSkeleton.LoadCHRModel(strReferenceSkeleton, &listener);
		if (res)
    {
      //fill list with filenames
      std::vector<string> fileList;
      string dirPath(cc.sourceFolder);
      string destPath;
	  dirPath = PathHelpers::AddSeparator(PathHelpers::Join(dirPath, loader.m_Definitions[i].GetAnimationPathWithoutSlash()));
      destPath = dirPath;
      std::vector<uint32> sizes;
      ScanDirectoryRecursive(dirPath.c_str(),string(""), string("*.caf"), fileList, true, sizes);

      if (bDestFolder)
      {
        destPath = destFolder;
		destPath = PathHelpers::AddSeparator(PathHelpers::Join(destPath, loader.m_Definitions[i].GetAnimationPathWithoutSlash()));
      }

      // clear trackstorage
      if (m_pTrackStorage.get())
      {
        m_pTrackStorage->Clear();

      }


      string dbFile;

      bool yep = true;
      for (uint32 anim = 0; anim < fileList.size(); ++anim)
      {

        string animFile(dirPath);
		animFile = PathHelpers::Join(animFile, fileList[anim]);
        FixString(animFile);
        string destFile(destPath);
		destFile = PathHelpers::Join(destFile, fileList[anim]);

        const SAnimationDefinition * pDefinition = loader.GetAnimationDefinition(animFile);
        if (!pDefinition)
        {
          //FIXME: Add message - definition not found!
          continue;
        }


				if (bDestFolder) {
					dbFile = destFolder;

				}
        else
          dbFile = cc.sourceFolder;

		dbFile = PathHelpers::Join(dbFile, pDefinition->m_DBName);

        if (m_pTrackStorage.get() && m_Behavior.m_bNeedLoadDBA && yep)
        {
          yep = false;
          m_pTrackStorage->LoadDataBase(dbFile.c_str());
        }

        SAnimationDesc pDesc;

        if ( (!bSingleFile) || (bSingleFile && (strstr(/*fileList[anim]*/animFile.c_str(), singleFile) || strstr(singleFile, /*fileList[anim]*/animFile.c_str()))))
        {
          pDesc =  pDefinition->GetAnimationDesc(string(fileList[anim]));

          if (bSingleFile && strlen(singleFile) == 0)
            continue;
          if (bSingleFile && strlen(fileList[anim]) == 0)
            continue;

					cc.pLog->Log("Reference Model: %s",strReferenceSkeleton);

          int resSize = ProcessSingleAnimation(&currentSkeleton, animFile, destFile, &listener, cc, pDesc, sizes[anim]);//, bCleanup, bOnlynew );

          if(resSize)
            bProcessed = true;

          //if (sizes[anim] > 0)
          //{
          //	float currentRatio = 100.0f - (float)resSize * 100.0f / (float)sizes[anim];
          //	cc.pLog->Log("Processed %s. Old size %d. New size %d. Compress ratio %.1f%%", destFile.c_str(), sizes[anim], resSize, currentRatio );
          //	total += currentRatio;
          //}
          //count++;
          //average = total/ (float)count;

          if (bSingleFile)
          {
            cc.pLog->Log("Processed %d files. Average compress ratio %f%%", m_iCount, m_fAverage);
            return true;
          }
        }
      }

			WaitAllThreads();

      if (m_pTrackStorage.get())
      {
        if (m_Behavior.m_bNeedSwapEndian)
          m_pTrackStorage->SaveDataBase904(dbFile.c_str());
        else
          m_pTrackStorage->SaveDataBase903(dbFile.c_str());
      }
    } 
		else
    {
      if (bSingleFile && strstr(singleFile, loader.m_Definitions[i].GetAnimationPath().c_str())) {
        errorModel = "Probably you have missed character file" + filename;
        errorState = eMissedModel;
      }
    }
  } 


  if (!bProcessed && bSingleFile)
  {
    //	// that mean we find no model or something
    //	// try to process with default param
    //	SAnimationDesc pDesc;
    //	int resSize = ProcessSingleAnimation(0, string(singleFile), string(singleFile), &listener, cc, pDesc, bCleanup, bOnlynew );
    switch (errorState)
    {
    case eMissedModel:
      cc.pLog->LogError(errorModel.c_str());
      break;
    case eUnknown:
      cc.pLog->LogError("Probably the animation(%s) was exported to an external path.\nTo export compressed data please save your file to:\nJ:\\Game02\\Game\\Animations", singleFile);
    }

    DeleteFile(singleFile);
  }

  cc.pLog->Log("Processed %d files. Average compress ratio %.1f%%.  ", m_iCount, m_fAverage  );

  if (m_pTrackStorage.get())
  {
    cc.pLog->Log("All models shared %d tracks. Total tracks %d. Shared memory %d. Total Memory %d", m_TotalShared, m_TotalTracks, m_TotalMemoryShared , m_TotalMemory  );

  }

  return true;
}

bool CAnimationCompiler::GetOutputFile(ConvertContext &cc)
{
  cc.outputFile = cc.sourceFileFinal;
  if (!cc.masterFolder.empty())
	  cc.m_sOutputFolder = PathHelpers::Join(cc.masterFolder, cc.m_sOutputFolder);
  return true;
}

ICompiler* CAnimationCompiler::CreateCompiler()
{
  // Only ever return one compiler, since we don't support multithreading. Since
  // the compiler is just this object, we can tell whether we have already returned
  // a compiler by checking the ref count.
  if (m_refCount >= 2)
    return 0;

  // Until we support multithreading for this convertor, the compiler and the
  // convertor may as well just be the same object.
  ++m_refCount;
  return this;
}

bool CAnimationCompiler::SupportsMultithreading() const
{
  return false;
}

//////////////////////////////////////////////////////////////////////////
// the paths must have trailing slash
static bool ScanDirectoryRecursive( const string &root,const string &path,const string &file,std::vector<string> &files, bool recursive, std::vector<uint32>& sizes )
{
  __finddata64_t c_file;
  intptr_t hFile;

  bool anyFound = false;

  string fullPath = root + path + file;
  if ( (hFile = _findfirst64( fullPath.c_str(), &c_file )) != -1L )
  {
    // Find the rest of the files.
    do {
      if (!(c_file.attrib & _A_SUBDIR))
      {
        anyFound = true;
        string ext(c_file.name);
        int pos = int(ext.rfind('.'));
        if (pos != -1 && _stricmp(".caf", ext.Right(strlen(ext) -  pos) ) == 0)	 {
          files.push_back( path + c_file.name );
          sizes.push_back((unsigned int)c_file.size);
        }
      }
    }	while (_findnext64( hFile, &c_file ) == 0);
    _findclose( hFile );
  }

  if (recursive)
  {
    fullPath = root + path + "*.*";
    if( (hFile = _findfirst64( fullPath.c_str(), &c_file )) != -1L )
    {
      // Find directories.
      do {
        if (c_file.attrib & _A_SUBDIR)
        {
          // If recursive.
          if (c_file.name[0] != '.')
          {
            if (ScanDirectoryRecursive( root,path + c_file.name + "\\",file,files,recursive, sizes ))
              anyFound = true;
          }
        }
      }	while (_findnext64( hFile, &c_file ) == 0);
      _findclose( hFile );
    }
  }

  return anyFound;
}
