////////////////////////////////////////////////////////////////////////////
//
//  CryEngine Source File.
//  Copyright (C), Crytek
// -------------------------------------------------------------------------
//  File name:   XMLBinaryWriter.cpp
//  Created:     21/04/2006 by Michael Smith.
//  Description: 
// -------------------------------------------------------------------------
//  History:
//     8/1/2008 - Modified by Timur
////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "XMLBinaryWriter.h"
#include <algorithm>
#include "..\ResourceCompiler\SwapEndianness.h"

const char* XMLBinary::BinaryFileHeader::sk_szCorrectSignature = "CryXMLB";

//////////////////////////////////////////////////////////////////////////
namespace XMLBinary
{
	void SwapEndianness_Node( Node &t )
	{
		SwapEndianness(t.nTagStringOffset);
		SwapEndianness(t.nContentStringOffset);
		SwapEndianness(t.nParentIndex);
		SwapEndianness(t.nFirstAttributeIndex);
		SwapEndianness(t.nAttributeCount);
		SwapEndianness(t.nFirstChildIndex);
		SwapEndianness(t.nChildCount);
	}

	void SwapEndianness_Attribute( Attribute &t )
	{
		SwapEndianness(t.nKeyStringOffset);
		SwapEndianness(t.nValueStringOffset);
	}

	void SwapEndianness_Header( BinaryFileHeader &t )
	{
		SwapEndianness(t.nXMLSize);
		SwapEndianness(t.nNodeTablePosition);
		SwapEndianness(t.nNodeCount);
		SwapEndianness(t.nAttributeTablePosition);
		SwapEndianness(t.nAttributeCount);
		SwapEndianness(t.nChildTablePosition);
		SwapEndianness(t.nChildCount);
		SwapEndianness(t.nStringDataPosition);
		SwapEndianness(t.nStringDataSize);
	}
}

//////////////////////////////////////////////////////////////////////////
XMLBinary::CXMLBinaryWriter::CXMLBinaryWriter()
{
	m_nStringDataSize = 0;
}

//////////////////////////////////////////////////////////////////////////
bool XMLBinary::CXMLBinaryWriter::WriteNode(IDataWriter* pFile, XmlNodeRef node)
{
	// Scan the node tree, building a flat node list, attribute list and string table.
	m_nStringDataSize = 0;
	
	CompileTables(node);

	// Initialize the file header.
	int nTheoreticalPosition = 0;
	BinaryFileHeader header;
	std::strncpy(header.szSignature, BinaryFileHeader::sk_szCorrectSignature, sizeof(header.szSignature));
	nTheoreticalPosition += sizeof(header);
	header.nNodeTablePosition = nTheoreticalPosition;
	header.nNodeCount = int(m_nodes.size());
	nTheoreticalPosition += header.nNodeCount * sizeof(Node);
	
	header.nChildTablePosition = nTheoreticalPosition;
	header.nChildCount = int(m_childs.size());
	nTheoreticalPosition += header.nChildCount * sizeof(uint16);

	header.nAttributeTablePosition = nTheoreticalPosition;
	header.nAttributeCount = int(m_attributes.size());
	nTheoreticalPosition += header.nAttributeCount * sizeof(Attribute);

	header.nStringDataPosition = nTheoreticalPosition;
	header.nStringDataSize = m_nStringDataSize;

	nTheoreticalPosition += header.nStringDataSize;

	header.nXMLSize = nTheoreticalPosition;

	// Swap endianess of structures
	if (g_bNeedSwapEndian)
	{
		SwapEndianness_Header(header);
		std::for_each( m_nodes.begin(),m_nodes.end(),SwapEndianness_Node );
		std::for_each( m_attributes.begin(),m_attributes.end(),SwapEndianness_Attribute );
		for (size_t i = 0,num = m_childs.size(); i < num; i++) SwapEndianness(m_childs[i]);
	}

	// Write out the file header.
	pFile->Write(&header, sizeof(header));

	// Write out the node table.
	if (!m_nodes.empty())
		pFile->Write(&m_nodes[0], sizeof(m_nodes[0]) * m_nodes.size());

	// Write out the childs table.
	if (!m_childs.empty())
		pFile->Write(&m_childs[0], sizeof(m_childs[0]) * m_childs.size());

	// Write out the attribute table.
	if (!m_attributes.empty())
		pFile->Write(&m_attributes[0], sizeof(m_attributes[0]) * m_attributes.size());

	// Write out the data of all the m_strings.
	for (int nString = 0; nString < m_strings.size(); ++nString)
		pFile->Write(m_strings[nString].c_str(), m_strings[nString].size() + 1);

	return true;
}

void XMLBinary::CXMLBinaryWriter::CompileTables(XmlNodeRef node)
{
	CompileTablesForNode(node, -1);
	CompileChildTable(node);
}

//////////////////////////////////////////////////////////////////////////
void XMLBinary::CXMLBinaryWriter::CompileTablesForNode(XmlNodeRef node, int nParentIndex)
{
	// Add the tag to the string table.
	int nTagStringIndex = AddString(node->getTag());

	// Add the content string to the string table.
	int nContentStringIndex = AddString(node->getContent());

	// Add all the attributes to the attributes table.
	const char* szKey;
	const char* szValue;
	int nFirstAttributeIndex = int(m_attributes.size());
	for (int i = 0,attrCount = node->getNumAttributes(); i < attrCount; i++)
	{
		if (node->getAttributeByIndex( i,&szKey,&szValue ))
		{
			// Add the key and the value to the string table.
			Attribute attribute;
			attribute.nKeyStringOffset = AddString(szKey);
			attribute.nValueStringOffset = AddString(szValue);

			// Add the attribute to the attribute table.
			m_attributes.push_back(attribute);
		}
	}
	int nAttributeCount = int(m_attributes.size()) - nFirstAttributeIndex;

	// Add ourselves to the node list.
	int nIndex = int(m_nodes.size());

	Node nd;
	memset(&nd,0,sizeof(nd));

	nd.nTagStringOffset = nTagStringIndex;
	nd.nContentStringOffset = nContentStringIndex;
	nd.nParentIndex = nParentIndex;
	nd.nFirstAttributeIndex = nFirstAttributeIndex;
	nd.nAttributeCount = nAttributeCount;
	m_nodes.push_back(nd);

	m_nodesMap.insert( NodesMap::value_type(node,nIndex) );

	// Recurse to the child nodes.
	for (int nChild = 0,numChilds = node->getChildCount(); nChild < numChilds; ++nChild)
		CompileTablesForNode(node->getChild(nChild), nIndex );
}

//////////////////////////////////////////////////////////////////////////
void XMLBinary::CXMLBinaryWriter::CompileChildTable(XmlNodeRef node)
{
	int nIndex = m_nodesMap.find(node)->second; // Assume node always exist in map.

	int nFirstChildIndex = (int)m_childs.size();
	int nChildCount = (int)node->getChildCount();
	for (int i = 0; i < nChildCount; i++)
	{
		XmlNodeRef childNode = node->getChild(i);
		int nChildIndex = m_nodesMap.find(childNode)->second; // Assume node always exist in map.
		m_childs.push_back(nChildIndex);
	}

	Node &nd = m_nodes[nIndex];
	nd.nFirstChildIndex = nFirstChildIndex;
	nd.nChildCount = nChildCount;

	// Recurse to the child nodes.
	for (int nChild = 0; nChild < node->getChildCount(); ++nChild)
		CompileChildTable(node->getChild(nChild));
}

int XMLBinary::CXMLBinaryWriter::AddString( const XmlString& sString )
{
	// Look for the string in the string map.
	StringMap::const_iterator itStringEntry = m_stringMap.find(sString);

	// Check whether we found the string.
	if (itStringEntry == m_stringMap.end())
	{
		// The string wasn't in the string map, so we should add it to the table.
		int nOffset = m_nStringDataSize;
		m_strings.push_back(sString);
		itStringEntry = m_stringMap.insert(StringMap::value_type(sString, nOffset)).first;
		m_nStringDataSize += (int)sString.length() + 1;
	}

	// Return the index of the string in the string table.
	return (*itStringEntry).second;
}
