#ifndef __FBFBX_H__
#define __FBFBX_H__
/**************************************************************************
 Copyright (c) 1994 - 2006 Autodesk, Inc. and/or its licensors.
 All Rights Reserved.
 
 The coded instructions, statements, computer programs, and/or related 
 material (collectively the "Data") in these files contain unpublished 
 information proprietary to Autodesk, Inc. and/or its licensors, which is 
 protected by Canada and United States of America federal copyright law 
 and by international treaties.
 
 The Data may not be disclosed or distributed to third parties, in whole 
 or in part, without the prior written consent of Autodesk, Inc. 
 ("Autodesk").
 
 THE DATA IS PROVIDED "AS IS" AND WITHOUT WARRANTY.
 ALL WARRANTIES ARE EXPRESSLY EXCLUDED AND DISCLAIMED. AUTODESK MAKES NO 
 WARRANTY OF ANY KIND WITH RESPECT TO THE DATA, EXPRESS, IMPLIED OR 
 ARISING BY CUSTOM OR TRADE USAGE, AND DISCLAIMS ANY IMPLIED WARRANTIES 
 OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR 
 PURPOSE OR USE. WITHOUT LIMITING THE FOREGOING, AUTODESK DOES NOT 
 WARRANT THAT THE OPERATION OF THE DATA WILL BE UNINTERRUPTED OR ERROR 
 FREE.
 
 IN NO EVENT SHALL AUTODESK, ITS AFFILIATES, PARENT COMPANIES, LICENSORS 
 OR SUPPLIERS ("AUTODESK GROUP") BE LIABLE FOR ANY LOSSES, DAMAGES OR 
 EXPENSES OF ANY KIND (INCLUDING WITHOUT LIMITATION PUNITIVE OR MULTIPLE 
 DAMAGES OR OTHER SPECIAL, DIRECT, INDIRECT, EXEMPLARY, INCIDENTAL, LOSS 
 OF PROFITS, REVENUE OR DATA, COST OF COVER OR CONSEQUENTIAL LOSSES OR 
 DAMAGES OF ANY KIND), HOWEVER CAUSED, AND REGARDLESS OF THE THEORY OF 
 LIABILITY, WHETHER DERIVED FROM CONTRACT, TORT (INCLUDING, BUT NOT 
 LIMITED TO, NEGLIGENCE), OR OTHERWISE, ARISING OUT OF OR RELATING TO THE 
 DATA OR ITS USE OR ANY OTHER PERFORMANCE, WHETHER OR NOT AUTODESK HAS 
 BEEN ADVISED OF THE POSSIBILITY OF SUCH LOSS OR DAMAGE.
 
**************************************************************************/

/**	\file fbfbx.h
*	FBX file format interface.
*	This file contains the necessary data structures and classes to
*	interface with the FBX file format, allowing the user to save device
*	settings, tool preferences and other customizable features in a
*	scene.
*/

#include <kaydaradef.h>
#ifndef FBSDK_DLL 
	#define FBSDK_DLL K_DLLIMPORT
#endif

#include <fbsdk/fbcomponent.h>

#ifdef FBSDKUseNamespace
	namespace FBSDKNamespace {
#endif

////////////////////////////////////////////////////////////////////////////////////
// FBFbx
////////////////////////////////////////////////////////////////////////////////////
FB_FORWARD( FBScene );
FB_FORWARD( FBCharacter );
FB_FORWARD( FBPlotOptions );
FB_DEFINE_COMPONENT( FBSDK_DLL, Scene );

__FB_FORWARD( FBFbxObject );
__FB_FORWARD( FBModel );

/** FBX file interface.
*	When an object is stored/retrieved, it receives a callback 
*	(FbxStore/FbxRetrieve). This callback receives an object of type
*	FBFbxObject as a parameter in order to interact with the FBX file.
*/
class FBSDK_DLL FBFbxObject : public FBComponent {
	__FBClassDeclare( FBFbxObject,FBComponent );
public:
	/**	Constructor.
    *   Client code cannot instantiate objects of this class. Instances of
    *   FBFbxObject are passed as parameters for the methods FbxStore and
    *   FbxRetrieve of some Open Reality classes.
	*	\param	pObject	For internal use only.
	*/
	FBFbxObject(HIObject pObject);

	/**	Begin writing a field named \e pFieldName.
	*	\param	pFieldName	Name to give to field in FBX file.
	*/
	void FieldWriteBegin(char *pFieldName);

	//!	End of field writing.
	void FieldWriteEnd();

	/**	Begin reading from a field named \e pFieldName.
	*	\param	pFieldName	Name of field to read from.
	*	\return \b true if the field was found.
	*/
	bool FieldReadBegin(char *pFieldName);
	
	//! End of field reading.
	void FieldReadEnd();

	/**	Get the number of items to read for the current field.
	*	\return The number of field items for the current field.
	*/
	int	 FieldReadGetCount();

	//@{
	/**	Write data to the current field.
	*	\param	pValue	Value (int,char,double) to write to current field.
	*/
	void	FieldWriteI( int pValue );
	void	FieldWriteC( char *pValue );
	void	FieldWriteD( double pValue );
	//@}

	//@{
	/**	Write data to field \e pFieldName.
	*	\param	pFieldName	Name of the field to write data to.
	*	\param	pValue	Value (int,char,double) to write to field.
	*/
	void	FieldWriteI( char *pFieldName, int pValue );
	void	FieldWriteC( char *pFieldName, char *pValue );
	void	FieldWriteD( char *pFieldName, double pValue );
	//@}

	//@{
	/**	Read data from the current field.
	*	\return	Value (int,char,double) to read from current field.
	*/
	int		FieldReadI();
	char   *FieldReadC();
	double	FieldReadD();
	//@}

	//@{
	/**	Read data from field \e pFieldName.
	*	\param	pFieldName	Name of the field to read data from.
	*	\return	Value (int,char,double) to read from field.
	*/
	int		FieldReadI(char *pFieldName);
	char   *FieldReadC(char *pFieldName);
	double	FieldReadD(char *pFieldName);
	//@}

	/**	Write an object reference (handle) to the field \e pFieldName.
	*	\param	pFieldName	Field name to assign to handle.
	*	\param	pObject		Handle to object to write to FBX.
	*/
	void   FieldWriteObjectReference (char *pFieldName,HFBComponent pObject);

	/**	Write an object reference (handle) to the current field.
	*	\param	pObject		Handle to object to write to FBX.
	*/
	void   FieldWriteObjectReference (HFBComponent pObject);

	/**	Read an object reference from the field \e pFieldName.\
    *   The returned reference is a HIObject which is an internal application object.
    *   Calling 'FBGetFBComponent()' will provide a FBComponent wrapper object with which it is possible to interact.
	*	\param	pFieldName	Field name to read from.
	*	\return	Handle to internal object.
	*/
	HIObject FieldReadObjectReference (char * pFieldName);

	/**	Read an object reference from the current field.
    *   The returned reference is a HIObject which is an internal application object.
    *   Calling 'FBGetFBComponent()' will provide a FBComponent wrapper object with which it is possible to interact.
	*	\return	Handle to internal object.
	*/
	HIObject FieldReadObjectReference ();

	/**	Begin writing a block in the FBX file.
	*	A block in an FBX file is to re-group different field together, in the same way a
	*	a namespace functions in C++. This permits the use of fields with the same name in 
	*	the FBX file, which is impossible unless they are in different blocks of data.
	*/
	void	FieldWriteBlockBegin ();

	/**	Begin writing a block in the FBX file.
	*	A block in an FBX file is to re-group different field together, in the same way a
	*	a namespace functions in C++. This permits the use of fields with the same name in 
	*	the FBX file, which is impossible unless they are in different blocks of data.
	*	\param pFieldName	Name to give to block.
	*/
	void	FieldWriteBlockBegin (char *pFieldName);

	//!	Finish writing a block of data.
	void	FieldWriteBlockEnd   ();

	/**	Is current reading a block of data?
	*	\return \b true if current data is a block.
	*/	
	bool	FieldReadIsBlock   ();

	/**	Begin reading a block of data.
	*	\return	\b true if current read is from a block.
	*/
	bool	FieldReadBlockBegin();

	//!	Finish reading data from a block
	void	FieldReadBlockEnd  ();

	/**	Get the full file path for \e pFilename.
	*	\param	pFilename	Name of file to get path for.
	*	\return	Path of file \e pFilename.
	*/
	char   *GetFullFilePath(char *pFilename);
};

__FB_FORWARD( FBFbxTake	);

/** Take class for Fbx import.
*	Take info for import of data from Fbx file.
*/
class FBSDK_DLL FBFbxTake : public FBComponent
{
	__FBClassDeclare( FBFbxTake, FBComponent );
public:
	FBFbxTake();
	
	FBPropertyBool		Import;			//!< <b>Read Write Property:</b> Import take?
	FBPropertyString	Name;			//!< <b>Read Write Property:</b> Name of take.
	FBPropertyString	Description;	//!< <b>Read Write Property:</b> Take description.
	FBPropertyString	ImportName;		//!< <b>Read Write Property:</b> Destination name for take upon import.

	friend class FBFbxManager;
};
FB_DEFINE_ARRAY( FBSDK_DLL, FbxTake );

typedef class FBSDK_DLL FBArrayTemplate<HFBModel> FBModelList;

/** Enumeration that describe the different actions available on a scene element
*   depending on the current context.
*/
enum FBElementAction { 
    kFBElementActionSave   ,       //!< Save the element (when saving).
    kFBElementActionAppend ,       //!< Append the elements to the current scene elements (when loading or merging).
    kFBElementActionMerge  ,       //!< Merge the elements from the file in the current scene (when merging).
    kFBElementActionDiscard,       //!< Do not consider the element (when loading, merging and saving).
};
FB_DEFINE_ENUM( FBSDK_DLL, ElementAction ); // FBPropertyElementAction


/** Enumeration that describe the different actions available on the animation of scene elements
*   depending on the current context.
*/
enum FBAnimationAction { 
    kFBAnimationActionConsider,       //!< Use the animation in the file (when loading or merging) or save it to the output file (saving).
    kFBAnimationActionDiscard ,       //!< Do not consider the animation.
};
FB_DEFINE_ENUM( FBSDK_DLL, AnimationAction ); // FBPropertyAnimationAction


/** This enumeration indicate the current state of the FBFbxManager.
*   The manager can only do one action at a time.
*/
enum FBFbxManagerState { kFBFbxManagerStateIdle   , //!< The manager is ready to start a process.
                         kFBFbxManagerStateLoading, //!< A load process is in progress.
                         kFBFbxManagerStateMerging, //!< A merge process is in progress.
                         kFBFbxManagerStateSaving , //!< A save process is in process.
};
FB_DEFINE_ENUM( FBSDK_DLL, FbxManagerState ); // FBPropertyFbxManagerState


/** This enumeration indicate the how to set the take start and end points on after a load.
*/
enum FBTakeSpanOnLoad { kFBLeaveAsIs     , //!< Use the current take's start and end point as defined before the load.
                        kFBImportFromFile, //!< Set the current take's span according what is set in the loaded file.
                        kFBFrameAnimation, //!< Have the take's span match the first and last key in the take.
};
FB_DEFINE_ENUM( FBSDK_DLL, TakeSpanOnLoad ); // FBPropertyTakeSpanOnLoad


/** This enumeration is used to choose how to load an animation file on a character.
*/
enum FBFbxManagerLoadAnimationMethod { kFBFbxManagerLoadConnect       , //!< Only connect the loaded character as an input.
               kFBFbxManagerLoadCopy          , //!< Copy keys from loaded character to target character.
               kFBFbxManagerLoadRetarget      , //!< Retarget (copy and correct) keys from loaded character to target character.
               kFBFbxManagerLoadPlotIfSampled , //!< If loaded animation seems sampled, plot animation from loaded character to target character; else retarget.
               kFBFbxManagerLoadPlot          , //!< Plot animation from loaded character to target character.
};


__FB_FORWARD( FBFbxManager	);
/**	FBX interface for loading/saving files into the application.
*	This class permits the loading and saving of scene without
*	using the user-interface. Each process is split into three steps, and
*	between the first and second, the developer can setup the different options for each
*	operation. For instance, when loading, LoadBegin() is called, the options
*	for the file can then be set, and then Load() is called to begin the
*	the actual import of the data from the scene.
*
*   \warning The manager cannot perform 2 actions at the same time!
*/
class FBSDK_DLL FBFbxManager : public FBComponent 
{
	//--- Open Reality Declaration
	__FBClassDeclare( FBFbxManager, FBComponent );
public:
	//! Constructor.
	FBFbxManager();


    //---------------------------------------
    // File Load.
    //---------------------------------------

	/**	Begin the file load process.
	*	This is the first call when loading a file via the SDK IN MEMORY. 
	*	\param	pFileAddress		Adress where the file is.
	*	\param	pFileMemoryLength	Length of the File.
    *   \param  pMerge      Indicate if we are simply loading or merging.
	*	\return	\b true if successful.
	*/
	bool LoadBegin( void* pFileAddress, kULong pFileMemoryLength, bool pMerge = false );

	/**	Begin the file load process.
	*	This is the first call when loading a file via the SDK. 
	*	\param	pFilename	Name of file to load.
    *   \param  pMerge      Indicate if we are simply loading or merging.
	*	\return	\b true if successful.
	*/
	bool LoadBegin( char* pFilename, bool pMerge = false );

	/**	File load process.
	*	If the load is asynchronous, this will return right away, otherwise, this
	*	is the actual loading process of the file.
	*	\return	\b true if successful.
	*/
	bool Load();

	/**	Query the file load process.
	*	\return \b true if file load is finished.
	*/
	bool LoadIsCompleted();

	/**	End the file load process.
	*	Regardless of the type of load (asynchronous or not, this will wait until the
	*	load is finished.
	*	\return	\b true if successful.
	*/
	bool LoadEnd();


    //---------------------------------------
    // File Save.
    //---------------------------------------

	/**	Begin the file save process.
	*	This is the first call when saving a file via the SDK.
	*	\param	pFilename	Name of file to save.
	*	\return	\b true if successful.
	*/
	bool SaveBegin( char* pFilename );

	/**	Actual file save process.
	*	\return	\b true if successful.
	*/
	bool Save();

	/**	End the file save process.
	*	\return	\b true if successful.
	*/
	bool SaveEnd();


#if !defined(K_NO_CHARACTER)
    //---------------------------------------
    // Character Retargeting.
    //---------------------------------------

    /** Save the rig and its animation in a file.
    *   \param  pFileName  File name.
    *   \param  pCharacter  Character to save.
    *   \param  pSaveCharacter  Set to true if the character should be saved.
    *   \param  pSaveRig  Set to true if the rig (and its children) should be saved.
    *   \param  pSaveExtensions  Set to true if the character extensions (and their children) should be saved.
	*/
    bool SaveCharacterRigAndAnimation( char* pFileName, HFBCharacter pCharacter, bool pSaveCharacter, bool pSaveRig, bool pSaveExtensions );

    /** Load a rig and its animation from a file.
    *   \param  pFileName  File name.
    *   \param  pCharacter  Target character.
    *   \param  pReplaceRig  Set to true if the current rig should be replaced by the rig in the file.
    *   \param  pCopyMissingExtensions  Set to true if the character extensions on the rig in the file should be copied to the target rig.
    *   \param  pUpdateLimits  Set to true if we should change the limits on the target rig.
    *   \param  pIgnoreConflicts  Set to true to ignore conflicts between objects in character extensions and objects in the scene. Conflicting objects will be merged in the extension.
    *   \param  pTransferMethod  How should the animation should be tranfered on the target rig.
    *   \param  pPlotOptions  If the animation should be plotted on the target rig, these plot options will be used. Set to NULL if animation will not be plotted.
    *   \param  pProcessAnimationOnExtensions  Set to true if animation on character extensions should also be transferred.
    *   \param  pRetargetOnBaseLayer  If the transfer method is retarget, set this parameter to control where the retarget correction will be made (on base layer or on another layer).
    *   \param  pResetHierarchy  Set to true if we should reset the character hierarchy.
    *   \return \b true if successful.
	*/
    bool LoadAnimationOnCharacter( char* pFileName, HFBCharacter pCharacter, bool pReplaceRig, bool pCopyMissingExtensions, bool pUpdateLimits, bool pIgnoreConflicts,
        FBFbxManagerLoadAnimationMethod pTransferMethod, HFBPlotOptions pPlotOptions, bool pProcessAnimationOnExtensions, bool pRetargetOnBaseLayer, bool pResetHierarchy );

#endif
    //---------------------------------------
    // Options related utilities.
    //---------------------------------------

    /** Use the options saved to a history config file to setup the current options.
    *   \param pHistoryFile Indicate the name of the history file to use. If NULL the default system file will be used.
    *                       This file will be created/read from the folder bin/config. Should the first character be a '@'
    *                       it will be replaced by the machine name. Using the default value is equivalent to using the
    *                       string "@History.txt" as an argument.
    *   \return \b true if the options were indeed set from the history file.
    */
    bool SetOptionsFromHistory( char* pHistoryFile = NULL );

    /** Update a history config file with the current options.
    *   \param pHistoryFile Indicate the name of the history file to use. If NULL the default system file will be used.
    *                       This file will be created/read from the folder bin/config. Should the first character be a '@'
    *                       it will be replaced by the machine name. Using the default value is equivalent to using the
    *                       string "@History.txt" as an argument.
    *   \return \b true if the options were indeed saved to the history file.
    */
    bool SetHistoryFromOptions( char* pHistoryFile = NULL );

    FBPropertyFbxManagerState   State;                  //!< <b>Read Only Property:</b> Current state of the manager.


    /** @name General options.
    */
    //@{
	FBPropertyBool				Async;					//!< <b>Read Write Property:</b> Use asynchronous operation. False by default.
    //@}


    /** @name Load specific options.
    *   \warning These options are not present in the history file. Thus will not be affected by 'SetOptionsFromHistory()' and 'SetHistoryFromOptions()'.
    */
    //@{
	FBPropertyBool				ClearBeforeLoad;		//!< <b>Read Write Property:</b> Clear the scene before load. False by default.
    FBPropertyTakeSpanOnLoad    TakeSpan;               //!< <b>Read Write Property:</b> Indicate how the take start and end point should be set. By default it is read from the file.
    //@}


    /** @name Save specific options.
    *   All of these options are false by default, except for 'EmbedMedia' which is true by default.
    *   \warning These options are not present in the history file. Thus will not be affected by 'SetOptionsFromHistory()' and 'SetHistoryFromOptions()'.
    */
    //@{
    FBPropertyBool              ConvertImageToTiff;     //!< <b>Read Write Property:</b> Only considered if EmbedMedia is true. Will convert the embeded media to tiff format when possible.
    FBPropertyBool              EmbedMedia;             //!< <b>Read Write Property:</b> Embed all media in the FBX file itself. When saving in ASCII mode it is not possible to embed media.
	FBPropertyBool				Selected;				//!< <b>Read Write Property:</b> Indicate that only the selected models will be saved.
    FBPropertyBool              UseASCIIFormat;         //!< <b>Read Write Property:</b> Indicate if the resulting FBX file will be in binary or ASCII mode.
    //@}


    /** @name Scene settings options.
    */
    //@{
    FBPropertyBool				BaseCameras;            //!< <b>Read Write Property:</b> Consider base camera settings.
	FBPropertyBool				CameraSwitcherSettings; //!< <b>Read Write Property:</b> Consider camera switcher settings.
    FBPropertyBool              CurrentCameraSettings;  //!< <b>Read Write Property:</b> Consider current camera settings.
	FBPropertyBool				GlobalLightingSettings; //!< <b>Read Write Property:</b> Consider global Lighting settings.
    FBPropertyBool				TransportSettings;      //!< <b>Read Write Property:</b> Consider transport control settings.
    //@}


    /** @name Element settings options.
    */
    //@{
    FBPropertyElementAction     Actors;               //!< <b>Read Write Property:</b> Handling of the Actors elements.
    FBPropertyElementAction     ActorFaces;           //!< <b>Read Write Property:</b> Handling of the Actor Faces elements.
    FBPropertyElementAction     Audio;                //!< <b>Read Write Property:</b> Handling of the Audio elements.
    FBPropertyElementAction     Cameras;              //!< <b>Read Write Property:</b> Handling of the Cameras elements.
    FBPropertyElementAction     Characters;           //!< <b>Read Write Property:</b> Handling of the Characters elements.
    FBPropertyElementAction     CharacterFaces;       //!< <b>Read Write Property:</b> Handling of the Character Faces elements.
    FBPropertyElementAction     CharacterExtensions;  //!< <b>Read Write Property:</b> Handling of the Character Extensions.
    FBPropertyElementAction     Constraints;          //!< <b>Read Write Property:</b> Handling of the Constraints elements.
    FBPropertyElementAction     Devices;              //!< <b>Read Write Property:</b> Handling of the Devices elements.
    FBPropertyElementAction     Groups;               //!< <b>Read Write Property:</b> Handling of the Groups elements.
    FBPropertyElementAction     Lights;               //!< <b>Read Write Property:</b> Handling of the Lights elements.
    FBPropertyElementAction     Materials;            //!< <b>Read Write Property:</b> Handling of the Materials elements.
    FBPropertyElementAction     Models;               //!< <b>Read Write Property:</b> Handling of the Models elements.
    FBPropertyElementAction     Notes;                //!< <b>Read Write Property:</b> Handling of the Notes elements.
    FBPropertyElementAction     Poses;                //!< <b>Read Write Property:</b> Handling of the Poses elements.
    FBPropertyElementAction     Scripts;              //!< <b>Read Write Property:</b> Handling of the Scripts elements.
    FBPropertyElementAction     Sets;                 //!< <b>Read Write Property:</b> Handling of the Sets elements.
    FBPropertyElementAction     Shaders;              //!< <b>Read Write Property:</b> Handling of the Shaders elements.
    FBPropertyElementAction     Story;                //!< <b>Read Write Property:</b> Handling of the Story elements.
    FBPropertyElementAction     Textures;             //!< <b>Read Write Property:</b> Handling of the Textures elements.
    FBPropertyElementAction     Video;                //!< <b>Read Write Property:</b> Handling of the Video elements.
    //@}


    /** @name Element animation settings options.
    *   Not all elements type have animation associated with them.
    */
    //@{
    FBPropertyAnimationAction   ActorFacesAnimation;        //!< <b>Read Write Property:</b> Handling of the Actor Faces animation.
    FBPropertyAnimationAction   CamerasAnimation;           //!< <b>Read Write Property:</b> Handling of the Cameras animation.
    FBPropertyAnimationAction   CharactersAnimation;        //!< <b>Read Write Property:</b> Handling of the Characters animation.
    FBPropertyAnimationAction   CharacterFacesAnimation;    //!< <b>Read Write Property:</b> Handling of the Character Faces animation.
    FBPropertyAnimationAction   ConstraintsAnimation;       //!< <b>Read Write Property:</b> Handling of the Constraints animation.
    FBPropertyAnimationAction   DevicesAnimation;           //!< <b>Read Write Property:</b> Handling of the Devices animation.
    FBPropertyAnimationAction   LightsAnimation;            //!< <b>Read Write Property:</b> Handling of the Lights animation.
    FBPropertyAnimationAction   MaterialsAnimation;         //!< <b>Read Write Property:</b> Handling of the Materials animation.
    FBPropertyAnimationAction   ModelsAnimation;            //!< <b>Read Write Property:</b> Handling of the Models animation.
    FBPropertyAnimationAction   NotesAnimation;             //!< <b>Read Write Property:</b> Handling of the Notes animation.
    FBPropertyAnimationAction   ShadersAnimation;           //!< <b>Read Write Property:</b> Handling of the Shaders animation.
    FBPropertyAnimationAction   TexturesAnimation;          //!< <b>Read Write Property:</b> Handling of the Textures animation.
    //@}


    FBArrayFbxTake				Takes;					//!< <b>List:</b> Scene takes.

	FBPropertyScene				Scene;					//!<  <b>Read Write Property:</b> This is the scene in wich you want to load
};

#ifdef FBSDKUseNamespace
	}
#endif

#endif
