#ifndef __FBPLUG_H__
#define __FBPLUG_H__
/**************************************************************************
 Copyright (c) 1994 - 2006 Autodesk, Inc. and/or its licensors.
 All Rights Reserved.
 
 The coded instructions, statements, computer programs, and/or related 
 material (collectively the "Data") in these files contain unpublished 
 information proprietary to Autodesk, Inc. and/or its licensors, which is 
 protected by Canada and United States of America federal copyright law 
 and by international treaties.
 
 The Data may not be disclosed or distributed to third parties, in whole 
 or in part, without the prior written consent of Autodesk, Inc. 
 ("Autodesk").
 
 THE DATA IS PROVIDED "AS IS" AND WITHOUT WARRANTY.
 ALL WARRANTIES ARE EXPRESSLY EXCLUDED AND DISCLAIMED. AUTODESK MAKES NO 
 WARRANTY OF ANY KIND WITH RESPECT TO THE DATA, EXPRESS, IMPLIED OR 
 ARISING BY CUSTOM OR TRADE USAGE, AND DISCLAIMS ANY IMPLIED WARRANTIES 
 OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR 
 PURPOSE OR USE. WITHOUT LIMITING THE FOREGOING, AUTODESK DOES NOT 
 WARRANT THAT THE OPERATION OF THE DATA WILL BE UNINTERRUPTED OR ERROR 
 FREE.
 
 IN NO EVENT SHALL AUTODESK, ITS AFFILIATES, PARENT COMPANIES, LICENSORS 
 OR SUPPLIERS ("AUTODESK GROUP") BE LIABLE FOR ANY LOSSES, DAMAGES OR 
 EXPENSES OF ANY KIND (INCLUDING WITHOUT LIMITATION PUNITIVE OR MULTIPLE 
 DAMAGES OR OTHER SPECIAL, DIRECT, INDIRECT, EXEMPLARY, INCIDENTAL, LOSS 
 OF PROFITS, REVENUE OR DATA, COST OF COVER OR CONSEQUENTIAL LOSSES OR 
 DAMAGES OF ANY KIND), HOWEVER CAUSED, AND REGARDLESS OF THE THEORY OF 
 LIABILITY, WHETHER DERIVED FROM CONTRACT, TORT (INCLUDING, BUT NOT 
 LIMITED TO, NEGLIGENCE), OR OTHERWISE, ARISING OUT OF OR RELATING TO THE 
 DATA OR ITS USE OR ANY OTHER PERFORMANCE, WHETHER OR NOT AUTODESK HAS 
 BEEN ADVISED OF THE POSSIBILITY OF SUCH LOSS OR DAMAGE.
 
**************************************************************************/

/** \file fbplug.h
*   Definition of the class FBPlug and related enums and utility functions.
*   All the Open Reality objects that expose application objects will inherit
*   from FBPlug.
*/

#include <kaydaradef.h>

#ifndef FBSDK_DLL 
    #define FBSDK_DLL K_DLLIMPORT
#endif

// Global includes
#include <object/i/icallback.h>
#include <fbsdk/fbdefines.h>

#ifdef FBSDKUseNamespace
    namespace FBSDKNamespace {
#endif

////////////////////////////////////////////////////////////////////////////////////
// FBPlug
////////////////////////////////////////////////////////////////////////////////////
FB_FORWARD( FBPlug );

//! Connection types available between plugs.
enum FBConnectionType
{
    kFBConnectionTypeNone,          //!< Default connection type.
    kFBConnectionTypeSystem,        //!< System connection type.
    kFBConnectionTypeUnidirectional //!< Unidirectional connection type.
};

/** Connections Basic Open Reality SDK Element.
*   Most elements that are available in the SDK inherit from this base class since
*   FBComponent and FBProperty inherit from FBPlug. Basically, all objects can be
*	connected together because they are all "plugs". To simplify the graph, you
*	can think of a "source" connection as a child, and a "destination" connection
*	as a parent. Also, it is correct to assume that a source affect/work on its
*	destination. For example, a shader applyed on an object would be seen as the
*	source while the object is the destination. So FBPlug is a set of functions
*	that enables you to control those connections with flexibility and ease.
*/
class FBSDK_DLL FBPlug : public ICallback
{
public:
    //! Constructor
    FBPlug();

    //! Destructor
    virtual ~FBPlug();

    // \internal Return an internal pointer.
    virtual HIObject GetHIObject();

    // \internal Assign component to an internal pointer.
    virtual bool SetHIObject( HIObject pObject, bool pSDKComponent );

    // \internal Return whether or not item is an SDK component.
    bool IsSDKComponent();

	/** Add a source connection.
	*   \param pSrc				Source plug.
	*   \param pConnectionType  Type of connection, taken from FBConnectionType. Default value should work in all cases.
	*   \return                 A boolean indicating success (True) or failure (False).
	*/
    bool                ConnectSrc( HFBPlug pSrc, FBConnectionType pConnectionType=kFBConnectionTypeNone );

	/** Add a destination connection.
	*   \param pDst				Destination plug.
	*   \param pConnectionType  Type of connection, taken from FBConnectionType. Default value should work in all cases.
	*   \return                 A boolean indicating success (True) or failure (False).
	*/
    bool                ConnectDst( HFBPlug pDst, FBConnectionType pConnectionType=kFBConnectionTypeNone );

	/** Add a source connection.
	*	\param pDst_SrcIndex	Index that tells where to add this source connection in the destination's connection list.
	*   \param pSrc				Source plug.
	*   \param pConnectionType  Type of connection, taken from FBConnectionType. Default value should work in all cases.
	*   \return                 A boolean indicating success (True) or failure (False).
	*/
    bool                ConnectSrcAt( int pDst_SrcIndex, HFBPlug pSrc, FBConnectionType pConnectionType=kFBConnectionTypeNone );

	/** Add a destination connection.
	*	\param pSrc_DstIndex	Index that tells where to add this destination connection in the source's connection list.
	*   \param pDst				Destination plug.
	*   \param pConnectionType  Type of connection, taken from FBConnectionType. Default value should work in all cases.
	*   \return                 A boolean indicating success (True) or failure (False).
	*/
    bool                ConnectDstAt( int pSrc_DstIndex, HFBPlug pDst, FBConnectionType pConnectionType=kFBConnectionTypeNone );

	/** Make connection between to plug, specifying the connection index. (Deprecated)
	*   \param pSrc				Source plug.
	*	\param pSrc_DstIndex	Index that tells where to add this destination connection in the source's connection list.
	*   \param pDst				Destination plug.
	*	\param pDst_SrcIndex	Index that tells where to add this source connection in the destination's connection list.
	*   \param pConnectionType  Type of connection, taken from FBConnectionType. Default value should work in all cases.
	*   \return                 A boolean indicating success (True) or failure (False).
	*/
    bool                ConnectAt( HFBPlug pSrc, int pSrc_DstIndex, HFBPlug pDst, int pDst_SrcIndex, FBConnectionType pConnectionType=kFBConnectionTypeNone );

	/** Remove a destination connection.
	*   \param pDst	Destination plug.
	*   \return		A boolean indicating success (True) or failure (False).
	*/
    bool                DisconnectDst( HFBPlug pDst );

	/** Remove a destination connection.
	*   \param pSrc	Source plug.
	*   \return		A boolean indicating success (True) or failure (False).
	*/
    bool                DisconnectSrc( HFBPlug pSrc );

	/** Remove all source connections. */
    void                DisconnectAllSrc();

	/** Remove all destination connections. */
    void                DisconnectAllDst();

	/** Remove a destination connection at a specifyed index.
	*   \param pIndex	Destination plug index.
	*   \return			A boolean indicating success (True) or failure (False).
	*/
    bool                DisconnectDstAt( int pIndex );

	/** Remove a source connection at a specifyed index.
	*   \param pIndex	Source plug index.
	*   \return			A boolean indicating success (True) or failure (False).
	*/
    bool                DisconnectSrcAt( int pIndex );

	/** Replace a destination connection at a specifyed index.
	*   \param pIndex	Destination plug index.
	*   \param pDst		Plug that will replace the other at index.
	*   \return			A boolean indicating success (True) or failure (False).
	*/
    bool                ReplaceDstAt( int pIndex, HFBPlug pDst );

	/** Replace a source connection at a specifyed index.
	*   \param pIndex	Source plug index.
	*   \param pSrc		Plug that will replace the other at index.
	*   \return			A boolean indicating success (True) or failure (False).
	*/
    bool                ReplaceSrcAt( int pIndex, HFBPlug pSrc );

	/** Swap source connection at index A with source connection at index B.
	*   \param pIndexA	Plug index.
	*   \param pIndexB	Other plug index.
	*   \return			A boolean indicating success (True) or failure (False).
	*/
    bool                SwapSrc( int pIndexA, int pIndexB );

	/** Move source connection at pIndex to pAtIndex.
	*   \param pIndex	Plug current index.
	*   \param pAtIndex	Plug new index.
	*   \return			A boolean indicating success (True) or failure (False).
	*	\remarks			This is not like the swap function since the connection at pAtIndex is untouched.
	*/
    bool                MoveSrcAt( int pIndex, int pAtIndex );

	/** Move source connection at pIndex to pAtIndex.
	*   \param pSrc		Plug.
	*   \param pAtSrc	Plug that mark where we want to insert (will insert before this one).
	*   \return			A boolean indicating success (True) or failure (False).
	*	\remarks			This is not like the swap function since the connection at pAtSrc is untouched.
	*/
    bool                MoveSrcAt( HFBPlug pSrc, HFBPlug pAtSrc );

	/** Get source connection count.
	*   \return	Total sources connections count.
	*/
    int                 GetSrcCount();

	/** Get a source connection's plug at specifyed index.
	*	\param	pIndex	Index of the source connection's plug.
	*   \return			Source plug at specifyed index.
	*/
    HFBPlug             GetSrc( int pIndex );

	/** Get a source connection's type at specifyed index.
	*	\param	pIndex	Index of the source connection's type.
	*   \return			Source connection's type at specifyed index.
	*/
    FBConnectionType    GetSrcType( int pIndex );

	/** Get destination connection count.
	*   \return	Total destinations connections count.
	*/
    int                 GetDstCount();

	/** Get a destination connection's plug at specifyed index.
	*	\param	pIndex	Index of the destination connection's plug.
	*   \return			Destination plug at specifyed index.
	*/
    HFBPlug             GetDst( int pIndex );

	/** Get a destination connection's type at specifyed index.
	*	\param	pIndex	Index of the destination connection's type.
	*   \return			Destination connection's type at specifyed index.
	*/
    FBConnectionType    GetDstType( int pIndex );

	/** Begins a change on multiple plugs.
	*   \return	A boolean indicating success (True) or failure (False).
	*/
    bool                BeginChange();

	/** Ends a change on multiple plugs. */
    void                EndChange();

	/** Get the owner of this plug. Very useful for properties since they are plugs too.
	*   \return	The owner of this plug.
	*/
    HFBPlug             GetOwner();

	/** Get the owner count of this plug. (Obsolete)
	*   \return	The owner count of this plug (will always return 1).
	*/
    int                 GetOwnerCount();

	/** Get the owned plug at specifyed index.
	*	\param	pIndex	Index of the owned plug to get.
	*   \return			The owned plug at specifyed index.
	*/
    HFBPlug             GetOwned( int pIndex );

	/** Get the owned plug count.
	*   \return	The owned plug count.
	*/
    int                 GetOwnedCount();

	//System vars
    virtual char*       ClassName();
    virtual bool        Is( int pTypeId );

    static int          TypeInfo;

protected:
    /** Actual destuctor for a FBPlug.
    *   It should never be called from client code. This is meant for internal use only!
    */
    virtual void        FBDelete();

    static int          mGlobalTypeInfo;
    HIObject            mObject;
    bool                mSDKComponent;
};


////////////////////////////////////////////////////////////////////////////////////
// Connection related utility functions.
////////////////////////////////////////////////////////////////////////////////////
/** Request the connection two FBPlug objects.
*   \param pSrc             Source plug.
*   \param pDst             Destination plug.
*   \param pConnectionType  Type of connection, taken from FBConnectionType.
*   \return                 A boolean indicating success (True) or failure (False).
*/
FBSDK_DLL bool FBConnect( HFBPlug pSrc, HFBPlug pDst, FBConnectionType pConnectionType=kFBConnectionTypeNone );

/** Connect two FBPlug objects.
*   \param pSrc             Source plug.
*   \param pDst             Destination plug.
*   \return                 A boolean indicating success (True) or failure (False).
*/
FBSDK_DLL bool FBDisconnect( HFBPlug pSrc, HFBPlug pDst );


#ifdef FBSDKUseNamespace
    }
#endif

#endif
