#ifndef __FBRENDERER_H__
#define __FBRENDERER_H__
/**************************************************************************
 Copyright (c) 1994 - 2006 Autodesk, Inc. and/or its licensors.
 All Rights Reserved.
 
 The coded instructions, statements, computer programs, and/or related 
 material (collectively the "Data") in these files contain unpublished 
 information proprietary to Autodesk, Inc. and/or its licensors, which is 
 protected by Canada and United States of America federal copyright law 
 and by international treaties.
 
 The Data may not be disclosed or distributed to third parties, in whole 
 or in part, without the prior written consent of Autodesk, Inc. 
 ("Autodesk").
 
 THE DATA IS PROVIDED "AS IS" AND WITHOUT WARRANTY.
 ALL WARRANTIES ARE EXPRESSLY EXCLUDED AND DISCLAIMED. AUTODESK MAKES NO 
 WARRANTY OF ANY KIND WITH RESPECT TO THE DATA, EXPRESS, IMPLIED OR 
 ARISING BY CUSTOM OR TRADE USAGE, AND DISCLAIMS ANY IMPLIED WARRANTIES 
 OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR 
 PURPOSE OR USE. WITHOUT LIMITING THE FOREGOING, AUTODESK DOES NOT 
 WARRANT THAT THE OPERATION OF THE DATA WILL BE UNINTERRUPTED OR ERROR 
 FREE.
 
 IN NO EVENT SHALL AUTODESK, ITS AFFILIATES, PARENT COMPANIES, LICENSORS 
 OR SUPPLIERS ("AUTODESK GROUP") BE LIABLE FOR ANY LOSSES, DAMAGES OR 
 EXPENSES OF ANY KIND (INCLUDING WITHOUT LIMITATION PUNITIVE OR MULTIPLE 
 DAMAGES OR OTHER SPECIAL, DIRECT, INDIRECT, EXEMPLARY, INCIDENTAL, LOSS 
 OF PROFITS, REVENUE OR DATA, COST OF COVER OR CONSEQUENTIAL LOSSES OR 
 DAMAGES OF ANY KIND), HOWEVER CAUSED, AND REGARDLESS OF THE THEORY OF 
 LIABILITY, WHETHER DERIVED FROM CONTRACT, TORT (INCLUDING, BUT NOT 
 LIMITED TO, NEGLIGENCE), OR OTHERWISE, ARISING OUT OF OR RELATING TO THE 
 DATA OR ITS USE OR ANY OTHER PERFORMANCE, WHETHER OR NOT AUTODESK HAS 
 BEEN ADVISED OF THE POSSIBILITY OF SUCH LOSS OR DAMAGE.
 
**************************************************************************/

/**	\file fbrenderer.h
*	Rendering engine.
*/

#include <kaydaradef.h>
#ifndef FBSDK_DLL 
	#define FBSDK_DLL K_DLLIMPORT
#endif

#if !defined(K_NO_MANIPULATOR)
	#include <fbsdk/fbmanipulator.h>
#endif

#include <fbsdk/fbcomponent.h>
#include <fbsdk/fbmodel.h>
#include <fbcontrols/fbcontrols.h>	// FBInputType

#ifdef FBSDKUseNamespace
	namespace FBSDKNamespace {
#endif

__FB_FORWARD( FBRenderer );
FB_FORWARD( FBView );

FB_DEFINE_COMPONENT( FBSDK_DLL, Renderer );

FB_FORWARD( FBViewingOptions );

//! Model display options.
enum FBDisplayMode {
	kFBDisplayModeDefault = 0,	//!< Use default display mode.
	kFBDisplayModeTexture,		//!< Textures are displayed.
	kFBDisplayModeHardShade,	//!< Hard shading.
	kFBDisplayModeFlatShade,	//!< Flat shading.
	kFBDisplayModeWireFrame,	//!< Wire-frame rendering.
	kFBDisplayModeCount			//!< End of enum, this value indicates the number of display modes available.
};

/** Model display mask
*	This mask determines what types of models are displayed by the renderer.
*/
enum FBDisplayWhat {
	kFBDisplayNone		= 0,			//!< Nothing is displayed.
	kFBDisplayNull		= (1 << 0),		//!< Null models are displayed.
	kFBDisplayMarker	= (1 << 1),		//!< Markers are displayed.
	kFBDisplaySkeleton	= (1 << 2),		//!< Skeletons and bones are displayed.
	kFBDisplayCenter	= (1 << 3),		//!< Centers are displayed.
	kFBDisplayLight		= (1 << 4),		//!< Lights are displayed.
	kFBDisplayCamera	= (1 << 5),		//!< Cameras are displayed.
	kFBDisplay3dIcon	= (1 << 6),		//!< 3D icons are displayed (3D icons are 3D elements that do not exist in the scene).
	kFBDisplayAll		= 0xff			//!< Everything is displayed.
};

//! 3D picking mode.
enum FBPickingMode {
	kFBPickingModeStandard = 0,		//!< Standard picking mode.
	kFBPickingModeXRay,				//!< X-Ray picking mode (obstructed models are displayed in overlay).
	kFBPickingModeModelsOnly,		//!< Models-only mode (no nulls or skeletons are displayed).
	kFBPickingModeCount				//!< End of enum, this valued indicates the number of picking modes available.
};

enum FBZSortMode {
	kFBZSortNone,								
	kFBZSortFirstLayer,							
	kFBZSortFirstLayerFastBufferRegion,
	kFBZSortConvexGeometry,
	kFBZSortConvexGeometryFastBufferRegion,
	kFBZSortFrontToBack,
	kFBZSortBackToFront,
	kFBZSortCount
};

//! Keyboard keys (for input).
enum FBDeviceKeyboardKey {
	kFBDKeyPageUp,			//!< Page Up.
	kFBDKeyPageDown,		//!< Page Down.
	kFBDKeyEnd,				//!< End.
	kFBDKeyHome,			//!< Home.
	kFBDKeyArrowLeft,		//!< Left.
	kFBDKeyArrowUp,			//!< Up.
	kFBDKeyArrowRight,		//!< Right.
	kFBDKeyArrowDown,		//!< Down.
	kFBDKeyReturn,			//!< Return.
	kFBDKeyEscape,			//!< Escape.
	kFBDKeySpace,			//!< Space bar.
	kFBDKey1,				//!< '1'
	kFBDKey2,				//!< '2'
	kFBDKey3,				//!< '3'
	kFBDKey4,				//!< '4'
	kFBDKey5,				//!< '5'
	kFBDKey6,				//!< '6'
	kFBDKey7,				//!< '7'
	kFBDKey8,				//!< '8'
	kFBDKey9,				//!< '9'
	kFBDKey0,				//!< '0'
	kFBDKeyF1,				//!< 'F1'
	kFBDKeyF2,				//!< 'F2'
	kFBDKeyF3,				//!< 'F3'
	kFBDKeyF4,				//!< 'F4'
	kFBDKeyF5,				//!< 'F5'
	kFBDKeyF6,				//!< 'F6'
	kFBDKeyF7,				//!< 'F7'
	kFBDKeyF8,				//!< 'F8'
	kFBDKeyF9,				//!< 'F9'
	kFBDKeyF10,				//!< 'F10'
	kFBDKeyF11,				//!< 'F11'
	kFBDKeyF12				//!< 'F12'
};
FB_DEFINE_ENUM( FBSDK_DLL, DeviceKeyboardKey );

/** Viewing options for rendering.
*	The FBRenderer allows to get and set those options.
*/
class FBSDK_DLL FBViewingOptions
{
  public:
	/**	Get a reference to the display mode.
	*	\return Reference to the current dispaly mode.
	*/
	int& DisplayMode();

	/** Get a reference to the display mask.
	*	\return Reference to the current display mask.
	*/
	int& DisplayWhat();

	/** Get a reference to the picking mode.
	*	\return Reference to the current picking mode.
	*/
	int& PickingMode();

	/** Show Time Code when rendering.
	*/
	bool& ShowTimeCode();

	/** Show Safe Area when rendering.
	*/
	bool& ShowSafeArea();

	/** Show Camera Label when rendering.
	*/
	bool& ShowCameraLabel();

  private:
	FBViewingOptions();
	FBViewingOptions(const FBViewingOptions &);
	FBViewingOptions & operator =(const FBViewingOptions &);
};

FB_DEFINE_ENUM( FBSDK_DLL, ZSortMode );





////////////////////////////////////////////////////////////////////////////
// FBPickInfos
////////////////////////////////////////////////////////////////////////////

/** Picking information structure.
*   This class holds information related to the picking action
*   on a renderer.
*
*   Sample C++ code:
*
*   @code
*       FBPickInfosList lList;
*       FBSystem lSystem;
*       HFBRenderer lRenderer = lSystem.Renderer;
*       lRenderer->Pick( 200, 200, lList );
*   
*       int lIdx;
*       for( lIdx = 0; lIdx < lList.GetCount(); ++lIdx )
*       {
*           FBString lName = "aNull ";
*           lName += (char*)lList[lIdx].mModel->Name;
*   
*           HFBModelNull lNull = new FBModelNull( lName );
*           lNull->Visible = true;
*           lNull->Translation = lList[lIdx].mPoint;
*       }
*   @endcode
*
*/
struct FBPickInfos {
    FBPickInfos( HFBModel pModel, FBVector3d pPoint ) : mModel( pModel ), mPoint( pPoint ) {};
    HFBModel   mModel;  //!< Model picked.
    FBVector3d mPoint;  //!< Location of the pick on the model.
};

bool operator==( const FBPickInfos& pLhs, const FBPickInfos& pRhs );

typedef class FBSDK_DLL FBArrayTemplate<FBPickInfos> FBPickInfosList;

////////////////////////////////////////////////////////////////////////////
// FBViewerInfos
////////////////////////////////////////////////////////////////////////////
__FB_FORWARD( FBViewerInfos ); 
FB_DEFINE_COMPONENT( FBSDK_DLL, ViewerInfos );

/** Scene information display.
*
*   By modifying the application configuration file, it is possible to display scene
*   related information to the screen. This information will also be present in the
*   rendered frames when creating AVIs or QuickTime files.
*
*   @note   To use this functionality, it is necessary to edit the config file
*           [MACHINE]-Application.txt located in folder [APPLICATION]/bin/config.
*           Look for the token 'ViewerInfoDisplay' in the section '[Display]' and
*           set its value to 'Enable'.
*   
*   @warning    Instances of this class can only be obtained via a FBRenderer object.
*               Do not attempt to create new instances.
*
*   @warning    Do not attempt to destroy instances of this class. The application is
*               the owner of these objects.
*/
class FBSDK_DLL FBViewerInfos : public FBComponent 
{
    //--- Open Reality declaration.
    __FBClassDeclare( FBViewerInfos, FBComponent );

    /** Disabled copy constructor.
    *
    *   Voluntarily making the copy constructor unavailable.
    */
    FBViewerInfos( const FBViewerInfos& );

    /** Disabled assignment operator.
    *
    *   Voluntarily making the assignment operator unavailable.
    */
    FBViewerInfos& operator=( const FBViewerInfos& );

public:

    //! Constructor.
    FBViewerInfos( HIObject pObject );

    FBPropertyBool Visibility;  //!< <b>Read Write Property:</b> Indicate if the information will be displayed or not.
};


////////////////////////////////////////////////////////////////////////////
// FBRenderer
////////////////////////////////////////////////////////////////////////////

//! Open Reality renderer interface.
class FBSDK_DLL FBRenderer : public FBComponent 
{
	//--- Open Reality declaration.
	__FBClassDeclare( FBRenderer,FBComponent );
  public:
	/**	Constructor.
    *   Client code cannot instantiate objects of this class. The FBSystem and
    *   FBScene classes provide access to the current renderer.
	*	\param	pObject	For internal use only.
	*/
	FBRenderer(HIObject pObject);

	//--- Rendering manipulation
	/** Must be called before inputing if the same renderer is used on multiple
	*   views/cameras in the same application 
	*	\param	pX	X position where to render.
	*	\param	pY	Y position where to render.
	*	\param	pW	Width of render area.
	*	\param	pH	Hight of render area.
	*/
	void SetViewport(int pX,int pY,int pW,int pH);

	/** RenderBegin
	*	\param	pX	X position where to render.
	*	\param	pY	Y position where to render.
	*	\param	pW	Width of render area.
	*	\param	pH	Hight of render area.
	*  
	*	must be called before rendering can happen
	*    
	*/
	bool RenderBegin(int pX,int pY,int pW,int pH);

	/** RenderEnd.
	*	\param		pView	If you want the renderer to draw artifacts, such as TimeCode, CameraLabel or SafeArea,
	*						you must provide the FBView on which the renderer draws on.
	*	\remarks	Must be called at the end of rendering.
	*/
	bool RenderEnd(HFBView pView=NULL);

	/** PreRenders one frame (needed for some shaders)
	*  This functions destroys the frame buffer content and must be called every time a render is called
	*    the typical order of call must be 
	*        Renderer->Prerender
	*        // at this point the frame buffer is garbage
	*        -Clear the ogl
	*        -Do your render functions
	*        Renderer->Render
    *        
	*	\param	pLayer	Rendering layer ID(default=-1).
	*	\return	\b true if successful.
	*/
	bool PreRender(int pLayer = -1);

	/** Renders one frame 
	*	\param	pLayer	Rendering layer ID(default=-1).
	*	\return	\b true if successful.
	*/
	bool Render(int pLayer = -1);

	/** Set the viewing options.
	*	\param pOptions See FBViewingOptions for more detail.
	*/
	bool SetViewingOptions(FBViewingOptions & pOptions);

    /** Obtain the current viewing options.
    *   \return A sctructure that can be queried and updated for a call to SetViewingOptions.
    */
	FBViewingOptions * GetViewingOptions();

	/** Frame the current camera either with all models or with the currently selected models.
	*	\param pAll  true to frame with all models.
	*	\return \b true if successful.
	*/
	bool FrameCurrentCameraWithModels(bool pAll);

	//--- Camera manipulation, Manipulators
	/** Mouse input.
	*	\param	pX				X position.
	*	\param	pY				Y position.
	*	\param	pInputType		Type of input.
	*	\param	pButtonKey		Button/Key pressed.
	*	\param	pModifier		Modifier pressed (CTRL/ALT/SHIFT).
	*	\param	pLayer			Rendering layer ID(default=-1).
	*	\return	\b true if successful.
	*/
	bool MouseInput(int pX,int pY,FBInputType pInputType,int pButtonKey,FBInputModifier pModifier,int pLayer = -1);

	/** Object picking.
	*	\param	pX				X position.
	*	\param	pY				Y position.
	*	\param	pPickInfosList  The list of models hit, and the location of the hit.
	*/
	bool Pick(int pX, int pY, FBPickInfosList& pPickInfosList );

	/** Keyboard input.
	*	\param	pKeyIndex		Key index. (See "enum FBDeviceKeyboardKey" above for supported keys)
	*	\param	pKeyState		Key state. (True == key is down, False == key is up)
	*	\param	pIsTrigger		When setting pKeyState to True, resets key state to False right after operation.
	*/
	void KeyboardInput(FBDeviceKeyboardKey pKeyIndex, bool pKeyState, bool pIsTrigger = false);

	// Properties
	FBPropertyCamera				CurrentCamera;			//!< <b>Read Write Property:</b> Current camera. 

#if !defined(K_NO_MANIPULATOR)
	FBPropertyManipulatorTransform  ManipulatorTransform;	//!< <b>Read Only Property:</b> Manipulator responsible of moving objects
	FBPropertyListManipulator		Manipulators;			//!< <b>List:</b> of manipulators.
#endif

	FBPropertyScene					Scene;					//!< <b>Read Write Property:</b> Scene that the renderer will use/draw
	FBPropertyBool					AutoEvaluate;			//!< <b>Read Write Property:</b> Indicate if a call to RenderBegin will also cause a re-evaluation of the scene.
	FBPropertyBool					Background;				//!< <b>Read Write Property:</b> The renderer.
	FBPropertyZSortMode				ZSortMode;				//!< <b>Read Write Property:</b> Z sorting method.
	FBPropertyInt					LayerCount;				//!< <b>Read Write Property:</b> Number of sorting layers.
	FBPropertyColor					ClearColor;				//!< <b>Read Write Property:</b> Clear color.
    FBPropertyViewerInfos           ViewerInfos;            //!< <b>Read Only Property:</b> Used to display information about the current scene. Will be present in any rendered images.
};

#ifdef FBSDKUseNamespace
	}
#endif
#endif /* this must be the last line of this file */
