#ifndef __FBSTORY_H__
#define __FBSTORY_H__
/**************************************************************************
 Copyright (c) 1994 - 2006 Autodesk, Inc. and/or its licensors.
 All Rights Reserved.
 
 The coded instructions, statements, computer programs, and/or related 
 material (collectively the "Data") in these files contain unpublished 
 information proprietary to Autodesk, Inc. and/or its licensors, which is 
 protected by Canada and United States of America federal copyright law 
 and by international treaties.
 
 The Data may not be disclosed or distributed to third parties, in whole 
 or in part, without the prior written consent of Autodesk, Inc. 
 ("Autodesk").
 
 THE DATA IS PROVIDED "AS IS" AND WITHOUT WARRANTY.
 ALL WARRANTIES ARE EXPRESSLY EXCLUDED AND DISCLAIMED. AUTODESK MAKES NO 
 WARRANTY OF ANY KIND WITH RESPECT TO THE DATA, EXPRESS, IMPLIED OR 
 ARISING BY CUSTOM OR TRADE USAGE, AND DISCLAIMS ANY IMPLIED WARRANTIES 
 OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR 
 PURPOSE OR USE. WITHOUT LIMITING THE FOREGOING, AUTODESK DOES NOT 
 WARRANT THAT THE OPERATION OF THE DATA WILL BE UNINTERRUPTED OR ERROR 
 FREE.
 
 IN NO EVENT SHALL AUTODESK, ITS AFFILIATES, PARENT COMPANIES, LICENSORS 
 OR SUPPLIERS ("AUTODESK GROUP") BE LIABLE FOR ANY LOSSES, DAMAGES OR 
 EXPENSES OF ANY KIND (INCLUDING WITHOUT LIMITATION PUNITIVE OR MULTIPLE 
 DAMAGES OR OTHER SPECIAL, DIRECT, INDIRECT, EXEMPLARY, INCIDENTAL, LOSS 
 OF PROFITS, REVENUE OR DATA, COST OF COVER OR CONSEQUENTIAL LOSSES OR 
 DAMAGES OF ANY KIND), HOWEVER CAUSED, AND REGARDLESS OF THE THEORY OF 
 LIABILITY, WHETHER DERIVED FROM CONTRACT, TORT (INCLUDING, BUT NOT 
 LIMITED TO, NEGLIGENCE), OR OTHERWISE, ARISING OUT OF OR RELATING TO THE 
 DATA OR ITS USE OR ANY OTHER PERFORMANCE, WHETHER OR NOT AUTODESK HAS 
 BEEN ADVISED OF THE POSSIBILITY OF SUCH LOSS OR DAMAGE.
 
**************************************************************************/

/**	\file fbstory.h
*	Story interface for FBSDK
*/

#include <kaydaradef.h>
#ifndef FBSDK_DLL 
	#define FBSDK_DLL K_DLLIMPORT
#endif

#include <fbsdk/fbcomponent.h>
#include <fbsdk/fbcore.h>	// FBBox
#include <fbsdk/fbvideo.h>	// FBPropertyVideo
#include <fbsdk/fbmodel.h>	// FBPropertyCamera

#ifdef FBSDKUseNamespace
	namespace FBSDKNamespace {
#endif

__FB_FORWARD( FBStory );
__FB_FORWARD( FBStoryFolder );
__FB_FORWARD( FBStoryTrack );
__FB_FORWARD( FBStoryClip );
FB_DEFINE_COMPONENT( FBSDK_DLL, Story );
FB_DEFINE_COMPONENT( FBSDK_DLL, StoryFolder );
FB_DEFINE_COMPONENT( FBSDK_DLL, StoryTrack );
FB_DEFINE_COMPONENT( FBSDK_DLL, StoryClip );
FB_DEFINE_COMPONENT( FBSDK_DLL, AnimationNode );
FB_DEFINE_COMPONENT( FBSDK_DLL, Video );
FB_DEFINE_COMPONENT( FBSDK_DLL, AudioClip );

//! Types for new story tracks.
enum FBStoryTrackType {
	kFBStoryTrackAnimation,		//!< Animation track.
	kFBStoryTrackCamera,		//!< Camera animation track.
	kFBStoryTrackCharacter,		//!< Character animation track.
	kFBStoryTrackConstraint,	//!< Constraint track.
	kFBStoryTrackCommand,		//!< Command track.
	kFBStoryTrackShot,			//!< Shot track.
	kFBStoryTrackAudio,			//!< Audio track.
	kFBStoryTrackVideo,			//!< Video track.
	kFBStoryTrackUnknown
};

//! References Modes for story animation tracks.
enum FBStoryTrackRefMode {
	kFBStoryTrackLocal,		//!< Reference mode is local.
	kFBStoryTrackAdditive	//!< Additive reference mode.
};

//! Body Parts for story track character.
enum FBStoryTrackBodyPart {
	kFBStoryTrackBodyPartNone,
	kFBStoryTrackBodyPartAll,
	kFBStoryTrackBodyPartUpperBody,
	kFBStoryTrackBodyPartLeftArm,
	kFBStoryTrackBodyPartLeftHand,
	kFBStoryTrackBodyPartRightArm,
	kFBStoryTrackBodyPartRightHand,
	kFBStoryTrackBodyPartHead,
	kFBStoryTrackBodyPartLowerBody,
	kFBStoryTrackBodyPartLeftLeg,
	kFBStoryTrackBodyPartLeftFoot,
	kFBStoryTrackBodyPartRightLeg,
	kFBStoryTrackBodyPartRightFoot,
	kFBStoryTrackBodyPartProps,
    kFBStoryTrackBodyPartExtensions
};

//! Show Ghost Modes for story animation clips.
enum FBStoryClipShowGhostMode {
	kFBStoryClipAlways,		//!< Always show the ghost.
	kFBStoryClipTimeCursor	//!< Show the ghost only on time cursor.
};

//! Solve Modes for story character clips.
enum FBStoryClipSolveMode {
	kFBStoryClipRetargetSkeleton,	//!< Solve retarget skeleton.
	kFBStoryClipAnimSkeleton,		//!< Solve skeleton animation.
	kFBStoryClipAnimFkIk,			//!< Solve forward and inverse kinematic animation.
	kFBStoryClipAnimSkeletonIk		//!< Solve skeleton inverse kinematic animation.
};

//! Compensation Modes for story character clips.
enum FBStoryClipCompMode {
	kFBStoryClipOff,	//!< No compensation.
	kFBStoryClipAuto,	//!< Automatic compensation.
	kFBStoryClipUser	//!< User defined compensation.
};

FB_DEFINE_ENUM(FBSDK_DLL, StoryTrackType);
FB_DEFINE_ENUM(FBSDK_DLL, StoryTrackRefMode);
FB_DEFINE_ENUM(FBSDK_DLL, StoryTrackBodyPart);
FB_DEFINE_ENUM(FBSDK_DLL, StoryClipShowGhostMode);
FB_DEFINE_ENUM(FBSDK_DLL, StoryClipSolveMode);
FB_DEFINE_ENUM(FBSDK_DLL, StoryClipCompMode);

typedef class FBSDK_DLL FBPropertyBaseList< HFBStoryFolder > FBPropertyBaseListStoryFolder;
typedef class FBSDK_DLL FBPropertyBaseList< HFBStoryTrack > FBPropertyBaseListStoryTrack;
typedef class FBSDK_DLL FBPropertyBaseList< HFBStoryTrack > FBPropertyBaseListStorySubTrack;
typedef class FBSDK_DLL FBPropertyBaseList< HFBStoryClip > FBPropertyBaseListStoryClip;
typedef class FBSDK_DLL FBPropertyBaseList< HFBComponent > FBPropertyBaseListStoryDetails;
typedef class FBSDK_DLL FBPropertyBaseList< HFBModel > FBPropertyBaseListPivot;

FB_FORWARD( FBPropertyListStoryFolder );
//! <b>List:</b> StoryFolder
class FBSDK_DLL FBPropertyListStoryFolder : public FBPropertyBaseList< HFBStoryFolder >
{
public:
	/**	Add a folder to the property list.
	*	\param	pItem	Folder to add to list.
	*	\return	Number of items in list after operation.
	*	\remark	A folder can't be in two or more separate folders at once nor multiple times in the same one.
	*/
	virtual int Add(HFBStoryFolder pItem);

	/**	Remove folder \e pItem from property list.
	*	\param	pItem	Folder to remove from list.
	*	\return Number of items in list after operation.
	*	\remark	A folder can't be in two or more separate folders at once nor multiple times in the same one.
	*/
    virtual int Remove(HFBStoryFolder pItem);

	/** Remove the folder at \e pIndex.
	*	\param pIndex	Index of folder to remove.
	*/
    virtual void RemoveAt(int pIndex);

	/**	Get the folder at \e pIndex.
	*	\param	pIndex	Index of folder to get.
	*	\return Folder at \e pIndex.
	*/
	virtual HFBStoryFolder operator[](int pIndex);

	/** Get the number of folder.
	*	\return Number of folders.
	*/
	virtual int GetCount();
};

FB_FORWARD( FBPropertyListStoryTrack );
//! <b>List:</b> StoryTrack
class FBSDK_DLL FBPropertyListStoryTrack : public FBPropertyBaseList< HFBStoryTrack >
{
public:
	/**	Add a track to the property list.
	*	\param	pItem	Track to add to list.
	*	\return	Number of items in list after operation.
	*	\remark	A track can't be in two or more separate folders at once nor multiple times in the same one.
	*/
	virtual int Add(HFBStoryTrack pItem);

	/**	Remove track \e pItem from property list.
	*	\param	pItem	Track to remove from list.
	*	\return Number of items in list after operation.
	*	\remark	A track can't be in two or more separate folders at once nor multiple times in the same one.
	*/
    virtual int Remove(HFBStoryTrack pItem);

	/** Remove the track at \e pIndex.
	*	\param pIndex	Index of track to remove.
	*/
    virtual void RemoveAt(int pIndex);

	/**	Get the track at \e pIndex.
	*	\param	pIndex	Index of track to get.
	*	\return Track at \e pIndex.
	*/
	virtual HFBStoryTrack operator[](int pIndex);

	/** Get the number of track.
	*	\return Number of tracks.
	*/
	virtual int GetCount();
};

FB_FORWARD( FBPropertyListStorySubTrack );
//! <b>List:</b> StorySubTrack
class FBSDK_DLL FBPropertyListStorySubTrack : public FBPropertyBaseList< HFBStoryTrack >
{
public:
	/**	Add a sub-track to the property list.
	*	\param	pItem	Sub-Track to add to list.
	*	\return	Number of items in list after operation.
	*	\remark	A sub-track can't be in two or more separate track at once nor multiple times in the same one.
	*/
	virtual int Add(HFBStoryTrack pItem);

	/**	Remove sub-track \e pItem from property list.
	*	\param	pItem	Sub-Track to remove from list.
	*	\return Number of items in list after operation.
	*	\remark	A sub-track can't be in two or more separate track at once nor multiple times in the same one.
	*/
    virtual int Remove(HFBStoryTrack pItem);

	/** Remove the sub-track at \e pIndex.
	*	\param pIndex	Index of sub-track to remove.
	*/
    virtual void RemoveAt(int pIndex);

	/**	Get the sub-track at \e pIndex.
	*	\param	pIndex	Index of sub-track to get.
	*	\return Sub-Track at \e pIndex.
	*/
	virtual HFBStoryTrack operator[](int pIndex);

	/** Get the number of sub-track.
	*	\return Number of sub-tracks.
	*/
	virtual int GetCount();
};

FB_FORWARD( FBPropertyListStoryClip );
//! <b>List:</b> StoryClip
class FBSDK_DLL FBPropertyListStoryClip : public FBPropertyBaseList< HFBStoryClip >
{
public:
	/**	Add a clip to the property list.
	*	\param	pItem	Clip to add to list.
	*	\return	Number of items in list after operation.
	*/
	virtual int Add(HFBStoryClip pItem);

	/**	Remove clip \e pItem from property list.
	*	\param	pItem	Clip to remove from list.
	*	\return Number of items in list after operation.
	*/
    virtual int Remove(HFBStoryClip pItem);

	/** Remove the clip at \e pIndex.
	*	\param pIndex	Index of clip to remove.
	*/
    virtual void RemoveAt(int pIndex);

	/**	Get the clip at \e pIndex.
	*	\param	pIndex	Index of clip to get.
	*	\return Clip at \e pIndex.
	*/
	virtual HFBStoryClip operator[](int pIndex);

	/** Get the number of clip.
	*	\return Number of clips.
	*/
	virtual int GetCount();
};

FB_FORWARD( FBPropertyListStoryDetails );
//! <b>List:</b> Story track Details
class FBSDK_DLL FBPropertyListStoryDetails : public FBPropertyBaseList< HFBComponent >
{
public:
	/**	Add a object to the property list.
	*	\param	pItem	Object to add to list.
	*	\return	Number of items in list after operation.
	*/
	virtual int Add(HFBComponent pItem);

	/**	Remove object \e pItem from property list.
	*	\param	pItem	Object to remove from list.
	*	\return Number of items in list after operation.
	*/
    virtual int Remove(HFBComponent pItem);

	/** Remove the object at \e pIndex.
	*	\param pIndex	Index of object to remove.
	*/
    virtual void RemoveAt(int pIndex);

	/**	Get the object at \e pIndex.
	*	\param	pIndex	Index of object to get.
	*	\return Object at \e pIndex.
	*/
	virtual HFBComponent operator[](int pIndex);

	/** Get the number of object.
	*	\return Number of objects.
	*/
	virtual int GetCount();
};

FB_FORWARD( FBPropertyListPivot );
//! <b>List:</b> Story Clip pivot models
class FBSDK_DLL FBPropertyListPivot : public FBPropertyBaseList< HFBModel >
{
public:
	/**	Add a object to the property list.
	*	\param	pItem	Object to add to list.
	*	\return	Number of items in list after operation.
	*/
	virtual int Add(HFBModel pItem);

	/**	Remove object \e pItem from property list.
	*	\param	pItem	Object to remove from list.
	*	\return Number of items in list after operation.
	*/
    virtual int Remove(HFBModel pItem);

	/** Remove the object at \e pIndex.
	*	\param pIndex	Index of object to remove.
	*/
    virtual void RemoveAt(int pIndex);

	/**	Get the object at \e pIndex.
	*	\param	pIndex	Index of object to get.
	*	\return Object at \e pIndex.
	*/
	virtual HFBModel operator[](int pIndex);

	/** Get the number of object.
	*	\return Number of objects.
	*/
	virtual int GetCount();
};

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// FBStory
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/** Story Management class.
*	This class serve as a management control for the Story global settings and members.
*/
class FBSDK_DLL FBStory : public FBComponent 
{
	//--- Open Reality declaration.
	__FBClassDeclare(FBStory, FBComponent);
public:

	/**	Constructor.
	*/
	FBStory();

	FBPropertyBool			Mute;				//!< <b>Read Write Property:</b> If true, the Story mode will be globally disabled.
	FBPropertyBool			LockedShot;			//!< <b>Read Write Property:</b> If true, shots will be locked (no time discontinuity).
	FBPropertyStoryFolder	RootFolder;			//!< <b>Read Only Property:</b> Story's root folder
	FBPropertyStoryFolder	RootEditFolder;		//!< <b>Read Only Property:</b> Story's root edit folder
};

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// FBStoryFolder
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/** Story Folder class.
*	With folders, you can group tracks together and create different timelines.
*/
class FBSDK_DLL FBStoryFolder : public FBComponent
{
	//--- Open Reality declaration.
	__FBClassDeclare(FBStoryFolder, FBComponent);
public:

	/**	Constructor.
	*	\param	pParentFolder	If NULL, parent will be the global root folder, according to its type.
	*	\param	pObject			For internal use only.
	*	\remark	You can't create a folder with the RootEditFolder as parent.
	*/
	FBStoryFolder(HFBStoryFolder pParentFolder=NULL, HIObject pObject=NULL);

	/** Virtual FBDelete function.
	*/
	virtual void FBDelete();

	FBPropertyString			Label;		//!< <b>Read Write Property:</b> Label to display for this story folder.
	FBPropertyBool				Solo;		//!< <b>Read Write Property:</b> If true, this story folder will be the only one to play. 
	FBPropertyBool				Mute;		//!< <b>Read Write Property:</b> If true, this story folder will be muted.
	FBPropertyStoryFolder		Parent;		//!< <b>Read Only Property:</b> Object pointing to the folder's parent.
	FBPropertyListStoryFolder	Childs;		//!< <b>List:</b> Children folders of this folder.
	FBPropertyListStoryTrack	Tracks;		//!< <b>List:</b> Tracks of this folder.
};

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// FBStoryTrack
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/** Story Track class.
*	Tracks are containers for clips (medias), have a specific type which offer different functions.
*/
class FBSDK_DLL FBStoryTrack : public FBComponent
{
	//--- Open Reality declaration.
	__FBClassDeclare(FBStoryTrack, FBComponent);
public:

	/**	Constructor.
	*	\param	pTrackType	Type of the track to be created.
	*	\param	pFolder		If NULL, parent will be the global root folder.
	*	\param	pObject		For internal use only.
	*/
	FBStoryTrack(FBStoryTrackType pTrackType, HFBStoryFolder pFolder=NULL, HIObject pObject=NULL);

	/**	Constructor.
	*	\param	pSource		Source of the track to be created based on media component type.
	*	\param	pFolder		If NULL, parent will be the global root folder.
	*	\param	pObject		For internal use only.
	*/
	FBStoryTrack(HFBComponent pSource, HFBStoryFolder pFolder=NULL, HIObject pObject=NULL);

	/** Virtual FBDelete function.
	*/
	virtual void FBDelete();

	/**	ChangeDetailsBegin.
	*	You must call this function before adding/removing any object to the Details list or it won't work.
	*/
	void ChangeDetailsBegin();

	/**	ChangeDetailsEnd.
	*	You must call this function after adding/removing any object to the Details list or it won't work.
	*/
	void ChangeDetailsEnd();

	//--- Character track functions -----------------------------------------------------------------------------------------------------
	/**	EnableBodyPart.
	*	\param	pPart	Which part to enable/disable.
	*	\param	pEnable	If True, this will enable the body part solving while false will disable it.
	*	Enable a specific body part for character solving.
	*/
	void EnableBodyPart(FBStoryTrackBodyPart pPart, bool pEnable);

	/**	IsBodyPartEnabled.
	*	Is a specific body part is enabled.
	*/
	bool IsBodyPartEnabled(FBStoryTrackBodyPart pPart);

	//--- All track properties ----------------------------------------------------------------------------------------------------------
	FBPropertyStoryTrackType	Type;			//!< <b>Read Only Property:</b> Type of the track
	FBPropertyString			Label;			//!< <b>Read Write Property:</b> Label to display for this story track.
	FBPropertyBool				Mute;			//!< <b>Read Write Property:</b> If true, this track wont' play.
	FBPropertyBool				Solo;			//!< <b>Read Write Property:</b> If true, this track will be the only one to play.
	FBPropertyStoryFolder		ParentFolder;	//!< <b>Read Only Property:</b> Parent folder.
	FBPropertyStoryTrack		ParentTrack;	//!< <b>Read Only Property:</b> Parent track, if the track is of Character or Animation type.
	FBPropertyListStorySubTrack	SubTracks;		//!< <b>List:</b> Only Character and Animation tracks can have sub-tracks.
	FBPropertyListStoryClip		Clips;			//!< <b>List:</b> Clips contained in this track.
	FBPropertyListStoryDetails	Details;		//!< <b>List:</b> All objects associated to this track for processing.

	//--- Animation and Constraint track properties -------------------------------------------------------------------------------------
	FBPropertyDouble			Blend;			//!< <b>Read Write Property:</b> Control the blend amount.

	//--- Animation track properties ----------------------------------------------------------------------------------------------------
	FBPropertyBool				Ghost;			//!< <b>Read Write Property:</b>
	FBPropertyBool				GhostModel;		//!< <b>Read Write Property:</b>
	FBPropertyBool				GhostTravelling;//!< <b>Read Write Property:</b>
	FBPropertyBool				GhostPivot;		//!< <b>Read Write Property:</b>
	FBPropertyBool				AcceptKey;		//!< <b>Read Write Property:</b>
	FBPropertyStoryTrackRefMode	ReferenceMode;	//!< <b>Read Write Property:</b>
	FBPropertyBool				OffsetEnable;	//!< <b>Read Write Property:</b>
	FBPropertyBool				PassThrough;	//!< <b>Read Write Property:</b>

	//--- Character track properties ----------------------------------------------------------------------------------------------------
	FBPropertyInt				CharacterIndex;	//!< <b>Read Write Property:</b> Character's index to use.

	//--- Shot track properties ---------------------------------------------------------------------------------------------------------
	FBPropertyBool				ShowBackplate;	//!< <b>Read Write Property:</b> If true, the backplate will be shown.

	//--- Audio track properties --------------------------------------------------------------------------------------------------------
	FBPropertyInt				AudioOutIndex;	//!< <b>Read Write Property:</b> Audio Output's index to use.

	//--- Video track properties --------------------------------------------------------------------------------------------------------
	FBPropertyVideo				TrackVideo;		//!< <b>Read Only Property:</b> This FBVideo can be used as a texture.
};

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// FBStoryClip
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/** Story Clip class.
*	Clips represents media, at a specific time, for a specific duration, in a track.
*/
class FBSDK_DLL FBStoryClip : public FBComponent
{
	//--- Open Reality declaration.
	__FBClassDeclare(FBStoryClip, FBComponent);
public:

	/**	Constructor.
	*	\param	pClipObject	Object (media data) for the clip.
	*	\param	pTrack		The track in which we create the clip.
	*	\param	pTime		Time where the clip should begin.
	*	\param	pObject		For internal use only.
	*/
	FBStoryClip(HFBComponent pClipObject, HFBStoryTrack pTrack, FBTime pTime, HIObject pObject=NULL);

	/**	Constructor.
	*	\param	pFilePath	Media file path to create clip with.
	*	\param	pTrack		The track in which we create the clip.
	*	\param	pTime		Time where the clip should begin.
	*	\param	pObject		For internal use only.
	*/
	FBStoryClip(char* pFilePath, HFBStoryTrack pTrack, FBTime pTime, HIObject pObject=NULL);

	/** Virtual FBDelete function.
	*/
	virtual void FBDelete();

	/**	Move.
	*	Move the clip of a delta offset.
	*	\param	pDelta	Delta time to apply to the clip.
	*	\param	pForce	Force clip to find the nearest position if the move fail.
	*	\return Return the delta between the new and previous clip's position.
	*/
	FBTime Move(FBTime pDelta, bool pForce=true);

	/**	MoveTo.
	*	Move the clip to the specified time.
	*	\param	pTime	Time where to put the clip.
	*	\param	pForce	Force clip to find the nearest position if the move fail.
	*	\return Returns the new clip's time position.
	*/
	FBTime MoveTo(FBTime pTime, bool pForce=true);

	/**	Razor.
	*	Cut (razor) the clip at the specified time.
	*	\param	pTime	Time where to cut. This time is local to the track, not to the clip.
	*	\return Returns the new clip generated by the razor action (right part).
	*/
	HFBStoryClip Razor(FBTime pTime);

	//--- Animation clip functions ------------------------------------------------------------------------------------------------------
	/**	Match.
	*	Match the animation clip with the specified pivot property.
	*/
	void Match();

	/**	ExportToFile.
	*	Export animation clip to disk file.
	*	\param	pOutputFile	Output file path name.
	*	\return	Returns true if successful.
	*/
	bool ExportToFile(FBString pOutputFile);

	/**	GetAffectedObjects.
	*	Get the list of objects affected by this story clip.
	*	\param	pAffectedObjects	Array of affected objects, will be filled by the function.
	*/
	void GetAffectedObjects( FBArrayTemplate<HFBComponent>* pAffectedObjects );

	/**	GetAffectedAnimationNodes.
	*	Get the list of animation nodes affected by this story clip, for a specific object.
	*	\param	pAffectedAnimationNodes	Array of affected animation nodes, will be filled by the function.
	*	\param	pClipObject				The object for which we search for affected animation nodes.
	*/
	void GetAffectedAnimationNodes( FBArrayTemplate<HFBAnimationNode>* pAffectedAnimationNodes, HFBComponent pClipObject );

	//--- All clip properties -----------------------------------------------------------------------------------------------------------
	FBPropertyColor		Color;			//!< <b>Read Write Property:</b> Color of the clip.
	FBPropertyTime		Start;			//!< <b>Read Write Property:</b> Start time of the clip local to its track.
	FBPropertyTime		Stop;			//!< <b>Read Write Property:</b> Stop time of the clip local to its track.
	FBPropertyTime		MarkIn;			//!< <b>Read Write Property:</b> Start time inside the clip.
	FBPropertyTime		MarkOut;		//!< <b>Read Write Property:</b> Stop time inside the clip.
	FBPropertyTimeSpan	PreBlend;		//!< <b>Read Write Property:</b> Start/Stop time of the pre-blend phase.
	FBPropertyTimeSpan	PostBlend;		//!< <b>Read Write Property:</b> Start/Stop time of the post-blend phase.
	FBPropertyTime		Offset;			//!< <b>Read Write Property:</b> First loop time offset.
	FBPropertyDouble	Speed;			//!< <b>Read Write Property:</b> Speed of the clip.

	//--- Shot clip properties ----------------------------------------------------------------------------------------------------------
	FBPropertyCamera	ShotCamera;				//!< <b>Read Write Property:</b> The camera used for that specific shot.
	FBPropertyVideo		ShotBackplate;			//!< <b>Read Write Property:</b> The backplate used for that specific shot.
	FBPropertyBool		ShowBackplate;			//!< <b>Read Write Property:</b> Enable/Disable the shot backplate.
	FBPropertyTime		ShotActionStart;		//!< <b>Read Write Property:</b> If not in locked shot mode (time discontinuity enabled), this time can be different from the Clip->Start property.
	FBPropertyTime		ShotActionStop;			//!< <b>Read Write Property:</b> If not in locked shot mode (time discontinuity enabled), this time can be different from the Clip->Start property.

	//--- Audio clip properties ---------------------------------------------------------------------------------------------------------
	FBPropertyAudioClip	AudioClip;				//!< <b>Read Only Property:</b> The audio clip used by this StoryClip.

	//--- Animation clip properties -----------------------------------------------------------------------------------------------------
	FBPropertyDouble					Scale;				//!< <b>Read Write Property:</b> Animation clip's scaling (some don't support this property)
	FBPropertyVector3d					Translation;		//!< <b>Read Write Property:</b> Animation clip's translation offset.
	FBPropertyVector3d					Rotation;			//!< <b>Read Write Property:</b> Animation clip's rotation offset.
	FBPropertyVector3d					LoopTranslation;	//!< <b>Read Write Property:</b> Animation clip's loop translation.
	FBPropertyBool						AutoLoop;			//!< <b>Read Write Property:</b> If true, clip will automatically loop
	FBPropertyBool						Loop;				//!< <b>Read Write Property:</b> If true, loop clip's animation
	FBPropertyBool						Ghost;				//!< <b>Read Write Property:</b>
	FBPropertyBool						GhostModel;			//!< <b>Read Write Property:</b>
	FBPropertyBool						GhostTravelling;	//!< <b>Read Write Property:</b>
	FBPropertyBool						GhostPivot;			//!< <b>Read Write Property:</b>
	FBPropertyStoryClipShowGhostMode	ShowGhostClipMode;	//!< <b>Read Write Property:</b>
	FBPropertyAnimationNode				PreBlendData;		//!< <b>Read Only Property:</b>
	FBPropertyAnimationNode				PostBlendData;		//!< <b>Read Only Property:</b>
	FBPropertyListPivot					Pivots;				//!< <b>List:</b> Pivots models (Generally, only one model is necessary)

	//--- Character clip properties -----------------------------------------------------------------------------------------------------
	FBPropertyStoryClipSolveMode	SolvingMode;						//!< <b>Read Write Property:</b>
};

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef FBSDKUseNamespace
	}
#endif
#endif /* this must be the last line of this file */
