#ifndef __FBTIME_H__
#define __FBTIME_H__
/**************************************************************************
 Copyright (c) 1994 - 2006 Autodesk, Inc. and/or its licensors.
 All Rights Reserved.
 
 The coded instructions, statements, computer programs, and/or related 
 material (collectively the "Data") in these files contain unpublished 
 information proprietary to Autodesk, Inc. and/or its licensors, which is 
 protected by Canada and United States of America federal copyright law 
 and by international treaties.
 
 The Data may not be disclosed or distributed to third parties, in whole 
 or in part, without the prior written consent of Autodesk, Inc. 
 ("Autodesk").
 
 THE DATA IS PROVIDED "AS IS" AND WITHOUT WARRANTY.
 ALL WARRANTIES ARE EXPRESSLY EXCLUDED AND DISCLAIMED. AUTODESK MAKES NO 
 WARRANTY OF ANY KIND WITH RESPECT TO THE DATA, EXPRESS, IMPLIED OR 
 ARISING BY CUSTOM OR TRADE USAGE, AND DISCLAIMS ANY IMPLIED WARRANTIES 
 OF TITLE, NON-INFRINGEMENT, MERCHANTABILITY OR FITNESS FOR A PARTICULAR 
 PURPOSE OR USE. WITHOUT LIMITING THE FOREGOING, AUTODESK DOES NOT 
 WARRANT THAT THE OPERATION OF THE DATA WILL BE UNINTERRUPTED OR ERROR 
 FREE.
 
 IN NO EVENT SHALL AUTODESK, ITS AFFILIATES, PARENT COMPANIES, LICENSORS 
 OR SUPPLIERS ("AUTODESK GROUP") BE LIABLE FOR ANY LOSSES, DAMAGES OR 
 EXPENSES OF ANY KIND (INCLUDING WITHOUT LIMITATION PUNITIVE OR MULTIPLE 
 DAMAGES OR OTHER SPECIAL, DIRECT, INDIRECT, EXEMPLARY, INCIDENTAL, LOSS 
 OF PROFITS, REVENUE OR DATA, COST OF COVER OR CONSEQUENTIAL LOSSES OR 
 DAMAGES OF ANY KIND), HOWEVER CAUSED, AND REGARDLESS OF THE THEORY OF 
 LIABILITY, WHETHER DERIVED FROM CONTRACT, TORT (INCLUDING, BUT NOT 
 LIMITED TO, NEGLIGENCE), OR OTHERWISE, ARISING OUT OF OR RELATING TO THE 
 DATA OR ITS USE OR ANY OTHER PERFORMANCE, WHETHER OR NOT AUTODESK HAS 
 BEEN ADVISED OF THE POSSIBILITY OF SUCH LOSS OR DAMAGE.
 
**************************************************************************/

/**	\file fbtime.h
*	Time classes.
*/

#include <kaydaradef.h>

#ifndef FBSDK_DLL
  #define FBSDK_DLL K_DLLIMPORT
#endif

#include <fbsdk/fbstring.h>

#ifdef FBSDKUseNamespace
	namespace FBSDKNamespace {
#endif

////////////////////////////////////////////////////////////////////////////////////
// FBTime
////////////////////////////////////////////////////////////////////////////////////

//! Different time modes available.
enum FBTimeMode {	
	kFBTimeModeDefault, 		//!< 30 fps
	kFBTimeModeCinema,			//!< 24 
	kFBTimeModePAL, 			//!< 25 
	kFBTimeMode30Frames,		//!< 30 
	kFBTimeModeNTSC_Drop,		//!< 29.97002617 
	kFBTimeMode50Frames,		//!< 50
	kFBTimeMode60Frames,		//!< 60
	kFBTimeMode100Frames,		//!< 100
	kFBTimeMode120Frames, 		//!< 120
	kFBTimeModeFullFrame, 		//!< Full frame 
	kFBTimeModeCinemaND,		    //!< 23.976020936 fps
    kFBTimeModeCustom           //!< Custom framerate
};

FB_FORWARD( FBTime );

//! Time data structure.
class FBSDK_DLL FBTime 
{
  private:
	kLongLong	mTime;			//!< Time member.
  public:
	/*	Constructor.
	*	\param	pTime	Time to initialize with(default=0).
	*/
    FBTime( kLongLong pTime=0);

	/**	Constructor.
	*	\param	pHour		Hour value.
	*	\param	pMinute		Minute value.
	*	\param	pSecond		Second value.
	*	\param	pFrame		Frame value.
	*	\param	pField		Field value.
	*	\param	pTimeMode	Time mode(default=kFBTimeModeDefault).
    *	\param	pFramerate	Custom framerate value in case of pTimeMode = kFBTimeModeCustom.
	*/
	FBTime (int pHour,int pMinute,int pSecond=0,int pFrame=0,int pField=0, FBTimeMode pTimeMode=kFBTimeModeDefault, double pFramerate = 0.0);

	/**	Get time as a string.
	*	\return String value of time.
	*/
	FBString	GetTimeString();

	/**	Set time from string.
	*	\param	pTime	String to set time from.
	*/
    void		SetTimeString(char *pTime);

	/**	Get time (filling separate values)
	*	\retval	pHour			Hour value.
	*	\retval	pMinute			Minute value.
	*	\retval	pSecond			Second value.
	*	\retval	pFrame			Frame value.
	*	\retval	pField			Field value.
	*	\retval	pMilliSecond	MilliSecond value.
	*	\param	pTimeMode		Time mode to get time as.
    *	\param	pFramerate	Custom framerate value in case of pTimeMode = kFBTimeModeCustom.
	*	\return	\b true if an acceptable \e TimeMode value was chosen.
	*/
	bool		GetTime(kLongLong &pHour,  kLongLong &pMinute, kLongLong &pSecond, kLongLong &pFrame, kLongLong &pField,  kLongLong &pMilliSecond, FBTimeMode pTimeMode=kFBTimeModeDefault, double pFramerate =0.0 );
	/**	Get milliseconds for time.
	*	\return	MilliSeconds value.
	*/
	kLongLong	GetMilliSeconds();

	/**	Set milliseconds time.
	*	\param	pMilliSeconds	MilliSeconds value.
	*/
	void		SetMilliSeconds(kLongLong pMilliSeconds);

	/**	Get time value (long)
	*	\return Time value as long.
	*/
    kLongLong&	Get();

	/** Set time value from a long.
	*	\param	pTime	Time value to set.
	*/
    void		Set( kLongLong pTime );

	/**	Get seconds as double.
	*	\return Seconds in double form.
	*/
	double		GetSecondDouble();

	/**	Set seconds from double.
	*	\param	pTime	Time to set seconds from.
	*/
	void		SetSecondDouble(double pTime);

	/**	Set time (from separate values)
	*	\param	pHour			Hour value.
	*	\param	pMinute			Minute value(default=0).
	*	\param	pSecond			Second value(default=0).
	*	\param	pFrame			Frame value(default=0).
	*	\param	pField			Field value(default=0).
	*	\param	pTimeMode		Time mode to get time as(default=kFBTimeModeDefault).
    *	\param	pFramerate	Custom framerate value in case of pTimeMode = kFBTimeModeCustom.
	*/
	void		SetTime (int pHour,int pMinute=0,int pSecond=0,int pFrame=0,int pField=0, FBTimeMode pTimeMode=kFBTimeModeDefault, double pFramerate = 0.0);

	//@{
	/**	Overloaded assignment operators with FBTime objects.
	*	\param	pTime	Time to assign with operator.
	*	\return	Resulting time from operation.
	*/
	FBTime& operator=	(const FBTime& pTime);
	FBTime& operator+=	(const FBTime& pTime);
	FBTime& operator-=	(const FBTime& pTime);
	FBTime&	operator*=	(const FBTime& pTime);
	FBTime&	operator/=	(const FBTime& pTime);
	//@}

	//@{
	/**	Overloaded assignment operators with constants.
	*	\param	pConstant	Time to assign with operator.
	*	\return	Resulting time from operation.
	*/
	FBTime& operator=	(double pConstant);
	FBTime& operator+=	(double pConstant);
	FBTime& operator-=	(double pConstant);
	FBTime&	operator*=	(double pConstant);
	FBTime&	operator/=	(double pConstant);
	//@}

	//@{
	/**	Overloaded arithmetic operators with FBTime objects.
	*	\param	pTime	Time to use in operation.
	*	\return	Result in FBTime data.
	*/
    FBTime	operator- (const FBTime &pTime);
    FBTime	operator+ (const FBTime &pTime);
	FBTime	operator/ (const FBTime &pTime);
	FBTime	operator* (const FBTime &pTime);
	//@}

	//@{
	/**	Overloaded arithmetic operators with constants.
	*	\param	pConstant	Double constant to add to time.
	*	\return	Result in FBTime data.
	*/
    FBTime	operator- (double pConstant);
    FBTime	operator+ (double pConstant);
	FBTime	operator/ (double pConstant);
	FBTime	operator* (double pConstant);
	//@}

	//@{
	/** Overloaded comparison operators.
	*	\param	pTime	Time to compare with.
	*	\return	Result of comparison.
	*/
	bool	operator==	(const FBTime& pTime);
	bool	operator!=	(const FBTime& pTime);
	bool	operator>=	(const FBTime& pTime);
	bool	operator<=	(const FBTime& pTime);
	bool	operator>	(const FBTime& pTime);
	bool	operator<	(const FBTime& pTime);
	//@}

	/**	Get the frame count.
	*	With this function, it is possible to obtain the cumulative and local frame counts.
	*	\param	pCummul		Cumulative frame count (\b true ), or local frame count (\b false )(default is false).
	*	\param	pTimeMode	Time mode to get the constant (default is kFBTimeModeDefault).
    *	\param	pFramerate	Custom framerate value in case of pTimeMode = kFBTimeModeCustom.
	*	\return	Frames per second constant for the specified time mode.
	*/
	kLongLong	GetFrame(bool pCummul=false, FBTimeMode pTimeMode=kFBTimeModeDefault, double pFramerate=0.0);

	//! Time constant: Infinity, the largest time value.
	static const FBTime Infinity;

	//! Time constant: Minus Infinity, the lowest negative time value.
	static const FBTime MinusInfinity;

	//! Time constant: Zero.
	static const FBTime Zero;

	//! Time constant: Epsilon, the smallest time increment.
	static const FBTime Epsilon;

	//! Time constant: One Second.
	static const FBTime OneSecond;

	//! Time constant: One Minute.
	static const FBTime OneMinute;

	//! Time constant: One Hour.
	static const FBTime OneHour;
};

////////////////////////////////////////////////////////////////////////////////////
// FBTimeSpan
////////////////////////////////////////////////////////////////////////////////////

FB_FORWARD( FBTimeSpan )

//! TimeSpan class.
class FBSDK_DLL FBTimeSpan 
{
  private:
    FBTime       mStart;		//!< Start time.
    FBTime       mStop;			//!< Stop time.
  public:
	/**	Constructor.
	*	\param	pStart	Start time(default=0).
	*	\param	pStop	Stop time(default=0).
	*/
    FBTimeSpan( FBTime pStart=0,FBTime pStop=0);

	//@{
	/**	Get the start/stop time.
	*	\return Start/Stop time.
	*/
	FBTime& GetStart();
	FBTime& GetStop();
	//@}

	/**	Set the TimeSpan
	*	\param	pStart	Start time.
	*	\param	pStop	Stop time.
	*/
	void	Set( FBTime pStart, FBTime pStop );

	/**	Get the unsigned duration value of a timespan.
	*	\return \b Unsigned duration of the timespan.
	*/
	FBTime	GetDuration();		

	/**	Get the signed duration value of a timespan.
	*	\return \b Signed duration of the timespan.
	*/
	FBTime	GetSignedDuration(); 

	/**	Get the direction of the timespan.
	*	Returns 1 if positive, -1 otherwise.
	*	\return Direction of timespan.
	*/
	int		GetDirection();

	/**	Verify if \e pTime is contained within the timespan.
	*	\param	pTime Time to verify if within the timespan.
	*	\return \b true if \e pTime is within the timespan.
	*/
	bool	operator& (FBTime &pTime);
	
	/**	Calculate the intersection of two timespans.
	*	\return	Timespan representing the intersection of the two timespans.
	*/
	FBTimeSpan operator& (FBTimeSpan &pTimeSpan);	
};
///////////////////////////////////////////////////////////////////////////////////
// Global
////////////////////////////////////////////////////////////////////////////////////
#ifdef FBSDKUseNamespace
	}
#endif

#endif

