//*****************************************************************************
/*!
   \file xsi_renderer.h
   \brief Renderer class declaration.

    Copyright 1998-2006 Avid Technology, Inc. and its licensors. All rights
   reserved. This file contains confidential and proprietary information of
   Avid Technology, Inc., and is subject to the terms of the SOFTIMAGE|XSI
   end user license agreement (or EULA).
*/
//*****************************************************************************

#if (_MSC_VER > 1000) || defined(SGI_COMPILER)
#pragma once
#endif

#ifndef __XSIRENDERER_H__
#define __XSIRENDERER_H__

#include <xsi_status.h>
#include <xsi_longarray.h>

namespace XSI {

//*****************************************************************************
/*! \class Renderer xsi_renderer.h
	\brief This object is provided as the source object available from the
	RendererContext for a Plugin-based \xt cus_render Custom %Renderer \endxt object.

	\sa PluginRegistrar::RegisterRenderer
	\since 6.01
*/
//*****************************************************************************

class SICPPSDKDECL Renderer : public SIObject
{
public:

	/*! Default constructor. */
	Renderer();

	/*! Default destructor. */
	~Renderer();

	/*! Constructor.
	\param in_ref constant reference object.
	*/
	Renderer(const CRef& in_ref);

	/*! Copy constructor.
	\param in_obj constant class object.
	*/
	Renderer(const Renderer& in_obj);

	/*! Returns true if a given class type is compatible with this API class.
	\param in_ClassID class type.
	\return true if the class is compatible, false otherwise.
	*/
	bool IsA( siClassID in_ClassID) const;

	/*! Returns the type of the API class.
	\return The class type.
	*/
	siClassID GetClassID() const;

	/*! Creates an object from another object.
	\param in_obj constant class object.
	\return The new Renderer object.
	*/
	Renderer& operator=(const Renderer& in_obj);

	/*! Creates an object from a reference object. The newly created object is
	set to empty if the input reference object is not compatible.
	\param in_ref constant class object.
	\return The new Renderer object.
	*/
	Renderer& operator=(const CRef& in_ref);


	/*! Sets the types of processes this rendering engine can handle as a list
	of \link siRenderProcessType enumeration values.
	\note This function should only be called from the renderer's
		\xt cb_Renderer_Init Init \endxt callback. In any other callback
		it has no effect.
	\param in_types List of process types supported by this renderer.
	\return CStatus::OK Process types set successfully
	\return CStatus::Fail No valid process types given.
	\return CStatus::AccessDenied Not in initialization mode.
	*/
	CStatus PutProcessTypes( const CLongArray &in_types );

	/*! Sets the name of the archive format and the default extension if the renderer
	supports render archives. The caller can also indicate whether the archive format
	supports multi-frame output to a single archive file.
	\note This function should only be called from the renderer's
		\xt cb_Renderer_Init Init \endxt callback. In any other callback
		it has no effect.
	\param in_strName The name of the format to use. This is used for display
		purposes, such as browsing.
	\param in_strExtension The standard file extension used for the format. This
		is appended automatically when creating archive output file paths from the
		archive filename template.
	\param in_bMultiFrame True if the archive format supports multiple frames in a
		single file. Otherwise %XSI always writes out separate files for each frame.
	\return CStatus::OK Archive settings set successfully.
	\return CStatus::AccessDenied Not in initialization mode.
	*/
	CStatus PutArchiveFormat( const CString &in_strName, const CString &in_strExtension, bool in_bMultiFrame );

	/*! Adds a new property that should be created for the specified type. The last property
	added for a specified type will be the one used.
	\note This function should only be called from the renderer's
		\xt cb_Renderer_Init Init \endxt callback. In any other callback
		it has no effect.
	\param in_eType The type of render property associated with the property given.
	\param in_strName The name of the property to create for the render property type.
	\return CStatus::OK Property type added successfully.
	\return CStatus::AccessDenied Not in initialization mode.
	*/
	CStatus AddProperty( siRenderPropertyType in_eType, const CString &in_strName );

	/*! Adds a new output image format that is supported by this renderer. The
	format will be available to the user to select for controlling which output
	format to write the rendered framebuffers out as. Used with Renderer::AddOutputImageFormatSubType
	to create a complete output image format description.
	\note This function should only be called from the renderer's
		\xt cb_Renderer_Init Init \endxt callback. In any other callback
		it has no effect.
	\param in_strName The name of the format as shown to the user.
	\param in_strExtension The format's extension.
	\return CStatus::OK Image format definition begin accepted.
	\return CStatus::AccessDenied Not in initialization mode.
	*/
	CStatus AddOutputImageFormat( const CString &in_strName, const CString &in_strExtension );

	/*! Adds a new sub-type to the image format. The sub-type is based on a render channel type
	and controls which formats, and sub-types thereof, will be offered to the user based on the
	render channel type of the render channel on which a framebuffer is based.
	\note This function should only be called from the renderer's
		\xt cb_Renderer_Init Init \endxt callback. In any other callback
		it has no effect.
	\param in_eChannelType Render channel type to associate this image format sub-type with.
	\param in_strDataType The data type being output from the render channel.
	\param in_eBitDepth The resolution of each individual component of the data type.
	\return CStatus::OK Sub-type successfully added.
	\return CStatus::AccessDenied Not in initialization mode or an output image format has not
		been added yet.
	*/
	CStatus AddOutputImageFormatSubType( siRenderChannelType in_eChannelType, const CString &in_strDataType, siImageBitDepth in_eBitDepth );

	/*! Locks the scene data from modification while the render data is being pushed. It is very
	important that this function be called before any other SDK access function is called, otherwise
	other running threads could potentially modify the data while the rendering thread is trying
	to access it.

	If the function returns CStatus::Abort the render callback should return with CStatus::Abort to
	indicate to the render controller that an abort has been successfully received. In that case
	Renderer::UnlockSceneData should not be called, since the lock has been rejected anyway.
	\note This function should only be called from the renderer's
		\xt cb_Renderer_Init Init \endxt callback. In any other callback
		it has no effect.
	\return CStatus::OK The scene data is locked.
	\return CStatus::Abort The rendering engine is being aborted and should return immediately.
	*/
	CStatus LockSceneData( );

	/*! Unlocks the scene data after locking. It is important that Renderer::LockSceneData
	be paired with this function, otherwise the scene database could be left locked after the
	render process is done, leaving %XSI in a locked up state.
	\note This function should only be called from the renderer's
		\xt cb_Renderer_Init Init \endxt callback. In any other callback
		it has no effect.
	\return CStatus:OK The scene data is now unlocked.
	*/
	CStatus UnlockSceneData( );

	private:
	Renderer * operator&() const;
	Renderer * operator&();
};

};
#endif // __XSIRENDERER_H__
