/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2004.
-------------------------------------------------------------------------
$Id$
$DateTime$

-------------------------------------------------------------------------
History:
- 01:2006		: Vista Essentials created by Jan Mller
- 23/01/2007: Modified by  Timur Davidenko.

*************************************************************************/
#include "StdAfx.h"
#include "Setup.h"

#include <rpcsal.h>
#include <gameux.h>
#include <strsafe.h>
#include <shlobj.h>

#include "LanguageMap.h"

using std::string;

//#define SHOW_DEBUG_MSGBOXES 

char* ConvertLPWSTRToLPSTR (LPWSTR lpwszStrIn, LPSTR pszOut)
{
	if (lpwszStrIn != NULL)
	{
		int nInputStrLen = int(wcslen (lpwszStrIn));

		// Double NULL Termination
		int nOutputStrLen = WideCharToMultiByte (CP_ACP, 0, lpwszStrIn, nInputStrLen, NULL, 0, 0, 0) + 2;
		pszOut = new char [nOutputStrLen];

		if (pszOut)
		{
			memset (pszOut, 0x00, nOutputStrLen);
			WideCharToMultiByte(CP_ACP, 0, lpwszStrIn, nInputStrLen, pszOut, nOutputStrLen, 0, 0);
		}
	}
	return pszOut;
}

HRESULT ConvertStringToGUID( const WCHAR* strSrc, GUID* pGuidDest )
{
	UINT aiTmp[10];

	if( swscanf_s( strSrc, L"{%8X-%4X-%4X-%2X%2X-%2X%2X%2X%2X%2X%2X}",
		&pGuidDest->Data1, 
		&aiTmp[0], &aiTmp[1], 
		&aiTmp[2], &aiTmp[3],
		&aiTmp[4], &aiTmp[5],
		&aiTmp[6], &aiTmp[7],
		&aiTmp[8], &aiTmp[9] ) != 11 )
	{
		ZeroMemory( pGuidDest, sizeof(GUID) );
		return E_FAIL;
	}
	else
	{
		pGuidDest->Data2       = (USHORT) aiTmp[0];
		pGuidDest->Data3       = (USHORT) aiTmp[1];
		pGuidDest->Data4[0]    = (BYTE) aiTmp[2];
		pGuidDest->Data4[1]    = (BYTE) aiTmp[3];
		pGuidDest->Data4[2]    = (BYTE) aiTmp[4];
		pGuidDest->Data4[3]    = (BYTE) aiTmp[5];
		pGuidDest->Data4[4]    = (BYTE) aiTmp[6];
		pGuidDest->Data4[5]    = (BYTE) aiTmp[7];
		pGuidDest->Data4[6]    = (BYTE) aiTmp[8];
		pGuidDest->Data4[7]    = (BYTE) aiTmp[9];
		return S_OK;
	}
}

//creates a link (game explorer tasks)
HRESULT CreateLink(WCHAR *lpszPathObj, WCHAR *lpszPathLink, WCHAR *lpszDesc, WCHAR *lpszArgs = NULL)
{
	HRESULT hres;
	IShellLinkW* psl;

	// Get a pointer to the IShellLink interface.
	hres = CoInitialize( 0 );
	hres = CoCreateInstance(CLSID_ShellLink, NULL, CLSCTX_INPROC_SERVER,
		IID_IShellLinkW, (LPVOID*)&psl);
	if (SUCCEEDED(hres))
	{
		IPersistFile* ppf;

		// Set the path to the shortcut target and add the description.
		psl->SetPath(lpszPathObj);
		psl->SetDescription(lpszDesc);
		if(lpszArgs)
			psl->SetArguments(lpszArgs);

		// Query IShellLink for the IPersistFile interface for saving the
		// shortcut in persistent storage.
		hres = psl->QueryInterface(IID_IPersistFile, (LPVOID*)&ppf);

		if (SUCCEEDED(hres))
		{
			// Save the link by calling IPersistFile::Save.
			hres = ppf->Save(lpszPathLink, TRUE);
			ppf->Release();
		}
		psl->Release();
	}
	return hres;
}

//creates a link (game explorer tasks)
HRESULT CreateLinkA(LPCSTR lpszPathObj, LPCSTR lpszPathLink, LPCSTR lpszDesc, LPCSTR lpszArgs = NULL)
{
	HRESULT hres;
	IShellLink* psl;

	// Get a pointer to the IShellLink interface.
	hres = CoInitialize( 0 );
	hres = CoCreateInstance(CLSID_ShellLink, NULL, CLSCTX_INPROC_SERVER,
		IID_IShellLink, (LPVOID*)&psl);
	if (SUCCEEDED(hres))
	{
		IPersistFile* ppf;

		// Set the path to the shortcut target and add the description.
		psl->SetPath(lpszPathObj);
		psl->SetDescription(lpszDesc);
		if(lpszArgs)
			psl->SetArguments(lpszArgs);

		// Query IShellLink for the IPersistFile interface for saving the
		// shortcut in persistent storage.
		hres = psl->QueryInterface(IID_IPersistFile, (LPVOID*)&ppf);

		if (SUCCEEDED(hres))
		{
			WCHAR wsz[MAX_PATH];

			// Ensure that the string is Unicode.
			MultiByteToWideChar(CP_ACP, 0, lpszPathLink, -1, wsz, MAX_PATH);

			// Save the link by calling IPersistFile::Save.
			hres = ppf->Save(wsz, TRUE);
			ppf->Release();
		}
		psl->Release();
	}
	return hres;
}

//--------------------------------------------------------------------------------------
// Adds a game to the Game Explorer
//--------------------------------------------------------------------------------------
STDAPI AddToGameExplorer( WCHAR* strGDFBinPath, WCHAR *strGameInstallPath, 
												 GAME_INSTALL_SCOPE InstallScope, WCHAR *strRegistryPath)
{
#ifdef SHOW_DEBUG_MSGBOXES
	MessageBox( NULL,"AddToGameExplorer","Debug",MB_OK );
#endif


	HRESULT hr = E_FAIL;
	bool    bCleanupCOM = false;
	BOOL    bHasAccess = FALSE;
	BSTR    bstrGDFBinPath = NULL;
	BSTR    bstrGameInstallPath = NULL;
	IGameExplorer* pFwGameExplorer = NULL;

	if( strGDFBinPath == NULL || strGameInstallPath == NULL )
	{
		assert( false );
		return E_INVALIDARG;
	}

	bstrGDFBinPath = SysAllocString( strGDFBinPath );
	bstrGameInstallPath = SysAllocString( strGameInstallPath );
	if( bstrGDFBinPath == NULL || bstrGameInstallPath == NULL )
	{
		hr = E_OUTOFMEMORY;
		goto LCleanup;
	}


	hr = CoInitialize( 0 );
	bCleanupCOM = SUCCEEDED(hr); 

	GUID m_GUID;
	CoCreateGuid( &m_GUID ); 
	printf("Created GUID \n");

	// Create an instance of the Game Explorer Interface
	hr = CoCreateInstance( __uuidof(GameExplorer), NULL, CLSCTX_INPROC_SERVER, 
		__uuidof(IGameExplorer), (void**) &pFwGameExplorer );

	if( FAILED(hr) || pFwGameExplorer == NULL )
	{
		printf("Failed or no game explorer created: \n");
		// Depending on GAME_INSTALL_SCOPE, write to:
		//      HKLM\Software\Microsoft\Windows\CurrentVersion\GameUX\GamesToFindOnWindowsUpgrade\{GUID}\
		// or
		//      HKCU\Software\Classes\Software\Microsoft\Windows\CurrentVersion\GameUX\GamesToFindOnWindowsUpgrade\{GUID}\
		// and write there these 2 string values: GDFBinaryPath and GameInstallPath 
		//
		HKEY hKeyGamesToFind = NULL, hKeyGame = NULL;
		LONG lResult;
		DWORD dwDisposition;
		if( InstallScope == GIS_CURRENT_USER )
			lResult = RegCreateKeyEx( HKEY_CURRENT_USER, "Software\\Classes\\Software\\Microsoft\\Windows\\CurrentVersion\\GameUX\\GamesToFindOnWindowsUpgrade", 
			0, NULL, 0, KEY_WRITE, NULL, &hKeyGamesToFind, &dwDisposition );
		else
			lResult = RegCreateKeyEx( HKEY_LOCAL_MACHINE, "Software\\Microsoft\\Windows\\CurrentVersion\\GameUX\\GamesToFindOnWindowsUpgrade", 
			0, NULL, 0, KEY_WRITE, NULL, &hKeyGamesToFind, &dwDisposition );

		if(lResult == ERROR_SUCCESS) 
		{
			WCHAR strGameInstanceGUID[256];
			StringFromGUID2(m_GUID, strGameInstanceGUID, 256);

			lResult = RegCreateKeyExW( hKeyGamesToFind, strGameInstanceGUID, 0, NULL, 0, KEY_WRITE, NULL, &hKeyGame, &dwDisposition );
			if(lResult == ERROR_SUCCESS) 
			{
				printf("Setting up Windows XP registry for update to Vista");
				size_t nGDFBinPath = 0, nGameInstallPath = 0;

				StringCchLengthW( strGDFBinPath, MAX_PATH, &nGDFBinPath );
				StringCchLengthW( strGameInstallPath, MAX_PATH, &nGameInstallPath );

				RegSetValueExW( hKeyGame, L"GDFBinaryPath", 0, REG_SZ, (const BYTE*)strGDFBinPath, (DWORD)((nGDFBinPath + 1)*sizeof(WCHAR)) );
				RegSetValueExW( hKeyGame, L"GameInstallPath", 0, REG_SZ, (const BYTE*)strGameInstallPath, (DWORD)((nGameInstallPath + 1)*sizeof(WCHAR)) );

				//write GUID in registry ...
				HKEY hk;
				DWORD dwDisp;
				if(InstallScope == GIS_ALL_USERS)
					hr = RegCreateKeyExW(HKEY_LOCAL_MACHINE, L"Software", 0, NULL, REG_OPTION_NON_VOLATILE,
					KEY_WRITE, NULL, &hk, &dwDisp);
				else
					hr = RegCreateKeyExW(HKEY_CURRENT_USER, L"Software", 0, NULL, REG_OPTION_NON_VOLATILE,
					KEY_WRITE, NULL, &hk, &dwDisp);

				hr = RegCreateKeyEx(hk, "Crytek", 0, NULL, REG_OPTION_NON_VOLATILE,
					KEY_WRITE, NULL, &hk, &dwDisp);
				hr = RegCreateKeyExW(hk, strRegistryPath, 0, NULL, REG_OPTION_NON_VOLATILE,
					KEY_WRITE, NULL, &hk, &dwDisp);
				size_t nguidStringW = 0;
				StringCchLengthW( strGameInstanceGUID, MAX_PATH, &nguidStringW );
				hr = RegSetValueExW(hk, L"GUID", 0, REG_SZ, (const BYTE*)(strGameInstanceGUID), (DWORD)((nguidStringW + 1)*sizeof(WCHAR)));
			}
			if( hKeyGame ) RegCloseKey( hKeyGame );
		}
		if( hKeyGamesToFind )
			RegCloseKey( hKeyGamesToFind );
	}
	else
	{			//Windows Vista Game Explorer path
		printf("Game explorer available \n");
		hr = pFwGameExplorer->VerifyAccess( bstrGDFBinPath, &bHasAccess );
		printf("HRESULT: %X\n",hr);
		if( SUCCEEDED(hr) || bHasAccess )
		{
			printf("Game explorer access verified - adding game \n");
			hr = pFwGameExplorer->AddGame( bstrGDFBinPath, bstrGameInstallPath, 
				InstallScope, &m_GUID );	//add Crysis to game explorer

			if( SUCCEEDED(hr))	//now save GUID to the registry
			{
				printf("Writing game instance ID in registry ...\n");

				HKEY hk;
				DWORD dwDisp;
				if(InstallScope == GIS_ALL_USERS)
					hr = RegCreateKeyEx(HKEY_LOCAL_MACHINE, "Software", 0, NULL, REG_OPTION_NON_VOLATILE,
					KEY_WRITE, NULL, &hk, &dwDisp);
				else
					hr = RegCreateKeyEx(HKEY_CURRENT_USER, "Software", 0, NULL, REG_OPTION_NON_VOLATILE,
					KEY_WRITE, NULL, &hk, &dwDisp);

				hr = RegCreateKeyEx(hk, "Crytek", 0, NULL, REG_OPTION_NON_VOLATILE,
					KEY_WRITE, NULL, &hk, &dwDisp);
				hr = RegCreateKeyExW(hk, strRegistryPath, 0, NULL, REG_OPTION_NON_VOLATILE,
					KEY_WRITE, NULL, &hk, &dwDisp);
				WCHAR guidStringW[256];
				StringFromGUID2(m_GUID, guidStringW, 256);
				size_t nguidStringW = 0;
				StringCchLengthW( guidStringW, MAX_PATH, &nguidStringW );
				hr = RegSetValueExW(hk, L"GUID", 0, REG_SZ, (const BYTE*)(guidStringW), (DWORD)((nguidStringW + 1)*sizeof(WCHAR)));
			}
			//else
			//	MessageBox(NULL, "Game explorer refused to add the game.", "ERROR", MB_OK);
		}
		else
			MessageBox(NULL, "Game explorer access failed.", "ERROR", MB_OK);
	}

LCleanup:
	if( bstrGDFBinPath ) SysFreeString( bstrGDFBinPath );
	if( bstrGameInstallPath ) SysFreeString( bstrGameInstallPath );
	if( pFwGameExplorer ) pFwGameExplorer->Release();
	if( bCleanupCOM ) CoUninitialize();

	return hr;
}

//--------------------------------------------------------------------------------------
// Removes a game from the Game Explorer
//--------------------------------------------------------------------------------------
STDAPI RemoveFromGameExplorer( GUID *pInstanceGUID )
{   
#ifdef SHOW_DEBUG_MSGBOXES
	MessageBox( NULL,"RemoveFromGameExplorer","Debug",MB_OK );
#endif


	HRESULT hr = E_FAIL;
	bool    bCleanupCOM = false;
	IGameExplorer* pFwGameExplorer = NULL;

	hr = CoInitialize( 0 );
	bCleanupCOM = SUCCEEDED(hr); 

	// Create an instance of the Game Explorer Interface
	hr = CoCreateInstance( __uuidof(GameExplorer), NULL, CLSCTX_INPROC_SERVER, 
		__uuidof(IGameExplorer), (void**) &pFwGameExplorer );
	if( FAILED(hr) || pFwGameExplorer == NULL )
	{
		hr = E_FAIL;
		goto LCleanup;
	}

	// Remove the game from the Game Explorer
	hr = pFwGameExplorer->RemoveGame( *pInstanceGUID );

LCleanup:
	if( pFwGameExplorer ) pFwGameExplorer->Release();
	if( bCleanupCOM ) CoUninitialize();

	return hr;
}


//this adds game tasks to the game explorer and returns true on success
bool CreateGameExplorerTasksA(WCHAR *gameBinaryPath, GAME_INSTALL_SCOPE InstallScope, WCHAR *strRegistryPath)
{
#ifdef SHOW_DEBUG_MSGBOXES
	MessageBox( NULL,"CreateGameExplorerTasks","Debug",MB_OK );
#endif

	//first retrieve game GUID
	char guidString[256];

	HKEY hk;
	DWORD dwDisp;
	HRESULT hr;
	if(InstallScope == GIS_ALL_USERS)
		hr = RegCreateKeyEx(HKEY_LOCAL_MACHINE, "Software", 0, NULL, REG_OPTION_NON_VOLATILE,
		KEY_QUERY_VALUE, NULL, &hk, &dwDisp);
	else
		hr = RegCreateKeyEx(HKEY_CURRENT_USER, "Software", 0, NULL, REG_OPTION_NON_VOLATILE,
		KEY_QUERY_VALUE, NULL, &hk, &dwDisp);

	hr = RegCreateKeyEx(hk, "Crytek", 0, NULL, REG_OPTION_NON_VOLATILE,
		KEY_QUERY_VALUE, NULL, &hk, &dwDisp);
	hr = RegCreateKeyExW(hk, strRegistryPath, 0, NULL, REG_OPTION_NON_VOLATILE,
		KEY_QUERY_VALUE, NULL, &hk, &dwDisp);

	//get the guid from the registry
	DWORD size = 256;
	DWORD type;
	hr = RegQueryValueExA(hk, "GUID", 0, &type, (BYTE*)guidString, &size);

	if(FAILED(hr))
		return false;

	char folderString[MAX_PATH];
	//get folder path ...
	if(InstallScope == GIS_CURRENT_USER)
		SHGetFolderPath(NULL, CSIDL_LOCAL_APPDATA, NULL, 0, LPSTR(&folderString));
	else if(InstallScope == GIS_ALL_USERS)
		SHGetFolderPath(NULL, CSIDL_COMMON_APPDATA, NULL, 0, LPSTR(&folderString));
	else
		return false;

	string fullPath(folderString);		//this will be our game's task directory
	fullPath.append("\\Microsoft\\Windows\\GameExplorer\\");
	fullPath.append(guidString);

	string playTasks = fullPath;			//the full playTasks path
	playTasks.append("\\PlayTasks\\0\\");

	//create default play game task directory and all subdirectories ...
	int result = SHCreateDirectoryEx(NULL,playTasks.c_str(), NULL);

	//add default double click link
	string defaultPlayTask = playTasks;
	defaultPlayTask.append("Play.lnk");

	char* temp = 0;
	char* binaryPath = ConvertLPWSTRToLPSTR(gameBinaryPath, temp);
	delete[] temp;

	if(FAILED(CreateLinkA(binaryPath, defaultPlayTask.c_str(), "Play Crysis!", "")))
		return false;

	//****************************************************

	playTasks = fullPath;			//the full playTasks path
	playTasks.append("\\PlayTasks\\1\\");

	//create default play game task directory and all subdirectories ...
	result = SHCreateDirectoryEx(NULL,playTasks.c_str(), NULL);

	defaultPlayTask = playTasks;
	defaultPlayTask.append("PlayDX10.lnk");

	if(FAILED(CreateLinkA(binaryPath, defaultPlayTask.c_str(), "Play Crysis!", "-DX10")))
		return false;

	//****************************************************

	playTasks = fullPath;			//the full playTasks path
	playTasks.append("\\PlayTasks\\2\\");

	//create default play game task directory and all subdirectories ...
	result = SHCreateDirectoryEx(NULL,playTasks.c_str(), NULL);

	defaultPlayTask = playTasks;
	defaultPlayTask.append("PlayDX9.lnk");

	if(FAILED(CreateLinkA(binaryPath, defaultPlayTask.c_str(), "Play Crysis!", "-DX9")))
		return false;

	//****************************************************

	//create task to open hele folder
	string supportTasks = fullPath;
	supportTasks.append("\\PlayTasks\\3\\");
	//create support directory
	result = SHCreateDirectoryEx(NULL,supportTasks.c_str(), NULL);

	string helpFolderTask = supportTasks;
	helpFolderTask.append("HelpFolder.lnk");

	string helpFolder = binaryPath;
	size_t pos = helpFolder.rfind("\\");
	if(pos != string::npos)
		helpFolder = helpFolder.substr(0, pos);
	pos = helpFolder.rfind("\\");
	if(pos != string::npos)
	{
		helpFolder = helpFolder.substr(0, pos);
		helpFolder.append("\\support\\");
	}

	if(FAILED(CreateLinkA(helpFolder.c_str(), helpFolderTask.c_str(), "Open help files.", "")))
		return false;

	//****************************************************

	//add further task links
	//->uninstall task?

	return true;
}

//this adds game tasks to the game explorer and returns true on success
bool CreateGameExplorerTasks(WCHAR *gameBinaryPath, GAME_INSTALL_SCOPE InstallScope, WCHAR *strRegistryPath, LocTasks &tasks, WCHAR *strGamePath)
{
#ifdef SHOW_DEBUG_MSGBOXES
	MessageBox( NULL,"CreateGameExplorerTasks","Debug",MB_OK );
#endif

	//first retrieve game GUID
	WCHAR guidString[MAX_PATH];

	HKEY hk;
	DWORD dwDisp;
	HRESULT hr;
	if(InstallScope == GIS_ALL_USERS)
		hr = RegCreateKeyEx(HKEY_LOCAL_MACHINE, "Software", 0, NULL, REG_OPTION_NON_VOLATILE,
		KEY_QUERY_VALUE, NULL, &hk, &dwDisp);
	else
		hr = RegCreateKeyEx(HKEY_CURRENT_USER, "Software", 0, NULL, REG_OPTION_NON_VOLATILE,
		KEY_QUERY_VALUE, NULL, &hk, &dwDisp);

	hr = RegCreateKeyEx(hk, "Crytek", 0, NULL, REG_OPTION_NON_VOLATILE,
		KEY_QUERY_VALUE, NULL, &hk, &dwDisp);
	hr = RegCreateKeyExW(hk, strRegistryPath, 0, NULL, REG_OPTION_NON_VOLATILE,
		KEY_QUERY_VALUE, NULL, &hk, &dwDisp);

	//get the guid from the registry
	DWORD size = 256;
	DWORD type;
	hr = RegQueryValueExW(hk, L"GUID", 0, &type, (BYTE*)guidString, &size);

	if(FAILED(hr))
		return false;

	WCHAR folderString[MAX_PATH];
	//get folder path ...
	if(InstallScope == GIS_CURRENT_USER)
		SHGetFolderPathW(NULL, CSIDL_LOCAL_APPDATA, NULL, 0, folderString);
	else if(InstallScope == GIS_ALL_USERS)
		SHGetFolderPathW(NULL, CSIDL_COMMON_APPDATA, NULL, 0, folderString);
	else
		return false;

	wstring fullPath(folderString);		//this will be our game's task directory
	fullPath.append(L"\\Microsoft\\Windows\\GameExplorer\\");
	fullPath.append(guidString);

	wstring playTasks = fullPath;			//the full playTasks path
	playTasks.append(L"\\PlayTasks\\0\\");

	//create default play game task directory and all subdirectories ...
	int result = SHCreateDirectoryExW(NULL,playTasks.c_str(), NULL);

	//add default double click link
	wstring defaultPlayTask = playTasks;
	defaultPlayTask.append(tasks.play);
	defaultPlayTask.append(L".lnk");

	if(FAILED(CreateLink(gameBinaryPath, (WCHAR*)(defaultPlayTask.c_str()), (WCHAR*)(tasks.play.c_str()), L"")))
	{
		MessageBox(NULL, "Failed creating game explorer tasks", "ERROR", MB_OK);
		return false;
	}

	//****************************************************

	playTasks = fullPath;			//the full playTasks path
	playTasks.append(L"\\PlayTasks\\1\\");

	//create default play game task directory and all subdirectories ...
	result = SHCreateDirectoryExW(NULL,playTasks.c_str(), NULL);

	defaultPlayTask = playTasks;
	defaultPlayTask.append(tasks.playDX9);
	defaultPlayTask.append(L".lnk");

	if(FAILED(CreateLink(gameBinaryPath, (WCHAR*)(defaultPlayTask.c_str()), (WCHAR*)(tasks.playDX9.c_str()), L"-DX9")))
	{
		MessageBox(NULL, "Failed creating 2nd game explorer tasks", "ERROR", MB_OK);
		return false;
	}

	//****************************************************

	playTasks = fullPath;			//the full playTasks path
	playTasks.append(L"\\PlayTasks\\2\\");

	//create default play game task directory and all subdirectories ...
	result = SHCreateDirectoryExW(NULL,playTasks.c_str(), NULL);

	defaultPlayTask = playTasks;
	defaultPlayTask.append(tasks.playDX10);
	defaultPlayTask.append(L".lnk");

	if(FAILED(CreateLink(gameBinaryPath, (WCHAR*)(defaultPlayTask.c_str()), (WCHAR*)(tasks.playDX10.c_str()), L"-DX10")))
	{
		MessageBox(NULL, "Failed creating 3rd game explorer tasks", "ERROR", MB_OK);
		return false;
	}

	//****************************************************

	//create task to open hele folder
	wstring supportTasks = fullPath;
	supportTasks.append(L"\\PlayTasks\\3\\");
	//create support directory
	result = SHCreateDirectoryExW(NULL,supportTasks.c_str(), NULL);

	wstring helpFolderTask = supportTasks;
	helpFolderTask.append(tasks.help);
	helpFolderTask.append(L".lnk");

	wstring gamePath = strGamePath;
	gamePath.append(L"Support\\");
	gamePath = L"\"" + gamePath + L"\"";

	WCHAR strWindowsDirectory[2048];
	SHGetFolderPathW(NULL, CSIDL_WINDOWS, NULL, 0, LPWSTR(strWindowsDirectory));
	wstring explorerPath = wstring(strWindowsDirectory) + L"\\explorer.exe";

	if(FAILED(CreateLink((WCHAR*)(explorerPath.c_str()), (WCHAR*)(helpFolderTask.c_str()), (WCHAR*)(tasks.help.c_str()), (WCHAR*)(gamePath.c_str()))))
	{
		MessageBox(NULL, "Failed creating game explorer help link.", "ERROR", MB_OK);
		return false;
	}

	//****************************************************

	//add further task links
	//->uninstall task?

	return true;
}

// strSavedGameExtension should begin with a period. ex: .ExampleSaveGame
// strLaunchPath should be enclosed in quotes.  ex: "%ProgramFiles%\ExampleGame\ExampleGame.exe"
// strCommandLineToLaunchSavedGame should be enclosed in quotes.  ex: "%1".  If NULL, it defaults to "%1"
//-----------------------------------------------------------------------------
STDAPI SetupRichSavedGames( WCHAR* strSavedGameExtension, WCHAR* strLaunchPath, 
													 WCHAR* strCommandLineToLaunchSavedGame ) 
{
#ifdef SHOW_DEBUG_MSGBOXES
	MessageBox( NULL,"SetupRichSavedGames","Debug",MB_OK );
#endif


	HKEY hKey = NULL;
	LONG lResult;
	DWORD dwDisposition;
	WCHAR strExt[256];
	WCHAR strType[256];
	WCHAR strCmdLine[256];
	WCHAR strTemp[512];
	size_t nStrLength = 0;

	// Validate args 
	if( strLaunchPath == NULL || strSavedGameExtension == NULL )
	{
		assert( false );
		return E_INVALIDARG;
	}

	// Setup saved game extension arg - make sure there's a period at the start
	if( strSavedGameExtension[0] == L'.' )
	{
		StringCchCopyW( strExt, 256, strSavedGameExtension );
		StringCchPrintfW( strType, 256, L"%sType", strSavedGameExtension+1 );
	}
	else
	{
		StringCchPrintfW( strExt, 256, L".%s", strSavedGameExtension );
		StringCchPrintfW( strType, 256, L"%sType", strSavedGameExtension );
	}

	// Create default command line arg if none supplied
	if( strCommandLineToLaunchSavedGame )
		StringCchCopyW( strCmdLine, 256, strCommandLineToLaunchSavedGame );
	else
		StringCchCopyW( strCmdLine, 256, L"load\"%1\"" );

	// Create file association & metadata regkeys
	lResult = RegCreateKeyExW( HKEY_CLASSES_ROOT, strExt, 0, NULL, 0, KEY_WRITE, NULL, &hKey, &dwDisposition );
	if( ERROR_SUCCESS == lResult ) 
	{
		// Create the following regkeys:
		//
		// [HKEY_CLASSES_ROOT\.ExampleGameSave]
		// (Default)="ExampleGameSaveFileType"
		//
		StringCchLengthW( strType, 256, &nStrLength );
		RegSetValueExW( hKey, L"", 0, REG_SZ, (BYTE*)strType, (DWORD)((nStrLength + 1)*sizeof(WCHAR)) );

		// Create the following regkeys:
		//
		// [HKEY_CLASSES_ROOT\.ExampleGameSave\ShellEx\{BB2E617C-0920-11d1-9A0B-00C04FC2D6C1}]
		// (Default)="{4E5BFBF8-F59A-4e87-9805-1F9B42CC254A}"
		//
		HKEY hSubKey = NULL;
		lResult = RegCreateKeyExW( hKey, L"ShellEx\\{BB2E617C-0920-11d1-9A0B-00C04FC2D6C1}", 0, NULL, 0, KEY_WRITE, NULL, &hSubKey, &dwDisposition );
		if( ERROR_SUCCESS == lResult ) 
		{
			StringCchPrintfW( strTemp, 512, L"{4E5BFBF8-F59A-4e87-9805-1F9B42CC254A}" );
			StringCchLengthW( strTemp, 256, &nStrLength );
			RegSetValueExW( hSubKey, L"", 0, REG_SZ, (BYTE*)strTemp, (DWORD)((nStrLength + 1)*sizeof(WCHAR)) );
		}
		if( hSubKey ) RegCloseKey( hSubKey );
	}
	else
	{
		char resultMessage[256];
		FormatMessageA(FORMAT_MESSAGE_FROM_SYSTEM, NULL, lResult, 0, resultMessage, 256, NULL);
		printf("Error: %s, could not create file association in registry!", resultMessage);
	}

	if( hKey ) 
		RegCloseKey( hKey );

	lResult = RegCreateKeyExW( HKEY_CLASSES_ROOT, strType, 0, NULL, 0, KEY_WRITE, NULL, &hKey, &dwDisposition );
	if( ERROR_SUCCESS == lResult ) 
	{
		// Create the following regkeys:
		//
		// [HKEY_CLASSES_ROOT\ExampleGameSaveFileType]
		// PreviewTitle="prop:System.Game.RichSaveName;System.Game.RichApplicationName"
		// PreviewDetails="prop:System.Game.RichLevel;System.DateChanged;System.Game.RichComment;System.DisplayName;System.DisplayType"
		//
		size_t nPreviewDetails = 0, nPreviewTitle = 0;
		WCHAR* strPreviewTitle = L"prop:System.Game.RichSaveName;System.Game.RichApplicationName";
		WCHAR* strPreviewDetails = L"prop:System.Game.RichLevel;System.DateChanged;System.Game.RichComment;System.DisplayName;System.DisplayType";
		StringCchLengthW( strPreviewTitle, 256, &nPreviewTitle );
		StringCchLengthW( strPreviewDetails, 256, &nPreviewDetails );
		RegSetValueExW( hKey, L"PreviewTitle", 0, REG_SZ, (BYTE*)strPreviewTitle, (DWORD)((nPreviewTitle + 1)*sizeof(WCHAR)) );
		RegSetValueExW( hKey, L"PreviewDetails", 0, REG_SZ, (BYTE*)strPreviewDetails, (DWORD)((nPreviewDetails + 1)*sizeof(WCHAR)) );

		// Create the following regkeys:
		//
		// [HKEY_CLASSES_ROOT\ExampleGameSaveFileType\Shell\Open\Command]
		// (Default)=""%ProgramFiles%\ExampleGame.exe" "%1""
		//
		HKEY hSubKey = NULL;
		lResult = RegCreateKeyExW( hKey, L"Shell\\Open\\Command", 0, NULL, 0, KEY_WRITE, NULL, &hSubKey, &dwDisposition );
		if( ERROR_SUCCESS == lResult ) 
		{
			StringCchPrintfW( strTemp, 512, L"%s %s", strLaunchPath, strCmdLine );
			StringCchLengthW( strTemp, 256, &nStrLength );
			RegSetValueExW( hSubKey, L"", 0, REG_SZ, (BYTE*)strTemp, (DWORD)((nStrLength + 1)*sizeof(WCHAR)) );
		}
		if( hSubKey ) RegCloseKey( hSubKey );
	}
	if( hKey ) RegCloseKey( hKey );

	// Create the following regkeys:
	//
	// [HKEY_LOCAL_MACHINE\Software\Microsoft\Windows\CurrentVersion\PropertySystem\PropertyHandlers\.ExampleGameSave]
	// (Default)="{ECDD6472-2B9B-4b4b-AE36-F316DF3C8D60}"
	//
	StringCchPrintfW( strTemp, 512, L"Software\\Microsoft\\Windows\\CurrentVersion\\PropertySystem\\PropertyHandlers\\%s", strExt );
	lResult = RegCreateKeyExW( HKEY_LOCAL_MACHINE, strTemp, 0, NULL, 0, KEY_WRITE, NULL, &hKey, &dwDisposition );
	if( ERROR_SUCCESS == lResult ) 
	{
		StringCchCopyW( strTemp, 512, L"{ECDD6472-2B9B-4B4B-AE36-F316DF3C8D60}" );
		StringCchLengthW( strTemp, 256, &nStrLength );
		RegSetValueExW( hKey, L"", 0, REG_SZ, (BYTE*)strTemp, (DWORD)((nStrLength + 1)*sizeof(WCHAR)) );
	}
	if( hKey ) RegCloseKey( hKey );

	return S_OK;
}

//-----------------------------------------------------------------------------
// Removes the registry keys to enable rich saved games.  
//
// strSavedGameExtension should begin with a period. ex: .ExampleSaveGame
//-----------------------------------------------------------------------------
STDAPI RemoveRichSavedGames( WCHAR* strSavedGameExtension ) 
{
#ifdef SHOW_DEBUG_MSGBOXES
	MessageBox( NULL,"RemoveRichSavedGames","Debug",MB_OK );
#endif


	WCHAR strExt[256];
	WCHAR strType[256];
	WCHAR strTemp[512];

	// Validate args 
	if( strSavedGameExtension == NULL )
	{
		assert( false );
		return E_INVALIDARG;
	}

	// Setup saved game extension arg - make sure there's a period at the start
	if( strSavedGameExtension[0] == L'.' )
	{
		StringCchCopyW( strExt, 256, strSavedGameExtension );
		StringCchPrintfW( strType, 256, L"%sType", strSavedGameExtension+1 );
	}
	else
	{
		StringCchPrintfW( strExt, 256, L".%s", strSavedGameExtension );
		StringCchPrintfW( strType, 256, L"%sType", strSavedGameExtension );
	}

	// Delete the following regkeys:
	//
	// [HKEY_CLASSES_ROOT\.ExampleGameSave]
	// [HKEY_CLASSES_ROOT\ExampleGameSaveFileType]
	// [HKEY_LOCAL_MACHINE\Software\Microsoft\Windows\CurrentVersion\PropertySystem\PropertyHandlers\.ExampleGameSave]
	RegDeleteKeyW( HKEY_CLASSES_ROOT, strExt );
	RegDeleteKeyW( HKEY_CLASSES_ROOT, strType );
	StringCchPrintfW( strTemp, 512, L"Software\\Microsoft\\Windows\\CurrentVersion\\PropertySystem\\PropertyHandlers\\%s", strExt );
	RegDeleteKeyW( HKEY_LOCAL_MACHINE, strTemp );

	return S_OK;
}

//installs to game to the game explorer and registry (not copying files ...)
STDAPI InstallGame( bool use64BitBinaries, GAME_INSTALL_SCOPE InstallScope, WCHAR *strRegistryPath = L"Crysis", WCHAR *strLanguage = L"" )
{
#ifdef SHOW_DEBUG_MSGBOXES
	MessageBox( NULL,"InstallGame","Debug",MB_OK );
#endif

	// Look up the product key in the registry.
	WCHAR strSoftwareKey[1024];
	StringCchPrintfW(strSoftwareKey, sizeof(strSoftwareKey) / sizeof(strSoftwareKey[0]), L"Software\\Crytek\\%s", strRegistryPath);
	HKEY hKey;
	if (ERROR_SUCCESS != RegOpenKeyExW(HKEY_LOCAL_MACHINE, strSoftwareKey, 0, KEY_READ, &hKey))
		hKey = 0;

	// Look up the install path in the registry.
	WCHAR strGameInstallPath[1024] = L"";
	if (hKey)
	{
		DWORD bufferSize = sizeof(strGameInstallPath) / sizeof(strGameInstallPath[0]);
		DWORD valueType;
		if (ERROR_SUCCESS != RegQueryValueExW(hKey, L"InstallDir", NULL, &valueType, reinterpret_cast<BYTE*>(strGameInstallPath), &bufferSize))
			strGameInstallPath[0] = 0;
	}

	// Determine the gdf path from the install path.
	WCHAR strGDFBinPath[1024] = L"";
	if (strGameInstallPath[0])
		StringCchPrintfW(strGDFBinPath, sizeof(strGDFBinPath) / sizeof(strGDFBinPath[0]), L"%sTools\\VistaEssentials.dll", strGameInstallPath);

	// Choose the directory name in which we'll find the binaries (relative to the install path).
	const WCHAR* binFolderName = (use64BitBinaries ? L"Bin64" : L"Bin32");

	// Determine the binaries path from the install path.
	WCHAR strExecutablePath[1024] = L"";
	if (strGameInstallPath[0])
		StringCchPrintfW(strExecutablePath, sizeof(strExecutablePath) / sizeof(strExecutablePath[0]), L"%s%s\\Crysis.exe", strGameInstallPath, binFolderName);

	if (strExecutablePath[0] && strGDFBinPath[0] && strGameInstallPath[0])
	{
		HRESULT hr = AddToGameExplorer(strGDFBinPath, strGameInstallPath, InstallScope, strRegistryPath);
		if(hr != S_OK)
			return hr;
		LanguageMap selectLanguage;
		bool taskFoldersCreated = false;
		taskFoldersCreated = CreateGameExplorerTasks(strExecutablePath, InstallScope, strRegistryPath, selectLanguage.getTasks(strLanguage), strGameInstallPath);
		//bool taskFoldersCreated = CreateGameExplorerTasksA(strExecutablePath, InstallScope, strRegistryPath);
		hr = SetupRichSavedGames(L".CRYSISJMSF", strExecutablePath, NULL);
		if(hr == S_OK && taskFoldersCreated)
			return S_OK;
		return hr;
	}

	return -1;
}

//removes all entries
STDAPI RemoveGame(GAME_INSTALL_SCOPE InstallScope, WCHAR *strRegistryPath = L"Crysis", WCHAR *strLanguage = L"" )
{
#ifdef SHOW_DEBUG_MSGBOXES
	MessageBox( NULL,"RemoveGame","Debug",MB_OK );
#endif

	LanguageMap selectLanguage;
	LocTasks tasks = selectLanguage.getTasks(strLanguage);

	//first retrieve game GUID
	WCHAR guidString[256];

	HRESULT S_FAILED = (-1);

	HKEY hk;
	HKEY sub;
	DWORD dwDisp;
	HRESULT hr;
	if(InstallScope == GIS_ALL_USERS)
		hr = RegCreateKeyExW(HKEY_LOCAL_MACHINE, L"Software", 0, NULL, REG_OPTION_NON_VOLATILE,
		KEY_QUERY_VALUE, NULL, &hk, &dwDisp);
	else
		hr = RegCreateKeyExW(HKEY_CURRENT_USER, L"Software", 0, NULL, REG_OPTION_NON_VOLATILE,
		KEY_QUERY_VALUE, NULL, &hk, &dwDisp);

	hr = RegCreateKeyExW(hk, L"Crytek", 0, NULL, REG_OPTION_NON_VOLATILE,
		KEY_QUERY_VALUE, NULL, &hk, &dwDisp);
	hr = RegCreateKeyExW(hk, strRegistryPath, 0, NULL, REG_OPTION_NON_VOLATILE,
		KEY_QUERY_VALUE, NULL, &hk, &dwDisp);
	sub = hk;

	//get the guid from the registry
	DWORD size = 256;
	DWORD type;
	hr = RegQueryValueExW(hk, L"GUID", 0, &type, (BYTE*)guidString, &size);

	if(hr!=S_OK)
	{
		printf("Could not find game GUID in registry - reinstall game by running this executable with one parameter..\n");
		return S_OK;
	}

	GUID guid;
	hr = ConvertStringToGUID(guidString, &guid);

	//delete game in game explorer (windows vista only)
	hr = RemoveFromGameExplorer(&guid);

	if(hr != S_OK)		//probably WinxP - remove WinXP upgrade entries
	{
		HKEY hKeyGamesToFind;
		DWORD dwDisposition;
		if( InstallScope == GIS_CURRENT_USER )
			RegCreateKeyExW( HKEY_CURRENT_USER, L"Software\\Classes\\Software\\Microsoft\\Windows\\CurrentVersion\\GameUX\\GamesToFindOnWindowsUpgrade", 
			0, NULL, 0, KEY_QUERY_VALUE, NULL, &hKeyGamesToFind, &dwDisposition );
		else
			RegCreateKeyExW( HKEY_LOCAL_MACHINE, L"Software\\Microsoft\\Windows\\CurrentVersion\\GameUX\\GamesToFindOnWindowsUpgrade", 
			0, NULL, 0, KEY_QUERY_VALUE, NULL, &hKeyGamesToFind, &dwDisposition );

		hr = RegDeleteKeyW(hKeyGamesToFind, guidString);
	}

	//uninstall rich save game registration
	hr = RemoveRichSavedGames(L".CRYSISJMSF");

	//remove game registry entry
	//hr = RegDeleteKeyW(sub, L"Crysis"); //Is handeled by the installer

	//remove task directories
	WCHAR folderString[MAX_PATH];
	//get folder path ...
	if(InstallScope == GIS_CURRENT_USER)
		SHGetFolderPathW(NULL, CSIDL_LOCAL_APPDATA, NULL, 0, LPWSTR(&folderString));
	else if(InstallScope == GIS_ALL_USERS)
		SHGetFolderPathW(NULL, CSIDL_COMMON_APPDATA, NULL, 0, LPWSTR(&folderString));
	else
		return S_FAILED;

	wstring fullPath(folderString);		//this will be our game's task directory
	fullPath.append(L"\\Microsoft\\Windows\\GameExplorer\\");
	fullPath.append(guidString);

	wstring playTasks = fullPath;			//the full playTasks path
	playTasks.append(L"\\PlayTasks\\");
	wstring defaultPlay = fullPath;
	defaultPlay.append(L"\\PlayTasks\\0\\");

	wstring tempTask = defaultPlay;
	tempTask.append(tasks.play);
	tempTask.append(L".lnk");
	BOOL worked = DeleteFileW(tempTask.c_str());

	worked = RemoveDirectoryW(defaultPlay.c_str());

	//***********************************

	defaultPlay = fullPath;
	defaultPlay.append(L"\\PlayTasks\\1\\");

	tempTask = defaultPlay;
	tempTask.append(tasks.playDX9);
	tempTask.append(L".lnk");
	worked = DeleteFileW(tempTask.c_str());

	worked = RemoveDirectoryW(defaultPlay.c_str());

	//***********************************

	defaultPlay = fullPath;
	defaultPlay.append(L"\\PlayTasks\\2\\");

	tempTask = defaultPlay;
	tempTask.append(tasks.playDX10);
	tempTask.append(L".lnk");
	worked = DeleteFileW(tempTask.c_str());

	worked = RemoveDirectoryW(defaultPlay.c_str());

	//***********************************

	wstring supportTask = fullPath;
	supportTask.append(L"\\PlayTasks\\3\\");

	wstring helpFilesTask = supportTask;
	helpFilesTask.append(tasks.help);
	helpFilesTask.append(L".lnk");

	worked = DeleteFileW(helpFilesTask.c_str());

	worked = RemoveDirectoryW(defaultPlay.c_str());
	worked = RemoveDirectoryW(supportTask.c_str());
	worked = RemoveDirectoryW(playTasks.c_str());

	//remove other directories ...

	worked = RemoveDirectoryW(fullPath.c_str());
	if(!worked)
		return S_FAILED;
	return S_OK;
}

static const WCHAR* keyName = L"SOFTWARE\\Policies\\Microsoft\\Windows\\Installer";
static const WCHAR* valueName = L"AllowLockdownMedia";
static const WCHAR* backupValueName = L"AllowLockdownMediaBACKUP";

STDAPI SetAllowLockdownMedia()
{
	bool success = true;

	// Open the key that contains the policy value.
	HKEY hKey;
	DWORD dwDisposition;
	if (ERROR_SUCCESS != RegCreateKeyExW(HKEY_LOCAL_MACHINE, keyName, 0, NULL, 0, KEY_WRITE|KEY_READ, NULL, &hKey, &dwDisposition))
	{
		success = false;
		hKey = 0;
	}

	// If the key was successfully opened, attempt to backup the current policy value.
	if (hKey)
	{
		// Query first with no buffer - function will then tell us how much space we must allocated.
		DWORD valueType;
		DWORD bufferSize;
		if (ERROR_SUCCESS != RegQueryValueExW(hKey, valueName, NULL, &valueType, NULL, &bufferSize))
		{
			bufferSize = 0;
			// This situation is ok - it just means the previous value was unset. We handle this by leaving the
			// backup value unset - when we restore, we will delete the value.
		}

		// Allocate a buffer that is large enough to hold the value.
		std::vector<unsigned char> valueBuffer(bufferSize);
		if (bufferSize > 0)
		{
			if (ERROR_SUCCESS != RegQueryValueExW(hKey, valueName, NULL, &valueType, &valueBuffer[0], &bufferSize))
				success = false;
		}

		// Copy the value to the backup location.
		if (bufferSize > 0)
		{
			if (ERROR_SUCCESS != RegSetValueExW(hKey, backupValueName, 0, valueType, &valueBuffer[0], bufferSize))
				success = false; // This must mean the value exists, but there's something else wrong, so fail.
		}
	}

	if (success && hKey)
	{
		// Set the policy to 1.
		int policy = 1;
		if (ERROR_SUCCESS != RegSetValueExW(hKey, valueName, 0, REG_DWORD, reinterpret_cast<unsigned char*>(&policy), sizeof(policy)))
			success = false;
	}

	if (hKey)
		RegCloseKey(hKey);

	return (success ? S_OK : -1);
}

STDAPI RollbackAllowLockdownMedia()
{
	bool success = true;

	// Open the key that contains the policy value.
	HKEY hKey;
	DWORD dwDisposition;
	if (ERROR_SUCCESS != RegCreateKeyExW(HKEY_LOCAL_MACHINE, keyName, 0, NULL, 0, KEY_WRITE|KEY_READ, NULL, &hKey, &dwDisposition))
	{
		success = false;
		hKey = 0;
	}

	// If the key was successfully opened, attempt to restore the backed-up policy value.
	if (hKey)
	{
		// Query first with no buffer - function will then tell us how much space we must allocated.
		DWORD valueType;
		DWORD bufferSize;
		if (ERROR_SUCCESS != RegQueryValueExW(hKey, backupValueName, NULL, &valueType, NULL, &bufferSize))
		{
			bufferSize = 0;

			// This situation is ok - it means that we didn't back up the value because it wasn't originally set. We handle this
			// by deleting the original key to restore the original policies.
		}

		// Allocate a buffer that is large enough to hold the value.
		std::vector<unsigned char> valueBuffer(bufferSize);
		if (bufferSize > 0)
		{
			if (ERROR_SUCCESS != RegQueryValueExW(hKey, backupValueName, NULL, &valueType, &valueBuffer[0], &bufferSize))
				success = false;
		}

		// Copy the value to the original location.
		if (bufferSize > 0)
		{
			if (ERROR_SUCCESS != RegSetValueExW(hKey, valueName, 0, valueType, &valueBuffer[0], bufferSize))
				success = false;
		}
		else if (success)
		{
			// If we are here, it means that the backup value was missing, so we should delete the original key to match it.
			if (ERROR_SUCCESS != RegDeleteValueW(hKey, valueName))
				success = false;
		}

		// Delete the backup value, if it exists.
		if (bufferSize > 0)
		{
			if (ERROR_SUCCESS != RegDeleteValueW(hKey, backupValueName))
				success = false;
		}
	}

	if (hKey)
		RegCloseKey(hKey);

	return (success ? S_OK : -1);
}

STDAPI GetInstallPath(WCHAR* strRegistryPath, WCHAR* strInstallPathBuffer, DWORD bufferSize)
{
	bool success = true;

	// Look up the product key in the registry.
	WCHAR strSoftwareKey[1024];
	StringCchPrintfW(strSoftwareKey, sizeof(strSoftwareKey) / sizeof(strSoftwareKey[0]), L"Software\\Crytek\\%s", strRegistryPath);
	HKEY hKey;
	if (ERROR_SUCCESS != RegOpenKeyExW(HKEY_LOCAL_MACHINE, strSoftwareKey, 0, KEY_READ, &hKey))
		hKey = 0;

	// Look up the install path in the registry.
	if (hKey)
	{
		DWORD valueType;
		if (ERROR_SUCCESS != RegQueryValueExW(hKey, L"InstallDir", NULL, &valueType, reinterpret_cast<BYTE*>(strInstallPathBuffer), &bufferSize))
		{
			strInstallPathBuffer[0] = 0;
			success = false;
		}
	}

	return (success ? S_OK : -1);
}

STDAPI CreateDefaultLanguageFile(WCHAR* strProductLanguage, WCHAR* strRegistryPath)
{
	bool success = true;

	// Look up the product key in the registry.
	WCHAR strInstallPathBuffer[2048];
	{
		WCHAR strSoftwareKey[1024];
		StringCchPrintfW(strSoftwareKey, sizeof(strSoftwareKey) / sizeof(strSoftwareKey[0]), L"Software\\Crytek\\%s", strRegistryPath);
		HKEY hKey;
		if (ERROR_SUCCESS != RegOpenKeyExW(HKEY_LOCAL_MACHINE, strSoftwareKey, 0, KEY_READ, &hKey))
			hKey = 0;

		// Look up the install path in the registry.
		if (hKey)
		{
			DWORD valueType;
			DWORD bufferSize = sizeof(strInstallPathBuffer) / sizeof(strInstallPathBuffer[0]);
			if (ERROR_SUCCESS != RegQueryValueExW(hKey, L"InstallDir", NULL, &valueType, reinterpret_cast<BYTE*>(strInstallPathBuffer), &bufferSize))
			{
				strInstallPathBuffer[0] = 0;
				success = false;
			}
		}
	}

	// Add game/localized to the path.
	WCHAR strLocalizedPath[2048] = L"";
	if (strInstallPathBuffer[0])
		StringCchPrintfW(strLocalizedPath, sizeof(strLocalizedPath) / sizeof(strLocalizedPath[0]), L"%sGame\\Localized\\Default.lng", strInstallPathBuffer);

	FILE* languageFile = 0;
	if (strLocalizedPath[0])
	{
		languageFile = _wfopen(strLocalizedPath, L"w");
		if (!languageFile)
			success = false;
	}

	if (languageFile)
	{
		char productLanguageAnsi[2048] = "";
		WideCharToMultiByte(CP_ACP, 0, strProductLanguage, int(wcslen(strProductLanguage)), productLanguageAnsi, sizeof(productLanguageAnsi) / sizeof(productLanguageAnsi[0]), 0, 0);

		fprintf(languageFile, "Language = \"%s\"\n", productLanguageAnsi);
		fprintf(languageFile, "PAK = \"%s.pak\"\n", productLanguageAnsi);
	}

	if (languageFile)
		fclose(languageFile);

	return (success ? S_OK : -1);
}

bool DeletePath(WCHAR* strPath, bool recursive)
{
	bool success = true;

	SHFILEOPSTRUCTW fileOps = {
		0,                                                   //HWND hwnd;
		FO_DELETE,                                           //UINT wFunc;
		strPath,                                             //LPCTSTR pFrom;
		0,                                                   //LPCTSTR pTo;
		FOF_NOERRORUI | FOF_SILENT | FOF_NOCONFIRMATION,     //FILEOP_FLAGS fFlags;
		false,                                               //BOOL fAnyOperationsAborted;
		0,                                                   //LPVOID hNameMappings;
		0                                                    //LPCTSTR lpszProgressTitle;
	};

	fileOps.fFlags |= (recursive ? 0 : FOF_NORECURSION | FOF_FILESONLY);

	if (0 != SHFileOperationW(&fileOps))
		success = false;

	return success;
}

STDAPI DeleteUserData(WCHAR* strRegistryPath)
{
	bool success = true;

	// Look up the product key in the registry.
	WCHAR strInstallPathBuffer[2048];
	{
		WCHAR strSoftwareKey[1024];
		StringCchPrintfW(strSoftwareKey, sizeof(strSoftwareKey) / sizeof(strSoftwareKey[0]), L"Software\\Crytek\\%s", strRegistryPath);
		HKEY hKey;
		if (ERROR_SUCCESS != RegOpenKeyExW(HKEY_LOCAL_MACHINE, strSoftwareKey, 0, KEY_READ, &hKey))
			hKey = 0;

		// Look up the install path in the registry.
		if (hKey)
		{
			DWORD valueType;
			DWORD bufferSize = sizeof(strInstallPathBuffer) / sizeof(strInstallPathBuffer[0]);
			if (ERROR_SUCCESS != RegQueryValueExW(hKey, L"InstallDir", NULL, &valueType, reinterpret_cast<BYTE*>(strInstallPathBuffer), &bufferSize))
			{
				strInstallPathBuffer[0] = 0;
				success = false;
			}
		}
	}

	// Find My Documents.
	WCHAR strMyDocumentsPath[MAX_PATH];
	SHGetFolderPathW(NULL, CSIDL_PERSONAL, NULL, 0, strMyDocumentsPath);

	// Delete all files in the root directory.
	if (strInstallPathBuffer[0])
	{
		WCHAR wildcard[2048] = L"";
		StringCchPrintfW(wildcard, sizeof(wildcard) / sizeof(wildcard[0]), L"%s*.log", strInstallPathBuffer);
		if (!DeletePath(wildcard, false))
			success = false;

		WCHAR aiSignals[2048] = L"";
		StringCchPrintfW(aiSignals, sizeof(aiSignals) / sizeof(aiSignals[0]), L"%sAISignals.csv", strInstallPathBuffer);
		if (!DeletePath(aiSignals, false))
			success = false;

		WCHAR errorBMP[2048] = L"";
		StringCchPrintfW(errorBMP, sizeof(errorBMP) / sizeof(errorBMP[0]), L"%sError.bmp", strInstallPathBuffer);
		if (!DeletePath(errorBMP, false))
			success = false;
	}

	// Delete the LogBackups folder.
	if (strInstallPathBuffer[0])
	{
		WCHAR folder[2048] = L"";
		StringCchPrintfW(folder, sizeof(folder) / sizeof(folder[0]), L"%sLogBackups", strInstallPathBuffer);
		DeletePath(folder, true);
	}

	// Delete the screenshots folder.
	if (strInstallPathBuffer[0])
	{
		WCHAR folder[2048] = L"";
		StringCchPrintfW(folder, sizeof(folder) / sizeof(folder[0]), L"%sScreenshots", strInstallPathBuffer);
		DeletePath(folder, true);
	}

	// Delete the folder from My Games.
	if (strMyDocumentsPath[0])
	{
		WCHAR folder[2048] = L"";
		StringCchPrintfW(folder, sizeof(folder) / sizeof(folder[0]), L"%s\\My Games\\%s", strMyDocumentsPath, strRegistryPath);
		DeletePath(folder, true);
	}

	return (success ? S_OK : -1);
}

STDAPI ShowReadmeFile(WCHAR* strRegistryPath, WCHAR* strFileName)
{
	bool success = true;

	// Look up the product key in the registry.
	WCHAR strInstallPathBuffer[2048];
	{
		WCHAR strSoftwareKey[1024];
		StringCchPrintfW(strSoftwareKey, sizeof(strSoftwareKey) / sizeof(strSoftwareKey[0]), L"Software\\Crytek\\%s", strRegistryPath);
		HKEY hKey;
		if (ERROR_SUCCESS != RegOpenKeyExW(HKEY_LOCAL_MACHINE, strSoftwareKey, 0, KEY_READ, &hKey))
			hKey = 0;

		// Look up the install path in the registry.
		if (hKey)
		{
			DWORD valueType;
			DWORD bufferSize = sizeof(strInstallPathBuffer) / sizeof(strInstallPathBuffer[0]);
			if (ERROR_SUCCESS != RegQueryValueExW(hKey, L"InstallDir", NULL, &valueType, reinterpret_cast<BYTE*>(strInstallPathBuffer), &bufferSize))
			{
				strInstallPathBuffer[0] = 0;
				success = false;
			}
		}
	}
	
	WCHAR readmePath[2048] = L"";
	if (strInstallPathBuffer[0])
		StringCchPrintfW(readmePath, sizeof(readmePath) / sizeof(readmePath[0]), L"%sSupport\\%s", strInstallPathBuffer, strFileName);

	if (readmePath[0])
	{
		if (32 >= (int)ShellExecuteW(
			0,                //HWND hwnd,
			0,                //LPCTSTR lpOperation,
			L"notepad.exe",    //LPCTSTR lpFile,
			readmePath,       //LPCTSTR lpParameters
			0,                //LPCTSTR lpDirectory
			SW_SHOW))         //INT nShowCmd
		{
			success = false;
		}
	}

	return (success ? S_OK : -1);
}
