//***************************************************************************************
//
// File supervisor: Softimage Rendering & Pipeline team
//
// (c) Copyright 2001-2005 Avid Technology, Inc. . All rights reserved.
//
//***************************************************************************************

/**************************************************************************************
THIS CODE IS PUBLISHED AS A SAMPLE ONLY AND IS PROVIDED "AS IS".
IN NO EVENT SHALL SOFTIMAGE, AVID TECHNOLOGY, INC. AND/OR THEIR RESPECTIVE
SUPPLIERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
CONNECTION WITH THE USE OR PERFORMANCE OF THIS CODE .

COPYRIGHT NOTICE. Copyright  1999-2005 Avid Technology Inc. . All rights reserved. 

SOFTIMAGE is a registered trademark of Avid Technology Inc. or its subsidiaries 
or divisions. Windows NT is a registered trademark of Microsoft Corp. All other
trademarks contained herein are the property of their respective owners. 
****************************************************************************************/
/*! \file cnv_camera.cpp
/*!
	implementation file for camera related classes
*/

#include "stdafx.h"
#include "cnv_camera.h"
#include <xsi_x3dobject.h>
#include <xsi_camera.h>
#include <xsi_time.h>
#include <xsi_kinematics.h>
#include <xsi_kinematicstate.h>
#include <xsi_math.h>
#include <xsi_constraint.h>
#include <Model.h>
#include <Camera.h>
#include <XSICamera.h>
#include "cmdstubs.h"

/**************************************************************************************
CCameraFromXSI
**************************************************************************************/
CCameraFromXSI::CCameraFromXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CCameraFromXSI::~CCameraFromXSI() {};

CStatus CCameraFromXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;

	X3DObject l_XSIModel = (X3DObject) *io_pXSIModel;

	if(wcscmp(l_XSIModel.GetType().GetWideString(), L"camera") == 0)
	{
		CSLModel *l_pNewModel;
		CSLModel *l_pParent = (CSLModel *) in_pFTKParent;
		CTime	l_Time;

		l_pNewModel = l_pParent->AddCamera();
		*io_pFTKModel = l_pNewModel;
		l_pNewModel->AttachUserData("CREF", in_pContext->AddCRef(*io_pXSIModel));

		// fill up the camera information now
		Camera l_Camera = (Camera) l_XSIModel;

		// get the near and far planes
		CSLCamera *l_pCamera = (CSLCamera *) l_pNewModel->Primitive();
		l_pCamera->SetNearPlane(l_Camera.GetParameterValue(CString(L"near"), CValue(l_Time.GetTime())));
		l_pCamera->SetFarPlane(l_Camera.GetParameterValue(CString(L"far"), CValue(l_Time.GetTime())));

		// get the field of view
		l_pCamera->SetFieldOfView(in_pContext->AngleToFTK(l_Camera.GetParameterValue(CString(L"fov"), CValue(l_Time.GetTime())), false));

		// get the position from the kinematic state
		MATH::CVector3	l_XSIVector;
		CSIBCVector3D	l_FTKVector;

		// this code gets the local kinematics transform
		l_XSIVector = l_XSIModel.GetKinematics().GetLocal().GetTransform().GetTranslation();
		l_FTKVector.Set((float) l_XSIVector.GetX(), (float) l_XSIVector.GetY(), (float) l_XSIVector.GetZ());
		l_pCamera->SetPosition(l_FTKVector);
		
		// ok now we just fill in the interest position, but we should really put
		// a direction constraint somewhere in there two

		X3DObject l_Interest = l_Camera.GetInterest();
		if(l_Interest.IsValid())
		{
			l_XSIVector = l_Interest.GetKinematics().GetLocal().GetTransform().GetTranslation();
			l_FTKVector.Set((float) l_XSIVector.GetX(), (float) l_XSIVector.GetY(), (float) l_XSIVector.GetZ());
			l_pCamera->SetInterestPosition(l_FTKVector);

			// we have an interest so we must have a direction constraint here, we might not be able
			// to deal with more convoluted setups, but we'll manage with the default (99% case)
			// to do this, we'll just use a simple getvalue command and see if it returns something
			// valid and use it.

			CString l_RollParamName = l_Camera.GetFullName() + CString(L".kine.dircns.roll");
			CValue	l_RollValue = COMMANDS::GetValue(l_RollParamName, l_Time.GetTime());
			if(!l_RollValue.IsEmpty())
			{
				l_pCamera->SetRoll(in_pContext->AngleToFTK(l_RollValue, false));
			}
		}

		// TODO: there is an issue with lonely cameras that are not constrained to an interest
	}	
	
	return status;
}

wchar_t *CCameraFromXSI::GetClassID(){return L"CCameraFromXSI";}


/**************************************************************************************
CCameraToXSI
**************************************************************************************/
CCameraToXSI::CCameraToXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CCameraToXSI::~CCameraToXSI() {};

CStatus CCameraToXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;

	CSLModel *l_pFTKModel = (CSLModel *) *io_pFTKModel;

	if(l_pFTKModel->Primitive() && (l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_CAMERA))
	{
		CTime		l_Time;

		X3DObject	l_XSIParent = (X3DObject) in_XSIParent;
		Camera		l_XSICamera;
		CSLCamera	*l_pCamera = (CSLCamera *) l_pFTKModel->Primitive();

		// create the camera
		_XSI_CALL(l_XSIParent.AddCamera(CString(L"Camera"), CString(L"unknown"), l_XSICamera), L"Failed to add a Camera object");
		*io_pXSIModel = l_XSICamera;

		// set the position and interest
		// this code gets the local kinematics transform
		MATH::CTransformation	l_Transformation;
		MATH::CVector3			l_XSIVector;
		CSIBCVector3D			l_FTKVector;

		l_FTKVector = l_pCamera->GetPosition();
		l_Transformation.SetTranslationFromValues(l_FTKVector.m_fX, l_FTKVector.m_fY, l_FTKVector.m_fZ);
		l_XSICamera.GetKinematics().GetLocal().PutTransform(l_Transformation, CValue(l_Time.GetTime()));
		
		l_FTKVector = l_pCamera->GetInterestPosition();
		l_Transformation.SetTranslationFromValues(l_FTKVector.m_fX, l_FTKVector.m_fY, l_FTKVector.m_fZ);
		l_XSICamera.GetInterest().GetKinematics().GetLocal().PutTransform(l_Transformation, CValue(l_Time.GetTime()));		
		l_XSICamera.GetInterest().PutName(L"Camera_Interest");

		// set the near and far planes
		l_XSICamera.PutParameterValue(CString(L"near"), CValue(l_pCamera->GetNearPlane()), CValue(l_Time.GetTime()));
		l_XSICamera.PutParameterValue(CString(L"far"), CValue(l_pCamera->GetFarPlane()), CValue(l_Time.GetTime()));

		// set the field of view
		l_XSICamera.PutParameterValue(CString(L"fov"), CValue(in_pContext->AngleFromFTK(l_pCamera->GetFieldOfView(), false)), CValue(l_Time.GetTime()));

		// we have an interest so we must have a direction constraint here, we might not be able
		// to deal with more convoluted setups, but we'll manage with the default (99% case)
		// to do this, we'll just use a simple getvalue command and see if it returns something
		// valid and use it.

		CString l_RollParamName = l_XSICamera.GetFullName() + CString(L".kine.dircns.roll");
		COMMANDS::SetValue(l_RollParamName, CValue(in_pContext->AngleFromFTK(l_pCamera->GetRoll(), false)), l_Time.GetTime());

		l_pFTKModel->AttachUserData("CREF", in_pContext->AddCRef(l_XSICamera));
	}	

	return status;
}

wchar_t *CCameraToXSI::GetClassID(){return L"CCameraToXSI";}

/**************************************************************************************
CXSI_CameraFromXSI
**************************************************************************************/
CXSI_CameraFromXSI::CXSI_CameraFromXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CXSI_CameraFromXSI::~CXSI_CameraFromXSI() {};

CStatus CXSI_CameraFromXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;

	X3DObject l_XSIModel = (X3DObject) *io_pXSIModel;

	if(wcscmp(l_XSIModel.GetType().GetWideString(), L"camera") == 0)
	{
		CSLModel *l_pNewModel;
		CSLModel *l_pParent = (CSLModel *) in_pFTKParent;
		CTime	l_Time;

		l_pNewModel = l_pParent->AddXSICamera();
		*io_pFTKModel = l_pNewModel;
		l_pNewModel->AttachUserData("CREF", in_pContext->AddCRef(*io_pXSIModel));

		// punch in the parameters now
		Camera l_Camera = (Camera) l_XSIModel;
		CParameterRefArray l_Parameters = l_Camera.GetParameters();

		CSLXSICamera* l_pFTKCamera = (CSLXSICamera*) l_pNewModel->Primitive();
		l_pFTKCamera->SetStandard(l_Parameters.GetValue(L"std", 0));
		l_pFTKCamera->SetPictureRatio(l_Parameters.GetValue(L"aspect", 0));
		l_pFTKCamera->SetFieldOfView(in_pContext->AngleToFTK(l_Parameters.GetValue(L"fov", 0), false));
		l_pFTKCamera->SetFieldOfViewType(l_Parameters.GetValue(L"fovtype", 0));
		l_pFTKCamera->SetProjectionType(l_Parameters.GetValue(L"proj", 0));
		l_pFTKCamera->SetOrthoHeight(l_Parameters.GetValue(L"orthoheight", 0));
		l_pFTKCamera->SetInterestDistance(l_Parameters.GetValue(L"interestdist", 0));
		l_pFTKCamera->SetNearPlane(l_Parameters.GetValue(L"near", 0));
		l_pFTKCamera->SetFarPlane(l_Parameters.GetValue(L"far", 0));
		l_pFTKCamera->SetProjectionPlaneEnable(l_Parameters.GetValue(L"projplane", 0));
		l_pFTKCamera->SetProjPlaneWidth(l_Parameters.GetValue(L"projplanewidth", 0));
		l_pFTKCamera->SetProjPlaneHeight(l_Parameters.GetValue(L"projplaneheight", 0));
		l_pFTKCamera->SetProjPlaneOffsetX(l_Parameters.GetValue(L"projplaneoffx", 0));
		l_pFTKCamera->SetProjPlaneOffsetY(l_Parameters.GetValue(L"projplaneoffy", 0));
	}	
	
	return status;
}

wchar_t *CXSI_CameraFromXSI::GetClassID(){return L"CXSI_CameraFromXSI";}


/**************************************************************************************
CXSI_CameraToXSI
**************************************************************************************/
CXSI_CameraToXSI::CXSI_CameraToXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CXSI_CameraToXSI::~CXSI_CameraToXSI() {};

CStatus CXSI_CameraToXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;

	CSLModel *l_pFTKModel = (CSLModel *) *io_pFTKModel;

	if(l_pFTKModel->Primitive() && (l_pFTKModel->Primitive()->Type() == CSLTemplate::XSI_CAMERA))
	{
		CTime			l_Time;

		X3DObject		l_XSIParent = (X3DObject) in_XSIParent;
		Camera			l_XSICamera;
		CSLXSICamera	*l_pCamera = (CSLXSICamera *) l_pFTKModel->Primitive();

		// create the camera
		_XSI_CALL(l_XSIParent.AddCamera(CString(L"Camera"), CString(L"unknown"), l_XSICamera), L"Failed to add a Camera object");
		*io_pXSIModel = l_XSICamera;

		// break the camera rig now
		X3DObject		l_XSICameraInterest = l_XSICamera.GetInterest();
		COMMANDS::DeleteObj(l_XSICameraInterest.GetFullName());

		l_pFTKModel->AttachUserData("CREF", in_pContext->AddCRef(l_XSICamera));

		// punch in the parameters
		CParameterRefArray l_Parameters = l_XSICamera.GetParameters();

		l_Parameters.PutValue(L"aspect", CValue(l_pCamera->GetPictureRatio()) ,0);
		l_Parameters.PutValue(L"proj", CValue(l_pCamera->GetProjectionType()) ,0);
		l_Parameters.PutValue(L"orthoheight", CValue(l_pCamera->GetOrthoHeight()) ,0);
		l_Parameters.PutValue(L"interestdist", CValue(l_pCamera->GetInterestDistance()) ,0);
		l_Parameters.PutValue(L"near", CValue(l_pCamera->GetNearPlane()) ,0);
		l_Parameters.PutValue(L"far", CValue(l_pCamera->GetFarPlane()) ,0);
		l_Parameters.PutValue(L"projplane", CValue(l_pCamera->GetProjectionPlaneEnable()) ,0);
		l_Parameters.PutValue(L"projplanewidth", CValue(l_pCamera->GetProjPlaneWidth()) ,0);
		l_Parameters.PutValue(L"projplaneheight", CValue(l_pCamera->GetProjPlaneHeight()) ,0);
		l_Parameters.PutValue(L"projplaneoffx", CValue(l_pCamera->GetProjPlaneOffsetX()) ,0);
		l_Parameters.PutValue(L"projplaneoffy", CValue(l_pCamera->GetProjPlaneOffsetY()) ,0);		                         
		l_Parameters.PutValue(L"fov", in_pContext->AngleFromFTK(CValue(l_pCamera->GetFieldOfView()), false) ,0);
		l_Parameters.PutValue(L"fovtype", CValue(l_pCamera->GetFieldOfViewType()) ,0);
		l_Parameters.PutValue(L"std", CValue(l_pCamera->GetStandard()) ,0);
	}	

	return status;
}

wchar_t *CXSI_CameraToXSI::GetClassID(){return L"CXSI_CameraToXSI";}

