//***************************************************************************************
//
// File supervisor: Softimage Rendering & Pipeline team
//
// (c) Copyright 2001-2005 Avid Technology, Inc. . All rights reserved.
//
//***************************************************************************************

/**************************************************************************************
THIS CODE IS PUBLISHED AS A SAMPLE ONLY AND IS PROVIDED "AS IS".
IN NO EVENT SHALL SOFTIMAGE, AVID TECHNOLOGY, INC. AND/OR THEIR RESPECTIVE
SUPPLIERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
CONNECTION WITH THE USE OR PERFORMANCE OF THIS CODE .

COPYRIGHT NOTICE. Copyright  1999-2005 Avid Technology Inc. . All rights reserved. 

SOFTIMAGE is a registered trademark of Avid Technology Inc. or its subsidiaries 
or divisions. Windows NT is a registered trademark of Microsoft Corp. All other
trademarks contained herein are the property of their respective owners. 
****************************************************************************************/
/*! \file cnv_envelope.cpp
/*!
	implementation file for envelope related classes
*/

#include "stdafx.h"
#include "cnv_envelope.h"
#include <xsi_application.h>
#include <xsi_x3dobject.h>
#include <xsi_envelope.h>
#include <xsi_polygonmesh.h>
#include <xsi_primitive.h>
#include <xsi_geometryaccessor.h>
#include <xsi_envelopeweight.h>
#include <xsi_clusterpropertybuilder.h>

#include <Envelope.h>
#include <EnvelopeList.h>
#include <XSIMesh.h>
#include <IK.h>
#include <Model.h>
#include <xsi_statickinematicstate.h>
#include <xsi_kinematicstate.h>
#include <xsi_kinematics.h>
#include <XSITransform.h>

/**************************************************************************************
CEnvelopeFromXSI
**************************************************************************************/
CEnvelopeFromXSI::CEnvelopeFromXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CEnvelopeFromXSI::~CEnvelopeFromXSI() {};

CStatus CEnvelopeFromXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;

	X3DObject XSIModel = (X3DObject) *io_pXSIModel;
	SceneItem XSISceneItem = (SceneItem)XSIModel;

	// validate some SDK objects
	if ( !XSISceneItem.IsValid()  )
		return CStatus::OK;

	// model enveloped?
	if ( !XSISceneItem.GetEnvelopes().GetCount() )
		return CStatus::OK;	// model not enveloped.

	// get the semantic layer model of the XSI envelope
	CSLModel* l_SLEnvelope = (CSLModel *) *io_pFTKModel;

	if ( !l_SLEnvelope )
	{
		// this is bad
		logmessage ( L"Error exporting envelope: Mesh %s not found.", XSIModel.GetName());
		return CStatus::False;
	}

	PolygonMesh			l_XSIMesh = (PolygonMesh) XSIModel.GetActivePrimitive().GetGeometry();

	if(!l_XSIMesh.IsValid())
	{
		return status;
	}

	//
	// get the geometry accessor created in cnv_mesh
	//
	CGeometryAccessor	l_XSIGeometryAccessor;
	CSLXSIMesh			*l_pMesh = (CSLXSIMesh *) l_SLEnvelope->Primitive();

	CSIBCUserData *l_pUserData = l_pMesh->FindUserData("GEOMETRYACCESSOR");
	if(l_pUserData)
	{
		CRef *l_pCRef = (CRef*)(l_pUserData->GetData());
		l_XSIGeometryAccessor = (CGeometryAccessor) *l_pCRef;
	} else {

		// this is terrible
		logmessage ( L"Error exporting envelope: Geometry Accessor not found for mesh %s.", XSIModel.GetName());
		return CStatus::Fail;
	}


	// export envelope templates
	CRefArray XSIEnvelopeList = l_XSIGeometryAccessor.GetEnvelopeWeights();

	CSIBCArray<CSLEnvelope*> l_aLocalEnvelopeList;

	for (LONG i=0;i<XSIEnvelopeList.GetCount();i++)
	{
		EnvelopeWeight l_Envelope ( XSIEnvelopeList[i] ); 
		CRefArray l_pDeformers = l_Envelope.GetDeformers();

		if ( !l_pDeformers.GetCount() )
			break;	// no deformer connected to envelope.

		// Create Semantic Layer Envelope List
        CSLScene* l_pSLScene = in_pContext->ftkscene();
		CSLEnvelopeList* l_pEnvelopeList = l_pSLScene->EnvelopeList ();

		if ( !l_pEnvelopeList )
		{
			l_pEnvelopeList = l_pSLScene->CreateEnvelopeList();
		}

		//
		// loop for all deformers and create the semantic layer
		// envelope templates, connecting all models together
		//

		for (LONG d=0;d<l_pDeformers.GetCount();d++)
		{
			X3DObject l_node = X3DObject ( l_pDeformers[d] );

			// Find the associated Semantic Layer Model for this deformer
			CSLModel* l_SLDeformer = in_pContext->FindModelByCRef ( NULL, l_node.GetRef () );
			if ( !l_SLDeformer )
			{
				// this is very not cool.
				logmessage ( L"Error exporting envelope: Deformer %S not found.", l_node.GetName());
				continue;
			}

			CSLEnvelope*	l_pEnvelope = l_pEnvelopeList->AddEnvelope();
			l_aLocalEnvelopeList.Add ( l_pEnvelope );

			l_pEnvelope->SetName ( l_SLEnvelope->GetName());
			l_pEnvelope->SetEnvelope ( l_SLEnvelope );
			l_pEnvelope->SetDeformer ( l_SLDeformer );
			CSLEnvelope::CSLVertexWeightArray* l_aWeightList = l_pEnvelope->GetVertexWeightList();
			l_aWeightList->Extend ( l_XSIGeometryAccessor.GetVertexCount() );

		}

		//
		// copy the weights to the semantic layer
		//

		CSIBCArray<float>	*l_cachedWeights = new CSIBCArray<float>[l_pDeformers.GetCount()];
		CFloatArray weightArray;
		l_Envelope.GetValues ( weightArray );
		LONG countDeformers = l_aLocalEnvelopeList.GetUsed();
		for (LONG v=0;v<l_XSIGeometryAccessor.GetVertexCount();v++)
		{
			for (LONG d=0;d<countDeformers;d++)
			{
				CSLEnvelope*	l_pEnvelope = l_aLocalEnvelopeList[d];
				CSLEnvelope::CSLVertexWeightArray* l_aWeightList = l_pEnvelope->GetVertexWeightList();
				SLVertexWeight*	l_pCurrentWeight = &(*l_aWeightList)[v];
				l_pCurrentWeight->m_fVertexIndex = (float)v;
				l_pCurrentWeight->m_fWeight = weightArray[(v*countDeformers)+d];
			}
		}
	}

	return status;
}

wchar_t *CEnvelopeFromXSI::GetClassID(){return L"CEnvelopeFromXSI";}

/**************************************************************************************
CEnvelopeToXSI
**************************************************************************************/
CEnvelopeToXSI::CEnvelopeToXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CEnvelopeToXSI::~CEnvelopeToXSI() {};

CStatus CEnvelopeToXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;
	
	CSLModel *l_pFTKModel = (CSLModel *) *io_pFTKModel;
	
	// Check if this model is enveloped
	CSLScene* l_pScene = l_pFTKModel->Scene ();
	CSLEnvelopeList* l_pEnvelopeList = l_pScene->EnvelopeList ();

	if ( !l_pEnvelopeList )
		return CStatus::OK;

	bool found = false;
	CRefArray	l_Deformers;
	CSIBCArray<CSLEnvelope*> l_EnvelopeMap;
	LONG loop;

	if((l_pFTKModel->Primitive() == NULL) || (l_pFTKModel->Primitive()->Type() != CSLTemplate::XSI_MESH))
		return status;

	//
	// check all envelopes in the semantic layer scene
	// to see if one references this model
	//

	for (loop=0;loop<l_pEnvelopeList->GetEnvelopeCount();loop++)
	{
		CSLEnvelope* l_pEnvelope = l_pEnvelopeList->Envelopes ()[loop];
		if ( l_pEnvelope->GetEnvelope () == l_pFTKModel )
		{
			// we found one envelope referencing this model
			// add it to our local list
			found = true;
			
			// get the deformer
			CSIBCUserData *l_pModelUserData = NULL; 
			if ( l_pEnvelope->GetDeformer() && (l_pModelUserData = l_pEnvelope->GetDeformer ()->FindUserData( "CREF" )) )
			{
				CRef *l_pCRef = (CRef*)(l_pModelUserData->GetData());
				l_Deformers.Add ( *l_pCRef );
				l_EnvelopeMap.Add ( l_pEnvelope );
			} else {

				// this is bad
				logmessage ( L"Error importing envelope: Unable to find a deformer" );
				found = false;
				break;
			}
		}
	}

	// this model is not enveloped
	if ( !found )
		return CStatus::OK;


	// apply the envelope
	CSIBCUserData *l_pModelUserData = l_pFTKModel->FindUserData ( "CREF" );
	X3DObject X3DEnvelope;
	if ( l_pModelUserData )
	{
		CRef *l_pCRef = (CRef*)(l_pModelUserData->GetData());
		X3DEnvelope = (X3DObject)(*l_pCRef);
	}

	if ( !X3DEnvelope.IsValid())
	{
		logmessage ( L"Error exporting envelope: Envelope mesh not found!");
		return CStatus::Fail;
	}

	PolygonMesh	l_XSIMesh = X3DEnvelope.GetActivePrimitive().GetGeometry(); 

	// now import the cluster property (Normals, UVs, Vertex Colors, Texture Coordinates, Weight Maps)
	CClusterPropertyBuilder l_XSIClusterPropBuilder = l_XSIMesh.GetClusterPropertyBuilder();	
	EnvelopeWeight l_EnvelopeWeights = l_XSIClusterPropBuilder.AddEnvelopeWeight(l_Deformers);

	// add normalized weights
	for (loop=0;loop<l_Deformers.GetCount();loop++)
	{
		CSLEnvelope* l_pCurrentEnvelope = l_EnvelopeMap[loop];

		if (l_pCurrentEnvelope->GetVertexWeightCount())
		{
			float *l_pWeightList = new float[l_pCurrentEnvelope->GetVertexWeightCount()];
			LONG *l_pWeightIndex = new LONG[l_pCurrentEnvelope->GetVertexWeightCount()];

			int loop2;
			for(loop2 = 0; loop2 < l_pCurrentEnvelope->GetVertexWeightCount(); loop2++)
			{	
				l_pWeightList[loop2] = l_pCurrentEnvelope->GetVertexWeightListPtr()[loop2].m_fWeight;
				l_pWeightIndex[loop2] = (LONG) l_pCurrentEnvelope->GetVertexWeightListPtr()[loop2].m_fVertexIndex;
			}	

			l_EnvelopeWeights.SetValues
			(
				l_Deformers[loop], 
				l_pWeightIndex,
				l_pWeightList, 
				l_pCurrentEnvelope->GetVertexWeightCount()
			);

			delete l_pWeightList;
			delete l_pWeightIndex;
		}
	}

	// set the static kinematic state (AKA base pose)
	for (loop=0;loop<=l_Deformers.GetCount();loop++)
	{
		XSI::X3DObject	l_X3DObject;
		CSLModel		*l_pFTKModelTemp;
		

		// this is just a little trick to avoid duplicating the same code for the enveloped model
		if(loop<l_Deformers.GetCount())
		{
			l_X3DObject = l_Deformers[loop];
			l_pFTKModelTemp = l_EnvelopeMap[loop]->GetDeformer();
		}
		else
		{
			l_X3DObject = (X3DObject) X3DEnvelope;
			l_pFTKModelTemp = l_pFTKModel;
		}


		// Fill in the transform now
		if(l_pFTKModelTemp->GetXSIBasePose() != NULL)
		{
			CSIBCVector3D l_FTKVector;

			// this code sets the base pose transform
			StaticKinematicState l_XSIKineState = l_X3DObject.GetStaticKinematicState();
			
			// position
			l_FTKVector = l_pFTKModelTemp->GetXSIBasePose()->GetTranslation();

			l_XSIKineState.GetParameter(L"posx").PutValue(l_FTKVector.m_fX);
			l_XSIKineState.GetParameter(L"posy").PutValue(l_FTKVector.m_fY);
			l_XSIKineState.GetParameter(L"posz").PutValue(l_FTKVector.m_fZ);

			// rotation
			l_FTKVector = l_pFTKModelTemp->GetXSIBasePose()->GetEulerRotation();

			l_XSIKineState.GetParameter(L"orix").PutValue(in_pContext->AngleFromFTK(l_FTKVector.m_fX, false));
			l_XSIKineState.GetParameter(L"oriy").PutValue(in_pContext->AngleFromFTK(l_FTKVector.m_fY, false));
			l_XSIKineState.GetParameter(L"oriz").PutValue(in_pContext->AngleFromFTK(l_FTKVector.m_fZ, false));

			// scaling
			l_FTKVector = l_pFTKModelTemp->GetXSIBasePose()->GetScale();

			l_XSIKineState.GetParameter(L"sclx").PutValue(l_FTKVector.m_fX);
			l_XSIKineState.GetParameter(L"scly").PutValue(l_FTKVector.m_fY);
			l_XSIKineState.GetParameter(L"sclz").PutValue(l_FTKVector.m_fZ);


			if(IsZUp(in_pContext->ftkscene()->CoordinateSystem()))
			{
				MATH::CTransformation l_BasePoseTransform;
				MATH::CTransformation l_ZUPConverter;

				l_BasePoseTransform = l_XSIKineState.GetTransform();
				l_ZUPConverter.SetRotationFromXYZAnglesValues(-1.570796f, 0.0f, 0.0f);

				l_BasePoseTransform.MulInPlace(l_ZUPConverter);
				l_XSIKineState.PutTransform(l_BasePoseTransform);

			}
		}
	}

	return status;
}

wchar_t *CEnvelopeToXSI::GetClassID(){return L"CEnvelopeToXSI";}

