//***************************************************************************************
//
// File supervisor: Softimage Rendering & Pipeline team
//
// (c) Copyright 2001-2005 Avid Technology, Inc. . All rights reserved.
//
//***************************************************************************************

/**************************************************************************************
THIS CODE IS PUBLISHED AS A SAMPLE ONLY AND IS PROVIDED "AS IS".
IN NO EVENT SHALL SOFTIMAGE, AVID TECHNOLOGY, INC. AND/OR THEIR RESPECTIVE
SUPPLIERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
CONNECTION WITH THE USE OR PERFORMANCE OF THIS CODE .

COPYRIGHT NOTICE. Copyright  1999-2005 Avid Technology Inc. . All rights reserved. 

SOFTIMAGE is a registered trademark of Avid Technology Inc. or its subsidiaries 
or divisions. Windows NT is a registered trademark of Microsoft Corp. All other
trademarks contained herein are the property of their respective owners. 
****************************************************************************************/
/*! \file cnv_image.cpp
/*!
	implementation file for image classes
*/

#include "stdafx.h"
#include "cnv_image.h"
#include <xsi_source.h>
#include <xsi_clip.h>
#include <xsi_imageclip2.h>
#include <xsi_image.h>
#include <xsi_material.h>
#include <xsi_time.h>
#include <xsi_library.h>
#include <xsi_timecontrol.h>
#include <xsi_utils.h>
#include <XSIImageLibrary.h>
#include <XSIImage.h>
#include <XSIImageFX.h>
#include <TimeControl.h>
#include <MaterialLibrary.h>
#include "FTKUtil.h"
#include "cmdstubs.h"
#ifdef unix
 #include <unistd.h>
#else
 #include <io.h> 
#endif

/**************************************************************************************
CXSIImageLibraryFromXSI
**************************************************************************************/
CXSIImageLibraryFromXSI::CXSIImageLibraryFromXSI() {};

CXSIImageLibraryFromXSI::~CXSIImageLibraryFromXSI() {};

CStatus CXSIImageLibraryFromXSI::Execute(CdotXSIConverter *in_pContext)
{
	CStatus status = CStatus::OK;
	CXSIImageFromXSI	l_ImageConverter(CRY_FILE_TYPE_NONE);
	CSLImageLibrary		*l_pFTKImageLibrary = in_pContext->ftkscene()->GetImageLibrary();

	// here is how to get all the image clips in the scene
	CRefArray l_ImageClips = in_pContext->application().GetActiveProject().GetActiveScene().GetImageClips();

	if ( l_ImageClips.GetCount() )
	{
		// create a new image library if it doesn't exist yet
		if(l_pFTKImageLibrary == NULL)
			l_pFTKImageLibrary = in_pContext->ftkscene()->CreateImageLibrary();
	}

	int loop;
	for(loop = 0; loop < l_ImageClips.GetCount(); loop++)
	{
		CRef l_ImageClip = l_ImageClips[loop];
		_XSI_CALL(l_ImageConverter.Execute(in_pContext, l_ImageClip, l_pFTKImageLibrary, NULL, &l_ImageClip, NULL), L"Failed to convert image clip");
	}

	return status;
}

wchar_t *CXSIImageLibraryFromXSI::GetClassID(){return L"CXSIImageLibraryFromXSI";}

/**************************************************************************************
CXSIImageLibraryFromXSI_RemoveUnused
**************************************************************************************/
CXSIImageLibraryFromXSI_RemoveUnused::CXSIImageLibraryFromXSI_RemoveUnused() {};

CXSIImageLibraryFromXSI_RemoveUnused::~CXSIImageLibraryFromXSI_RemoveUnused() {};

CStatus CXSIImageLibraryFromXSI_RemoveUnused::Execute(CdotXSIConverter *in_pContext)
{
	CStatus status = CStatus::OK;

	CSLImageLibrary		*l_pFTKImageLibrary = in_pContext->ftkscene()->GetImageLibrary();

	if (l_pFTKImageLibrary)
	{
		LONG loop;

		for (loop = 0; loop < l_pFTKImageLibrary->GetImageCount(); loop++)
		{
			CSLImage **l_pImageArray = l_pFTKImageLibrary->GetImageList();

			CSLImage *l_pImage = l_pImageArray[loop];

			CSIBCUserData *l_pUserData = l_pImage->FindUserData("USED");

			if (!l_pUserData)
			{
				l_pFTKImageLibrary->RemoveImage(loop);
				loop--;
			}
		}
	}

	return status;
}

wchar_t *CXSIImageLibraryFromXSI_RemoveUnused::GetClassID(){return L"CXSIImageLibraryFromXSI_RemoveUnused";}

/**************************************************************************************
CXSIImageFromXSI
**************************************************************************************/
CXSIImageFromXSI::CXSIImageFromXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CXSIImageFromXSI::~CXSIImageFromXSI() {};

CStatus CXSIImageFromXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;

	// the first thing to do is look if this image clip is already in the library
	ImageClip2 l_ImageClip = (ImageClip2) *io_pXSIModel;
	CSLImage *l_pFTKImageClip = in_pContext->ftkscene()->GetImageLibrary()->FindImage((char*)l_ImageClip.GetName().GetAsciiString());

	// if it doesn't exist we create it
	if(l_pFTKImageClip == NULL)
	{
		CSLImageLibrary *l_pFTKImageLibrary = (CSLImageLibrary *) in_pFTKParent;
		l_pFTKImageClip = l_pFTKImageLibrary->AddImage();
		
		// set the name
		l_pFTKImageClip->SetName((char*)l_ImageClip.GetName().GetAsciiString());

		// set the CREF
		l_pFTKImageClip->AttachUserData("CREF", in_pContext->AddCRef(*io_pXSIModel));
		
		// get the image
		Image l_XSIImage;
		l_XSIImage = l_ImageClip.GetImage();

		// get the source
		Source l_XSISource;
		l_XSISource = l_ImageClip.GetSource();

		// set the image info

		// set the image file path
		CParameterRefArray params = l_XSISource.GetParameters();
		CString l_SourceFileName = l_ImageClip.GetFileName();

		CSIBCString RelativeImageName;

		CString l_ExportPath = in_pContext->exportproperty().GetParameterValue(L"Filename");
		CValue l_PathRelative = in_pContext->exportproperty().GetParameterValue(L"PathRelative");

		if(l_PathRelative == true)
		{
			FTKUPathAbsoluteToRelative((char*)l_SourceFileName.GetAsciiString(), (char*) l_ExportPath.GetAsciiString(), RelativeImageName);
			l_pFTKImageClip->SetSourceFile(RelativeImageName.GetText());
		}
		else
		{
			l_pFTKImageClip->SetSourceFile((char*)	l_SourceFileName.GetAsciiString());
		}

		// set the image dimension
		l_pFTKImageClip->SetWidth((LONG)params.GetValue(L"XRes"));
		l_pFTKImageClip->SetHeight((LONG)params.GetValue(L"YRes"));

		// set the image depth and color space
		l_pFTKImageClip->SetChannels((l_XSIImage.GetNumChannels() == 3) ? CSLImage::SI_RGB : CSLImage::SI_RGBA);

		int l_ColorDepth = l_XSIImage.GetChannelSize() * l_XSIImage.GetNumChannels();

		if(l_ColorDepth == 24)
		{
			l_pFTKImageClip->SetColorDepth(CSLImage::SI_24BITS);
		}
		else if(l_ColorDepth == 32)
		{
			l_pFTKImageClip->SetColorDepth(CSLImage::SI_32BITS);
		}
		// set sequence info
		params = l_ImageClip.GetParameters();
		l_pFTKImageClip->SetFirstFrame((LONG) params.GetValue(L"FrameStart"));
		l_pFTKImageClip->SetLastFrame((LONG) params.GetValue(L"FrameEnd"));
		l_pFTKImageClip->SetFrameRate((float)((double) params.GetValue(L"FrameRate")));


		// set cropping and effects
		bool l_EffectsDisabled = params.GetValue(L"ImageDefinitionType");

		if(!l_EffectsDisabled)
		{
			l_pFTKImageClip->SetCropMinX((float)((double) params.GetValue(L"Xmin")));
			l_pFTKImageClip->SetCropMaxX((float)((double) params.GetValue(L"Xmax")));
			l_pFTKImageClip->SetCropMinY((float)((double) params.GetValue(L"Ymin")));
			l_pFTKImageClip->SetCropMaxY((float)((double) params.GetValue(L"Ymax")));

			// let's create our image fx template now
			CSLImageFX *l_pImageFX = l_pFTKImageClip->CreateImageFX();
			l_pImageFX->SetBlurAlpha((bool) params.GetValue(L"BlurAlpha"));
			l_pImageFX->SetBlurAmount((float) ((double) params.GetValue(L"Amount")));
			l_pImageFX->SetBlurRadius((float) ((double) params.GetValue(L"Radius")));
			l_pImageFX->SetBrightness((float) ((double) params.GetValue(L"Brightness")));
			l_pImageFX->SetConvertTo16Bits((bool) params.GetValue(L"SixteenBitsPerChannel"));
			l_pImageFX->SetFlipHorizontal((bool) params.GetValue(L"FlipX"));
			l_pImageFX->SetFlipVertical((bool) params.GetValue(L"FlipY"));
			l_pImageFX->SetGain((float) ((double) params.GetValue(L"Gain")));


			l_pImageFX->SetHue((float)((double) params.GetValue(L"Hue")));
			l_pImageFX->SetRGBA2Greyscale((bool) params.GetValue(L"GrayScale"));
			l_pImageFX->SetSaturation((float)((double) params.GetValue(L"Saturation")));

			l_pImageFX->SetScalingType((CSLImageFX::EScalingType)((LONG) params.GetValue(L"Type")));

			float xscale = (float) ((double) params.GetValue(L"ResX"));
			float yscale = (float) ((double) params.GetValue(L"ResY"));

			xscale /= (float) l_pFTKImageClip->GetWidth();
			yscale /= (float) l_pFTKImageClip->GetHeight();

			l_pImageFX->SetHorizontalScale(xscale); 
			l_pImageFX->SetVerticalScale(yscale); 		
		}
		else
		{
			l_pFTKImageClip->SetCropMinX(0.0f);
			l_pFTKImageClip->SetCropMaxX(1.0f);
			l_pFTKImageClip->SetCropMinY(0.0f);
			l_pFTKImageClip->SetCropMaxY(1.0f);
		}

		// if we have a TimeControl on this image clip then export it
		TimeControl l_timeControl = l_ImageClip.GetTimeControl();
		if (l_timeControl.IsValid())
		{
			// found one... add it to the SemanticLayer
			CSLTimeControl* l_pSLTimeControl = l_pFTKImageClip->AddTimeControl();
			params = l_ImageClip.GetParameters();

			// Set the sequence info
			CParameterRefArray l_timeControlParams = l_timeControl.GetParameters();

			// set the values
			l_pSLTimeControl->SetIn(l_timeControlParams.GetValue(L"clipin"));
			l_pSLTimeControl->SetOut(l_timeControlParams.GetValue(L"clipout"));
			l_pSLTimeControl->SetStartOffset(l_timeControlParams.GetValue(L"startoffset"));
			l_pSLTimeControl->SetScale(l_timeControlParams.GetValue(L"scale"));

			// do the extrapolations
			siTimeControlExtrapolationType	l_type = siTimeControlExtrapolationNone;
			double							l_value = 0;

			l_timeControl.GetExtrapolationBeforeType(l_type);
			l_timeControl.GetExtrapolationBeforeValue(l_value);

			CSLExtrapolation *l_pBeforeExtrapolation = l_pSLTimeControl->CreateExtrapolation(CSLTimeControl::SI_BEFORE, 
				                                                                            (CSLExtrapolation::EExtrapolationType)l_type);
			l_pBeforeExtrapolation->SetValue((float)l_value);

			l_timeControl.GetExtrapolationAfterType(l_type);
			l_timeControl.GetExtrapolationAfterValue(l_value);

			CSLExtrapolation *l_pAfterExtrapolation = l_pSLTimeControl->CreateExtrapolation(CSLTimeControl::SI_AFTER, 
				                                                                           (CSLExtrapolation::EExtrapolationType)l_type);
			l_pAfterExtrapolation->SetValue((float)l_value);
		}
	}

	return status;
}

wchar_t *CXSIImageFromXSI::GetClassID(){return L"CXSIImageFromXSI";}

/**************************************************************************************
CXSIImageLibraryToXSI
**************************************************************************************/
CXSIImageLibraryToXSI::CXSIImageLibraryToXSI() {};

CXSIImageLibraryToXSI::~CXSIImageLibraryToXSI() {};

CStatus CXSIImageLibraryToXSI::Execute(CdotXSIConverter *in_pContext)
{
	CStatus status = CStatus::OK;
	int loop;
	CXSIImageToXSI	l_ImageConverter(CRY_FILE_TYPE_NONE);

	CSLImageLibrary *l_pFTKImageLibrary = in_pContext->ftkscene()->GetImageLibrary();

	if(l_pFTKImageLibrary)
	{
		for(loop = 0; loop < l_pFTKImageLibrary->GetImageCount(); loop++)
		{
			CRef l_ImageClip;
			CSLTemplate *l_pImage = l_pFTKImageLibrary->GetImageList()[loop];		
			_XSI_CALL(l_ImageConverter.Execute(in_pContext, l_ImageClip, l_pFTKImageLibrary, NULL, &l_ImageClip, &l_pImage), L"Failed to convert image clip");
		}
	}

	return status;
}

wchar_t *CXSIImageLibraryToXSI::GetClassID(){return L"CXSIImageLibraryToXSI";}

/**************************************************************************************
CXSIImageToXSI
**************************************************************************************/
CXSIImageToXSI::CXSIImageToXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CXSIImageToXSI::~CXSIImageToXSI() {};

CStatus CXSIImageToXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;

	CSLImage *l_pImage = (CSLImage *) *io_pFTKModel;
		
	CString ClipName, FileName;
	ClipName.PutAsciiString(l_pImage->GetName());

	CSIBCString l_SearchFileName;


	CSIBCString SourceFileAbsolutePath;

	FTKUPathRelativeToAbsolute(l_pImage->GetSourceFile(), in_pContext->ftkscene()->Parser()->FileName().GetText(), SourceFileAbsolutePath);
	char *l_pSourceFile = SourceFileAbsolutePath.GetText();

	if(_access(l_pSourceFile, 4) != 0)
	{
		// find the directory of the COLLADA file
		char *l_pLastSlash = NULL;
		CSIBCString l_Directory( in_pContext->ftkscene()->Parser()->FileName().GetText() );

		l_pLastSlash = strrchr(l_Directory.GetText(), '/');
		if(l_pLastSlash == NULL)
			l_pLastSlash = strrchr(l_Directory.GetText(), '\\');

		if(l_pLastSlash != NULL)
		{
			*l_pLastSlash = 0; // Crop the filename, keep only directory (without ending \)

			// try a relative path
			char *l_pSourceFileRelativeDir = l_pImage->GetSourceFile();	

			l_SearchFileName.SetText(l_Directory.GetText());
			l_SearchFileName.Concat("/");
			l_SearchFileName.Concat(l_pSourceFileRelativeDir);

			if(_access(l_SearchFileName.GetText(), 4) == 0)
			{
				l_pSourceFile = l_SearchFileName.GetText();
			}
			else
			{
				// try to find the file into the directory of COLLADA file
				char *l_pSourceFileInColladaDir = l_pImage->GetSourceFile(); // Get the image filename

				l_pLastSlash = strrchr(l_pSourceFileInColladaDir, '/');
				if(l_pLastSlash == NULL)
					l_pLastSlash = strrchr(l_pSourceFileInColladaDir, '\\');
				if(l_pLastSlash != NULL)
					l_pSourceFileInColladaDir = l_pLastSlash + 1; // Position Next to the last slash

				l_SearchFileName.SetText(l_Directory.GetText());
				l_SearchFileName.Concat("/");
				l_SearchFileName.Concat(l_pSourceFileInColladaDir);

				if(_access(l_SearchFileName.GetText(), 4) == 0)
				{
					l_pSourceFile = l_SearchFileName.GetText();
				}
			}
		}
	}

	FileName.PutAsciiString(l_pSourceFile);

	// Create our clip
	CString l_Value;
	ImageClip2 l_XSIImage = (ImageClip2) COMMANDS::CreateImageClip(FileName, ClipName, l_Value);

	l_pImage->AttachUserData("CREF", in_pContext->AddCRef(l_XSIImage));

	// dimensions, bit depth and other stuff are set by the image source itself 
	// so there is no need to set them in XSI here

	// Set the sequence info
	CParameterRefArray imageParams = l_XSIImage.GetParameters();
	imageParams.PutValue(L"FrameStart", (LONG)l_pImage->GetFirstFrame());
	imageParams.PutValue(L"FrameEnd", (LONG)l_pImage->GetLastFrame());
	imageParams.PutValue(L"FrameRate", (double)l_pImage->GetFrameRate());

	// Set cropping and effects
	bool l_EffectsDisabled = (l_pImage->GetCropMaxX() == 1.0f) && (l_pImage->GetCropMaxY() == 1.0f) &&
		(l_pImage->GetCropMinX() == 0.0f) && (l_pImage->GetCropMinY() == 0.0f);
	

	if(!l_EffectsDisabled)
	{
		imageParams.PutValue(L"ImageDefinitionType", false);

		imageParams.PutValue(L"Xmin", (double)l_pImage->GetCropMinX());
		imageParams.PutValue(L"Xmax", (double)l_pImage->GetCropMaxX());
		imageParams.PutValue(L"Ymin", (double)l_pImage->GetCropMinY());
		imageParams.PutValue(L"Ymax", (double)l_pImage->GetCropMaxY());

		// set our image effects
		CSLImageFX *l_pImageFX = l_pImage->ImageFX();
		if(l_pImageFX)
		{
			imageParams.PutValue(L"BlurAlpha", (bool)(l_pImageFX->GetBlurAlpha() ? true : false));

			imageParams.PutValue(L"Amount", (double)l_pImageFX->GetBlurAmount());
			imageParams.PutValue(L"Radius", (double)l_pImageFX->GetBlurRadius());
			imageParams.PutValue(L"Brightness", (double)l_pImageFX->GetBrightness());

			imageParams.PutValue(L"SixteenBitsPerChannel", (bool)(l_pImageFX->GetConvertTo16Bits() ? true : false));
			imageParams.PutValue(L"FlipX", (bool)(l_pImageFX->GetFlipHorizontal() ? true : false));
			imageParams.PutValue(L"FlipY", (bool)(l_pImageFX->GetFlipVertical() ? true : false));


			imageParams.PutValue(L"Gain", (double)l_pImageFX->GetGain());
			imageParams.PutValue(L"Hue", (double)l_pImageFX->GetHue());

			imageParams.PutValue(L"GrayScale", (bool)(l_pImageFX->GetRGBA2Greyscale() ? true : false));
			imageParams.PutValue(L"Brightness", (double)l_pImageFX->GetBrightness());
			imageParams.PutValue(L"Saturation", (double)l_pImageFX->GetSaturation());

			imageParams.PutValue(L"Type", (LONG)l_pImageFX->GetScalingType());

			imageParams.PutValue(L"ResX", (double)l_pImageFX->GetHorizontalScale() * l_pImage->GetWidth());
			imageParams.PutValue(L"ResY", (double)l_pImageFX->GetVerticalScale() * l_pImage->GetHeight());
		}
	}

	// set our timecontrol
	CSLTimeControl *l_pSLTimeControl = l_pImage->TimeControl();
	if(l_pSLTimeControl)
	{
		// found one... populate through the SDK
		TimeControl l_sdkTimeControl = l_XSIImage.GetTimeControl();

		// Set the sequence info
		CParameterRefArray l_timeControlParams = l_sdkTimeControl.GetParameters();

		// set the values
		l_timeControlParams.PutValue(L"clipin", l_pSLTimeControl->GetIn());
		l_timeControlParams.PutValue(L"clipout", l_pSLTimeControl->GetOut());
		l_timeControlParams.PutValue(L"startoffset", l_pSLTimeControl->GetStartOffset());
		l_timeControlParams.PutValue(L"scale", l_pSLTimeControl->GetScale());

		// set the BEFORE extrapolations
		CSLExtrapolation * l_pBeforeExtrap = l_pSLTimeControl->GetExtrapolation(CSLTimeControl::SI_BEFORE);

		if (l_pBeforeExtrap)
		{
			CSLExtrapolation::EExtrapolationType l_beforeType = l_pBeforeExtrap->GetExtrapolationType();
			l_timeControlParams.PutValue(L"extrapbef_type", l_beforeType);
			switch (l_beforeType)
			{
				case CSLExtrapolation::SI_NO_CONTRIBUTION:
				{
					l_timeControlParams.PutValue(L"extrapbef_timehold", 0 );
					l_timeControlParams.PutValue(L"extrapbef_nbcycles", 0 );
					l_timeControlParams.PutValue(L"extrapbef_nbbounces", 0 );
					break;
				}
				case CSLExtrapolation::SI_HOLD:
				{
					l_timeControlParams.PutValue(L"extrapbef_timehold", l_pBeforeExtrap->GetValue() );
					l_timeControlParams.PutValue(L"extrapbef_nbcycles", 0 );
					l_timeControlParams.PutValue(L"extrapbef_nbbounces", 0 );
					break;
				}
				case CSLExtrapolation::SI_BOUNCE:
				{
					l_timeControlParams.PutValue(L"extrapbef_timehold", 0 );
					l_timeControlParams.PutValue(L"extrapbef_nbcycles", 0 );
					l_timeControlParams.PutValue(L"extrapbef_nbbounces", l_pBeforeExtrap->GetValue() );
					break;
				}
				case CSLExtrapolation::SI_CYCLE:
				{
					l_timeControlParams.PutValue(L"extrapbef_timehold", 0 );
					l_timeControlParams.PutValue(L"extrapbef_nbcycles", l_pBeforeExtrap->GetValue() );
					l_timeControlParams.PutValue(L"extrapbef_nbbounces", 0 );
					break;
				}
				default:
				{
					_SI_ASSERT(FALSE); // unsupported type
					break;
				}
			}
		}

		// set the AFTER extrapolations
		CSLExtrapolation * l_pAfterExtrap = l_pSLTimeControl->GetExtrapolation(CSLTimeControl::SI_AFTER);

		if (l_pAfterExtrap)
		{
			CSLExtrapolation::EExtrapolationType l_afterType = l_pAfterExtrap->GetExtrapolationType();
			l_timeControlParams.PutValue(L"extrapaft_type", l_afterType);
			switch (l_afterType)
			{
				case CSLExtrapolation::SI_NO_CONTRIBUTION:
				{
					l_timeControlParams.PutValue(L"extrapaft_timehold", 0 );
					l_timeControlParams.PutValue(L"extrapaft_nbcycles", 0 );
					l_timeControlParams.PutValue(L"extrapaft_nbbounces", 0 );
					break;
				}
				case CSLExtrapolation::SI_HOLD:
				{
					l_timeControlParams.PutValue(L"extrapaft_timehold", l_pAfterExtrap->GetValue() );
					l_timeControlParams.PutValue(L"extrapaft_nbcycles", 0 );
					l_timeControlParams.PutValue(L"extrapaft_nbbounces", 0 );
					break;
				}
				case CSLExtrapolation::SI_BOUNCE:
				{
					l_timeControlParams.PutValue(L"extrapaft_timehold", 0 );
					l_timeControlParams.PutValue(L"extrapaft_nbcycles", 0 );
					l_timeControlParams.PutValue(L"extrapaft_nbbounces", l_pAfterExtrap->GetValue() );
					break;
				}
				case CSLExtrapolation::SI_CYCLE:
				{
					l_timeControlParams.PutValue(L"extrapaft_timehold", 0 );
					l_timeControlParams.PutValue(L"extrapaft_nbcycles", l_pAfterExtrap->GetValue() );
					l_timeControlParams.PutValue(L"extrapaft_nbbounces", 0 );
					break;
				}
				default:
				{
					_SI_ASSERT(FALSE); // unsupported type
					break;
				}
			}
		}
	}

	return status;
}

wchar_t *CXSIImageToXSI::GetClassID(){return L"CXSIImageToXSI";}



