//***************************************************************************************
//
// File supervisor: Softimage Rendering & Pipeline team
//
// (c) Copyright 2001-2005 Avid Technology, Inc. . All rights reserved.
//
//***************************************************************************************

/**************************************************************************************
THIS CODE IS PUBLISHED AS A SAMPLE ONLY AND IS PROVIDED "AS IS".
IN NO EVENT SHALL SOFTIMAGE, AVID TECHNOLOGY, INC. AND/OR THEIR RESPECTIVE
SUPPLIERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
CONNECTION WITH THE USE OR PERFORMANCE OF THIS CODE .

COPYRIGHT NOTICE. Copyright  1999-2005 Avid Technology Inc. . All rights reserved. 

SOFTIMAGE is a registered trademark of Avid Technology Inc. or its subsidiaries 
or divisions. Windows NT is a registered trademark of Microsoft Corp. All other
trademarks contained herein are the property of their respective owners. 
****************************************************************************************/
/*! \file cnv_light.cpp
/*!
	light conversion classes
*/

#include "stdafx.h"
#include "cnv_light.h"
#include <xsi_x3dobject.h>
#include <xsi_light.h>
#include <xsi_time.h>
#include <Model.h>
#include <Light.h>
#include <PointLight.h>
#include <SpotLight.h>
#include <InfiniteLight.h>
#include <DirectionalLight.h>
#include <xsi_shader.h>
#include <xsi_kinematics.h>
#include <xsi_kinematicstate.h>
#include <xsi_directed.h>


/**************************************************************************************
CLightFromXSI
**************************************************************************************/
CLightFromXSI::CLightFromXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CLightFromXSI::~CLightFromXSI() {};

CStatus CLightFromXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;
	X3DObject l_XSIModel = (X3DObject) *io_pXSIModel;
	if(wcscmp(l_XSIModel.GetType().GetWideString(), L"light") == 0)
	{
		CSLModel	*l_pNewModel = NULL;
		CSLModel	*l_pParent = (CSLModel *) in_pFTKParent;
		Light		l_Light = (Light) l_XSIModel;
		CTime		l_Time;
		Shader		l_LightShader;
		CRefArray	l_ShaderArray = l_Light.GetShaders();

		if(l_ShaderArray.GetCount() > 0)
		{
			Shader l_tmpShader = (Shader) l_ShaderArray[0];
			CString l_ProgID = l_tmpShader.GetProgID(); // we only support softimage.soft_light.1
			if(l_ProgID == CString(L"Softimage.soft_light.1"))
			{
				l_LightShader = l_tmpShader;
			}
		}
		CParameterRefArray lightShaderParams = l_LightShader.GetParameters();

		// create our light depending on the type
		LONG l_LightType = l_Light.GetParameterValue(L"Type");
		switch(l_LightType)
		{
			case 0: // point light
				l_pNewModel = l_pParent->AddPointLight();
				break;

			case 1: // directional light
				l_pNewModel = l_pParent->AddDirectionalLight();
				break;

			case 2: // spot light
				l_pNewModel = l_pParent->AddSpotLight();
				CSLSpotLight *l_pSpot = (CSLSpotLight *) l_pNewModel->Primitive();
				l_pSpot->SetConeAngle(l_Light.GetParameterValue(L"LightCone"));

				if(l_LightShader.IsValid())
				{
					l_pSpot->SetSpreadAngle(lightShaderParams.GetValue(L"spread"));
				}
				break;
		}

		// grab the other info
		CSLLight *l_pBaseLight = (CSLLight *) l_pNewModel->Primitive();

		CSLLightInfo *l_pLightInfo = l_pBaseLight->LightInfo();
		if(l_pLightInfo == NULL)
			l_pLightInfo = l_pBaseLight->CreateLightInfo();

		if(l_LightShader.IsValid())
		{
			Parameter l_ColorParam = l_LightShader.GetParameter(L"color");
			CSIBCColorf l_Color;

			l_Color.m_fR = l_ColorParam.GetParameterValue(L"red");
			l_Color.m_fG = l_ColorParam.GetParameterValue(L"green");
			l_Color.m_fB = l_ColorParam.GetParameterValue(L"blue");

			l_pBaseLight->SetColor(l_Color);

			l_pLightInfo->SetIntensity(lightShaderParams.GetValue(L"intensity"));
			l_pLightInfo->SetFallofActive(lightShaderParams.GetValue(L"atten"));
			l_pLightInfo->SetFallofMode((CSLLightInfo::EFallofMode)((LONG)lightShaderParams.GetValue(L"mode")));
			l_pLightInfo->SetStart(lightShaderParams.GetValue(L"start"));
			l_pLightInfo->SetEnd(lightShaderParams.GetValue(L"stop"));
			l_pLightInfo->SetShadowActive(lightShaderParams.GetValue(L"shadow"));
			l_pLightInfo->SetUmbra(lightShaderParams.GetValue(L"factor"));
			l_pLightInfo->SetLightAsEnergy(lightShaderParams.GetValue(L"use_color"));
			l_pLightInfo->SetPhotonFactor(lightShaderParams.GetValue(L"energy_factor"));

		}

		// now the kinestate data
		// this code gets the local kinematics transform
		MATH::CVector3	l_XSIVector;
		CSIBCVector3D	l_FTKVector;

		l_XSIVector = l_XSIModel.GetKinematics().GetLocal().GetTransform().GetTranslation();
		l_FTKVector.Set((float) l_XSIVector.GetX(), (float) l_XSIVector.GetY(), (float) l_XSIVector.GetZ());
		l_pBaseLight->SetPosition(l_FTKVector);

		if(l_LightType == 1)
		{
			CSLDirectionalLight *l_pDirLight = (CSLDirectionalLight *) l_pNewModel->Primitive();
			MATH::CRotation l_Euler = l_XSIModel.GetKinematics().GetLocal().GetTransform().GetRotation();
			l_XSIVector = l_Euler.GetXYZAngles();

			l_FTKVector.Set
			(
				in_pContext->AngleToFTK((float) l_XSIVector.GetX(), true), 
				in_pContext->AngleToFTK((float) l_XSIVector.GetY(), true), 
				in_pContext->AngleToFTK((float) l_XSIVector.GetZ(), true)
			);

			l_pDirLight->SetDirection(l_FTKVector);
		}

		if(l_LightType == 2)
		{
			Directed l_SpotLight = (Directed) l_XSIModel;
			CSLSpotLight *l_pSpot = (CSLSpotLight *) l_pNewModel->Primitive();

			// ok now we just fill in the interest position, but we should really put
			// a direction constraint somewhere in there two

			X3DObject l_Interest = l_SpotLight.GetInterest();
			if(l_Interest.IsValid())
			{
				l_XSIVector = l_Interest.GetKinematics().GetLocal().GetTransform().GetTranslation();
				l_FTKVector.Set((float) l_XSIVector.GetX(), (float) l_XSIVector.GetY(), (float) l_XSIVector.GetZ());
				l_pSpot->SetInterestPosition(l_FTKVector);
			}
		}

		*io_pFTKModel = l_pNewModel;
		l_pNewModel->AttachUserData("CREF", in_pContext->AddCRef(*io_pXSIModel));

	}

	return status;
}

wchar_t *CLightFromXSI::GetClassID(){return L"CLightFromXSI";}


/**************************************************************************************
CLightToXSI
**************************************************************************************/
CLightToXSI::CLightToXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CLightToXSI::~CLightToXSI() {};

CStatus CLightToXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;
	CSLModel *l_pFTKModel = (CSLModel *) *io_pFTKModel;

	if(	l_pFTKModel->Primitive() && (
		(l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_POINT_LIGHT) || 
		(l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_INFINITE_LIGHT) || 
		(l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_DIRECTIONAL_LIGHT) || 
		(l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_SPOT_LIGHT)))
	{
		CTime		l_Time;

		X3DObject	l_XSIParent = (X3DObject) in_XSIParent;
		Light		l_XSILight;
		CString		l_LightName;
		Shader		l_LightShader;

		l_LightName.PutAsciiString(l_pFTKModel->GetName());

		// create the light
		if(l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_POINT_LIGHT)
		{
			l_XSIParent.AddLight(CString(L"Point.Preset"), false, l_LightName, l_XSILight);
		}
		else if(l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_INFINITE_LIGHT)
		{
			l_XSIParent.AddLight(CString(L"Infinite.Preset"), false, l_LightName, l_XSILight);
		}
		else if(l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_DIRECTIONAL_LIGHT)
		{
			l_XSIParent.AddLight(CString(L"Infinite.Preset"), false, l_LightName, l_XSILight);
		}
		else if(l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_SPOT_LIGHT)
		{
			l_XSIParent.AddLight(CString(L"Spot.Preset"), true, l_LightName, l_XSILight);
		}

		// now the kinestate data
		// this code gets the local kinematics transform
		MATH::CVector3			l_XSIVector;
		MATH::CTransformation	l_Transformation;
		CSIBCVector3D			l_FTKVector;
		CSLLight				*l_pBaseLight = (CSLLight *) l_pFTKModel->Primitive();

		l_FTKVector = l_pBaseLight->GetPosition();

		l_XSIVector.Set((double) l_FTKVector.GetX(), (double) l_FTKVector.GetY(), (double) l_FTKVector.GetZ());
		l_Transformation = l_XSILight.GetKinematics().GetLocal().GetTransform();
		l_Transformation.SetTranslation(l_XSIVector);

		// treat direction now

		if(l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_DIRECTIONAL_LIGHT)
		{
			MATH::CRotation		l_Rotation;
			CSLDirectionalLight	*l_pDirectionalLight = (CSLDirectionalLight*) l_pFTKModel->Primitive();
			l_FTKVector = l_pDirectionalLight->GetDirection();
			l_Rotation.SetFromXYZAngles
			(
				(double) in_pContext->AngleFromFTK(l_FTKVector.GetX(), true),
				(double) in_pContext->AngleFromFTK(l_FTKVector.GetY(), true),
				(double) in_pContext->AngleFromFTK(l_FTKVector.GetZ(), true)
			);

			l_Transformation.SetRotation(l_Rotation);

		}

		// treat spot interest

		if(l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_SPOT_LIGHT)
		{
			Directed l_SpotLight = (Directed) l_XSILight;
			CSLSpotLight *l_pSpot = (CSLSpotLight *) l_pFTKModel->Primitive();

			// ok now we just fill in the interest position, but we should really put
			// a direction constraint somewhere in there two

			X3DObject l_Interest = l_SpotLight.GetInterest();
			if(l_Interest.IsValid())
			{
				MATH::CTransformation	l_InterestTransformation;

				l_FTKVector = l_pSpot->GetInterestPosition();
				l_InterestTransformation = l_Interest.GetKinematics().GetLocal().GetTransform();
				l_XSIVector.Set((double) l_FTKVector.GetX(), (double) l_FTKVector.GetY(), (double) l_FTKVector.GetZ());
				l_InterestTransformation.SetTranslation(l_XSIVector);
				l_Interest.GetKinematics().GetLocal().PutTransform(l_InterestTransformation);
			}
		}


		l_XSILight.GetKinematics().GetLocal().PutTransform(l_Transformation);

		*io_pXSIModel = l_XSILight;

		CRefArray	l_ShaderArray = l_XSILight.GetShaders();

		if(l_ShaderArray.GetCount() > 0)
		{
			Shader l_tmpShader = (Shader) l_ShaderArray[0];
			CString l_ProgID = l_tmpShader.GetProgID(); // we only support softimage.soft_light.1
			if(l_ProgID == CString(L"Softimage.soft_light.1"))
			{
				l_LightShader = l_tmpShader;
			}
		}

		// we treat the other light information

		if(l_pFTKModel->Primitive()->Type() == CSLTemplate::SI_SPOT_LIGHT)
		{
			CSLSpotLight *l_pSpot = (CSLSpotLight *) l_pFTKModel->Primitive();
			l_LightShader.GetParameter(CString(L"spread")).PutValue(CValue(l_pSpot->GetSpreadAngle()));
		}

		Parameter l_ColorParam = l_LightShader.GetParameter(CString(L"color"));
		CSIBCColorf l_Color = l_pBaseLight->GetColor();

		l_ColorParam.GetParameter(CString(L"red")).PutValue(CValue(l_Color.m_fR));
		l_ColorParam.GetParameter(CString(L"green")).PutValue(CValue(l_Color.m_fG));
		l_ColorParam.GetParameter(CString(L"blue")).PutValue(CValue(l_Color.m_fB));

		CSLLightInfo *l_pLightInfo = l_pBaseLight->LightInfo();
		if(l_pLightInfo)
		{
			l_LightShader.GetParameter(CString(L"intensity")).PutValue(CValue(l_pLightInfo->GetIntensity()));
			l_LightShader.GetParameter(CString(L"atten")).PutValue(CValue(l_pLightInfo->GetFallofActive()));
			l_LightShader.GetParameter(CString(L"mode")).PutValue(CValue((LONG)l_pLightInfo->GetFallofMode()));
			l_LightShader.GetParameter(CString(L"start")).PutValue(CValue(l_pLightInfo->GetStart()));
			l_LightShader.GetParameter(CString(L"stop")).PutValue(CValue(l_pLightInfo->GetEnd()));
			l_LightShader.GetParameter(CString(L"shadow")).PutValue(CValue(l_pLightInfo->GetShadowActive()));
			l_LightShader.GetParameter(CString(L"factor")).PutValue(CValue(l_pLightInfo->GetUmbra()));
			l_LightShader.GetParameter(CString(L"use_color")).PutValue(CValue(l_pLightInfo->GetLightAsEnergy()));
			l_LightShader.GetParameter(CString(L"energy_factor")).PutValue(CValue(l_pLightInfo->GetPhotonFactor()));
		}

		l_pFTKModel->AttachUserData("CREF", in_pContext->AddCRef(l_XSILight));

	}	
	return status;
}

wchar_t *CLightToXSI::GetClassID(){return L"CLightToXSI";}

