//***************************************************************************************
//
// File supervisor: Softimage Rendering & Pipeline team
//
// (c) Copyright 2001-2005 Avid Technology, Inc. . All rights reserved.
//
//***************************************************************************************

/**************************************************************************************
THIS CODE IS PUBLISHED AS A SAMPLE ONLY AND IS PROVIDED "AS IS".
IN NO EVENT SHALL SOFTIMAGE, AVID TECHNOLOGY, INC. AND/OR THEIR RESPECTIVE
SUPPLIERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
CONNECTION WITH THE USE OR PERFORMANCE OF THIS CODE .

COPYRIGHT NOTICE. Copyright  1999-2005 Avid Technology Inc. . All rights reserved. 

SOFTIMAGE is a registered trademark of Avid Technology Inc. or its subsidiaries 
or divisions. Windows NT is a registered trademark of Microsoft Corp. All other
trademarks contained herein are the property of their respective owners. 
****************************************************************************************/
/*! \file cnv_mesh.cpp
/*!
	Mesh conversion classes
*/

#include "stdafx.h"
#include "cnv_mesh.h"
#include <xsi_polygonmesh.h>
#include <xsi_geometryaccessor.h>
#include <xsi_meshbuilder.h>
#include <XSIMesh.h>
#include <XSIShape.h>
#include <xsi_x3dobject.h>
#include <xsi_time.h>
#include <xsi_primitive.h>
#include <xsi_cluster.h>
#include <Model.h>
#include <XSISubComponentAttributeList.h>
#include <XSIVertexList.h>
#include <MaterialLibrary.h>
#include <Material.h>
#include <xsi_material.h>
#include <XSIPolygonList.h>
#include <XSITriangleList.h>
#include <GlobalMaterial.h>
#include <xsi_clusterpropertybuilder.h>
#include <xsi_polygonface.h>
#include <xsi_shader.h>
#include <XSIIndexList.h>
#include <map>
#include "cmdstubs.h"


/**************************************************************************************
CMeshFromXSI
**************************************************************************************/
CMeshFromXSI::CMeshFromXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CMeshFromXSI::~CMeshFromXSI() {};

CStatus CMeshFromXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;

	X3DObject l_XSIModel = (X3DObject) *io_pXSIModel;

	if(wcscmp(l_XSIModel.GetType().GetWideString(), L"polymsh") == 0)
	{
		Material l_Material = l_XSIModel.GetMaterial();

		Parameter l_parameter = l_Material.GetParameter(L"surface");
		// get the shader itself
		Shader l_Shader = l_parameter.GetSource();
		CString l_ProgID = l_Shader.GetProgID();
		char* l_strProgID = (char*)l_ProgID.GetAsciiString();
		if (!strstr(l_strProgID, "CryShader.1"))
		{
			Application app;
			app.LogMessage(L"No CryShader on this object: " + l_XSIModel.GetName(), siErrorMsg);

			return status;
		}

		CSLModel *l_pNewModel;
		CSLModel *l_pParent = (CSLModel *) in_pFTKParent;
		CTime	l_Time;

		l_pNewModel = l_pParent->AddXSIMesh();
		*io_pFTKModel = l_pNewModel;
		l_pNewModel->AttachUserData("CREF", in_pContext->AddCRef(*io_pXSIModel));
	
		CSLXSIMesh			*l_pFTKMesh = (CSLXSIMesh *) l_pNewModel->Primitive();
		CSLXSIShape			*l_pOriginalShape = l_pFTKMesh->XSIShape();
		PolygonMesh			l_XSIMesh = (PolygonMesh) l_XSIModel.GetActivePrimitive().GetGeometry();

		// Here we extract the ApplySubdivisionToGeometry parameter value
		Property exportProp = in_pContext->exportproperty();		

		bool l_bApplySubdivisionToGeometry = exportProp.GetParameterValue(L"ApplySubdivisionToGeometry");

		// Here we extract the Triangulate parameter value
		bool l_bTriangulate = exportProp.GetParameterValue(L"Triangulate");

		// Get the geometry approximation properties
		Property l_Geomapprox = l_XSIModel.GetProperties().GetItem(L"Geometry Approximation");
			
		// grab the discontinuity properties for this mesh
		bool   l_bAutoDiscontinuity = true;
		double l_DiscontinuityAngle = 60.0;

		if (l_Geomapprox.IsValid())
		{
			// if object has it's own copy of these properties
			l_bAutoDiscontinuity = l_Geomapprox.GetParameterValue(L"gapproxmoad");
			l_DiscontinuityAngle = l_Geomapprox.GetParameterValue(L"gapproxmoan");
		}

		CGeometryAccessor	l_XSIGeometryAccessor;
		
		if (l_bApplySubdivisionToGeometry)
		{
			// get the current subdivision option for this mesh
			LONG l_SubdivisionLevel = 0;
			LONG l_SubdivisionRule = 0;
			bool l_bUseLoopForTriangles = false;

			if (l_Geomapprox.IsValid())
			{
				l_SubdivisionLevel = l_Geomapprox.GetParameterValue(L"gapproxmosl");
			}

			l_SubdivisionRule = COMMANDS::GetValue(l_XSIModel.GetFullName() + CString(L".polymsh.subdrule"), l_Time.GetTime());
			l_bUseLoopForTriangles = COMMANDS::GetValue(l_XSIModel.GetFullName() + CString(L".polymsh.subdloop"), l_Time.GetTime());

			// create a Geometry accessor object for the mesh with current subdivition setting
			l_XSIGeometryAccessor = l_XSIMesh.GetGeometryAccessor( siConstructionModeModeling,
																   (siSubdivisionRuleType)l_SubdivisionRule,
																   l_SubdivisionLevel,
																   l_bUseLoopForTriangles,
																   l_bAutoDiscontinuity,
																   l_DiscontinuityAngle);
		}
		else
		{
			// create a Geometry accessor object for the original mesh (no subdivition)
			l_XSIGeometryAccessor = l_XSIMesh.GetGeometryAccessor( siConstructionModeModeling,
																   siCatmullClark, /* default value */
																   0, /* default value */ 
																   false, /* default value */
																   l_bAutoDiscontinuity,
																   l_DiscontinuityAngle);
		}

		if(l_XSIGeometryAccessor.IsValid())
		{
			// the mesh CREF will hold the XSIGeometryAccessor
			l_pFTKMesh->AttachUserData("GEOMETRYACCESSOR", in_pContext->AddCRef(l_XSIGeometryAccessor));

			// Start filling in the vertex information
			int l_nbVertices = l_XSIGeometryAccessor.GetVertexCount();

			// create vertex positions
			CSLXSISubComponentAttributeList *l_pComponentList = l_pOriginalShape->AddVertexPositionList();
			CSLXSISubComponentAttributeList::CSLFloatArray *l_pFTKPosition = l_pComponentList->GetAttributeArray();
			l_pComponentList->Template()->InstanceName().SetText("position");
			
			l_pFTKPosition->Extend(l_nbVertices * 3);

			int loop, loop2;

			// start filling in the position array
			CDoubleArray l_PositionArray;
			l_XSIGeometryAccessor.GetVertexPositions(l_PositionArray);

			for(loop = 0; loop < (l_nbVertices * 3); loop++)
			{
				(*l_pFTKPosition)[loop] = (float)l_PositionArray[loop];
			}	

			// add the required number of vertices
			CSLXSIVertexList *l_pFTKVertexList = l_pFTKMesh->XSIVertexList();
			l_pFTKVertexList->SetCount(l_nbVertices);

			// add the position attribute
			l_pFTKVertexList->AddAttribute("position");

			// add the weight map attributes
			CRefArray l_XSIWeightMapPropArray = l_XSIGeometryAccessor.GetWeightMaps();

			for (loop2 = 0; loop2 < l_XSIWeightMapPropArray.GetCount(); loop2++)
			{
				ClusterProperty		l_XSIWeightMapProp(l_XSIWeightMapPropArray.GetItem(loop2));

				l_pFTKVertexList->AddAttribute((char*)l_XSIWeightMapProp.GetName().GetAsciiString());
			}

            // fill the attributes indicies
			for (loop = 0; loop < l_pFTKVertexList->GetAttributeCount(); loop++)
			{
				CSLXSISubComponentList::CSLIntArray *l_pAttributeIndices = l_pFTKVertexList->GetAttributeIndices(loop);

				for (loop2 = 0; loop2 < l_nbVertices; loop2++)
				{
					(*l_pAttributeIndices)[loop2] = loop2;
				}	
			}

			CRefArray						l_XSIMaterialList = l_XSIGeometryAccessor.GetMaterials();
			CSIBCArray<CSLBaseMaterial*>	l_FTKMaterialList;
			l_FTKMaterialList.Extend(l_XSIMaterialList.GetCount());

			CLongArray						l_MatIndices;
			l_XSIGeometryAccessor.GetPolygonMaterialIndices(l_MatIndices);

			// build our list of FTK materials at the same time
			for(loop = 0; loop < l_XSIMaterialList.GetCount(); loop++)
			{
				for(loop2 = 0; loop2 < in_pContext->ftkscene()->GetMaterialLibrary()->GetMaterialCount(); loop2++)
				{
					CSLBaseMaterial *l_pFTKMaterial = in_pContext->ftkscene()->GetMaterialLibrary()->GetMaterialList()[loop2];
					CSIBCUserData *l_pUserData = l_pFTKMaterial->FindUserData("CREF");

					if(l_pUserData)
					{
						CRef *l_pCRef = (CRef*)(l_pUserData->GetData());
						Material l_XSIMaterial = (Material) *l_pCRef;
						Material l_XSIMaterial2 = (Material) l_XSIMaterialList[loop];
						if(l_XSIMaterial == l_XSIMaterial2)
						{
							l_FTKMaterialList[loop] = l_pFTKMaterial;

							// CrySpecific: Flag material as "USED"
							CSIBCUserData *l_pUserData = l_pFTKMaterial->FindUserData("USED");
							
							if (!l_pUserData)
							{
								l_pFTKMaterial->AttachUserData("USED", 0);
							}

							break;
						}
					}
				}
			}

			// Triangulate
			if (l_bTriangulate)
			{
				CLongArray		l_TriangleNodeIndices;
				CLongArray		l_TriangleVertexIndices;
				CLongArray		l_PolygonTriangleIndices;

				LONG			l_TriangleCount = l_XSIGeometryAccessor.GetTriangleCount();

				l_XSIGeometryAccessor.GetTriangleNodeIndices( l_TriangleNodeIndices );
				l_XSIGeometryAccessor.GetTriangleVertexIndices( l_TriangleVertexIndices );
				l_XSIGeometryAccessor.GetPolygonTriangleIndices( l_PolygonTriangleIndices );

				// create an arrays to hole the number of triangle per triangle list and the number of nodes per triangle list (nbr of triangle *3)
				CSIBCArray<int>	l_TriangleCountPerTriangleList;
				CSIBCArray<int>	l_PolynodeCountPerTriangleList;

				l_TriangleCountPerTriangleList.Extend(l_XSIMaterialList.GetCount());
				l_PolynodeCountPerTriangleList.Extend(l_XSIMaterialList.GetCount());

				int l_nMatIndice;

				for(loop = 0; loop < l_XSIGeometryAccessor.GetPolygonCount(); loop++)
				{
					l_nMatIndice = l_MatIndices[loop];
				}

				// clear the counts to 0
				for(loop = 0; loop < l_XSIMaterialList.GetCount(); loop++)
				{
					l_TriangleCountPerTriangleList[loop] = 0;
					l_PolynodeCountPerTriangleList[loop] = 0;
				}

				// now fill in the counts
				for(loop = 0; loop < l_TriangleCount; loop++)
				{
					// the triangle X correspond in fact to the polygon Y
					// the material indices of the triangle is the one of it's corresponding polygon
					int l_nPolygonIndice = l_PolygonTriangleIndices[loop];
					int l_nTriangleListIndex = l_MatIndices[l_PolygonTriangleIndices[loop]];
					l_TriangleCountPerTriangleList[l_nTriangleListIndex]++;
					l_PolynodeCountPerTriangleList[l_nTriangleListIndex] += 3;
				}
		
				// we do not create/export empty FTK TriangleList so we have to create and set
				// a table to get the proper FTK TriangleList that correspong to XSIMaterial since they
				// may not correspong directly in the case where one material in the list isn't used
				// by any triangle (ex.: when all the triangle of a mesh are in a cluster, the object global
				// material is still in the list even if it's not associated to any triangle).
				LONG* l_pMatToFTKTriangleListIndices = new LONG[l_XSIMaterialList.GetCount()];
				LONG l_nCurrentFTKTriangleListIndex = 0;

				// and create the triangle lists
				for(loop = 0; loop < l_XSIMaterialList.GetCount(); loop++)
				{
					if (l_TriangleCountPerTriangleList[loop] > 0)
					{
						l_pMatToFTKTriangleListIndices[loop] = l_nCurrentFTKTriangleListIndex++;

						CSLXSITriangleList *l_pFTKTriangleList = l_pFTKMesh->AddXSITriangleList();

						// set the triangle list name
						char l_pName[32];
						sprintf(l_pName, "TriList%02d", loop);
						l_pFTKTriangleList->SetName(l_pName);

						l_pFTKTriangleList->SetCount(l_PolynodeCountPerTriangleList[loop]);

						// now add the user normal attributes
						CRefArray l_XSIUserNormalPropArray = l_XSIGeometryAccessor.GetUserNormals();

						bool l_bExportXSINormals = true;

						for (loop2 = 0; loop2 < l_XSIUserNormalPropArray.GetCount(); loop2++)
						{
							ClusterProperty		l_XSIUserNormalProp(l_XSIUserNormalPropArray.GetItem(loop2));

							l_pFTKTriangleList->AddAttribute((char*)l_XSIUserNormalProp.GetName().GetAsciiString());

							if (strcmp("XSINormal",(char*)l_XSIUserNormalProp.GetName().GetAsciiString()) == 0)
							{
								l_bExportXSINormals = false;
							}
						}

						if (l_bExportXSINormals)
							l_bExportXSINormals = in_pContext->exportproperty().GetParameterValue(L"ExportXSINormals");

						// XSI Normal
						if (l_bExportXSINormals)
						{
							l_pFTKTriangleList->AddAttribute("XSINormal");
						}

						// now add the texture coordinate (UV) attributes
						CRefArray l_XSIUVPropArray = l_XSIGeometryAccessor.GetUVs();

						for (loop2 = 0; loop2 < l_XSIUVPropArray.GetCount(); loop2++)
						{
							ClusterProperty		l_XSIUVProp(l_XSIUVPropArray.GetItem(loop2));

							l_pFTKTriangleList->AddAttribute((char*)l_XSIUVProp.GetName().GetAsciiString());
						}

						// now add the vertex color attributes
						CRefArray l_XSIVertexColorPropArray = l_XSIGeometryAccessor.GetVertexColors();

						for (loop2 = 0; loop2 < l_XSIVertexColorPropArray.GetCount(); loop2++)
						{
							ClusterProperty		l_XSIVertexColorProp(l_XSIVertexColorPropArray.GetItem(loop2));

							l_pFTKTriangleList->AddAttribute((char*)l_XSIVertexColorProp.GetName().GetAsciiString());
						}

						// we'll empty this one since we will reuse it 
						l_PolynodeCountPerTriangleList[loop] = 0;

						l_pFTKTriangleList->SetMaterial(l_FTKMaterialList[loop]);
					}
					else
					{
						l_pMatToFTKTriangleListIndices[loop] = -1;
					}
				}

				//  now let's set the node counts and node ids
				int l_CurrentNode = 0;
				int l_NodeCount = 0;

				CLongArray l_XSITriangleNodeIndices;
				l_XSIGeometryAccessor.GetTriangleNodeIndices(l_XSITriangleNodeIndices);

				for(loop = 0; loop < l_TriangleCount; loop++)
				{
					int l_nMatListIndex = l_MatIndices[l_PolygonTriangleIndices[loop]];
					int l_nFTKTriangleListIndex = l_pMatToFTKTriangleListIndices[l_nMatListIndex];

					if (l_nFTKTriangleListIndex >= 0)
					{
						CSLXSITriangleList *l_pFTKTriangleList = l_pFTKMesh->XSITriangleLists()[l_nFTKTriangleListIndex];

						CSLXSIPolygonList::CSLIntArray* l_pFTKPolynodeVertexIndices = l_pFTKTriangleList->GetVertexIndices();

						// set the attribute indices
						CSIBCArray<CSLXSISubComponentList::CSLIntArray*> *l_pFTKAttributeIndices = l_pFTKTriangleList->GetAttributeIndicesArray();

						for(int loop2 = 0; loop2 < 3; loop2++)
						{
							(*l_pFTKPolynodeVertexIndices)[l_PolynodeCountPerTriangleList[l_nMatListIndex]] = l_TriangleVertexIndices[l_CurrentNode];

							for (int loop3 = 0; loop3 < l_pFTKAttributeIndices->GetUsed(); loop3++)
							{
								(*((*l_pFTKAttributeIndices)[loop3]))[l_PolynodeCountPerTriangleList[l_nMatListIndex]] = l_XSITriangleNodeIndices[l_CurrentNode];
							}

							l_PolynodeCountPerTriangleList[l_nMatListIndex]++;
							l_CurrentNode ++;
						}
					}
				}

				l_NodeCount = l_CurrentNode;

				// clear our markers
				for(loop = 0; loop < l_XSIMaterialList.GetCount(); loop++)
				{
					l_PolynodeCountPerTriangleList[loop] = 0;
					l_TriangleCountPerTriangleList[loop] = 0;
				}

				delete [] l_pMatToFTKTriangleListIndices;
			}
			else
			{
				// create our polygon lists
				CRefArray						l_XSIMaterialList = l_XSIGeometryAccessor.GetMaterials();
				CSIBCArray<int>					l_PolygonCountList;
				LONG							l_PolygonCount = l_XSIGeometryAccessor.GetPolygonCount();
				CLongArray						l_PolygonVertexCount;
				CLongArray						l_PolygonVertexIndices;
				CLongArray						l_PolygonNodeIndices;
				CSIBCArray<int>					l_PolygonCountPerPolygonList;
				CSIBCArray<int>					l_PolynodeCountPerPolygonList;
				
				l_XSIGeometryAccessor.GetPolygonVerticesCount(l_PolygonVertexCount);
				l_XSIGeometryAccessor.GetVertexIndices(l_PolygonVertexIndices);
				l_XSIGeometryAccessor.GetNodeIndices(l_PolygonNodeIndices);

				l_PolygonCountPerPolygonList.Extend(l_XSIMaterialList.GetCount());
				l_PolynodeCountPerPolygonList.Extend(l_XSIMaterialList.GetCount());
				l_PolygonCountList.Extend(l_XSIMaterialList.GetCount());
			
				// clear the counts to 0
				for(loop = 0; loop < l_XSIMaterialList.GetCount(); loop++)
				{
					l_PolygonCountList[loop] = 0;
					l_PolynodeCountPerPolygonList[loop] = 0;
				}

				// now fill in the counts
				for(loop = 0; loop < l_PolygonCount; loop++)
				{
					int l_nPolygonListIndex = l_MatIndices[loop];
					l_PolygonCountList[l_nPolygonListIndex] ++;
					l_PolynodeCountPerPolygonList[l_nPolygonListIndex] += l_PolygonVertexCount[loop];
				}
		
				// we do not create/export empty FTKPolygonList so we have to create and set
				// a table to get the proper FTKPolygonList that correspong to XSIMaterial since they
				// may not correspong directly in the case where one material in the list isn't used
				// by any polygon (ex.: when all the polygon of a mesh are in a cluster, the object global
				// material is still in the list even if it's not associated to any polygon).
				LONG* l_pMatToFTKPolygonListIndices = new LONG[l_XSIMaterialList.GetCount()];
				LONG l_nCurrentFTKPolygonListIndex = 0;

				// and create the polygon lists
				for(loop = 0; loop < l_XSIMaterialList.GetCount(); loop++)
				{
					if (l_PolygonCountList[loop] > 0)
					{
						l_pMatToFTKPolygonListIndices[loop] = l_nCurrentFTKPolygonListIndex++;

						CSLXSIPolygonList *l_pFTKPolygonList = l_pFTKMesh->AddXSIPolygonList();

						// set the polygon list name
						char l_pName[32];
						sprintf(l_pName, "PolyList%02d", loop);
						l_pFTKPolygonList->SetName(l_pName);

						l_pFTKPolygonList->SetCount(l_PolynodeCountPerPolygonList[loop]);
						l_PolynodeCountPerPolygonList[loop] = 0;
						l_PolygonCountPerPolygonList[loop] = 0;
						l_pFTKPolygonList->GetPolygonNodeCountArray()->Extend(l_PolygonCountList[loop]);

						// now add the user normal attributes
						CRefArray l_XSIUserNormalPropArray = l_XSIGeometryAccessor.GetUserNormals();

						bool l_bExportXSINormals = true;

						for (loop2 = 0; loop2 < l_XSIUserNormalPropArray.GetCount(); loop2++)
						{
							ClusterProperty		l_XSIUserNormalProp(l_XSIUserNormalPropArray.GetItem(loop2));

							l_pFTKPolygonList->AddAttribute((char*)l_XSIUserNormalProp.GetName().GetAsciiString());

							if (strcmp("XSINormal",(char*)l_XSIUserNormalProp.GetName().GetAsciiString()) == 0)
							{
								l_bExportXSINormals = false;
							}
						}

						if (l_bExportXSINormals)
							l_bExportXSINormals = in_pContext->exportproperty().GetParameterValue(L"ExportXSINormals");

						// XSI Normal
						if (l_bExportXSINormals)
						{
							l_pFTKPolygonList->AddAttribute("XSINormal");
						}

						// now add the texture coordinate (UV) attributes
						CRefArray l_XSIUVPropArray = l_XSIGeometryAccessor.GetUVs();

						for (loop2 = 0; loop2 < l_XSIUVPropArray.GetCount(); loop2++)
						{
							ClusterProperty		l_XSIUVProp(l_XSIUVPropArray.GetItem(loop2));

							l_pFTKPolygonList->AddAttribute((char*)l_XSIUVProp.GetName().GetAsciiString());
						}

						// now add the vertex color attributes
						CRefArray l_XSIVertexColorPropArray = l_XSIGeometryAccessor.GetVertexColors();

						for (loop2 = 0; loop2 < l_XSIVertexColorPropArray.GetCount(); loop2++)
						{
							ClusterProperty		l_XSIVertexColorProp(l_XSIVertexColorPropArray.GetItem(loop2));

							l_pFTKPolygonList->AddAttribute((char*)l_XSIVertexColorProp.GetName().GetAsciiString());
						}

						l_pFTKPolygonList->SetMaterial(l_FTKMaterialList[loop]);

						// now add the index list
						// if there are no polygon clusters, don't add the list
						// we currently don't support export of polygon clusters if there is triangulation or subdivision
						CRefArray l_XSIPolyClusters = l_XSIGeometryAccessor.GetClusters(siClusterPolygonType);
						if (!l_bApplySubdivisionToGeometry && l_XSIPolyClusters && l_XSIPolyClusters.GetCount() > 0) 
						{
							CSLXSIIndexList* l_pFTKIndexList = l_pFTKPolygonList->AddIndexList();
							if (l_pFTKIndexList) 
							{
								CSLXSIIndexList::CSLIntArray *l_pFTKIndexArray = l_pFTKIndexList->GetIndexArray();
								l_pFTKIndexArray->Extend(l_PolygonCountList[loop]);

								// grab the index of the polygons, in the order they've been associated with the materials
								int l_PolyIndexCount = 0;
								for (int loop2 = 0; loop2 < l_PolygonCount; loop2++)
								{
									// loop is the current material index (aka the current poly list index)
									// loop2 is the polygon index
									if (l_MatIndices[loop2] == loop)
									{
										(*l_pFTKIndexArray)[l_PolyIndexCount] = loop2;
										l_PolyIndexCount++;
									}
								}
								assert (l_PolyIndexCount == l_PolygonCountList[loop]);
							}
						}
					}
					else
					{
						l_pMatToFTKPolygonListIndices[loop] = -1;
					}
				}

				//  now let's set the node counts and node ids
				int l_CurrentNode = 0;
				int l_NodeCount = 0;
				for(loop = 0; loop < l_PolygonCount; loop++)
				{
					int l_nMatListIndex = l_MatIndices[loop];
					int l_nFTKPolygonListIndex = l_pMatToFTKPolygonListIndices[l_nMatListIndex];

					if (l_nFTKPolygonListIndex >= 0)
					{
						CSLXSIPolygonList *l_pFTKPolygonList = l_pFTKMesh->XSIPolygonLists()[l_nFTKPolygonListIndex];
						CSLXSIPolygonList::CSLIntArray *l_pPolynodeCountArray = l_pFTKPolygonList->GetPolygonNodeCountArray();

						(*l_pPolynodeCountArray)[l_PolygonCountPerPolygonList[l_nMatListIndex]] = l_PolygonVertexCount[loop];
						CSLXSIPolygonList::CSLIntArray* l_pFTKPolynodeVertexIndices = l_pFTKPolygonList->GetVertexIndices();

						CSIBCArray<CSLXSISubComponentList::CSLIntArray*> *l_pFTKAttributeIndices = l_pFTKPolygonList->GetAttributeIndicesArray();

						for(loop2 = 0; loop2 < l_PolygonVertexCount[loop]; loop2++)
						{
							(*l_pFTKPolynodeVertexIndices)[l_PolynodeCountPerPolygonList[l_nMatListIndex]] = l_PolygonVertexIndices[l_CurrentNode];

							for (int loop3 = 0; loop3 < l_pFTKAttributeIndices->GetUsed(); loop3++)
							{
								(*((*l_pFTKAttributeIndices)[loop3]))[l_PolynodeCountPerPolygonList[l_nMatListIndex]] = l_PolygonNodeIndices[l_CurrentNode];
							}

							l_PolynodeCountPerPolygonList[l_nMatListIndex] ++;
							l_CurrentNode ++;
						}

						l_PolygonCountPerPolygonList[l_nMatListIndex] ++;
					}
				}

				l_NodeCount = l_CurrentNode;

				// clear our markers
				for(loop = 0; loop < l_XSIMaterialList.GetCount(); loop++)
				{
					l_PolynodeCountPerPolygonList[loop] = 0;
					l_PolygonCountPerPolygonList[loop] = 0;
				}

				delete [] l_pMatToFTKPolygonListIndices;
			}
		}
	}

	return status;
}

wchar_t *CMeshFromXSI::GetClassID(){return L"CMeshFromXSI";}

struct HelperPoly {
	int count;
	LONG* indices;
};

/**************************************************************************************
CMeshToXSI
**************************************************************************************/
CMeshToXSI::CMeshToXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CMeshToXSI::~CMeshToXSI() {};

CStatus CMeshToXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;
	LONG l_lMaxPolygon, loop, loop2;

	if (*io_pFTKModel)
	{
		CSLModel *l_pFTKModel = (CSLModel *)(*io_pFTKModel);

		if (l_pFTKModel->Primitive() && l_pFTKModel->Primitive()->Type() == CSLTemplate::XSI_MESH)
		{
			CSLXSIMesh* l_pFTKMesh = (CSLXSIMesh*)l_pFTKModel->Primitive();

			if(l_pFTKMesh->XSIVertexList()->GetCount() == 0)
			{
				return status;
			}

			X3DObject	l_XSIParent = (X3DObject) in_XSIParent;
		
			CString l_Name;
			l_Name.PutAsciiString(l_pFTKModel->GetName());

			MATH::CVector3Array l_Vertices;
			CLongArray l_Polygons;
			X3DObject l_XSIObject;

			_XSI_CALL(l_XSIParent.AddPolygonMesh( l_Vertices, l_Polygons , l_Name, l_XSIObject ), L"Failed to add a Mesh object");
			PolygonMesh	l_XSIMesh = l_XSIObject.GetActivePrimitive().GetGeometry(); 
			CMeshBuilder l_XSIMeshBuilder = l_XSIMesh.GetMeshBuilder();

			if (l_XSIMeshBuilder.IsValid() && l_pFTKMesh->XSIVertexList()
				&& (l_pFTKMesh->XSIVertexList()->GetCount() > 0) && l_pFTKMesh->XSIShape())
			{
				CSLXSIVertexList* l_pFTKVertexList = l_pFTKMesh->XSIVertexList();
				CSLXSIShape* l_pFTKShape = l_pFTKMesh->XSIShape();

				// get the position array from the FTK
				float* l_pPositionArray = l_pFTKShape->GetVertexPositionList()->GetAttributeArray()->ArrayPtr();

				// get the vertex index into the position array from the FTK
				int* l_pPositionIndices = l_pFTKVertexList->GetAttributeByName("position")->ArrayPtr();

				LONG l_nVertexCount = l_pFTKVertexList->GetCount();

				// add the vertex
				l_XSIMeshBuilder.AddVertices(l_nVertexCount, NULL);

				// set the vertex position
				double l_Position[3];
				int index;

				for (loop = 0; loop < l_nVertexCount; loop++)
				{
					index = l_pPositionIndices[loop]*3;
					l_Position[0] = l_pPositionArray[index++];
					l_Position[1] = l_pPositionArray[index++];
					l_Position[2] = l_pPositionArray[index++];

					l_XSIMeshBuilder.SetVertexPositions(1, l_Position, loop);
				}

				// we need to keep track of how many polys/triangles we think we've added
				// so that if there's a difference between the number we think we've added and 
				// the number that were actually added, we can notify the client and avoid problems
				// importing polynode properties like normals
				ULONG l_expectedPolyCount = 0;
				ULONG l_VtxIndexCount = 0;
				std::map<int, HelperPoly*> l_polyMap;

				// IF there exists an index list, and IF there are the right number of indices in the list
				// THEN we'll use the lovely index list. Otherwise, we won't.
				// this boolean represents whether we'll use index lists or not
				bool bPolyClusterSupport = true;
				for (loop = 0; bPolyClusterSupport && loop < l_pFTKMesh->GetXSIPolygonListCount(); loop++)
				{
					// grab poly list stuff
					CSLXSIPolygonList *l_pFTKPolygonList = l_pFTKMesh->XSIPolygonLists()[loop];
					ULONG l_polyCount = l_pFTKPolygonList->GetPolygonCount();

					// set up the index list
					CSLXSIIndexList *l_pFTKIndexList = l_pFTKPolygonList->GetIndexList();
					if (!l_pFTKIndexList) {
						// index list not found
						bPolyClusterSupport = false;
						break;
					}
					int l_indexListCount = l_pFTKIndexList->GetCount();
					bPolyClusterSupport = (l_indexListCount == l_polyCount);
				}


				// if we can order the polygons (ie. there is an index list), then construct an extensive 
				// array of all the polygons, and place in continuous, ordered list.
				// otherwise, if no index list, then we'll add them in the order we encounter them
				for (loop = 0; loop < l_pFTKMesh->GetXSIPolygonListCount(); loop++)
				{
					CSLXSIPolygonList *l_pFTKPolygonList = l_pFTKMesh->XSIPolygonLists()[loop];
					if (l_pFTKPolygonList->GetPolygonCount() > 0)
					{
						// set up polygon arrays
						ULONG l_polyCount = l_pFTKPolygonList->GetPolygonCount();
						l_expectedPolyCount += l_polyCount;
						if (!bPolyClusterSupport)
						{
							l_XSIMeshBuilder.AddPolygons(l_polyCount, (LONG*)l_pFTKPolygonList->GetPolygonNodeCountArray()->ArrayPtr(), (LONG*)l_pFTKPolygonList->GetVertexIndices()->ArrayPtr());
						}
						else
						{
							const LONG* l_pPolyVertCounts = (LONG*)l_pFTKPolygonList->GetPolygonNodeCountArray()->ArrayPtr();
							const LONG* l_pVtxIndices = (LONG*)l_pFTKPolygonList->GetVertexIndices()->ArrayPtr();

							// set up the index list
							CSLXSIIndexList *l_pFTKIndexList = NULL;
							l_pFTKIndexList = l_pFTKPolygonList->GetIndexList();
							int l_indexListCount = 0;
							CSLXSIIndexList::CSLIntArray *l_pFTKIndexArray = NULL;
							if (l_pFTKIndexList)
							{
								l_indexListCount = l_pFTKIndexList->GetCount();
								assert (l_indexListCount == l_polyCount);
								l_pFTKIndexArray = l_pFTKIndexList->GetIndexArray();
							}

							// add to the list
							int nOffset = 0;
							for (unsigned int loop2 = 0; loop2 < l_polyCount; loop2++)
							{
								HelperPoly *p = new HelperPoly;
								p->count = l_pPolyVertCounts[loop2];
								p->indices = new LONG[p->count];
								for (int i = 0; i < p->count; i++)
									p->indices[i] = l_pVtxIndices[nOffset+i];
								nOffset += p->count;

								// add to list
								l_polyMap[(*l_pFTKIndexArray)[loop2]] = p;

							}
							l_VtxIndexCount += nOffset;
						}
					}
				}

				if (bPolyClusterSupport)
				{
					// now that we've created our sorted pile, flatten into a whole array
					// and tack the unsorted at the end
					LONG* l_pPolyVertCounts = new LONG[l_expectedPolyCount]; 
					LONG* l_pVtxIndices = new LONG[l_VtxIndexCount]; 
					int pCount = 0;
					int vCount = 0;
					for (loop = 0; loop < (LONG)l_polyMap.size(); loop++)
					{
						HelperPoly *p = l_polyMap[loop];
						l_pPolyVertCounts[pCount++] = p->count;
						for (loop2 = 0; loop2 < p->count; loop2++ )
							l_pVtxIndices[vCount++] = p->indices[loop2];
					}

					if (l_expectedPolyCount > 0)
					{
						l_XSIMeshBuilder.AddPolygons(l_expectedPolyCount, l_pPolyVertCounts, l_pVtxIndices);
					}

					// clean up all the newly created polygon objects and arrays
					for (loop = 0; loop < (LONG)l_polyMap.size(); loop++)
					{
						HelperPoly *p = l_polyMap[loop];
						delete [] p->indices;
						delete p;
					}
					l_polyMap.clear();
					delete [] l_pPolyVertCounts;
					delete [] l_pVtxIndices;
				}

				// add the triangles

				// create a temporary polygon count array filled with 3 for the triangle addition.
				
				// count the total number of triangle
				l_lMaxPolygon = 0;
				for (loop = 0; loop < l_pFTKMesh->GetXSITriangleListCount(); loop++)
				{
					if (l_lMaxPolygon < l_pFTKMesh->XSITriangleLists()[loop]->GetTriangleCount())
					{
						l_lMaxPolygon = l_pFTKMesh->XSITriangleLists()[loop]->GetTriangleCount();
					}
				}

				if (l_lMaxPolygon > 0)
				{
					// allocate the temporary polygon count array
					LONG *l_pTempPolygonCount = new LONG[l_lMaxPolygon];

					// set all the value to 3
					for (loop = 0; loop < l_lMaxPolygon; loop++)
					{
						l_pTempPolygonCount[loop] = 3;
					}

					// add the triangle
					for (loop = 0; loop < l_pFTKMesh->GetXSITriangleListCount(); loop++)
					{
						CSLXSITriangleList *l_pXSITriangleList = l_pFTKMesh->XSITriangleLists()[loop];

						ULONG l_triCount = l_pXSITriangleList->GetTriangleCount();
						l_expectedPolyCount += l_triCount;

						if (l_triCount > 0)
						{
							l_XSIMeshBuilder.AddPolygons(l_triCount, l_pTempPolygonCount, (LONG*)l_pXSITriangleList->GetVertexIndices()->ArrayPtr());
						}
					}

					delete [] l_pTempPolygonCount;
				}
 
				// TODO: triangle strips import is not implemented yet.
	
				CMeshBuilder::CErrorDescriptor l_Error = l_XSIMeshBuilder.Build(FALSE);

				// check the actual number of polygons/triangles created and compare to the expected number
				// - if there's a discrepancy, it means XSI threw out some of the polygons for reasons such as a 
				//   degenerate triangle that only contains 2 unique vertices
				CPolygonFaceRefArray l_polygonArray = l_XSIMesh.GetPolygons();
				ULONG l_actualPolyCount = l_polygonArray.GetCount();
				if (l_expectedPolyCount != l_actualPolyCount) {
					// log a warning and set a value in the hierarchyelementinfo so that we 
					// know not to load polynode settings like normals
					l_pFTKMesh->AttachUserData("BADPOLYS", new CSIBCUserData("BADPOLYS", (void *) 0, NULL) );
					logmessage( L"Degenerate polygons in object: %s - cluster properties will not be imported.", l_XSIObject.GetName() );
				}

				if (l_Error.GetCode()==CStatus::Fail)
				{
					logmessage( L"Error generating the mesh: %s - %s", l_XSIObject.GetName(), l_Error.GetDescription() );
				}
				else
				{
					// set the material

					// create a temporary indices array of the size of the larger polygon list
					l_lMaxPolygon = 0;
					for (loop = 0; loop < l_pFTKMesh->GetXSIPolygonListCount(); loop++)
					{
						if (l_lMaxPolygon < l_pFTKMesh->XSIPolygonLists()[loop]->GetPolygonCount())
						{
							l_lMaxPolygon = l_pFTKMesh->XSIPolygonLists()[loop]->GetPolygonCount();
						}
					}
					for (loop = 0; loop < l_pFTKMesh->GetXSITriangleListCount(); loop++)
					{
						if (l_lMaxPolygon < l_pFTKMesh->XSITriangleLists()[loop]->GetTriangleCount())
						{
							l_lMaxPolygon = l_pFTKMesh->XSITriangleLists()[loop]->GetTriangleCount();
						}
					}

					if (l_lMaxPolygon > 0)
					{
						LONG l_lCurrentPolyIndice = 0;
						LONG* l_pTempIndices = new LONG[l_lMaxPolygon];

						// set the material for the polygon list
						for (loop = 0; loop < l_pFTKMesh->GetXSIPolygonListCount(); loop++)
						{
							CSLXSIPolygonList *l_pFTKPolygonList = l_pFTKMesh->XSIPolygonLists()[loop];

							Material l_XSIMaterial;
							CSLBaseMaterial* l_pFTKMaterial = l_pFTKPolygonList->GetMaterial();

							if (l_pFTKMaterial)
							{
								CSIBCUserData *l_pUserData = l_pFTKMaterial->FindUserData("CREF");
								if(l_pUserData)
								{
									CRef *l_pCRef = (CRef*)(l_pUserData->GetData());
									l_XSIMaterial = (Material) *l_pCRef;

									if (l_XSIMaterial.IsValid())
									{
										// if we can order via indexlists, then do so
										CSLXSIIndexList::CSLIntArray *l_pFTKIndexArray = NULL;
										if (bPolyClusterSupport)
										{
											CSLXSIIndexList *l_pFTKIndexList = l_pFTKPolygonList->GetIndexList();
											l_pFTKIndexArray = l_pFTKIndexList->GetIndexArray();
										}

										for (loop2 = 0; loop2 < l_pFTKPolygonList->GetPolygonCount(); loop2++)
										{
											assert ( loop2 < l_lMaxPolygon ); 
											if (bPolyClusterSupport)
												l_pTempIndices[loop2] = (*l_pFTKIndexArray)[loop2];
											else
												l_pTempIndices[loop2] = l_lCurrentPolyIndice++;
										}

										if (l_pFTKPolygonList->GetPolygonCount() > 0 && l_pFTKModel->GlobalMaterial())
										{
											// do not add a cluster/material if it's using the global material.
											if (l_pFTKMaterial != l_pFTKModel->GlobalMaterial()->GetMaterial())
											{
                                                XSI::CString l_csTemp;
												l_XSIMeshBuilder.SetPolygonsMaterial(l_XSIMaterial, l_pFTKPolygonList->GetPolygonCount(), l_pTempIndices, l_csTemp);
											}
										}
									}
								}
							}

							
						}

						// now set the material for the triangle list
						for (loop = 0; loop < l_pFTKMesh->GetXSITriangleListCount(); loop++)
						{
							CSLXSITriangleList *l_pXSITriangleList = l_pFTKMesh->XSITriangleLists()[loop];

							Material l_XSIMaterial;
							CSLBaseMaterial* l_pFTKMaterial = l_pXSITriangleList->GetMaterial();

							if (l_pFTKMaterial)
							{
								CSIBCUserData *l_pUserData = l_pFTKMaterial->FindUserData("CREF");
								if(l_pUserData)
								{
									CRef *l_pCRef = (CRef*)(l_pUserData->GetData());
									l_XSIMaterial = (Material) *l_pCRef;

									if (l_XSIMaterial.IsValid())
									{
										for (loop2 = 0; loop2 < l_pXSITriangleList->GetTriangleCount(); loop2++)
										{
											l_pTempIndices[loop2] = l_lCurrentPolyIndice++;
										}

										if (l_pXSITriangleList->GetTriangleCount() > 0 && l_pFTKModel->GlobalMaterial())
										{
											// do not add a cluster/material if it's using the global material.
											if (l_pFTKMaterial != l_pFTKModel->GlobalMaterial()->GetMaterial())
											{
                                                XSI::CString l_csTemp;
												l_XSIMeshBuilder.SetPolygonsMaterial(l_XSIMaterial, l_pXSITriangleList->GetTriangleCount(), l_pTempIndices, l_csTemp);
											}
										}
									}
								}
							}
						}

						// TODO: triangle strips import is not implemented yet.

						delete [] l_pTempIndices;
					}
				}
			}

			*io_pXSIModel = l_XSIObject;

			l_pFTKModel->AttachUserData("CREF", in_pContext->AddCRef(l_XSIObject));
		}
	}

	return status;
}

wchar_t *CMeshToXSI::GetClassID(){return L"CMeshToXSI";}

