//***************************************************************************************
//
// File supervisor: Softimage Rendering & Pipeline team
//
// (c) Copyright 2001-2005 Avid Technology, Inc. . All rights reserved.
//
//***************************************************************************************

/**************************************************************************************
THIS CODE IS PUBLISHED AS A SAMPLE ONLY AND IS PROVIDED "AS IS".
IN NO EVENT SHALL SOFTIMAGE, AVID TECHNOLOGY, INC. AND/OR THEIR RESPECTIVE
SUPPLIERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
CONNECTION WITH THE USE OR PERFORMANCE OF THIS CODE .

COPYRIGHT NOTICE. Copyright  1999-2005 Avid Technology Inc. . All rights reserved. 

SOFTIMAGE is a registered trademark of Avid Technology Inc. or its subsidiaries 
or divisions. Windows NT is a registered trademark of Microsoft Corp. All other
trademarks contained herein are the property of their respective owners. 
****************************************************************************************/
/*! \file cnv_mixer.cpp
/*!
	implementation file for mixer related classes
*/

#include "stdafx.h"
#include "cmdstubs.h"
#include "cnv_mixer.h"
#include <xsi_application.h>
#include <xsi_x3dobject.h>
#include <xsi_model.h>
#include <xsi_mixer.h>
#include <xsi_actionsource.h>
#include <xsi_staticsource.h>
#include <xsi_animationsourceitem.h>
#include <xsi_fcurve.h>
#include <xsi_track.h>
#include <xsi_clip.h>
#include <xsi_clipcontainer.h>
#include <xsi_shapeclip.h>
#include <xsi_timecontrol.h>
#include <xsi_source.h>
#include <xsi_shapekey.h>
#include <xsi_cluster.h>
#include <xsi_preferences.h>
#include <Model.h>
#include <Primitive.h>
#include <Mixer.h>
#include <Action.h>
#include <ActionFCurve.h>
#include <StaticValue.h>
#include <ActionClip.h>
#include <ActionClipContainer.h>
#include <TimeControl.h>
#include <Extrapolation.h>
#include <FTKUtil.h>
#include <SceneInfo.h>
#include <XSIShape.h>
#include <XSIMesh.h>
#include <XSIShapeAnimation.h>
#include <XSIShapeReference.h>
#include "plugin_stub.h"
#include "helper.h"
#include "StringHelpers.h"
#include "STLHelpers.h"
#include <set>
#include <algorithm>

struct AnimationInfo
{
	string name;
	std::vector<string> rootNames;
};

void GetAnimationInfo(AnimationInfo& info, const string& animationName)
{
	size_t position = 0;

	// Read the animation name.
	size_t nameStart = position;
	position = animationName.find('-', position);
	string name = animationName.substr(nameStart, position - nameStart);

	std::vector<string> rootNames;
	for (;;)
	{
		position = animationName.find_first_not_of("-", position);
		if (position == string::npos)
			break;

		size_t keyStart = position;
		position = animationName.find('-', position);
		string key = StringHelpers::Trim(StringHelpers::MakeLowerCase(animationName.substr(keyStart, position - keyStart)));
		if (key == "root")
		{
			position = animationName.find_first_not_of("-", position);

			size_t valueStart = position;
			position = animationName.find('-', position);
			string value = StringHelpers::Trim(StringHelpers::MakeLowerCase(animationName.substr(valueStart, position - keyStart)));
			if (!value.empty())
				rootNames.push_back(value);
		}
	}

	info.name = name;
	info.rootNames = rootNames;
}

/**************************************************************************************
CMixerFromXSI
**************************************************************************************/
CMixerFromXSI::CMixerFromXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CMixerFromXSI::~CMixerFromXSI() {};

void ExportTrack(Track &in_XSITrack, CSLMixer *in_pCSLMixer, CSLActionClipContainer *in_pCSLActionClipContainer)
{
	CString l_TrackType = in_XSITrack.GetType();
	siClassID l_TrackClassID = in_XSITrack.GetClassID();

	CRefArray l_XSIClips = in_XSITrack.GetClips();

	LONG loop, loop2;

	bool l_bMuteValue = in_XSITrack.GetParameterValue(L"mute",0);
	bool l_bSoloValue = in_XSITrack.GetParameterValue(L"solo",0);
	bool l_bRippleValue = in_XSITrack.GetParameterValue(L"ripple",0);

	CSLTrack *l_pCSLTrack = NULL;

	if (in_pCSLActionClipContainer)
	{
		l_pCSLTrack = in_pCSLActionClipContainer->AddTrack();
	}
	else
	{
		l_pCSLTrack = in_pCSLMixer->AddTrack();
	}

	if (l_pCSLTrack)
	{
		l_pCSLTrack->SetName((char*)in_XSITrack.GetName().GetAsciiString());
		l_pCSLTrack->SetMute(l_bMuteValue);
		l_pCSLTrack->SetSolo(l_bSoloValue);
		l_pCSLTrack->SetRipple(l_bRippleValue);


		for (loop = 0; loop < l_XSIClips.GetCount(); loop++)
		{
			CSLActionClip *l_pCSLActionClip = NULL;
			
			CRef l_XSIItem = l_XSIClips.GetItem(loop);

			//siShapeClipID, siClipContainerID, siClipID

			siClassID l_XSIClipClassID = l_XSIItem.GetClassID();

			if ((l_XSIClipClassID == siClipID) || (l_XSIClipClassID == siClipContainerID))
			{
				Clip l_XSIClip = (Clip)l_XSIItem;

				if (l_XSIClipClassID == siClipID)
				{
					l_pCSLActionClip = l_pCSLTrack->AddActionClip();
				}
				else
				{
					l_pCSLActionClip = l_pCSLTrack->AddActionClipContainer();
				}

				if (l_pCSLActionClip)
				{
					l_pCSLActionClip->SetName((char*)l_XSIClip.GetName().GetAsciiString());

					CSLTimeControl *l_pCSLTimeControl = l_pCSLActionClip->AddTimeControl();

					TimeControl l_XSITimeControl = l_XSIClip.GetTimeControl();
					float l_dClipIn = (float)l_XSITimeControl.GetClipIn();
					float l_dClipOut = (float)l_XSITimeControl.GetClipOut();
					float l_dScale = (float)l_XSITimeControl.GetScale();
					float l_dStartOffset = (float)l_XSITimeControl.GetStartOffset();

					/////////////////////////////////////////////////////////////////////////////
					// set the time control parameter
					l_pCSLTimeControl->SetIn(l_dClipIn);
					l_pCSLTimeControl->SetOut(l_dClipOut);
					l_pCSLTimeControl->SetScale(l_dScale);
					l_pCSLTimeControl->SetStartOffset(l_dStartOffset);

					/////////////////////////////////////////////////////////////////////////////
					// set the action parameter
					l_pCSLActionClip->SetDuration((l_dClipOut-l_dClipIn)/l_dScale);
					l_pCSLActionClip->SetStartTime(l_dStartOffset);
					l_pCSLActionClip->SetFillAction(l_XSIClip.GetParameterValue(L"fillaction",0));
					l_pCSLActionClip->SetFillPriority(l_XSIClip.GetParameterValue(L"fillpriority",0));
					l_pCSLActionClip->SetWeight(l_XSIClip.GetParameterValue(L"weight",0));

					/////////////////////////////////////////////////////////////////////////////
					// export the weight FCurve
					Parameter l_Parameter = l_XSIClip.GetParameter(L"weight");
					CString l_ParamName = l_Parameter.GetScriptName();

					FTKUParameterAnimationFromXSI(l_pCSLActionClip, CSLFCurve::SI_PARAMETER, l_Parameter, /*m_pAnimationPlotter*/ NULL, l_ParamName.GetWideString());					

					/////////////////////////////////////////////////////////////////////////////
					// export the fillpriority FCurve
					l_Parameter = l_XSIClip.GetParameter(L"fillpriority");
					l_ParamName = l_Parameter.GetScriptName();

					FTKUParameterAnimationFromXSI(l_pCSLActionClip, CSLFCurve::SI_PARAMETER, l_Parameter, /*m_pAnimationPlotter*/ NULL, l_ParamName.GetWideString());					

					/////////////////////////////////////////////////////////////////////////////
					// get the corresponding source name
					Source l_XSISource = l_XSIClip.GetSource();
					const char *l_pSourceName = NULL;
					CString l_Name;

					l_Name = l_XSISource.GetName();
					l_pSourceName = l_Name.GetAsciiString();

					/////////////////////////////////////////////////////////////////////////////
					// find the corresponding source action and set it as the reference of this clip.
					l_pCSLActionClip->SetReference(in_pCSLMixer->FindAction((char*)l_pSourceName));

					/////////////////////////////////////////////////////////////////////////////
					// set the extrapolation Before type if needed
					siTimeControlExtrapolationType	l_ExtrapolationBeforeType;
					double l_ExtrapolationBeforeValue;

					l_XSITimeControl.GetExtrapolationBeforeType(l_ExtrapolationBeforeType);
					l_XSITimeControl.GetExtrapolationBeforeValue(l_ExtrapolationBeforeValue);

					if (l_ExtrapolationBeforeType != XSI::siTimeControlExtrapolationNone)
					{
						CSLExtrapolation *l_pExtrapolationBefore = l_pCSLTimeControl->CreateExtrapolation(CSLTimeControl::SI_BEFORE, (CSLExtrapolation::EExtrapolationType)l_ExtrapolationBeforeType);
						l_pExtrapolationBefore->SetValue((float)l_ExtrapolationBeforeValue);
					}

					/////////////////////////////////////////////////////////////////////////////
					// set the extrapolation After type if needed
					siTimeControlExtrapolationType	l_ExtrapolationAfterType;
					double l_ExtrapolationAfterValue;

					l_XSITimeControl.GetExtrapolationAfterType(l_ExtrapolationAfterType);
					l_XSITimeControl.GetExtrapolationAfterValue(l_ExtrapolationAfterValue);

					if (l_ExtrapolationAfterType != XSI::siTimeControlExtrapolationNone)
					{
						CSLExtrapolation *l_pExtrapolationAfter = l_pCSLTimeControl->CreateExtrapolation(CSLTimeControl::SI_AFTER, (CSLExtrapolation::EExtrapolationType)l_ExtrapolationAfterType);
						l_pExtrapolationAfter->SetValue((float)l_ExtrapolationAfterValue);
					}

					/////////////////////////////////////////////////////////////////////////////
					// if it's a clip container, export it's tracks
					if (l_XSIClipClassID == siClipContainerID)
					{
						ClipContainer l_XSIClipContainer = (ClipContainer)l_XSIItem;

						CRefArray l_XSITracks = l_XSIClipContainer.GetTracks();

						for (loop2 = 0; loop2 < l_XSITracks.GetCount(); loop2++)
						{
							XSI::Track l_Track(l_XSITracks.GetItem(loop2));
							ExportTrack(l_Track, in_pCSLMixer, (CSLActionClipContainer*)l_pCSLActionClip);
						}
					}
				}
			}
		}
	}
}

typedef std::set<string, STLHelpers::less_stricmp<string> > RootSet;
typedef std::set<string, STLHelpers::less_stricmp<string> > AnimationExportSet;

void GenerateAnimationNodeExportSetRecurseChildOfRoot(X3DObject l_XSIModel, const RootSet& roots, AnimationExportSet& animationsToExport)
{
	string name = l_XSIModel.GetName().GetAsciiString();
	animationsToExport.insert(name);
	CRefArray l_Children = l_XSIModel.GetChildren();
	for (LONG j = 0; j < l_Children.GetCount(); j++)
	{
		SIObject l_Obj(l_Children[j]);
		X3DObject l_Child(l_Obj);
		GenerateAnimationNodeExportSetRecurseChildOfRoot(l_Child, roots, animationsToExport);
	}
}

void GenerateAnimationNodeExportSetRecurseNotChildOfRoot(X3DObject l_XSIModel, const RootSet& roots, AnimationExportSet& animationsToExport)
{
	string name = l_XSIModel.GetName().GetAsciiString();
	if (roots.find(name) != roots.end())
	{
		GenerateAnimationNodeExportSetRecurseChildOfRoot(l_XSIModel, roots, animationsToExport);
	}
	else
	{
		CRefArray l_Children = l_XSIModel.GetChildren();
		for (LONG j = 0; j < l_Children.GetCount(); j++)
		{
			SIObject l_Obj(l_Children[j]);
			X3DObject l_Child(l_Obj);
			GenerateAnimationNodeExportSetRecurseNotChildOfRoot(l_Child, roots, animationsToExport);
		}
	}
}

void GenerateAnimationNodeExportSet(X3DObject l_XSIModel, const RootSet& roots, AnimationExportSet& animationsToExport)
{
	if (roots.empty())
		GenerateAnimationNodeExportSetRecurseChildOfRoot(l_XSIModel, roots, animationsToExport);
	else
		GenerateAnimationNodeExportSetRecurseNotChildOfRoot(l_XSIModel, roots, animationsToExport);
}

CStatus CMixerFromXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;

	if((wcscmp(((X3DObject)*io_pXSIModel).GetType().GetWideString(), L"#model") == 0) && io_pFTKModel && *io_pFTKModel)
	{
		CSLModel *l_pFTKModel = (CSLModel*)*io_pFTKModel;

		float	framerate = 1.0f;

		CTime::Unit l_Format;
		if(in_pFTKParent->Scene()->SceneInfo()->GetTimingType() == CSLSceneInfo::SI_FRAMES)
		{
			l_Format = CTime::Frames;
		}
		else
		{
			CTime l_Time;
			l_Format = CTime::Seconds;
			framerate = (float)l_Time.GetFrameRate();
		}

		Model l_XSIModel = (Model)*io_pXSIModel;

		CRefArray l_XSISources = l_XSIModel.GetSources();

		CSLMixer *l_pFTKMixer = NULL;

		LONG loop, loop2;
		for (loop = 0; loop < l_XSISources.GetCount(); loop++)
		{
			if (l_XSISources.GetItem(loop).IsA(siActionSourceID))
			{
				ActionSource l_XSIActionSource = (ActionSource)l_XSISources.GetItem(loop);

				// CrySpecific: Do not export the action if the DoNotExport CPSet is present
				CRefArray l_properties;
				bool l_bDoNotExport = false;

				l_properties = l_XSIActionSource.GetProperties();

				// go thru all properties
				for(loop2 = 0; loop2 < l_properties.GetCount(); loop2++)
				{
					CustomProperty l_property = l_properties.GetItem(loop2);

					// look for the "DoNotExport" property
					CString name = l_property.GetName();
					const char *asciiname = name.GetAsciiString();

					if  (strstr(asciiname, "DoNotExport"))
					{
						l_bDoNotExport = true;
						break;
					}
				}

				if (l_bDoNotExport)
					continue;

				if (!l_pFTKMixer)
				{
					CSLModel *l_pFTKModel = (CSLModel*)*io_pFTKModel;
					l_pFTKMixer = l_pFTKModel->CreateMixer();
					l_pFTKMixer->SetName(l_pFTKModel->Name().GetText());
				}

				CRefArray l_XSISourceItems = l_XSIActionSource.GetItems();

				// Only export fcurve actions
				if (l_XSISourceItems.GetCount() > 0)
				{
					AnimationSourceItem l_XSICurrItem(l_XSISourceItems[0]);

					// skip this action if it's not FCurveAnimItem
					if (wcscmp(l_XSICurrItem.GetType().GetWideString(), L"FCurveAnimItem") != 0)
						continue;
				}

				CSLAction* l_pFTKAction = l_pFTKMixer->AddAction();

				// CrySpecific : concatenation of the CryExportNode name
				CString l_Name = l_XSIModel.GetName();
				char l_strNewNodename[MAX_PATH];
				char *l_strNodename = (char*)(l_Name.GetAsciiString());
				char *l_strCEN = strstr(l_strNodename, "CryExportNode");
				if (l_strCEN)
				{
					strcpy(l_strNewNodename, l_strCEN);
					strcat(l_strNewNodename, "_");
					char l_strPost[MAX_PATH];
					memset(l_strPost, 0, MAX_PATH);
					memcpy(l_strPost, l_strNodename, l_strCEN-l_strNodename-1);
					strcat(l_strNewNodename, l_strPost);
				}
				l_Name.PutAsciiString(l_strNewNodename);

				AnimationInfo animationInfo;
				GetAnimationInfo(animationInfo, l_XSIActionSource.GetName().GetAsciiString());

				// Create a set of bones that should be exported. They should be exported either if:
				// - No roots were specified for this animation;
				// - The bone is a child of one of the specified roots.
				RootSet roots(animationInfo.rootNames.begin(), animationInfo.rootNames.end());
				AnimationExportSet animationsToExport;
				GenerateAnimationNodeExportSet(l_XSIModel, roots, animationsToExport);

				l_pFTKAction->SetName((char*)(animationInfo.name + "-" + l_Name.GetAsciiString()).c_str());

				CValue l_FrameRate;
				Application app;
				app.GetPreferences().GetPreferenceValue(L"output_format.frame_rate", l_FrameRate);

				float l_FrameStart = l_XSIActionSource.GetParameter(L"FrameStart").GetValue();
				float l_FrameEnd = l_XSIActionSource.GetParameter(L"FrameEnd").GetValue();

				float l_FR = l_FrameRate;
				l_pFTKAction->SetStartTime(l_FrameStart/l_FR);
				l_pFTKAction->SetEndTime(l_FrameEnd/l_FR);

				CSLAction::EActionType l_ActionType = CSLAction::SI_AT_NONE;
				bool l_bMixedActionType = false;

				for(loop2 = 0; loop2 < l_XSISourceItems.GetCount(); loop2++)
				{
					AnimationSourceItem l_XSICurrItem(l_XSISourceItems[loop2]);

					CString l_XSISourceItemType = l_XSICurrItem.GetType();
					const char* l_pXSISourceItemType = l_XSISourceItemType.GetAsciiString();

					if (l_XSISourceItemType == L"ShapeKeyAnimItem")
					{
						//////////////////////////////////////////////////////////////////////
						// Set the action type
						if (!l_bMixedActionType)
						{
							if (l_ActionType == CSLAction::SI_AT_NONE)
							{
								l_ActionType = CSLAction::SI_AT_CLUSTERKEY;
							}
							else if (l_ActionType != CSLAction::SI_AT_CLUSTERKEY)
							{
								l_bMixedActionType = true;
								l_ActionType = CSLAction::SI_AT_NONE;
							}
						}

						CRef l_XSIAnimationSource = l_XSICurrItem.GetSource();

						if (l_XSIAnimationSource.IsA(siShapeKeyID))
						{
							ShapeKey l_XSIShapeKey = l_XSIAnimationSource;
							CString l_XSIShapeKeyName = l_XSIShapeKey.GetFullName();

							CSLXSIShapeReference *l_pCSLXSIShapeReference = l_pFTKAction->AddXSIShapeReference();

							if (l_pCSLXSIShapeReference)
							{
								l_pCSLXSIShapeReference->SetReferenceXSIPath((char*)l_XSIShapeKeyName.GetAsciiString());
							}
						}
					}
					else if (l_XSISourceItemType == L"FCurveAnimItem")
					{
						//////////////////////////////////////////////////////////////////////
						// Set the action type
						if (!l_bMixedActionType)
						{
							if (l_ActionType == CSLAction::SI_AT_NONE)
							{
								l_ActionType = CSLAction::SI_AT_FCURVE;
							}
							else if (l_ActionType != CSLAction::SI_AT_FCURVE)
							{
								l_bMixedActionType = true;
								l_ActionType = CSLAction::SI_AT_NONE;
							}
						}

						CRef l_CRef = l_XSICurrItem.GetSource();

						CString l_Target = l_XSICurrItem.GetTarget();
						const char* l_AsciiString = l_Target.GetAsciiString();
						const char* l_Dot = strstr( l_AsciiString, "." );
						char l_NewString[MAX_PATH];
						memset( l_NewString, 0, MAX_PATH);
						memcpy( l_NewString, l_AsciiString, l_Dot-l_AsciiString );
						if (animationsToExport.find(l_NewString) == animationsToExport.end())
							continue;

						// check if this parameter is an FCurve
						if (l_CRef.IsA(siFCurveID))
						{
							FCurve l_XSIFCurve = l_CRef;
							CSLActionFCurve* l_pFTKFCurve;

					 		LONG l_lNumKeys = l_XSIFCurve.GetNumKeys();

							// create and fill the FTK FCurve depending on the type of the XSI FCurve
							switch (l_XSIFCurve.GetInterpolation())
							{
								case siConstantInterpolation:
								{
									l_pFTKFCurve = l_pFTKAction->AddAnimation(CSLActionFCurve::SI_CONSTANT);
									l_pFTKFCurve->SetParameterXSIPath((char*)l_XSICurrItem.GetTarget().GetAsciiString());

									if (l_pFTKFCurve)
									{
										CSLActionFCurve::CSLConstantKeyArray* l_pConstantKeyArray = l_pFTKFCurve->GetConstantKeyList();
										(*l_pConstantKeyArray).Resize(l_lNumKeys);

										for (LONG loop3 = 0; loop3 < l_lNumKeys; loop3++)
										{
											FCurveKey l_XSIFCurveKey = l_XSIFCurve.GetKeyAtIndex(loop3);

											(*l_pConstantKeyArray)[loop3].m_fTime = (float)l_XSIFCurveKey.GetTime().GetTime(l_Format);
											(*l_pConstantKeyArray)[loop3].m_fValue = (float)l_XSIFCurveKey.GetValue();
										}
									}
								} break;

								case siCubicInterpolation:
								{
									l_pFTKFCurve = l_pFTKAction->AddAnimation(CSLActionFCurve::SI_CUBIC);
									l_pFTKFCurve->SetParameterXSIPath((char*)l_XSICurrItem.GetTarget().GetAsciiString());

									if (l_pFTKFCurve)
									{
										CSLActionFCurve::CSLCubicKeyArray* l_pCubicKeyArray = l_pFTKFCurve->GetCubicKeyList();
										(*l_pCubicKeyArray).Resize(l_lNumKeys);

										for (LONG loop3 = 0; loop3 < l_lNumKeys; loop3++)
										{
											FCurveKey l_XSIFCurveKey = l_XSIFCurve.GetKeyAtIndex(loop3);

											(*l_pCubicKeyArray)[loop3].m_fTime = (float)l_XSIFCurveKey.GetTime().GetTime(l_Format);
											(*l_pCubicKeyArray)[loop3].m_fValue = (float)l_XSIFCurveKey.GetValue();
											(*l_pCubicKeyArray)[loop3].m_fLeftTanX = (float)l_XSIFCurveKey.GetLeftTanX() / framerate;
											(*l_pCubicKeyArray)[loop3].m_fLeftTanY = (float)l_XSIFCurveKey.GetLeftTanY();
											(*l_pCubicKeyArray)[loop3].m_fRightTanX = (float)l_XSIFCurveKey.GetRightTanX() / framerate;
											(*l_pCubicKeyArray)[loop3].m_fRightTanY = (float)l_XSIFCurveKey.GetRightTanY();
										}
									}
								} break;

								case siLinearInterpolation:
								default:
								{
									l_pFTKFCurve = l_pFTKAction->AddAnimation(CSLActionFCurve::SI_LINEAR);
									l_pFTKFCurve->SetParameterXSIPath((char*)l_XSICurrItem.GetTarget().GetAsciiString());

									if (l_pFTKFCurve)
									{
										CSLActionFCurve::CSLLinearKeyArray* l_pLinearKeyArray = l_pFTKFCurve->GetLinearKeyList();
										(*l_pLinearKeyArray).Resize(l_lNumKeys);

										for (LONG loop3 = 0; loop3 < l_lNumKeys; loop3++)
										{
											FCurveKey l_XSIFCurveKey = l_XSIFCurve.GetKeyAtIndex(loop3);

											(*l_pLinearKeyArray)[loop3].m_fTime = (float)l_XSIFCurveKey.GetTime().GetTime(l_Format);
											(*l_pLinearKeyArray)[loop3].m_fValue = (float)l_XSIFCurveKey.GetValue();
										}
									}
								} break;
							}

							// CrySpecific : concatenation of the CryExportNode name
							CString l_Name = l_XSIModel.GetName();
							char *l_strNodename = (char*)(l_Name.GetAsciiString());
							char *l_strCEN = strstr(l_strNodename, "CryExportNode");
							if (l_strCEN)
							{
								char l_strPost[MAX_PATH];
								memset(l_strPost, 0, MAX_PATH);
								memcpy(l_strPost, l_strNodename, l_strCEN-l_strNodename-1);
								strcat(l_strPost, &l_strCEN[13]);

								if (l_strPost)
								{
									strcat(l_NewString, "%");
									strcat(l_NewString, l_strPost);
									strcat(l_NewString, "%");

									CString l_Properties;

									CStringArray l_NodeNames;
									GetNodeNamesUnderNode( l_XSIModel, l_NodeNames );
									CRefArray l_Nodes;
									GetNodesUnderNode( l_XSIModel, l_Nodes );

									CStringArray l_SplitName = l_Target.Split(L".");
									for (int i = 0; i < l_Nodes.GetCount(); i++)
									{
										X3DObject l_X3DObject(l_Nodes.GetItem(i));
										if (l_X3DObject.GetName() == l_SplitName[0])
										{
											CustomProperty l_ObjectProperties(l_X3DObject.GetProperties().GetItem(L"ObjectProperties"));
											if (l_ObjectProperties.IsValid())
											{
												CString l_Props = l_ObjectProperties.GetParameterValue(L"Props");
												std::vector<std::wstring> properties;
												ConvertObjectProperties(l_Props, properties, l_NodeNames, l_SplitName[0]);
												if (!properties.empty())
												{
													l_Properties += L"--PRprops";
													for (int i = 0, count = int(properties.size()); i < count; ++i)
													{
														std::wstring safeProperty = properties[i];
														std::replace(safeProperty.begin(), safeProperty.end(), ' ', '*');
														l_Properties += (L"_" + safeProperty).c_str();
													}
													l_Properties += L"__";
												}

												strcat(l_NewString, l_Properties.GetAsciiString());
											}
										}
									}
								}
							}
							strcat( l_NewString, l_Dot );
//							_strlwr(l_NewString);
							l_pFTKFCurve->SetParameterXSIPath( l_NewString );
						}
					}
					else if (l_XSISourceItemType == L"StaticValueAnimItem")
					{
						//////////////////////////////////////////////////////////////////////
						// Set the action type
						if (!l_bMixedActionType)
						{
							if (l_ActionType == CSLAction::SI_AT_NONE)
							{
								l_ActionType = CSLAction::SI_AT_STATICVALUE;
							}
							else if (l_ActionType != CSLAction::SI_AT_STATICVALUE)
							{
								l_bMixedActionType = true;
								l_ActionType = CSLAction::SI_AT_NONE;
							}
						}

						CSLStaticValue* l_pFTKStaticValue = l_pFTKAction->AddStaticValue();

						StaticSource l_XSIVal = l_XSICurrItem.GetSource();
						
						l_pFTKStaticValue->SetParameterName((char*)l_XSICurrItem.GetTarget().GetAsciiString());
						l_pFTKStaticValue->SetValue(l_XSIVal.GetValue());
					}
					else if (l_XSISourceItemType == L"ExpressionAnimItem")
					{
						//////////////////////////////////////////////////////////////////////
						// Set the action type
						if (!l_bMixedActionType)
						{
							if (l_ActionType == CSLAction::SI_AT_NONE)
							{
								l_ActionType = CSLAction::SI_AT_EXPRESSION;
							}
							else if (l_ActionType != CSLAction::SI_AT_EXPRESSION)
							{
								l_bMixedActionType = true;
								l_ActionType = CSLAction::SI_AT_NONE;
							}
						}
					}
					else if (l_XSISourceItemType == L"ConstrainAnimItem")
					{
						//////////////////////////////////////////////////////////////////////
						// Set the action type
						if (!l_bMixedActionType)
						{
							if (l_ActionType == CSLAction::SI_AT_NONE)
							{
								l_ActionType = CSLAction::SI_AT_CONSTRAINT;
							}
							else if (l_ActionType != CSLAction::SI_AT_CONSTRAINT)
							{
								l_bMixedActionType = true;
								l_ActionType = CSLAction::SI_AT_NONE;
							}
						}
					}
					else if (l_XSISourceItemType == L"ShapeCompoundAnimItem")
					{
						//////////////////////////////////////////////////////////////////////
						// Set the action type
						if (!l_bMixedActionType)
						{
							if (l_ActionType == CSLAction::SI_AT_NONE)
							{
								l_ActionType = CSLAction::SI_AT_SHAPECOMPOUND;
							}
							else if (l_ActionType != CSLAction::SI_AT_SHAPECOMPOUND)
							{
								l_bMixedActionType = true;
								l_ActionType = CSLAction::SI_AT_NONE;
							}
						}
					}
					else if (l_XSISourceItemType == L"AnimCompoundAnimItem")
					{
						//////////////////////////////////////////////////////////////////////
						// Set the action type
						if (!l_bMixedActionType)
						{
							if (l_ActionType == CSLAction::SI_AT_NONE)
							{
								l_ActionType = CSLAction::SI_AT_COMPOUND;
							}
							else if (l_ActionType != CSLAction::SI_AT_COMPOUND)
							{
								l_bMixedActionType = true;
								l_ActionType = CSLAction::SI_AT_NONE;
							}
						}
					}
				}

				l_pFTKAction->SetActionType(l_ActionType);
			}
		}

		// now export the Tracks
		CRefArray l_XSITracks = l_XSIModel.GetMixer().GetTracks();

		for (loop = 0; loop < l_XSITracks.GetCount(); loop++)
		{
			XSI::Track l_track(l_XSITracks.GetItem(loop));
			if (!l_pFTKMixer)
			{
				CSLModel *l_pFTKModel = (CSLModel*)*io_pFTKModel;
				l_pFTKMixer = l_pFTKModel->CreateMixer();
				l_pFTKMixer->SetName(l_pFTKModel->Name().GetText());
			}

			ExportTrack(l_track, l_pFTKMixer, NULL);
		}
	}	

	return status;
}

wchar_t *CMixerFromXSI::GetClassID(){return L"CMixerFromXSI";}


/**************************************************************************************
CMixerToXSI
**************************************************************************************/
CMixerToXSI::CMixerToXSI(short in_CryFiletype) : CHierarchyTraverserCallback(in_CryFiletype) {};

CMixerToXSI::~CMixerToXSI() {};

CStatus CMixerToXSI::Execute(CdotXSIConverter *in_pContext, CRef in_XSIParent, CSLTemplate *in_pFTKParent, CHierarchyElementInfo* in_pInfo, CRef *io_pXSIModel, CSLTemplate **io_pFTKModel)
{
	CStatus status = CStatus::OK;
	LONG loop, loop2;
	Application app;
	
	CSLModel *l_pFTKModel = (CSLModel *) *io_pFTKModel;
	if(!l_pFTKModel->Primitive() && (l_pFTKModel != in_pContext->ftkscene()->Root()) && l_pFTKModel->Mixer())
	{
		CRef out_CRef;
		CustomProperty l_XSITempMixerToXSICPSet;
		Parameter l_XSITempParameter;
		FCurve l_XSIFCurve;

		CString l_TempMixerToXSICPSetName(L"TempMixerToXSICPSet");

		out_CRef = in_pContext->application().GetActiveSceneRoot().GetProperties().GetItem(l_TempMixerToXSICPSetName);

		if (out_CRef.IsValid())
		{
			l_XSITempMixerToXSICPSet = out_CRef;
			l_XSITempParameter = l_XSITempMixerToXSICPSet.GetParameter(L"Value");
			l_XSIFCurve = (FCurve)l_XSITempParameter.GetSource();
		}
		else
		{
			in_pContext->application().GetActiveSceneRoot().AddCustomProperty(l_TempMixerToXSICPSetName, false, l_XSITempMixerToXSICPSet);
			l_XSITempMixerToXSICPSet.AddParameter(L"Value", CValue::siFloat, siAnimatable, L"Value", L"Value", CValue(0.0), l_XSITempParameter);
			l_XSITempParameter.AddFCurve(siDefaultFCurve, l_XSIFCurve);
		}
	
		CSLMixer *l_pFTKMixer = l_pFTKModel->Mixer();

		Model l_XSIModel = (Model)(*io_pXSIModel);
		Mixer l_XSIMixer = l_XSIModel.GetMixer();
		if (!l_XSIMixer.IsValid())
			l_XSIMixer = l_XSIModel.AddMixer();

		/////////////////////////////////////////////////////////////////////////////////
		// Import the action source
		//
		for (loop = 0; loop < l_pFTKMixer->GetActionCount(); loop++)
		{
			CSLAction *l_pFTKAction = l_pFTKMixer->GetActionList()[loop];

			CSLAction::EActionType l_ActionType = l_pFTKAction->GetActionType();

			// do not import the action source if it's a shape key, we'll do it later.
		if ( (l_ActionType == CSLAction::SI_AT_CLUSTERKEY)
			|| (l_ActionType == CSLAction::SI_AT_SHAPECOMPOUND) )
				continue;

			CString l_ActionName;
			l_ActionName.PutAsciiString(l_pFTKAction->Name().GetText());

			ActionSource l_XSIActionSource = l_XSIModel.AddActionSource(l_ActionName);

			if (l_XSIActionSource.GetParameter(L"FrameStart").IsValid())
			{
				l_XSIActionSource.GetParameter(L"FrameStart").PutValue(l_pFTKAction->GetStartTime());
			}

			if (l_XSIActionSource.GetParameter(L"FrameEnd").IsValid())
			{
				l_XSIActionSource.GetParameter(L"FrameEnd").PutValue(l_pFTKAction->GetEndTime());
			}

			for (loop2 = 0; loop2 < l_pFTKAction->GetAnimationCount(); loop2++)
			{
				l_XSIFCurve.RemoveKeys();

				CSLActionFCurve* l_pFTKFCurve = l_pFTKAction->GetAnimationList()[loop2];

				CString l_ParameterName;
				l_ParameterName.PutAsciiString(l_pFTKFCurve->GetParameterXSIPath());

				FTKUFCurveToXSI(l_pFTKFCurve, l_XSIFCurve);

				l_XSIActionSource.AddSourceItem(l_ParameterName, l_XSIFCurve, true);
			}

			for (loop2 = 0; loop2 < l_pFTKAction->GetStaticValueCount(); loop2++)
			{
				CSLStaticValue* l_pFTKStaticValue = l_pFTKAction->GetStaticValueList()[loop2];

				CString l_ParameterName;
				l_ParameterName.PutAsciiString(l_pFTKStaticValue->GetParameterName());

				l_XSIActionSource.AddSourceItem(l_ParameterName, l_pFTKStaticValue->GetValue(), true);
			}
		}

		/////////////////////////////////////////////////////////////////////////////////
		// Import the track and clips
		//
		ImportTracks(in_pContext, l_pFTKMixer->GetTrackList(), l_pFTKMixer->GetTrackCount(), l_XSIModel, CValue());
	}
	
	return status;
}

void CMixerToXSI::ImportTracks(CdotXSIConverter *in_pContext, CSLTrack **in_pTrackList, int in_TrackCount, Model in_XSIModel, CValue in_Compound)
{
	CValue retval;
	int loop, loop2, loop3, loop4;
	Application app;
	Mixer l_XSIMixer = in_XSIModel.GetMixer();
	bool l_bIsValid;

	for (loop = in_TrackCount-1; loop >= 0; loop--)
	{
		int l_nTrackType = 0;
		CSLTrack *l_pTrack = in_pTrackList[loop];
		CSLAction::EActionType l_ActionType = CSLAction::SI_AT_NONE;

		if (l_pTrack->GetActionClipCount() > 0)
		{
			CSLTemplate::ETemplateType l_ActionClipType = l_pTrack->GetActionClipList()[0]->Type();

			if (l_pTrack->GetActionClipList()[0]->GetReference())
			{
				l_ActionType = l_pTrack->GetActionClipList()[0]->GetReference()->GetActionType();
			}
		}

		if ( (l_ActionType == CSLAction::SI_AT_CLUSTERKEY)
			|| (l_ActionType == CSLAction::SI_AT_SHAPECOMPOUND) )
		{
			l_nTrackType = 1;
		}

		CValueArray args(5);
		CValue retval;
		LONG b(0);

		CString l_Name;
		l_Name.PutAsciiString(l_pTrack->GetName());

		args[b++]= in_XSIModel;
		args[b++]= in_Compound;
		args[b++]= l_nTrackType;
		args[b++]= l_Name;
		args[b++]= 0;

		app.ExecuteCommand( L"AddTrack", args, retval );

		Track l_NewTrack(retval);
		l_bIsValid = l_NewTrack.IsValid();

		// set the track parameter
		l_NewTrack.PutParameterValue(L"mute", (bool)l_pTrack->GetMute());
		l_NewTrack.PutParameterValue(L"solo", (bool)l_pTrack->GetSolo());
		l_NewTrack.PutParameterValue(L"ripple", (bool)l_pTrack->GetRipple());

		l_pTrack->AttachUserData("CREF", in_pContext->AddCRef(l_NewTrack));

		int l_ActionClipCount = l_pTrack->GetActionClipCount();
		CSLActionClip **l_ActionClipList = l_pTrack->GetActionClipList();

		for (loop2 = 0; loop2 < l_ActionClipCount; loop2++)
		{
			CSLTemplate::ETemplateType l_ActionClipType = l_ActionClipList[loop2]->Type();

			// ActionClipContainer import
			if (l_ActionClipType == CSLTemplate::XSI_ACTIONCLIPCONTAINER)
			{
				CSLActionClipContainer* l_pActionClipContainer = (CSLActionClipContainer*)l_ActionClipList[loop2];

				CSLTrack** l_pTrackList = l_pActionClipContainer->GetTrackList();
				SI_Int l_nTrackCount = l_pActionClipContainer->GetTrackCount();

				ImportTracks(in_pContext, l_pTrackList, l_nTrackCount, in_XSIModel, CValue());

				CString l_ClipNameList;

				for (loop3 = 0; loop3 < l_nTrackCount; loop3++)
				{
					int l_TrackActionClipCount = l_pTrackList[loop3]->GetActionClipCount();
					CSLActionClip **l_TrackActionClipList = l_pTrackList[loop3]->GetActionClipList();

					for (loop4 = 0; loop4 < l_TrackActionClipCount; loop4++)
					{
						if (!l_ClipNameList.IsEmpty())
						{
							l_ClipNameList += L", ";
						}

						CRef* l_pCRefTrackActionClip;
						CSIBCUserData *l_pTrackActionClipUserData = l_TrackActionClipList[loop4]->FindUserData("CREF");
						if (l_pTrackActionClipUserData)
						{
							l_pCRefTrackActionClip = (CRef*)(l_pTrackActionClipUserData->GetData());

							Clip l_TrackActionClip(*l_pCRefTrackActionClip);

							l_ClipNameList += l_TrackActionClip.GetFullName();
						}
					}
				}

				l_Name.PutAsciiString(l_ActionClipList[loop2]->GetName());

				CValueArray args2(6);
				b = 0;
				args2[b++]= in_XSIModel; // Model
				args2[b++] = l_ClipNameList;// b++; // InputObj
				b++; // args2[b++] = in_Compound; // Compound
				args2[b++] = l_NewTrack; // Track
				args2[b++] = l_ActionClipList[loop2]->GetStartTime(); // Time
				args2[b++] = l_Name;

				app.ExecuteCommand( L"CreateCompound", args2, retval );

				ClipContainer l_ClipContainer(retval);

				// delete the original tracks
				for (loop3 = 0; loop3 < l_nTrackCount; loop3++)
				{
					CRef* l_pCRefTrack;

					CSIBCUserData *l_pTrackUserData = l_pTrackList[loop3]->FindUserData("CREF");
					if (l_pTrackUserData)
					{
						l_pCRefTrack = (CRef*)(l_pTrackUserData->GetData());

						Track l_pTrackUserData(*l_pCRefTrack);

						COMMANDS::DeleteObj(l_pTrackUserData.GetFullName());
					}
				}

				// re-set the new track parameters and name
				CRefArray l_XSITracks = l_ClipContainer.GetTracks();

				if (l_XSITracks.GetCount() == l_nTrackCount)
				{
					for (loop3 = 0; loop3 < l_XSITracks.GetCount(); loop3++)
					{
						Track l_Track(l_XSITracks.GetItem(loop3));
						char *l_pTrackName = l_pTrackList[loop3]->GetName();
						CString l_TrackName;
						l_TrackName.PutAsciiString(l_pTrackName);
						l_Track.PutName(l_TrackName);
						// set the track parameter
						l_Track.PutParameterValue(L"mute", (bool)l_pTrackList[loop3]->GetMute());
						l_Track.PutParameterValue(L"solo", (bool)l_pTrackList[loop3]->GetSolo());
						l_Track.PutParameterValue(L"ripple", (bool)l_pTrackList[loop3]->GetRipple());
					}
				}
			}
			// ActionClip import
			else
			{
				CSLAction *l_pSourceAction = l_ActionClipList[loop2]->GetReference();

				CString l_SourceName;
				const char *l_pName = l_SourceName.GetAsciiString();

				if (l_pSourceAction)
				{
					l_SourceName = l_XSIMixer.GetFullName();
					l_SourceName += L".";

					CString l_Temp;
					l_Temp.PutAsciiString(l_pSourceAction->Name().GetText());

					l_SourceName += l_Temp;

					l_pName = l_SourceName.GetAsciiString();
					l_Name.PutAsciiString(l_ActionClipList[loop2]->Name().GetText());
				}

				// if we're dealing with shape clip
				if (l_nTrackType)
				{
					CSLXSIShape *l_pXSIShape = NULL;

					CSLXSIShapeReference *l_pXSIShapeReference = l_pSourceAction->GetXSIShapeReference();

					if (l_pXSIShapeReference)
					{
						l_pXSIShape = l_pXSIShapeReference->GetReference();
					}

					if (l_pXSIShape)
					{
						CString l_ShapeName = in_XSIModel.GetFullName();
						
						CRef* l_pCRefShapeKey;

						CSIBCUserData *l_pShapeUserData = l_pXSIShape->FindUserData("CREF");
						if (l_pShapeUserData)
						{
							l_pCRefShapeKey = (CRef*)(l_pShapeUserData->GetData());
						}

						ShapeKey l_XSIShapeKey(*l_pCRefShapeKey);
						l_bIsValid = l_XSIShapeKey.IsValid();

						CValueArray args2(6);
						b = 0;
						args2[b++]= in_XSIModel;
						args2[b++]= l_XSIShapeKey;
						b++; // args2[b++]= in_Compound; // Compound
						args2[b++]= l_NewTrack;
						args2[b++]= l_ActionClipList[loop2]->GetStartTime();
						args2[b++]= l_Name;

						app.ExecuteCommand( L"AddClip", args2, retval );
					}
				}
				else
				{
					CValueArray args2(6);
					b = 0;
					args2[b++]= in_XSIModel;
					args2[b++]= l_SourceName;
					b++; // args2[b++]= in_Compound; // Compound
					args2[b++]= l_NewTrack;
					args2[b++]= l_ActionClipList[loop2]->GetStartTime();
					args2[b++]= l_Name;

					app.ExecuteCommand( L"AddClip", args2, retval );
				}
			}

			Clip l_XSIClip(retval);

			l_bIsValid = l_XSIClip.IsValid();

			l_ActionClipList[loop2]->AttachUserData("CREF", in_pContext->AddCRef(l_XSIClip));

			// set the clip property
			l_XSIClip.PutParameterValue(L"fillaction", (bool)l_ActionClipList[loop2]->GetFillAction());
			l_XSIClip.PutParameterValue(L"fillpriority", (LONG)l_ActionClipList[loop2]->GetFillPriority());
			l_XSIClip.PutParameterValue(L"weight", l_ActionClipList[loop2]->GetWeight());

			/////////////////////////////////////////////////////////////////////////////
			// import the weight FCurves
			CSLFCurve *l_pWeightFCurve = l_ActionClipList[loop2]->GetParameterFCurve("weight");
			if (l_pWeightFCurve)
			{
				CParameterRefArray l_XSIClipParameters = l_XSIClip.GetParameters();
				FTKUParameterAnimationToXSI(l_pWeightFCurve, l_XSIClipParameters, L"weight");
			}

			/////////////////////////////////////////////////////////////////////////////
			// import the fillpriority FCurves
			CSLFCurve *l_pFillPriorityFCurve = l_ActionClipList[loop2]->GetParameterFCurve("fillpriority");
			if (l_pFillPriorityFCurve)
			{
				CParameterRefArray l_XSIClipParameters = l_XSIClip.GetParameters();
				FTKUParameterAnimationToXSI(l_pFillPriorityFCurve, l_XSIClipParameters, L"fillpriority");
			}

			TimeControl l_XSITimeControl = l_XSIClip.GetTimeControl();

			// we will only consider the first one.
			if (l_ActionClipList[loop2]->GetTimeControlCount() > 0)
			{
				CSLTimeControl *l_pCSLTimeControl = l_ActionClipList[loop2]->GetTimeControlList()[0];

				l_XSITimeControl.PutParameterValue(L"clipin", l_pCSLTimeControl->GetIn());
				l_XSITimeControl.PutParameterValue(L"clipout", l_pCSLTimeControl->GetOut());
				l_XSITimeControl.PutParameterValue(L"startoffset", l_pCSLTimeControl->GetStartOffset());
				l_XSITimeControl.PutParameterValue(L"scale", l_pCSLTimeControl->GetScale());

				///////////////////////////////////////////////////////////////////////////////////////
				// Extrapolation Before Values
				//
				CSLExtrapolation *l_pExtrapolationBefore = l_pCSLTimeControl->GetExtrapolation(CSLTimeControl::SI_BEFORE);

				if (l_pExtrapolationBefore)
				{
					switch (l_pExtrapolationBefore->GetExtrapolationType())
					{
						case CSLExtrapolation::SI_NO_CONTRIBUTION:
						break;
						case CSLExtrapolation::SI_HOLD:
							l_XSITimeControl.PutParameterValue(L"extrapbef_type", (short)CSLExtrapolation::SI_HOLD);
							l_XSITimeControl.PutParameterValue(L"extrapbef_timehold", l_pExtrapolationBefore->GetValue());
						break;
						case CSLExtrapolation::SI_CYCLE:
							l_XSITimeControl.PutParameterValue(L"extrapbef_type", (short)CSLExtrapolation::SI_CYCLE);
							l_XSITimeControl.PutParameterValue(L"extrapbef_nbcycles", l_pExtrapolationBefore->GetValue());
						break;
						case CSLExtrapolation::SI_BOUNCE:
							l_XSITimeControl.PutParameterValue(L"extrapbef_type", (short)CSLExtrapolation::SI_BOUNCE);
							l_XSITimeControl.PutParameterValue(L"extrapbef_nbbounces", l_pExtrapolationBefore->GetValue());
						break;
					};
				}

				///////////////////////////////////////////////////////////////////////////////////////
				// Extrapolation After Values
				//
				CSLExtrapolation *l_pExtrapolationAfter = l_pCSLTimeControl->GetExtrapolation(CSLTimeControl::SI_AFTER);

				if (l_pExtrapolationAfter)
				{
					switch (l_pExtrapolationAfter->GetExtrapolationType())
					{
						case CSLExtrapolation::SI_NO_CONTRIBUTION:
						break;
						case CSLExtrapolation::SI_HOLD:
							l_XSITimeControl.PutParameterValue(L"extrapaft_type", (short)CSLExtrapolation::SI_HOLD);
							l_XSITimeControl.PutParameterValue(L"extrapaft_timehold", l_pExtrapolationAfter->GetValue());
						break;
						case CSLExtrapolation::SI_CYCLE:
							l_XSITimeControl.PutParameterValue(L"extrapaft_type", (short)CSLExtrapolation::SI_CYCLE);
							l_XSITimeControl.PutParameterValue(L"extrapaft_nbcycles", l_pExtrapolationAfter->GetValue());
						break;
						case CSLExtrapolation::SI_BOUNCE:
							l_XSITimeControl.PutParameterValue(L"extrapaft_type", (short)CSLExtrapolation::SI_BOUNCE);
							l_XSITimeControl.PutParameterValue(L"extrapaft_nbbounces", l_pExtrapolationAfter->GetValue());
						break;
					};
				}
			}
		}
	}
}

wchar_t *CMixerToXSI::GetClassID(){return L"CMixerToXSI";}

/**************************************************************************************
CMixerToXSIPostProcess
**************************************************************************************/
CMixerToXSIPostProcess::CMixerToXSIPostProcess() {};

CMixerToXSIPostProcess::~CMixerToXSIPostProcess() {};

CStatus CMixerToXSIPostProcess::Execute(CdotXSIConverter *in_pContext)
{
	CStatus status = CStatus::OK;

	CString l_TempMixerToXSICPSetName(L"TempMixerToXSICPSet");
    CRef out_CRef = in_pContext->application().GetActiveSceneRoot().GetProperties().GetItem(l_TempMixerToXSICPSetName);

	if (out_CRef.IsValid())
	{
		CustomProperty l_XSITempMixerToXSICPSet = out_CRef;
		COMMANDS::DeleteObj(l_XSITempMixerToXSICPSet.GetFullName());
	}

	return status;
}

wchar_t *CMixerToXSIPostProcess::GetClassID(){return L"CMixerToXSIPostProcess";}
