// ZipEncrypt.cpp : Defines the entry point for the console application.
//

#ifndef _WIN32_WINNT		// Allow use of features specific to Windows XP or later.                   
#define _WIN32_WINNT 0x0501	// Change this to the appropriate value to target other versions of Windows.
#endif						

#define _CRT_SECURE_NO_WARNINGS

#include <stdio.h>
#include <stdlib.h>
#include <tchar.h>
#include <assert.h>
#include <windows.h>
#include <vector>

#pragma pack(push)
#pragma pack(1)

typedef unsigned int ulong;
typedef unsigned short ushort;

// General-purpose bit field flags
enum {
	GPF_ENCRYPTED = 1, // If set, indicates that the file is encrypted.
	GPF_DATA_DESCRIPTOR = 1 << 3, // if set, the CRC32 and sizes aren't set in the file header, but only in the data descriptor following compressed data
	GPF_RESERVED_8_ENHANCED_DEFLATING = 1 << 4, // Reserved for use with method 8, for enhanced deflating.
	GPF_COMPRESSED_PATCHED = 1 << 5, // the file is compressed patched data
};

// compression methods
enum {
	METHOD_STORE  = 0, // The file is stored (no compression)
	METHOD_SHRINK = 1, // The file is Shrunk
	METHOD_REDUCE_1 = 2, // The file is Reduced with compression factor 1
	METHOD_REDUCE_2 = 3, // The file is Reduced with compression factor 2
	METHOD_REDUCE_3 = 4, // The file is Reduced with compression factor 3
	METHOD_REDUCE_4 = 5, // The file is Reduced with compression factor 4
	METHOD_IMPLODE  = 6, // The file is Imploded
	METHOD_TOKENIZE = 7, // Reserved for Tokenizing compression algorithm
	METHOD_DEFLATE  = 8, // The file is Deflated
	METHOD_DEFLATE64 = 9, // Enhanced Deflating using Deflate64(tm)
	METHOD_IMPLODE_PKWARE = 10// PKWARE Date Compression Library Imploding
};


// end of Central Directory Record
// followed by the .zip file comment (variable size, can be empty, obtained from nCommentLength)
struct CDREnd
{
	enum {SIGNATURE = 0x06054b50};
	ulong  lSignature;       // end of central dir signature    4 bytes  (0x06054b50)
	ushort nDisk;            // number of this disk             2 bytes
	ushort nCDRStartDisk;    // number of the disk with the start of the central directory  2 bytes
	ushort numEntriesOnDisk; // total number of entries in the central directory on this disk  2 bytes
	ushort numEntriesTotal;  // total number of entries in the central directory           2 bytes
	ulong  lCDRSize;          // size of the central directory   4 bytes
	ulong  lCDROffset;        // offset of start of central directory with respect to the starting disk number        4 bytes
	ushort nCommentLength;   // .ZIP file comment length        2 bytes

	// .ZIP file comment (variable size, can be empty) follows
};

// This descriptor exists only if bit 3 of the general
// purpose bit flag is set (see below).  It is byte aligned
// and immediately follows the last byte of compressed data.
// This descriptor is used only when it was not possible to
// seek in the output .ZIP file, e.g., when the output .ZIP file
// was standard output or a non seekable device.  For Zip64 format
// archives, the compressed and uncompressed sizes are 8 bytes each.
struct DataDescriptor
{
	ulong lCRC32;             // crc-32                          4 bytes
	ulong lSizeCompressed;    // compressed size                 4 bytes
	ulong lSizeUncompressed;  // uncompressed size               4 bytes

	bool operator == (const DataDescriptor& d)const
	{
		return lCRC32 == d.lCRC32 && lSizeCompressed == d.lSizeCompressed && lSizeUncompressed == d.lSizeUncompressed;
	}
	bool operator != (const DataDescriptor& d)const
	{
		return lCRC32 != d.lCRC32 || lSizeCompressed != d.lSizeCompressed || lSizeUncompressed != d.lSizeUncompressed;
	}
};

// the File Header as it appears in the CDR
// followed by:
//    file name (variable size)
//    extra field (variable size)
//    file comment (variable size)
struct CDRFileHeader
{
	enum {SIGNATURE = 0x02014b50};
	ulong  lSignature;         // central file header signature   4 bytes  (0x02014b50)
	ushort nVersionMadeBy;     // version made by                 2 bytes
	ushort nVersionNeeded;     // version needed to extract       2 bytes
	ushort nFlags;             // general purpose bit flag        2 bytes
	ushort nMethod;            // compression method              2 bytes
	ushort nLastModTime;       // last mod file time              2 bytes
	ushort nLastModDate;       // last mod file date              2 bytes
	DataDescriptor desc;
	ushort nFileNameLength;    // file name length                2 bytes
	ushort nExtraFieldLength;  // extra field length              2 bytes
	ushort nFileCommentLength; // file comment length             2 bytes
	ushort nDiskNumberStart;   // disk number start               2 bytes
	ushort nAttrInternal;      // internal file attributes        2 bytes
	ulong  lAttrExternal;      // external file attributes        4 bytes

	// This is the offset from the start of the first disk on
	// which this file appears, to where the local header should
	// be found.  If an archive is in zip64 format and the value
	// in this field is 0xFFFFFFFF, the size will be in the
	// corresponding 8 byte zip64 extended information extra field.
	enum {ZIP64_LOCAL_HEADER_OFFSET = 0xFFFFFFFF};
	ulong  lLocalHeaderOffset; // relative offset of local header 4 bytes
};


// this is the local file header that appears before the compressed data
// followed by:
//    file name (variable size)
//    extra field (variable size)
struct LocalFileHeader
{
	enum {SIGNATURE = 0x04034b50};
	ulong  lSignature;        // local file header signature     4 bytes  (0x04034b50)
	ushort nVersionNeeded;    // version needed to extract       2 bytes
	ushort nFlags;            // general purpose bit flag        2 bytes
	ushort nMethod;           // compression method              2 bytes
	ushort nLastModTime;      // last mod file time              2 bytes
	ushort nLastModDate;      // last mod file date              2 bytes
	DataDescriptor desc;
	ushort nFileNameLength;   // file name length                2 bytes
	ushort nExtraFieldLength; // extra field length              2 bytes
};

// compression methods
enum EExtraHeaderID {
	EXTRA_ZIP64  = 0x0001, //  ZIP64 extended information extra field
	EXTRA_NTFS   = 0x000a, //  NTFS 
};

//////////////////////////////////////////////////////////////////////////
// header1+data1 + header2+data2 . . .
// Each header should consist of:
//		Header ID - 2 bytes
//    Data Size - 2 bytes
struct ExtraFieldHeader
{
	ushort headerID;
	ushort dataSize;
};
struct ExtraNTFSHeader
{
	ulong reserved;  // 4 bytes.
	ushort attrTag;  // 2 bytes.
	ushort attrSize; // 2 bytes.
};

#pragma pack(pop)

//#define assert_exit(cond) {assert(cond); if (!(cond)) exit(-1);}
#define assert_exit(cond) {if (!(cond)) {printf("Error: assertion %s failed", #cond); exit(-1);}}

CDREnd CentralDirEnd;

char ZipSwapTable[64] =
 {
 	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 
 	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
 	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 
 	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F 
};

const unsigned long ZipEncryptCrcTable[256] =
{
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL,
	0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL, 0x00000000UL
};

void encrypt_buffer(unsigned char *buffer, int size, unsigned int seed)
{
	unsigned char tmpdata[64];

	seed ^= 0xcafebabeUL;

	for (int pos = 0; pos < size; pos += 64)
	{
		int len = min(size-pos, 64);
		int shift = (seed >> 16) & 63;

		if (len == 64)
		{
			for (int i=0; i<64; i++)
				tmpdata[(ZipSwapTable[i] + shift) & 63] = buffer[pos + i];
		}
		else
		{
			for (int i=0; i<len; i++)
				tmpdata[i] = buffer[pos + i];
		}
		
		for (int i=0; i<len; i++)
		{
			buffer[pos + i] = (seed ^ tmpdata[i]) & 0xFF;
			seed = ZipEncryptCrcTable[(seed ^ tmpdata[i]) & 0xFF] ^ (seed >> 8);
		}
	}
}

void decrypt_buffer(unsigned char *buffer, int size, unsigned int seed)
{
	unsigned char tmpdata[64];

	seed ^= 0xcafebabeUL;

	for (int pos = 0; pos < size; pos += 64)
	{
		int len = min(size-pos, 64);
		int shift = (seed >> 16) & 63;

		for (int i=0; i<len; i++)
		{
			tmpdata[i] = (seed ^ buffer[pos + i]) & 0xFF;
			seed = ZipEncryptCrcTable[buffer[pos + i]] ^ (seed >> 8);
		}

		if (len == 64)
		{
			for (int i=0; i<64; i++)
				buffer[pos + i] = tmpdata[(ZipSwapTable[i] + shift) & 63];
		}
		else
		{
			for (int i=0; i<len; i++)
				buffer[pos + i] = tmpdata[i];
		}
	}
}

int main(int argc, char* argv[])
{
	if (argc != 2)
	{
		printf("Usage: ZipEncrypt <pak-file-name>\n");
		return -1;
	}

	size_t res;
	FILE *zipfile = fopen(argv[1], "r+b");
	if (!zipfile)
	{
		printf("Error: Unable to open file: '%s'\n", argv[1]);
		return -1;
	}

	res = fseek(zipfile, 0, SEEK_END);
	assert_exit(res==0);
	int zip_length = ftell(zipfile);
	assert_exit(zip_length > sizeof(CDREnd));

	int search_pos = max(0, zip_length - 256);
	char buf[256];
	res = fseek(zipfile, search_pos, SEEK_SET);
	assert_exit(res==0);
	res = fread(buf, zip_length-search_pos, 1, zipfile);
	assert_exit(res==1);

	bool found = false;
	for (int i=0; i+sizeof(CDREnd) <= zip_length-search_pos; i++)
	{
		if (reinterpret_cast<CDREnd *>(buf + i)->lSignature == CDREnd::SIGNATURE)
		{
			memcpy(&CentralDirEnd, buf + i, sizeof(CDREnd));
			found = true;
			break;
		}
	}
	assert_exit(found);

	std::vector<char> central_dir;

	central_dir.resize(CentralDirEnd.lCDRSize);

	res = fseek(zipfile, CentralDirEnd.lCDROffset, SEEK_SET);
	assert_exit(res==0);
	res = fread(&central_dir[0], CentralDirEnd.lCDRSize, 1, zipfile);
	assert_exit(res==1);

	int num_files_encrypted = 0;
	int num_files_skipped = 0;

	CDRFileHeader *filehdr = reinterpret_cast<CDRFileHeader *>(&central_dir[0]);
	for (int i=0; i<CentralDirEnd.numEntriesTotal; i++)
	{
		char *filename = reinterpret_cast<char *>(filehdr+1);
		assert_exit(filehdr->lSignature == CDRFileHeader::SIGNATURE);

		if (filehdr->nMethod == METHOD_DEFLATE && filehdr->desc.lSizeCompressed >= 64)
		{
			// encrypt
			num_files_encrypted++;

			LocalFileHeader lfh;
			
			res = fseek(zipfile, filehdr->lLocalHeaderOffset, SEEK_SET);
			assert_exit(res==0);
			res = fread(&lfh, sizeof(lfh), 1, zipfile);
			assert_exit(res==1);

			assert_exit(lfh.lSignature == LocalFileHeader::SIGNATURE);
			assert_exit(filehdr->desc == lfh.desc);
			assert_exit(filehdr->nMethod == lfh.nMethod);
			assert_exit(filehdr->nFileNameLength == lfh.nFileNameLength);

			unsigned char *file_buffer = new unsigned char[lfh.desc.lSizeCompressed];
			assert_exit(file_buffer);

			size_t file_offset = filehdr->lLocalHeaderOffset + sizeof(LocalFileHeader) + lfh.nFileNameLength + lfh.nExtraFieldLength;

			res = fseek(zipfile, file_offset, SEEK_SET);
			assert_exit(res==0);
			res = fread(file_buffer, lfh.desc.lSizeCompressed, 1, zipfile);
			assert_exit(res==1);

			encrypt_buffer(file_buffer, lfh.desc.lSizeCompressed, lfh.desc.lCRC32);
			/*
			for (unsigned int i=0; i<lfh.desc.lSizeCompressed; i++)
				file_buffer[i] ^= 0x55;
			*/

			filehdr->nMethod = METHOD_IMPLODE;
			lfh.nMethod = METHOD_IMPLODE;

			res = fseek(zipfile, filehdr->lLocalHeaderOffset, SEEK_SET);
			assert_exit(res==0);
			res = fwrite(&lfh, sizeof(lfh), 1, zipfile);
			assert_exit(res==1);

			res = fseek(zipfile, file_offset, SEEK_SET);
			assert_exit(res==0);
			res = fwrite(file_buffer, lfh.desc.lSizeCompressed, 1, zipfile);
			assert_exit(res==1);

			delete file_buffer;
		}
		else
		{
			// skip
			num_files_skipped++;
		}

		filehdr = reinterpret_cast<CDRFileHeader *>(filename + filehdr->nFileNameLength + filehdr->nFileCommentLength + filehdr->nFileCommentLength);
	}

	res = fseek(zipfile, CentralDirEnd.lCDROffset, SEEK_SET);
	assert_exit(res==0);
	res = fwrite(&central_dir[0], CentralDirEnd.lCDRSize, 1, zipfile);
	assert_exit(res==1);

	printf("Encrypted %d files skipped %d files\n", num_files_encrypted, num_files_skipped);
	//getchar();

	return 0;
}

