#include "StdAfx.h"
#include "PredictionHub.h"
#include "DataReader.h"

CPredictionHub::PredictorMap CPredictionHub::s_predictors;

void	CPredictionHub::Get(CHTTPOutput *pOut, const IServlet::ParamsMap &params)
{
	ParamsMap::const_iterator it = params.find("image");
	if(it!=params.end())//send image
	{
		ShowResultsImage(pOut);
		return;
	}
	std::string file;
	it = params.find("file");
	if(it!=params.end())
		file = it->second;
	std::string predictor;
	it = params.find("predictor");
	if(it!=params.end())
		predictor = it->second;

	it = params.find("zoom");
	std::string zoom;
	if(it!=params.end())
		zoom = it->second;

	it = params.find("cap");
	std::string cap;
	if(it!=params.end())
		cap = it->second;
	
	if(file.empty()||predictor.empty())
		ShowSetupPage(pOut,"");
	else
	{
		try
		{
			ProcessData(file,predictor,true,zoom,cap);
			ShowResultsPage(pOut);
		}
		catch(std::exception& e)
		{
			ShowSetupPage(pOut,e.what());
		}
	}
}

static void EnumFiles( const char* mask, std::vector<std::string>& list )
{
	WIN32_FIND_DATAA fd;
	HANDLE f = FindFirstFileA(mask,&fd);
	if(f==INVALID_HANDLE_VALUE)
		return;
	do
	{
		if(fd.nFileSizeHigh==0 && fd.nFileSizeLow == 0)
			continue;
		list.push_back(std::string(fd.cFileName));
	}
	while(FindNextFileA(f,&fd));
	FindClose(f);
}

void	CPredictionHub::ShowSetupPage(CHTTPOutput *pOut, const std::string& err)
{
	pOut->Write("<HTML><BODY><FORM><BR>");
	if(!err.empty())
	{
		pOut->Write("<FONT color=red><B>Error : %s</B></FONT><BR>",err.c_str());
	}
	pOut->Write("Select file and predictor type<BR>");
	//File
	std::vector<std::string> lst;
	EnumFiles("wrld*.dat",lst);
	pOut->Write("<SELECT name=\"file\"><option>Select file");
	for(size_t i=0;i<lst.size();++i)
	{
		pOut->Write("<option>%s",lst[i].c_str());
	}
	pOut->Write("</SELECT><BR>");
	//Predictor
	pOut->Write("<SELECT name=\"predictor\"><option>Select predictor");
	for(PredictorMap::iterator it=s_predictors.begin();it!=s_predictors.end();++it)
	{
		pOut->Write("<option>%s",it->first.c_str());
	}
	pOut->Write("</SELECT><BR>");

	pOut->Write("Zoom:<input type=text name=\"zoom\"/><br>");
	pOut->Write("Cap:<input type=text name=\"cap\"/><br>");

	pOut->Write("<input type=\"submit\" value=\"Go\"/></FORM></BODY></HTML>");
}

void HSLtoRGB(float hsl[3], float rgb[3]);

class CPredictionHub::CDeltaError : public	CDataReader::IDataSink
{
public:
	CDeltaError(IPredictorCreator* p, CPredictionHub::SResults& res, bool reset):
			m_prd(p->Create()), m_res(res)
	{
		if (!reset)
			return;

		res.entities = 0;
		res.samples = 0;
		res.m_err.m_err.clear();
		res.tot_err = SVec3(0,0,0);
	}

	bool Data(uint32 sess_id, uint32 ent_id, const SVec3 *data, size_t size)
	{
		for(size_t i=0;i<size;++i)
		{
			SVec3 pv = data[i];
			if(m_prd->Predict(pv))
				CollectError(pv);
			m_prd->Value(data[i]);
		}
		m_res.entities++;
		return true;
	}



	void CollectError(const SVec3& v)
	{
		if(m_res.m_err.m_err.empty())
		{
			m_res.m_err.m_max = v;
			m_res.m_err.m_min = v;
		}
		else
		{
			for(int i=0;i<3;++i)
			{
				m_res.m_err.m_max[i] = max(v[i],m_res.m_err.m_max[i]);
				m_res.m_err.m_min[i] = min(v[i],m_res.m_err.m_min[i]);
			}
		}
		m_res.m_err.m_err.push_back(v);
		m_res.tot_err = m_res.tot_err + v;
	}
private:
	std::auto_ptr<IPredictor<SVec3>> m_prd;
	SVec3				m_max,m_min;
	CPredictionHub::SResults& m_res;
};


void CPredictionHub::Draw(const char* zoom, int cap)
{
	CImage<iRGB>& img = m_result.image;
	SVec3 origin = m_result.tot_err / m_result.m_err.m_err.size();
	SVec3 scale;
	for (int i=0; i<3; i++)
		scale[i] = max( m_result.m_err.m_max[i] - origin[i], origin[i] - m_result.m_err.m_min[i] );
	origin = origin-scale;
	
	scale = scale * 2;
	if(	zoom )
	{
		const char* z = zoom;
		while(*z)
		{
			if(*z>='1' && *z<='9')
			{
				int num = *z-'1';
				scale = scale/3.0f;
				origin.x += scale.x*(num%3);
				origin.y += scale.y*(num/3);
			}
			else
				break;
			z++;
		}
	}

	int w = img.GetWidth();
	int h = img.GetHeight();
	float scale_x = float(w-1)/scale.x;
	float scale_y = float(h-1)/scale.y;
	int	*count = new int[w*h];
	for(size_t i=0; i < w*h; i++)
		count[i] = 0;
	int mx=0,my=0;
	int max_c = 0;
	for(size_t i=0; i < m_result.m_err.m_err.size(); i++)
	{
		SVec3 rel = m_result.m_err.m_err[i]-origin;
		int x = int(rel.x*scale_x+0.5f);
		int y = img.GetHeight()-1-int(rel.y*scale_y+0.5f);
		if(x<0 || x>=img.GetWidth() || y<0 || y>=img.GetHeight())
			continue;
		count[x+y*w]++;
		if(max_c<count[x+y*w])
		{
			max_c = count[x+y*w];
			mx = x;
			my = y;
		}
	}

	float scale_c = 1.0f/float(max_c);
	if(cap && cap<max_c)
	{
		scale_c = 1.0f/float(cap);
	}
	else
		cap = max_c;

	uint32 total_c = 0;
	
	float hsl[3];
	hsl[1] = 1.0f;
	hsl[2] = 0.5f;
	for (int i=0; i<h; i++)
	{
		for (int j=0; j<w; j++)
		{
			total_c += count[i*w+j];
			float t = float(count[i*w+j])*scale_c;
			if(t>1.0f)
				t=1.0f;
			hsl[0] = 0.7f*t+0.3f;
			t = pow(t,0.3f);
			float rgb[3];
			HSLtoRGB(hsl,rgb);
			img(j,i).r = int(t*rgb[0]*255.0f);
			img(j,i).g = int(t*rgb[1]*255.0f);
			img(j,i).b = int(t*rgb[2]*255.0f);
			
		}
	}
	delete [] count;

	m_result.maxcount = max_c;
	m_result.maxshown = cap;
	m_result.samples = total_c;
	m_result.total_samples = m_result.m_err.m_err.size();
	m_result.min_err = origin;
	m_result.max_err = origin+scale;
}

void 	CPredictionHub::ProcessData(const string& filename, const string& predictor, bool reset, const string& zoom, const string& cap)
{
	SCOPED_GLOBAL_LOCK;

	IPredictorCreator* pred = NULL;
	PredictorMap::iterator it = s_predictors.find(predictor);
	if(it!=s_predictors.end())
		pred = it->second;
	if(!pred)
		throw std::exception("Predictor not registered.");

	CDataReader dr(filename);
	CDeltaError proc(pred,m_result,reset);
	dr.Process(&proc,-1);

	m_result.image.Clear(iRGB(0,0,0));
	Draw(zoom.c_str(),atoi(cap.c_str()));

	m_result.filename = filename;
	m_result.predictor = predictor;
}

void	CPredictionHub::ShowResultsPage(CHTTPOutput *pOut)
{
	pOut->Write("<HTML><BODY><BR>");
	pOut->Write("File : %s Predictor : %s<br>",m_result.filename.c_str(),m_result.predictor.c_str());
	pOut->Write("Entities : %d Samples : %d of %d<br>",m_result.entities,m_result.samples,m_result.total_samples);
	pOut->Write("Scale : 0 to %d. Actual maximum %d<br>",m_result.maxshown,m_result.maxcount);
	pOut->Write("<IMG src=\"spectrum.png\"/>");
	pOut->Write("<BR>Results range x(%.5f;%.5f) y(%.5f;%.5f) <BR>",m_result.min_err.x,m_result.max_err.x,m_result.min_err.y,m_result.max_err.y);
	pOut->Write("<IMG src=\"/predict.htm?image\"/>");
	pOut->Write("</BODY></HTML>");
}

void	CPredictionHub::ShowResultsImage(CHTTPOutput *pOut)
{
	pOut->WriteImage(m_result.image);
}

