/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2007.
-------------------------------------------------------------------------
$Id$
$DateTime$

-------------------------------------------------------------------------
History:
- 16:08:2007   : Created by Denisz Polgar

*************************************************************************/
#include "StdAfx.h"
#include "Single.h"
#include "Shotgun.h"
#include "BombLaunch.h"
#include "Item.h"
#include "Weapon.h"
#include "Projectile.h"
#include "Actor.h"
#include "Player.h"
#include "Game.h"
#include "GameCVars.h"
#include "HUD/HUD.h"
#include "HUD/HUDRadar.h"
#include "WeaponSystem.h"
#include <IEntitySystem.h>
#include "ISound.h"
#include <IVehicleSystem.h>
#include <IMaterialEffects.h>
#include "GameRules.h"
#include <Cry_GeoDistance.h>

#include "IronSight.h"

#include "IRenderer.h"
#include "IRenderAuxGeom.h"

const float RELOAD_TIME = 1000;
const float BOMB_SPACE = 0.7f;
const Vec3 BOMB_SLOTS[] =
{
	Vec3(-2*BOMB_SPACE, 0.5f, 0),
	Vec3(-BOMB_SPACE, 1, -BOMB_SPACE),
	Vec3(BOMB_SPACE, 1, -BOMB_SPACE),
	Vec3(2*BOMB_SPACE, 0.5f, 0),
	Vec3(-2*BOMB_SPACE, 1, BOMB_SPACE)
};

CBombLaunch::CBombLaunch() :
	m_loadingBombs(false),
	m_instaLoaded(false),
	m_instaLoadCount(-1)
{
}

CBombLaunch::~CBombLaunch()
{
	ResetBombs();
	ResetInstaLoad();
}

void CBombLaunch::Init(IWeapon *pWeapon, const IItemParamsNode *params)
{
	CShotgun::Init(pWeapon, params);

	ResetBombs();
	ResetInstaLoad();
}

bool CBombLaunch::Shoot(bool resetAnimation, bool autoreload/* =true */, bool noSound /* =false */)
{
	IEntityClass* ammo = m_fireparams.ammo_type_class;
	int ammoCount = m_pWeapon->GetAmmoCount(ammo);

	if (m_fireparams.clip_size==0)
		ammoCount = m_pWeapon->GetInventoryAmmoCount(ammo);

	CActor *pActor = m_pWeapon->GetOwnerActor();
	
	if (!CanFire(true))
	{
		if ((ammoCount <= 0) && (!m_reloading))
		{
			m_pWeapon->Reload();
		}	
		return false;
	}
	
	if (m_reloading || !pActor)
	{
		return false;
	}

	// Check target distance
	Vec3 pos=m_pWeapon->GetEntity()->GetWorldPos();
	IPipeUser* pPipeUser = CastToIPipeUserSafe(pActor->GetEntity()->GetAI());

	if (!pPipeUser)
		return false;
	
	IAIObject *pAITarget=pPipeUser->GetAttentionTarget();

	if (!pAITarget)
		return false;

	unsigned short type=pAITarget->GetAIType();
	if (type != AIOBJECT_PLAYER && type != AIOBJECT_PUPPET && 
			type != AIOBJECT_VEHICLE && type != AIOBJECT_WAYPOINT)
		return false;

	Vec3 dest=pAITarget->GetPos();
	// Effective range counts in 2D only!
	Vec3 trajectory=dest-pos;

	if (trajectory.GetLengthSquared2D() > m_bombLaunchParams.effective_range_sqr)
	{
		//gEnv->pLog->Log("<<<< Target out of effective bombardment range >>>>");	
		return false;
	}
	
	LaunchBomb(1.0f);

	m_fired = true;
	m_next_shot += m_next_shot_dt;
	m_zoomtimeout = m_next_shot + 0.5f;

	return true;
}

void CBombLaunch::LaunchBomb(float speedScale)
{
	IEntity *pOwner=m_pWeapon->GetOwner();
		
	if (!pOwner)
		return;

	IEntityClass* pAmmoType = GetAmmoType();

	if (!pAmmoType)
		return;

	int bombCount=GetAmmoCount();

	IAIObject* pAI = pOwner->GetAI();
	IAIActor* pAIActor=pAI ? pAI->CastToIAIActor() : NULL;

	if (pAIActor)
	{
		for(TLoadedBombs::iterator it=m_loadedBombs.begin(); it != m_loadedBombs.end(); ++it)
		{
			IEntity *pBomb=gEnv->pEntitySystem->GetEntity(*it);
			
			if (pBomb)
			{
				// Trigger bomb launch
				CProjectile *pBombProjectile=g_pGame->GetWeaponSystem()->GetProjectile(pBomb->GetId());

				if (pBombProjectile)
				{
					pe_action_update_constraint uc;
					uc.bRemove = 1;
					pBomb->GetPhysics()->Action(&uc);

					pBomb->DetachThis(IEntity::ATTACHMENT_KEEP_TRANSFORMATION, ENTITY_XFORM_USER);
					pBombProjectile->Launch(pBomb->GetWorldPos(), Vec3(0, 0, -1), Vec3(0, 0, -1), 1.0f);
					//gEnv->pLog->Log("Launch SB!");
				}

				*it=0;
				--bombCount;

				//gEnv->pLog->Log("Ammo count after launching bomb %d", bombCount);
				break;
			}
		}

		if (bombCount == 0)
			m_loadedBombs.resize(0);
	}

	if (GetClipSize() != -1)
	{
		if (GetClipSize()!=0)
			m_pWeapon->SetAmmoCount(pAmmoType, bombCount);
		else
			m_pWeapon->SetInventoryAmmoCount(pAmmoType, bombCount);
	}
}

// Reload shells
class CBombLaunch::ReloadOneShellAction
{
public:
	ReloadOneShellAction(CWeapon *_wep, int _zoomed)
	{
		pWep = _wep;
		rzoomed = _zoomed;
	}
	void execute(CItem *_this)
	{
		CBombLaunch *fm = (CBombLaunch *)pWep->GetFireMode(pWep->GetCurrentFireMode());

		if(fm->m_reload_was_broken)
			return;

		IEntityClass* pAmmoType = fm->GetAmmoType();

		if (pWep->IsServer())
		{
			int ammoCount = pWep->GetAmmoCount(pAmmoType);
			pWep->SetAmmoCount(pAmmoType, ammoCount+1);
			pWep->SetInventoryAmmoCount(pAmmoType, pWep->GetInventoryAmmoCount(pAmmoType)-1);
		}

		if (!fm->m_break_reload)
			fm->ReloadShell(rzoomed);
		else
			fm->EndReload(rzoomed);
	}
private:
	CWeapon *pWep;
	int rzoomed;
};


void CBombLaunch::StartReload(int zoomed)
{
	if (m_loadingBombs || m_loadedBombs.size() > 0)
		return;

	CShotgun::StartReload(zoomed);
	m_loadingBombs=true;

	
	CActor *pActor=m_pWeapon->GetOwnerActor();
}

void CBombLaunch::ReloadShell(int zoomed)
{
	if(m_reload_was_broken)
		return;

	CActor* pOwner = m_pWeapon->GetOwnerActor();
	bool isAI = pOwner && (pOwner->IsPlayer() == false);
	int ammoCount = GetAmmoCount();
	if (ammoCount < m_fireparams.clip_size && m_max_shells>0 && (isAI || m_pWeapon->GetInventoryAmmoCount(m_fireparams.ammo_type_class) > 0))
	{
		m_max_shells --;

		// EXP 1 Rev 2.: Has no bomb grabbed by tentacles anymore!!!
		// Perform actual grabbing
		IEntity *pOwner=m_pWeapon->GetOwner();
			
		if (!pOwner)
			return;

		IEntityClass* pAmmoType = GetAmmoType();

		if (!pAmmoType)
			return;

		IAIObject* pAI = pOwner->GetAI();
		IAIActor* pAIActor=pAI->CastToIAIActor();

		if (pAIActor)
		{
			CProjectile *pBomb=m_pWeapon->SpawnAmmo(pAmmoType);
				   
			if (pBomb && pBomb->GetGameObject())
			{
				IEntity* pBombEntity= pBomb->GetGameObject()->GetEntity();
				
				pBomb->SetParams(pOwner->GetId(), m_pWeapon->GetHostId(), m_pWeapon->GetEntityId(), m_pWeapon->GetCurrentFireMode(), GetDamage(), 0);
				pBomb->SetDestination(m_pWeapon->GetDestination());
				//pBomb->Launch(Vec3(ammoCount*0.5f-(m_fireparams.clip_size/2.0f)*0.5f, 1, -1), Vec3(0, 0, 0), Vec3(0, 0, 0));
				//pBomb->Launch(Vec3(-(m_fireparams.clip_size/2.0f)*BOMB_SPACE, 1, 0), Vec3(0, 0, 0), Vec3(0, 0, 0));
				pBomb->Launch(BOMB_SLOTS[4], Vec3(0, 0, 0), Vec3(0, 0, 0));
				
				//pBomb->GetGameObject()->SetAspectProfile(eEA_Physics, eAP_Linked);
				
				// Set ignore collision
				pe_action_add_constraint ac;
				ac.flags = constraint_inactive|constraint_ignore_buddy;
				ac.pBuddy = pOwner->GetPhysics();
				ac.pt[0].Set(0,0,0);

				pBombEntity->GetPhysics()->Action(&ac);
				pOwner->AttachChild(pBombEntity);

				m_loadedBombs.push_back(pBombEntity->GetId());
			}	
		}
		
		// Leave this much time to grab the bomb
		uint animTime = RELOAD_TIME;
		m_pWeapon->GetScheduler()->TimerAction(animTime, CSchedulerAction<ReloadOneShellAction>::Create(ReloadOneShellAction(m_pWeapon, zoomed)), false);
		// call this again
	}
	else
	{
		EndReload(zoomed);
	}
}

void CBombLaunch::EndReload(int zoomed)
{
	if (!m_loadingBombs)
		return;

	CShotgun::EndReload(zoomed);
	
	m_loadingBombs=false;
}

//------------------------------------------------------------------------
void CBombLaunch::ResetParams(const struct IItemParamsNode *params)
{
	CShotgun::ResetParams(params);

	const IItemParamsNode *bomb_launch = params?params->GetChild("bomb_launch"):0;
	m_bombLaunchParams.Reset(bomb_launch);
}

//------------------------------------------------------------------------
void CBombLaunch::PatchParams(const struct IItemParamsNode *patch)
{
	CShotgun::PatchParams(patch);

	const IItemParamsNode *bomb_launch = patch->GetChild("bomb_launch");
	m_bombLaunchParams.Reset(bomb_launch, false);
}

//------------------------------------------------------------------------
void CBombLaunch::Update(float frameTime, uint frameId)
{
	CShotgun::Update(frameTime, frameId);

	//int cnt=m_pWeapon->GetAmmoCount(m_fireparams.ammo_type_class);
	//int max=m_fireparams.clip_size;
	if (!m_instaLoaded)
	{
		IEntity *pOwner=m_pWeapon->GetOwner();
		if (pOwner && !pOwner->IsHidden() && pOwner->GetPhysics())
			InstaLoad();

	}
	else if (m_loadingBombs)
	{
		for(int i=0; i<m_loadedBombs.size(); ++i)
		{
			IEntity *pBomb=gEnv->pEntitySystem->GetEntity(m_loadedBombs[i]);

			if (pBomb)
			{
				Vec3 pos=pBomb->GetPos();
				Vec3 dpos(0, 0, 0);
				dpos=BOMB_SLOTS[m_loadedBombs.size()-i-1]-pos;
								
				// HACK: Make reload anim faster than reloading itself, so bombs surely get into position
				dpos*=frameTime/RELOAD_TIME*5000;
				pBomb->SetPos(pos+dpos, ENTITY_XFORM_USER);
			}
		}

		m_pWeapon->RequireUpdate(eIUS_FireMode);
	}
}
//------------------------------------------------------------------------
void CBombLaunch::ReleaseProjectiles()
{
	if (m_loadingBombs)
		CancelReload();

	for(TLoadedBombs::const_iterator it=m_loadedBombs.begin(); it != m_loadedBombs.end(); ++it)
	{
		IEntity *pBomb=gEnv->pEntitySystem->GetEntity(*it);
		
		if (pBomb)
		{
			// Trigger bomb launch
			CProjectile *pBombProjectile=g_pGame->GetWeaponSystem()->GetProjectile(pBomb->GetId());

			if (pBombProjectile)
			{
				pe_action_update_constraint uc;
				uc.bRemove = 1;
				pBomb->GetPhysics()->Action(&uc);

				pBomb->DetachThis(IEntity::ATTACHMENT_KEEP_TRANSFORMATION, ENTITY_XFORM_USER);
				pBombProjectile->Launch(pBomb->GetWorldPos(), Vec3(0, 0, -1), Vec3(0, 0, -1), 1.0f);
			}
		}
	}

	m_loadedBombs.resize(0);
	
	IEntityClass* pAmmoType = GetAmmoType();

	if (!pAmmoType)
		return;

	int bombCount=GetAmmoCount();

	if (GetClipSize() != -1)
	{
		if (GetClipSize()!=0)
			m_pWeapon->SetAmmoCount(pAmmoType, 0);
		else
			m_pWeapon->SetInventoryAmmoCount(pAmmoType, m_pWeapon->GetInventoryAmmoCount(pAmmoType)-bombCount);
	}
}
//------------------------------------------------------------------------
void CBombLaunch::ResetBombs()
{
		for(TLoadedBombs::iterator it=m_loadedBombs.begin(); it != m_loadedBombs.end(); ++it)
	{
		IEntity *pBomb=gEnv->pEntitySystem->GetEntity(*it);
		
		if (pBomb)
		{
			gEnv->pEntitySystem->RemoveEntity(pBomb->GetId(), true);
		}
	}

	m_loadedBombs.resize(0);
	m_loadingBombs=false;
}

//------------------------------------------------------------------------
void CBombLaunch::ResetInstaLoad()
{
	m_instaLoaded=false;
	m_instaLoadCount=-1;
	m_pWeapon->RequireUpdate(eIUS_FireMode);
}
//------------------------------------------------------------------------
void CBombLaunch::InstaLoad()
{
	//gEnv->pLog->Log("Requesting insta-load");
	IEntity *pOwner=m_pWeapon->GetOwner();
			
	IEntityClass* pAmmoType = GetAmmoType();

	if (!m_pWeapon || !pOwner || !pAmmoType)
		return;

	if (m_loadedBombs.size() > 0)
	{
		m_instaLoaded=true;	
		return;
	}

	int load_count=m_instaLoadCount > -1 ? m_instaLoadCount : m_fireparams.clip_size;

	//gEnv->pLog->Log("Executing insta-load");
	for (int i=0; i<m_fireparams.clip_size; ++i)
	{
		if (i < m_fireparams.clip_size-load_count)
		{
			m_loadedBombs.push_back(-1);
			continue;
		}

		CProjectile *pBomb=m_pWeapon->SpawnAmmo(pAmmoType);
		
		if (pBomb && pBomb->GetGameObject())
		{
			IEntity* pBombEntity= pBomb->GetGameObject()->GetEntity();
			
			pBomb->SetParams(pOwner->GetId(), m_pWeapon->GetHostId(), m_pWeapon->GetEntityId(), m_pWeapon->GetCurrentFireMode(), GetDamage(), 0);
			pBomb->SetDestination(m_pWeapon->GetDestination());
			pBomb->Launch(BOMB_SLOTS[m_fireparams.clip_size-i-1], Vec3(0, 0, 0), Vec3(0, 0, 0));
			
			// Set ignore collision
			pe_action_add_constraint ac;
			ac.flags = constraint_inactive|constraint_ignore_buddy;
			ac.pBuddy = pOwner->GetPhysics();
			ac.pt[0].Set(0,0,0);

			pBombEntity->GetPhysics()->Action(&ac);
			pOwner->AttachChild(pBombEntity);

			m_loadedBombs.push_back(pBombEntity->GetId());
		}	
	}

	m_max_shells=0;
	m_instaLoaded=true;

	if (m_pWeapon->IsServer())
	{
		m_pWeapon->SetAmmoCount(pAmmoType, load_count);
		m_pWeapon->SetInventoryAmmoCount(pAmmoType, m_pWeapon->GetInventoryAmmoCount(pAmmoType)-load_count);
	}
}
//------------------------------------------------------------------------
const char *CBombLaunch::GetType() const
{
	return "BombLaunch";
}
//------------------------------------------------------------------------
void CBombLaunch::Serialize(TSerialize ser)
{
	CShotgun::Serialize(ser);

	if(ser.GetSerializationTarget() != eST_Network)
	{
		ser.BeginGroup("smartmode");
		
		if (ser.IsWriting())
		{
			// Save bomb count for reload, and ids to reset them
			ser.Value("bombs", m_loadedBombs);
		}
		else
		{
			// Load count, reset bombs, prevent insta-loading
			ser.Value("bombs", m_loadedBombs);
			/*ResetBombs();
			m_instaLoaded=true;*/
		}

		ser.EndGroup();
	}
}

void CBombLaunch::PostSerialize()
{
	CShotgun::PostSerialize();

	int insta_count=GetAmmoCount();
	ResetBombs();
	ResetInstaLoad();
	m_instaLoadCount=insta_count > 0 ? insta_count : -1;
}