/*************************************************************************
Crytek Source File.
Copyright (C), Crytek Studios, 2001-2007.
-------------------------------------------------------------------------
$Id:$
$DateTime$
Description:  
-------------------------------------------------------------------------
History:
- 16:08:2007   : Created by Denisz Polgar

*************************************************************************/
#include "StdAfx.h"

#include <ICryAnimation.h>

#include "Projectile.h"
#include "SmartBomb.h"
#include "Actor.h"
#include "GameRules.h"

#include "Single.h"
#include "Shotgun.h"
#include "BombLaunch.h"

#include "IVehicleSystem.h"
#include "IActorSystem.h"

int CSmartBomb::m_sSmart=SMARTBOMB_SMART;
int CSmartBomb::m_sDumb=SMARTBOMB_DUMB;

CSmartBomb::CSmartBomb()
:CProjectile(),
/*
m_playerScale(1),
m_impactTime(0),
*/
m_launchedMass(0),
m_lastDetectionTime(0),
m_detectionInterval(0),
m_attractionRange(0),
m_attractionForce(0),
m_minDistance(0),
m_maxDistance(10),
m_explosionProximity(0),
m_bArmed(false),
m_armedEffect(0),
m_armedEffectId(-1),
m_armedSound(0),
m_armedSoundId(INVALID_SOUNDID),
m_bAutoArm(false),
m_bLateReactor(false),
m_bImpactExplode(false),
m_fScatter(0),
m_maxSpeed(0),
m_safetyTimeout(0),
m_searchTimeout(0),
m_targetId(0),
m_bReadyToLaunch(false),
m_bLaunched(false),
m_bPhysSetUp(false),
m_armTime(0),
m_playerAggression(0),
m_dynamicLifetimeScale(0),
m_trackInaccuracy(0),
m_trackInaccuracyTimeout(0),
m_lastTrackDivergence(ZERO),
m_nextTrackRecalculate(0),
m_jump(0)
{
}

//-------------------------------------------
CSmartBomb::~CSmartBomb()
{
}

//-------------------------------------------
bool CSmartBomb::Init(IGameObject *pGameObject)
{
	bool res=CProjectile::Init(pGameObject);

	m_launchedLifetime=GetParam("launched_lifetime", m_launchedLifetime);
	m_launchedMass=GetParam("launched_mass", m_launchedMass);
	m_attractionForce=GetParam("attractionforce", m_attractionForce);
	m_attractionRange=GetParam("attractionrange", m_attractionRange);
	m_minDistance=GetParam("mindistance", m_minDistance);
	m_maxDistance=m_attractionRange;
	m_maxDistance=GetParam("maxdistance", m_maxDistance);
	m_explosionProximity=GetParam("explosionproximity", m_explosionProximity);
	m_safetyTimeout=GetParam("safetytimeout", m_safetyTimeout);
	m_detectionInterval=GetParam("detectioninterval", m_detectionInterval);
	m_armedEffect=GetParam("armedeffect", m_armedEffect);
	m_armedSound=GetParam("armedsound", m_armedSound);
	m_bAutoArm=GetParam("autoarm", m_bAutoArm);
	m_bLateReactor=GetParam("latereactor", m_bLateReactor);
	m_bImpactExplode=GetParam("impact_explode", m_bImpactExplode);
	m_fScatter=GetParam("scatter", m_fScatter);
	m_maxSpeed=GetParam("maxspeed", m_maxSpeed);
	m_searchTimeout=GetParam("searchtimeout", m_searchTimeout);
	m_jump=GetParam("jump", m_jump);
	
	m_playerAggression=GetParam("player_aggression", m_playerAggression);
	m_dynamicLifetimeScale=GetParam("dynamic_lifetime_scale", m_dynamicLifetimeScale);
	m_trackInaccuracy=GetParam("track_inaccuracy", m_trackInaccuracy);
	m_trackInaccuracyTimeout=GetParam("track_inaccuracy_timeout", m_trackInaccuracyTimeout);

	//SetAspectProfile(eEA_Physics, ePT_None);
	
	return res;
}

typedef std::vector <EntityId> TPotentialTargets;

//------------------------------------------
void CSmartBomb::HandleEvent(const SGameObjectEvent &event)
{
	if (m_destroying)
		return;

	CProjectile::HandleEvent(event);

	switch (event.event)
	{
		case eGFE_OnCollision:
			if (m_bImpactExplode && m_bArmed)
			{
				EventPhysCollision *pColl=(EventPhysCollision *)event.ptr;
				if (pColl)
				{
					IEntity *pTarget = pColl->iForeignData[1]==PHYS_FOREIGN_ID_ENTITY ? (IEntity*)pColl->pForeignData[1] : 0;

					if (!pTarget || pTarget->GetClass() != GetEntity()->GetClass())
						Explode(true);
				}
			}
			else if (!m_bAutoArm && m_armTime == 0 && m_totalLifetime > m_safetyTimeout)
			{
				m_armTime=gEnv->pTimer->GetCurrTime()+m_searchTimeout;
			}
			break;
		case eGFE_OnPostStep:
		{
			EventPhysPostStep *pPost=(EventPhysPostStep *)event.ptr;
			// Maximize speed to increase maneuverability of projectile and prevent very high velocities
			if (m_pPhysicalEntity && m_maxSpeed > 0)
			{
				pe_status_dynamics dyn;
				pe_action_set_velocity vel;

				if (m_pPhysicalEntity->GetStatus(&dyn))
				{
					float speed=dyn.v.GetLength();
					if (speed > m_maxSpeed)
					{
						vel.v=dyn.v.GetNormalized()*m_maxSpeed;
						m_pPhysicalEntity->Action(&vel);
					}
				}
			}
			break;
		}
	}
}

void CSmartBomb::Update( SEntityUpdateContext &ctx, int updateSlot)
{
	CProjectile::Update(ctx, updateSlot);

	if (!m_bLaunched)
		m_totalLifetime=0;

	if (!m_pPhysicalEntity)
		return;

	if (m_destroying)
		return;

	if (m_bAutoArm && m_totalLifetime > m_safetyTimeout)
		m_bArmed=true;

	if (!m_bArmed && m_armTime > 0 && ctx.fCurrTime > m_armTime)
		m_bArmed=true;


	if (m_bLaunched && !m_bPhysSetUp)
	{
		if (m_maxSpeed > 0)
		{
			IGameObject *pgO=gEnv->pGame->GetIGameFramework()->GetGameObject(GetEntityId());

			if (pgO)
				pgO->EnablePhysicsEvent(true, eEPE_OnPostStepLogged);
		}

		m_bPhysSetUp=true;
	}

	if (m_targetId)
	{
		if (m_armTime > 0 && ctx.fCurrTime > m_armTime)
		{
			m_armTime=0;
			ArmedEffect(true);
			ArmedSound(true);
		}

		Vec3 centre=GetEntity()->GetWorldPos();
		IEntity *pTargetEntity=gEnv->pEntitySystem->GetEntity(m_targetId);

		if (pTargetEntity)
		{
			Vec3 target=pTargetEntity->GetWorldPos();
			target.z+=1.0f;

			if (m_trackInaccuracy > 0)
			{
				if (ctx.fCurrTime > m_nextTrackRecalculate)
				{
					float trckdst=target.GetDistance(centre);
					m_lastTrackDivergence=Random(Vec3(-m_trackInaccuracy, -m_trackInaccuracy, -m_trackInaccuracy)*trckdst, Vec3(m_trackInaccuracy, m_trackInaccuracy, m_trackInaccuracy)*trckdst);
					m_nextTrackRecalculate=ctx.fCurrTime+m_trackInaccuracyTimeout;
				}

				target+=m_lastTrackDivergence;
			}

			Vec3 pulldir=target-centre;
			float dst=pulldir.GetLength();
			float explosionProximity=m_explosionProximity;

			// Explode farther if target is in vehivicle
			IActor *pTargetActor=gEnv->pGame->GetIGameFramework()->GetIActorSystem()->GetActor(m_targetId);
			if (pTargetActor && pTargetActor->GetLinkedVehicle())
				explosionProximity+=2.0f;

			IAIObject *pAI=pTargetEntity->GetAI();
			if (pAI && pAI->GetAIType() == AIOBJECT_VEHICLE)
				explosionProximity+=2.0f;
			
			//gEnv->pLog->Log("<<< Proximity range: %f; explosion proximity: %f>>>", dst, explosionProximity);

			if (m_pPhysicalEntity)
			{
				pe_status_dynamics dyn;
				m_pPhysicalEntity->GetStatus(&dyn);

				pulldir.Normalize();
				// For gravity like behavior you'd need squared, but it ruins gameplay

				if (dst < explosionProximity)
				{
					Explode(true);
					return;
				}

				// To avoid very large forces arising at short distances
				if (dst < m_minDistance)
					dst=m_minDistance;
				else if (dst > m_maxDistance)
					dst=m_maxDistance;

				// Calculate distance effect
				if (m_bLateReactor)
					//dst*=dst;
					dst=dst;
				else
					dst=sqrt(dst);

				float force=(m_attractionForce*dyn.mass*ctx.fFrameTime)/dst;

				pe_action_impulse impulse;
				impulse.impulse=pulldir*force;

				m_pPhysicalEntity->Action(&impulse);
			}
			else
			{
				m_targetId=0;
				ArmedEffect(false);
				ArmedSound(false);
			}
			
		}
	}
	else if (m_bArmed && ctx.fCurrTime-m_lastDetectionTime > m_detectionInterval)
	{
		m_lastDetectionTime=ctx.fCurrTime;

		// HACK: Attack player aggressively
		if (m_playerAggression > 0 && Random(1.0f) < m_playerAggression)
		{
			IActor *pPlayer=gEnv->pGame->GetIGameFramework()->GetClientActor();

			if (pPlayer)
			{
				m_targetId=pPlayer->GetEntity()->GetId();
			}
		}

		if (!m_targetId)
		{
			// Find a suitable target
			Vec3 centre=GetEntity()->GetWorldPos();
			SEntityProximityQuery query;
			//query.nEntityFlags=ENTITY_FLAG_CALC_PHYSICS;
			query.box = AABB(centre-Vec3(m_attractionRange, m_attractionRange, m_attractionRange), centre+Vec3(m_attractionRange, m_attractionRange, m_attractionRange));
			
			TPotentialTargets targets;
			int count = gEnv->pEntitySystem->QueryProximity(query);

			IEntity *pOwnerEntity = gEnv->pEntitySystem->GetEntity(m_ownerId);
			IEntity *pOwnerWeapon = gEnv->pEntitySystem->GetEntity(m_weaponId);
			IAIObject *pOwnerAI=NULL;
	 
			if (pOwnerEntity)
			{
				pOwnerAI=pOwnerEntity->GetAI();
			}
			else if (pOwnerWeapon)
			{
				for (IEntityLink* pLink = pOwnerWeapon->GetEntityLinks(); pLink; pLink = pLink->next)
				{
					if (!strcmp(pLink->name, "owner"))
					{
						pOwnerEntity=gEnv->pEntitySystem->GetEntity(pLink->entityId);

						if (pOwnerEntity)
						{
							pOwnerAI=pOwnerEntity->GetAI();
							break;
						}
					}
				}
			}

			for(int i=0; i<query.nCount; ++i)
			{
				IEntity* pEntity = query.pEntities[i];

				if (pEntity && pEntity->GetId() != GetEntity()->GetId() && !pEntity->IsHidden())
				{
					IAIObject *pAI=pEntity->GetAI();

					if (pAI)
					{
						if (pAI->GetAIType() == AIOBJECT_VEHICLE)
						{
							IVehicleSystem *pVs=g_pGame->GetIGameFramework()->GetIVehicleSystem();
							IVehicle* pVehicle = pVs ?  pVs->GetVehicle(pEntity->GetId()) : NULL;
							if (pVehicle && !pVehicle->IsDestroyed())
								targets.push_back(pEntity->GetId());
						}
						else if (pAI->IsHostile(pOwnerAI))
						{
							IActorSystem *pAs = g_pGame->GetIGameFramework()->GetIActorSystem();
							IActor *pActor= pAs ? pAs->GetActor(pEntity->GetId()) : NULL;
							if (pActor && pActor->GetHealth() > 0)
								targets.push_back(pEntity->GetId());
						}
					}
				}
			}

			if (!targets.size())
				return;

			int chosen=Random((uint32)targets.size());

			m_targetId=targets[chosen];
		}

		if (m_pAmmoParams && m_pAmmoParams->altGeometry)
		{
			// Swap to open projectile
			GetEntity()->SetStatObj(m_pAmmoParams->altGeometry, 0, false);

			// Jump
			if (m_jump > 0)
			{
				pe_action_impulse impulse;
				impulse.impulse=Vec3(0, 0, m_jump);

				m_pPhysicalEntity->Action(&impulse);
			}
		}

		m_armTime=max(ctx.fCurrTime, m_armTime+m_safetyTimeout);

		IEntity *pTargetEntity=gEnv->pEntitySystem->GetEntity(m_targetId);
		if (m_dynamicLifetimeScale > 0 && pTargetEntity)
		{
			GetEntity()->KillTimer(ePTIMER_LIFETIME);
			float dst=pTargetEntity->GetWorldPos().GetDistance(GetEntity()->GetWorldPos());

			GetEntity()->SetTimer(ePTIMER_LIFETIME, (int)(m_dynamicLifetimeScale*dst*1000.0f));
		}
	}
}

void CSmartBomb::ProcessEvent(SEntityEvent &event)
{
	if (event.event == ENTITY_EVENT_TIMER && event.nParam[0])
	{
		switch (event.nParam[0])
		{
			case ePTIMER_LIFETIME:
				Timeout(FORWARD_DIRECTION);
				break;
		}
	}
}

void CSmartBomb::Launch(const Vec3 &pos, const Vec3 &dir, const Vec3 &velocity, float speedScale)
{
	m_bPhysSetUp=false;

	if (!m_bReadyToLaunch)
	{
		Vec3 mdir=dir;
		mdir.x+=Random(-m_fScatter, m_fScatter);
		mdir.y+=Random(-m_fScatter, m_fScatter);
		mdir.z+=Random(0, m_fScatter);
		CProjectile::Launch(pos, mdir, velocity, speedScale);

		pe_action_awake awake;
		awake.bAwake=false;
		m_pPhysicalEntity->Action(&awake);
	}

	if (m_bReadyToLaunch || m_bAutoArm)
	{
		if (m_pPhysicalEntity)
		{
			pe_params_part part;
			m_pPhysicalEntity->GetParams(&part);
			part.mass=m_launchedMass;
			m_pPhysicalEntity->SetParams(&part);

			pe_action_awake awake;
			m_pPhysicalEntity->Action(&awake);
		}

		m_bLaunched=true;

		if (m_launchedLifetime > 0.0f)
			GetEntity()->SetTimer(ePTIMER_LIFETIME, (int)(m_launchedLifetime*1000.0f));
	}

	if (!m_bReadyToLaunch)
	{
		m_bReadyToLaunch=true;
	}
}

void CSmartBomb::Explode(bool destroy, bool impact, const Vec3 &pos, const Vec3 &normal, const Vec3 &vel, EntityId targetId)
{
	if (!m_bLaunched)
	{
		CWeapon *pWeapon=GetWeapon();

		CBombLaunch *launch = static_cast <CBombLaunch *> (pWeapon->GetFireMode(pWeapon->GetCurrentFireMode()));

		if (launch)
		{
			launch->ReleaseProjectiles();
			//launch->ReleaseProjectile(GetEntity()->GetId());
		}
	}

	CProjectile::Explode(destroy, impact, pos, normal, vel, targetId);
}

void CSmartBomb::ArmedEffect(bool enable)
{
	if (enable)
	{
		if (m_armedEffectId < 0 && m_armedEffect)
		{
			if (m_trailEffectId > -1)
			{
				AttachEffect(false, m_trailEffectId);
				m_trailEffectId=-1;
			}

			m_armedEffectId=AttachEffect(true, 0, m_armedEffect, Vec3(0, 0, 0), Vec3(0, 1, 0), m_pAmmoParams->pTrail->scale);
		}
	}
	else
	{
		if (m_armedEffectId > -1)
		{
			AttachEffect(false, m_armedEffectId);
			m_armedEffectId=-1;

			if (m_trailEffectId<0 && m_pAmmoParams->pTrail)
			{
				m_trailEffectId = AttachEffect(true, 0, m_pAmmoParams->pTrail->effect, Vec3(0,0,0), Vec3(0,1,0), m_pAmmoParams->pTrail->scale, m_pAmmoParams->pTrail->prime);
			}
		}
	}
}

void CSmartBomb::FullSerialize(TSerialize ser )
{
	CProjectile::FullSerialize(ser);

	ser.BeginGroup("smart_projectile");
	ser.Value("armed", m_bArmed);
	ser.Value("target", m_targetId);
	ser.Value("ready", m_bReadyToLaunch);
	ser.Value("launched", m_bLaunched);
	ser.Value("armtime", m_armTime)	;

	ser.EndGroup();
}

void CSmartBomb::PostSerialize()
{
	CProjectile::PostSerialize();

	// Reset trail sound id so first update creates a new sound for it
	TrailSound(false);

	if (m_bLaunched && m_pPhysicalEntity)
	{
		pe_params_part part;
		m_pPhysicalEntity->GetParams(&part);
		part.mass=m_launchedMass;
		m_pPhysicalEntity->SetParams(&part);

		pe_action_awake awake;
		m_pPhysicalEntity->Action(&awake);

		if (m_maxSpeed > 0)
		{
			pe_params_flags fp;
			fp.flagsOR=pef_log_poststep;
			m_pPhysicalEntity->SetParams(&fp);
		}
	}

	if (m_bAutoArm && (m_bReadyToLaunch || m_bLaunched))
	{
		TrailEffect(true);
	}

	if (m_bArmed && m_targetId)
	{
		ArmedEffect(true);
		ArmedSound(true);
	}

	m_bPhysSetUp=false;
	m_lastTrackDivergence=Vec3(ZERO);
	m_nextTrackRecalculate=0;
}

void CSmartBomb::ArmedSound(bool enable)
{
	if (enable && m_armedSound)
	{
		m_armedSoundId = GetSoundProxy()->PlaySound(m_armedSound, Vec3(0,0,0), FORWARD_DIRECTION, FLAG_SOUND_DEFAULT_3D, eSoundSemantic_Projectile, 0, 0);
		if (m_armedSoundId != INVALID_SOUNDID)
		{
			ISound *pSound=GetSoundProxy()->GetSound(m_armedSoundId);
			if (pSound)
				pSound->SetLoopMode(true);
		}
	}
	else if (m_armedSoundId!=INVALID_SOUNDID)
	{
		GetSoundProxy()->StopSound(m_armedSoundId);
		m_armedSoundId=INVALID_SOUNDID;
	}
}