#include "StdAfx.h"
#include "FlowTrainNode.h"
#include "Player.h"
#include "Item.h"
#include "GameCVars.h"

#include <CopyProtection.h>

#define MAX_TRAIN_SPEED 8.f  //for sound
#define COACH_NUMBER_SOUND_TOGETHER 2

float CFlowTrainNode::m_playerMaxVelGround=0;

CTrainPathIterator::CTrainPathIterator()
{
	Invalidate();
}

void CTrainPathIterator::Invalidate()
{
	Segment = 0;
	LastDistance = 0;
}

CTrainPath::CTrainPath()
{
}

void CTrainPath::SetPath(const std::vector<Vec3> *path)
{
	if (!path)
	{
		Path.resize(0);
		return;
	}

	int num_path_nodes = path->size();
	Path.resize(num_path_nodes);
	float total_length = 0;
	for (int i=0; i<num_path_nodes; i++)
	{
		Path[i].Position = (*path)[i];
		if (i<num_path_nodes-1)
			Path[i].Length = (*path)[i].GetDistance((*path)[i+1]);
		else
			Path[i].Length = 0;

		Path[i].TotalLength = total_length;
		total_length += Path[i].Length;
	}
}

Vec3 CTrainPath::GetPointAlongPath(float dist, CTrainPathIterator &iterator)
{
	int size = Path.size();

	if (!size)
		return Vec3(0, 0, 0);
	if (dist <= 0)
		return Path[0].Position;
	if (dist < iterator.LastDistance)
		iterator.Invalidate();

	while (iterator.Segment+1 < size && dist > Path[iterator.Segment+1].TotalLength)
		iterator.Segment++;

	iterator.LastDistance = dist;

	if (iterator.Segment+1 == size)
		return Path[iterator.Segment].Position;

	return Vec3::CreateLerp(Path[iterator.Segment].Position, Path[iterator.Segment+1].Position, (dist - Path[iterator.Segment].TotalLength)/Path[iterator.Segment].Length);
}

CFlowTrainNode::STrainCoach::STrainCoach()
{
	m_pEntity=NULL;
	m_frontWheelBase=m_backWheelBase=-1;
	m_wheelDistance=0.f;
	m_coachOffset=0.f;
	m_distanceOnPath=0.f;
	m_pEntitySoundsProxy=NULL;
	m_runSoundID=INVALID_SOUNDID;
	m_breakSoundID=INVALID_SOUNDID;
}


int CFlowTrainNode::OnPhysicsPostStep_static(const EventPhys * pEvent)
{
	for (int i=0; i<gFlowTrainNodes.size(); ++i)
		gFlowTrainNodes[i]->OnPhysicsPostStep(pEvent);
	return 0;
}

CFlowTrainNode::CFlowTrainNode( SActivationInfo * pActInfo )
{
	m_distanceOnPath = 0;
	m_bFirstUpdate = true;
	m_speed = 0;
	m_splitSpeed = 0;
	m_coachIndex = -1;
	m_splitCoachIndex = 0;
	m_offTrainStartTime=0.f;
	m_splitDistanceOnPath=0;
	m_hornSoundID=INVALID_SOUNDID;
	m_engineStartSoundID=INVALID_SOUNDID;
	m_startBreakSoundShifted=0;
	pActInfo->pGraph->SetRegularlyUpdated( pActInfo->myID, true );
	if (gFlowTrainNodes.empty())
		gEnv->pPhysicalWorld->AddEventClient(EventPhysPostStep::id, OnPhysicsPostStep_static, 0);
	gFlowTrainNodes.push_back(this);

	ICVar* pICVar = gEnv->pConsole->GetCVar("es_UsePhysVisibilityChecks");
	if(pICVar) //prevent distance problem between coaches
		pICVar->Set(0);
}

CFlowTrainNode::~CFlowTrainNode()
{
	for (int i=0; i<gFlowTrainNodes.size(); ++i)
	{
		if (gFlowTrainNodes[i] == this)
		{
			gFlowTrainNodes.erase(gFlowTrainNodes.begin() + i);
			break;
		}
	}
	if (gFlowTrainNodes.empty())
		gEnv->pPhysicalWorld->RemoveEventClient(EventPhysPostStep::id, OnPhysicsPostStep_static, 0);

	ICVar* pICVar = gEnv->pConsole->GetCVar("es_UsePhysVisibilityChecks");
	if(pICVar)
		pICVar->Set(1);
}

IFlowNodePtr CFlowTrainNode::Clone( SActivationInfo * pActInfo )
{
	return new CFlowTrainNode(pActInfo);
}

void CFlowTrainNode::Serialize(SActivationInfo *, TSerialize ser)
{
	ser.BeginGroup("Local");
	ser.Value("m_distanceOnPath", m_distanceOnPath);
	ser.Value("m_speed", m_speed);
	ser.Value("m_offTrainStartTime", m_offTrainStartTime);
	ser.Value("m_coachIndex", m_coachIndex);
	ser.Value("m_splitCoachIndex", m_splitCoachIndex);
	ser.Value("m_splitSpeed", m_splitSpeed);
	ser.Value("m_splitDistanceOnPath", m_splitDistanceOnPath);
	ser.Value("m_startBreakSoundShifted", m_startBreakSoundShifted);
	ser.EndGroup();

	if(ser.IsReading()) //after load must do some initializing in first update
		m_bFirstUpdate = true;
}

void CFlowTrainNode::GetConfiguration( SFlowNodeConfig &config )
{
	static const SInputPortConfig in_config[] = {
		InputPortConfig<string>( "Path",_HELP("Path to move the train on") ),
		InputPortConfig<float>( "Speed",_HELP("Speed in m/s (not work with negative speed)") ),
		InputPortConfig<float>( "StartDistance",_HELP("Start distance of the last coach end (from the start of the path)") ),
		InputPortConfig<int>( "SplitCoachNumber",_HELP("If acticated, the train splits from this coach") ),
		InputPortConfig_Void( "PlayHornSound",_HELP("If acticated, train horn sound played") ),
		InputPortConfig_Void( "PlayBreakSound",_HELP("If acticated, train break sound played") ),
		{0}
	};
	static const SOutputPortConfig out_config[] = {
		OutputPortConfig<int> ("PlayerCoachIndex", _HELP("gives the coach index Player is standing on, (-1) if not on train for at least 4 seconds")),
		{0}
	};

	config.sDescription = _HELP( "Train node will move a special entity on an AIPath with speed" );
	config.pInputPorts = in_config;
	config.pOutputPorts = out_config;
	config.SetCategory(EFLN_WIP);
	config.nFlags |= EFLN_TARGET_ENTITY;
}

#define PATH_DERIVATION_TIME 0.1f

int CFlowTrainNode::OnPhysicsPostStep(const EventPhys * pEvent)
{
	FUNCTION_PROFILER(GetISystem(), PROFILE_GAME);

  if(m_bFirstUpdate)
    return 0; //after QuickLoad OnPhysicsPostStep called before ProcessEvent

	const EventPhysPostStep *pPhysPostStepEvent = (const EventPhysPostStep *)pEvent;

	IPhysicalEntity *pEventPhysEnt = pPhysPostStepEvent->pEntity;

	Vec3 posBack, posFront, posCenter;

	for (int i = m_coaches.size()-1; i >= 0; --i)
	{
		IPhysicalEntity *pPhysEnt = m_coaches[i].m_pEntity->GetPhysics();

		if (pEventPhysEnt == pPhysEnt)
		{
			float speed = (m_splitCoachIndex>0 && i>=m_splitCoachIndex) ? m_splitSpeed : m_speed;
			float distance = (m_coaches[i].m_distanceOnPath += pPhysPostStepEvent->dt * speed);
			
			if(m_splitCoachIndex>0)
			{//train splitted
				if(i==m_splitCoachIndex-1) // update m_distanceOnPath for serialization
					m_distanceOnPath=distance-m_coaches[i].m_coachOffset;
				else if(i==m_coaches.size()-1) // update m_splitDistanceOnPath for serialization
					m_splitDistanceOnPath=distance-m_coaches[i].m_coachOffset;
			}
			else
			{//train in one piece
				if(i==m_coaches.size()-1)// update m_distanceOnPath for serialization
					m_distanceOnPath=distance-m_coaches[i].m_coachOffset;
			}

			posBack  = m_path.GetPointAlongPath(distance - m_coaches[i].m_wheelDistance, m_coaches[i].m_backWheelIterator[0]);
			posFront = m_path.GetPointAlongPath(distance + m_coaches[i].m_wheelDistance, m_coaches[i].m_frontWheelIterator[0]);
			posCenter = (posBack+posFront)*0.5f;

			pe_params_pos ppos;
			ppos.pos = posCenter;
			ppos.q = Quat::CreateRotationVDir((posFront - posBack).GetNormalized());
			pPhysEnt->SetParams(&ppos, 1);

			Vec3 futurePositionBack, futurePositionFront;
			futurePositionBack  = m_path.GetPointAlongPath(distance + PATH_DERIVATION_TIME * speed - m_coaches[i].m_wheelDistance, m_coaches[i].m_backWheelIterator[1]);
			futurePositionFront = m_path.GetPointAlongPath(distance + PATH_DERIVATION_TIME * speed + m_coaches[i].m_wheelDistance, m_coaches[i].m_frontWheelIterator[1]);

			pe_action_set_velocity setVel;
			setVel.v = ((futurePositionBack+ futurePositionFront)*0.5f - posCenter) * (1.0f/PATH_DERIVATION_TIME);

			Vec3 dir=(posFront-posBack).GetNormalized();
			Vec3 future_dir=(futurePositionFront-futurePositionBack).GetNormalized();
			Vec3 w=dir.Cross(future_dir);
			float angle=cry_asinf(w.GetLength());
			setVel.w=(angle/PATH_DERIVATION_TIME)*w.GetNormalized();
			pPhysEnt->Action(&setVel, 1);		
			break;
		}
	}
	return 0;
}

bool CFlowTrainNode::PlayerIsOnaTrain()
{
	for(int i=0; i<gFlowTrainNodes.size(); ++i)
	{
		if(gFlowTrainNodes[i]->m_coachIndex>=0)
			return true;
	}
	return false;
}

void CFlowTrainNode::DiscoverTrainCoaches(IEntity *pEntity)
{
	m_coaches.resize(0);

	while (pEntity)
	{
		STrainCoach tc;

		ICharacterInstance *pCharacterInstance = pEntity->GetCharacter(0);
		if (!pCharacterInstance)
			break;
		ISkeletonPose *pSkeletonPose = pCharacterInstance->GetISkeletonPose();
		if (!pSkeletonPose)
			break;

		IPhysicalEntity* pPhysics = pEntity->GetPhysics();
		if(!pPhysics)
			break;

		tc.m_frontWheelBase = pSkeletonPose->GetJointIDByName("wheel_base1");
		QuatT qt1 = pSkeletonPose->GetRelJointByID(tc.m_frontWheelBase);
		tc.m_backWheelBase = pSkeletonPose->GetJointIDByName("wheel_base2");
		QuatT qt2 = pSkeletonPose->GetRelJointByID(tc.m_backWheelBase);
		tc.m_wheelDistance = qt1.t.GetDistance(qt2.t) * .5f;

		AABB bbox;
		pEntity->GetLocalBounds(bbox);

		tc.m_coachOffset = (bbox.max.y - bbox.min.y) * .5f;

		tc.m_pEntity = pEntity;
//    pEntity->SetTrainEntity();
//		for (int i = 0; i < pEntity->GetChildCount(); i++)
//			pEntity->GetChild(i)->SetTrainEntity();

		tc.m_pEntitySoundsProxy = (IEntitySoundProxy*) tc.m_pEntity->CreateProxy(ENTITY_PROXY_SOUND);
		assert(tc.m_pEntitySoundsProxy);

		m_coaches.push_back(tc);
		IEntityLink *pEntityLink = pEntity->GetEntityLinks();
		pEntity = pEntityLink ? gEnv->pEntitySystem->GetEntity(pEntityLink->entityId) : NULL;
	}
}


void CFlowTrainNode::InitTrainCoaches()
{
	float distance = m_splitCoachIndex>0 ? m_splitDistanceOnPath : m_distanceOnPath;

	for (int i=m_coaches.size()-1; i>=0; --i)
	{
		if(m_splitCoachIndex>0 && i==m_splitCoachIndex-1)
			distance=m_distanceOnPath; // find the first train part end distance
		
		distance += m_coaches[i].m_coachOffset;
		m_coaches[i].m_distanceOnPath=distance;			
		distance += m_coaches[i].m_coachOffset;

		IPhysicalEntity* pPhysics = m_coaches[i].m_pEntity->GetPhysics();
		pe_params_flags flags;
		flags.flagsOR = pef_monitor_poststep|pef_fixed_damping;
		pPhysics->SetParams(&flags);

		pe_params_foreign_data pfd;
		pfd.iForeignFlagsOR = PFF_ALWAYS_VISIBLE;
		pPhysics->SetParams(&pfd);

		pe_simulation_params sp;
		sp.mass = -1;
		sp.damping = sp.dampingFreefall = 0;
		pPhysics->SetParams(&sp);
	}
}

void CFlowTrainNode::AwakeCoaches() //awake the physics
{
	pe_action_awake aa; 
	aa.bAwake=1;
	for (int i=0; i<m_coaches.size(); ++i)
	{
		IPhysicalEntity* pPhysics = m_coaches[i].m_pEntity->GetPhysics();
		pPhysics->Action(&aa);
	}
}

int CFlowTrainNode::GetCoachIndexPlayerIsOn()
{
	IPhysicalEntity *pGroundCollider=NULL;
	IActor *pPlayerActor = gEnv->pGame->GetIGameFramework()->GetClientActor();

	// if player use a mounted weapon on the train, the GroundCollider check not good
	if(m_coachIndex>=0)
	{//player can catch mounted weapon on the train if he were on it before
		CItem *pCurrentItem=static_cast<CItem *>(pPlayerActor->GetCurrentItem());
		if ( pCurrentItem && pCurrentItem->IsMounted()) 
			return m_coachIndex; // give back the last m_coachIndex, it should be valid 		
	}

	IPhysicalEntity *pPhysicalEntity=pPlayerActor->GetEntity()->GetPhysics();
	if (pPhysicalEntity)
	{
		pe_status_living livStat;
		if(pPhysicalEntity->GetStatus(&livStat))
			pGroundCollider=livStat.pGroundCollider;
	}
	
	if(!pGroundCollider)
		return -1;

	for (int i=0; i<m_coaches.size(); ++i)
	{
		if(m_coaches[i].m_pEntity->GetPhysics()==pGroundCollider)
			return i;
		else
		{//attached objects
			IEntity *pCoachEntity=m_coaches[i].m_pEntity;
			for (int j = 0; j < pCoachEntity->GetChildCount(); ++j)
			{
				if(pCoachEntity->GetChild(j)->GetPhysics()==pGroundCollider)
					return i;
			}
		}
	}
	return -1;
}

int CFlowTrainNode::GetCoachIndexPlayerIsOn2()
{
	IActor *pPlayerActor = gEnv->pGame->GetIGameFramework()->GetClientActor();
	Vec3 playerPos=pPlayerActor->GetEntity()->GetWorldPos();
	for (int i=0; i<m_coaches.size(); ++i)
	{
		AABB bbox;
		m_coaches[i].m_pEntity->GetLocalBounds(bbox);
		Vec3 localpos=m_coaches[i].m_pEntity->GetWorldTM().GetInverted().TransformPoint(playerPos);
		if(bbox.min.x<=localpos.x && bbox.min.y<=localpos.y && bbox.max.x>=localpos.x && bbox.max.y>=localpos.y)
			return i;
	}
	return -1;
}


#define ENERGY_DRAIN_DURATION 10.f

void CFlowTrainNode::SuckCloakEnergyOnTrain()
{
	IActor *pPlayerActor = gEnv->pGame->GetIGameFramework()->GetClientActor();
	if (CNanoSuit* pSuit = ((CPlayer*)pPlayerActor)->GetNanoSuit())
	{
		if (const SNanoCloak *pCloak=pSuit->GetCloak())
		{
			if (pCloak->GetState()!=0)
			{
				float energy = pSuit->GetSuitEnergy();
				float frameTime = gEnv->pSystem->GetITimer()->GetFrameTime();
				float energyDrainRate = NANOSUIT_ENERGY / ENERGY_DRAIN_DURATION;
				energy -= (pSuit->GetEnergyRechargeRate() + energyDrainRate) * frameTime;
				pSuit->SetSuitEnergy(energy);
			}
		}
	}
}

float CFlowTrainNode::GetPlayerMaxVelGround()
{
	IActor *pPlayerActor = gEnv->pGame->GetIGameFramework()->GetClientActor();
	if(pPlayerActor)
	{
		IPhysicalEntity *pPhysicalEntity=pPlayerActor->GetEntity()->GetPhysics();
		if(pPhysicalEntity)
		{
			pe_player_dynamics ppd;
			pPhysicalEntity->GetParams(&ppd);
			return ppd.maxVelGround;
		}
	}
	return 0;
}

void CFlowTrainNode::SetPlayerMaxVelGround(float vel)
{
	IActor *pPlayerActor = gEnv->pGame->GetIGameFramework()->GetClientActor();
	if(pPlayerActor)
	{
		IPhysicalEntity *pPhysicalEntity=pPlayerActor->GetEntity()->GetPhysics();
		if(pPhysicalEntity)
		{
			pe_player_dynamics ppd;
			ppd.maxVelGround=vel;
			pPhysicalEntity->SetParams(&ppd);
		}
	}
}

#define SHIFT_FRAME 5

void CFlowTrainNode::StartBreakSoundShifted()
{
	if(m_startBreakSoundShifted%SHIFT_FRAME==1)
	{
		bool isPlaying=false;
		int index=m_startBreakSoundShifted/SHIFT_FRAME;
		if(m_coaches[index].m_breakSoundID!=INVALID_SOUNDID)
		{
			ISound *i_sound=m_coaches[index].m_pEntitySoundsProxy->GetSound(m_coaches[index].m_breakSoundID);
			if(i_sound && i_sound->IsPlaying())
				isPlaying=true;
		}
		if(!isPlaying)
			m_coaches[index].m_breakSoundID=m_coaches[index].m_pEntitySoundsProxy->PlaySound("sounds/vehicles_exp1:train:breaking", Vec3Constants<float>::fVec3_Zero, Vec3Constants<float>::fVec3_OneY, FLAG_SOUND_DEFAULT_3D, eSoundSemantic_Vehicle);
		if(index==m_coaches.size()-1)
			m_startBreakSoundShifted=-1;
	}
	m_startBreakSoundShifted++;
}

#define BIG_MAX_VEL_GROUND 1000000.f
#define SPLIT_ACCEL -0.5f  //splitted coaches acceleration
#define OFF_TRAIN_TIME 4.f

void CFlowTrainNode::ProcessEvent( EFlowEvent event, SActivationInfo *pActInfo )
{
	switch (event)
	{
	case eFE_Update:
		{
			if (m_bFirstUpdate)
			{
				DiscoverTrainCoaches(pActInfo->pEntity);
				InitTrainCoaches();
				AwakeCoaches();
				if(m_speed>0.f)
					StartSounds();
				if(!m_playerMaxVelGround)
					m_playerMaxVelGround=GetPlayerMaxVelGround();
				if(m_coachIndex<0)
					SetPlayerMaxVelGround(m_playerMaxVelGround);
				else
					SetPlayerMaxVelGround(BIG_MAX_VEL_GROUND);
				m_bFirstUpdate = false;
			}

			//int coachIndex=GetCoachIndexPlayerIsOn();
			int coachIndex=GetCoachIndexPlayerIsOn2();

			if(coachIndex==-1 && m_coachIndex!=-1) 
			{//before on the train but get off
				/*if(m_offTrainStartTime==0.f)
					m_offTrainStartTime = gEnv->pTimer->GetFrameStartTime();
				else
					if((gEnv->pTimer->GetFrameStartTime()-m_offTrainStartTime).GetSeconds()>OFF_TRAIN_TIME)*/
					{
						m_coachIndex=-1;
						m_offTrainStartTime=0.f;
						ActivateOutput(pActInfo, 0, m_coachIndex);
						SetPlayerMaxVelGround(m_playerMaxVelGround);
						break;
					}
			}
			else if(coachIndex!=m_coachIndex)
			{//changed coach or get on another
				m_coachIndex=coachIndex;
				ActivateOutput(pActInfo, 0, m_coachIndex);
				SetPlayerMaxVelGround(BIG_MAX_VEL_GROUND);
			}
			else if(m_offTrainStartTime!=0.f && coachIndex>-1)
				m_offTrainStartTime=0.f;// was off train but get on the same coach within time

			if(coachIndex>=0 && m_speed)//tilt cheating with cloak on train
				SuckCloakEnergyOnTrain();

			if(m_splitCoachIndex>0 && m_splitSpeed>0)
			{// slow down splitted coaches automatically
				m_splitSpeed+=SPLIT_ACCEL*gEnv->pTimer->GetFrameTime();
				if(m_splitSpeed<0)
					m_splitSpeed=0;
			}

			if(m_speed || m_splitSpeed)
			{
				SetSoundParams();
				UpdateLineSounds();
				if(m_startBreakSoundShifted)
					StartBreakSoundShifted();
			}
		}
		break;
	case eFE_Activate:
		{
			if (IsPortActive(pActInfo, IN_SPEED))
			{
				float speed=0;
#ifdef SECUROM_32
				SECUROM_MARKER_PERFORMANCE_ON(302)
				DWORD sp_result = 0;

				Sony_VM.code_ptr = SMS_INT;
				Sony_VM.std.Param4 = SPOT_CHECK_4;

				sp_result = SendMessage(SecuROM.sms_handle, SecuROM.SMS_spotcheck, NULL, (LPARAM) & Sony_VM);

				ICVar *pVar=gEnv->pConsole->GetCVar("sys_spec_Quality");

				// SECURITY CHECK AT MEDIUM
				if (sp_result == GetCurrentProcessId() || !pVar || pVar->GetRealIVal() != 2)
					speed=GetPortFloat(pActInfo, IN_SPEED);
				SECUROM_MARKER_PERFORMANCE_OFF(302)
#else
				speed=GetPortFloat(pActInfo, IN_SPEED);
#endif
				if(speed<0) //do not use negative speed
					speed=0;
				if(speed>m_speed && m_speed<0.01f)
				{
					AwakeCoaches(); //if speed is too small Physics go to sleep and callback does not work...
													//have to awake if train start again. should make more elegant...(keep awake somehow)
					StartSounds();
				}

				if(speed==0.f && m_speed>0.f)
					TrainStopSounds();

				m_speed=speed;				
			}
			else if(IsPortActive(pActInfo, IN_SPLIT_COACH))
			{
				if(!m_splitCoachIndex)
				{
					int splitCoachIndex=GetPortInt(pActInfo, IN_SPLIT_COACH);
					if(splitCoachIndex>0 && splitCoachIndex<m_coaches.size())
					{
						m_splitCoachIndex=splitCoachIndex;
						m_splitDistanceOnPath=m_distanceOnPath;
						m_distanceOnPath=m_coaches[m_splitCoachIndex-1].m_distanceOnPath-m_coaches[m_splitCoachIndex-1].m_coachOffset;
						m_splitSpeed=m_speed;
						if(!(m_splitCoachIndex&1))
							SplitLineSound(); //have to split the lineSound (there is 1 sound for 2 coaches)
					}
				}
				else
					CryWarning(VALIDATOR_MODULE_GAME, VALIDATOR_WARNING, "SplitCoachIndex has been set! Do not set it again! FlowTrainNode: %s", m_coaches[0].m_pEntity->GetName());
			}
			else if(IsPortActive(pActInfo, IN_HORN_SOUND))
			{
				bool isPlaying=false;
				if(m_hornSoundID!=INVALID_SOUNDID)
				{
					ISound *i_sound=m_coaches[0].m_pEntitySoundsProxy->GetSound(m_hornSoundID);
					if(i_sound && i_sound->IsPlaying())
						isPlaying=true;
				}
				if(!isPlaying)
					m_hornSoundID=m_coaches[0].m_pEntitySoundsProxy->PlaySound("sounds/vehicles_exp1:train:horn", Vec3Constants<float>::fVec3_Zero, Vec3Constants<float>::fVec3_OneY, FLAG_SOUND_DEFAULT_3D, eSoundSemantic_Vehicle);
			}
			else if(IsPortActive(pActInfo, IN_BREAK_SOUND) && m_speed>0.f)
				m_startBreakSoundShifted=1;
			else if(IsPortActive(pActInfo, IN_START_DISTANCE))
			{
				m_distanceOnPath = GetPortFloat(pActInfo, IN_START_DISTANCE);
				InitTrainCoaches();
			}
		}
		break;		
	case eFE_Initialize:
		m_distanceOnPath = GetPortFloat(pActInfo, IN_START_DISTANCE);
		m_speed = GetPortFloat(pActInfo, IN_SPEED);
		m_path.SetPath(gEnv->pAISystem->GetPath(GetPortString(pActInfo, IN_PATH)));
		m_splitCoachIndex = GetPortInt(pActInfo,IN_SPLIT_COACH);
		m_bFirstUpdate = true;
		StopAllSounds();
		break;
	}
}

 float CFlowTrainNode::GetSpeedSoundParam(int coachIndex)
{
	float speedParam;
	if(m_splitCoachIndex>0 && coachIndex>=m_splitCoachIndex)
		speedParam =m_splitSpeed;
	else
		speedParam = m_speed;

	speedParam=speedParam/MAX_TRAIN_SPEED;
	return speedParam>1.f ? 1.f : speedParam;
}

void CFlowTrainNode::SetSoundParams()
{
	for (int i=0; i<m_coaches.size(); ++i)
	{
		if(m_coaches[i].m_runSoundID!=INVALID_SOUNDID && m_coaches[i].m_pEntitySoundsProxy)
		{
			ISound *i_sound=m_coaches[i].m_pEntitySoundsProxy->GetSound(m_coaches[i].m_runSoundID);
			if(i_sound)
				i_sound->SetParam("speed",GetSpeedSoundParam(i), false);
			//else
				//CryLogAlways("No ISound:%d", i);
		}
	}
}

tSoundID CFlowTrainNode::PlayLineSound(int coachIndex, const char *sGroupAndSoundName, const Vec3 &vStart, const Vec3 &vEnd)
{
	IEntitySoundProxy* psoundProxy=m_coaches[coachIndex].m_pEntitySoundsProxy;
	EntityId SkipEntIDs[1];
	SkipEntIDs[0] = m_coaches[coachIndex].m_pEntity->GetId();

	tSoundID ID = INVALID_SOUNDID;

	if (!gEnv->pSoundSystem)
		return ID;

	ISound *pSound = gEnv->pSoundSystem->CreateLineSound(sGroupAndSoundName, FLAG_SOUND_DEFAULT_3D, vStart, vEnd);

	if (pSound)
	{
		pSound->SetSemantic(eSoundSemantic_Vehicle);
		pSound->SetPhysicsToBeSkipObstruction(SkipEntIDs, 1);

		if (psoundProxy->PlaySound(pSound, Vec3Constants<float>::fVec3_Zero, Vec3Constants<float>::fVec3_OneY))
			ID = pSound->GetId();
	}
	return ID;
}

void CFlowTrainNode::UpdateLineSounds()
{
	for (int i=0; i<m_coaches.size(); ++i)
	{
		if(m_coaches[i].m_runSoundID!=INVALID_SOUNDID && m_coaches[i].m_pEntitySoundsProxy)
		{
			ISound *i_sound=m_coaches[i].m_pEntitySoundsProxy->GetSound(m_coaches[i].m_runSoundID);
			if(i_sound)
			{				
				bool splitSound1=m_splitCoachIndex && !(m_splitCoachIndex&1) && (m_splitCoachIndex==i || m_splitCoachIndex==i+1);
				bool splitSound2=(splitSound1 && i==m_splitCoachIndex && m_splitCoachIndex+1==m_coaches.size()-1);
				Vec3 lineStart, lineEnd;
				if(!i || (splitSound1 && !splitSound2) ) 
				{//line sound for train engine or one split coach
					Vec3 pos=m_coaches[i].m_pEntity->GetWorldPos();
					Vec3 forwardDir=m_coaches[i].m_pEntity->GetWorldTM().GetColumn1();
					lineStart=pos-forwardDir*m_coaches[i].m_coachOffset;
					lineEnd=pos+forwardDir*m_coaches[i].m_coachOffset;
				}
				else 
				{
					Vec3 pos2;
					float halfLineLength;
					Vec3 pos1=m_coaches[i].m_pEntity->GetWorldPos();
					if(i+2==m_coaches.size()-1)
					{//line sound for two coaches and the last coach (3 together)
						pos2=m_coaches[i+2].m_pEntity->GetWorldPos();
						halfLineLength=m_coaches[i].m_coachOffset+m_coaches[i+1].m_coachOffset+m_coaches[i+2].m_coachOffset;
					}
					else
					{//line sound for two coaches
						pos2=m_coaches[i+1].m_pEntity->GetWorldPos();
						halfLineLength=m_coaches[i].m_coachOffset+m_coaches[i+1].m_coachOffset;
					}
					Vec3 forwardDir=(pos1-pos2).GetNormalizedSafe();

					lineStart=(pos1+pos2)*0.5f+forwardDir*halfLineLength;
					lineEnd=(pos1+pos2)*0.5f-forwardDir*halfLineLength;
				}

				i_sound->SetLineSpec(lineStart, lineEnd);
				//gEnv->pRenderer->GetIRenderAuxGeom()->DrawLine(lineStart+Vec3(0,0,2.f), ColorB(0,255,0,255), lineEnd+Vec3(0,0,2.f), ColorB(0,255,0,255), 2.0f);
			}
			//else
			//CryLogAlways("No ISound:%d", i);
		}
	}
}

void CFlowTrainNode::StartSounds()
{
	for (int i=0; i<m_coaches.size(); ++i)
	{
		if(m_coaches[i].m_pEntitySoundsProxy)
		{
			if(m_coaches[i].m_runSoundID != INVALID_SOUNDID)
				m_coaches[i].m_pEntitySoundsProxy->StopAllSounds();
			
			if(!i)
			{
				m_engineStartSoundID=m_coaches[0].m_pEntitySoundsProxy->PlaySound("sounds/vehicles_exp1:train:engine_start", Vec3Constants<float>::fVec3_Zero, Vec3Constants<float>::fVec3_OneY, FLAG_SOUND_DEFAULT_3D, eSoundSemantic_Vehicle);
				m_coaches[0].m_runSoundID=m_engineStartSoundID=PlayLineSound(0, "sounds/vehicles_exp1:train:engine", Vec3Constants<float>::fVec3_Zero, Vec3Constants<float>::fVec3_Zero);
			}//the LineSound line start and end parameter will be filled later in UpdateLineSound()
			else if(i&1 && i!=m_coaches.size()-1) //set sound for every second coach to prevent too much sound
				m_coaches[i].m_runSoundID=m_engineStartSoundID=PlayLineSound(i, "sounds/vehicles_exp1:train:run", Vec3Constants<float>::fVec3_Zero, Vec3Constants<float>::fVec3_Zero);

			if(m_coaches[i].m_runSoundID != INVALID_SOUNDID)
			{
				ISound *i_sound=m_coaches[i].m_pEntitySoundsProxy->GetSound(m_coaches[i].m_runSoundID);
				if(i_sound)
					i_sound->SetParam("speed",GetSpeedSoundParam(i), false);
			}
		}
	}
}

void CFlowTrainNode::SplitLineSound()
{
	if(m_coaches[m_splitCoachIndex].m_pEntitySoundsProxy && m_coaches[m_splitCoachIndex].m_runSoundID == INVALID_SOUNDID)
	{
		m_coaches[m_splitCoachIndex].m_runSoundID=m_engineStartSoundID=PlayLineSound(m_splitCoachIndex, "sounds/vehicles_exp1:train:run", Vec3Constants<float>::fVec3_Zero, Vec3Constants<float>::fVec3_Zero);
	}
}

void CFlowTrainNode::TrainStopSounds()
{
	for (int i=0; i<m_coaches.size(); ++i)
	{
		if(m_coaches[i].m_pEntitySoundsProxy && m_coaches[i].m_runSoundID != INVALID_SOUNDID)
			m_coaches[i].m_pEntitySoundsProxy->StopSound(m_coaches[i].m_runSoundID, ESoundStopMode_AtOnce);
		if(m_coaches[i].m_pEntitySoundsProxy && m_coaches[i].m_breakSoundID != INVALID_SOUNDID)
			m_coaches[i].m_pEntitySoundsProxy->StopSound(m_coaches[i].m_breakSoundID, ESoundStopMode_AtOnce);
		m_coaches[i].m_runSoundID = INVALID_SOUNDID;
		m_coaches[i].m_breakSoundID = INVALID_SOUNDID;
	}
	
	//train engine stop sound
	m_coaches[0].m_pEntitySoundsProxy->PlaySound("sounds/vehicles_exp1:train:engine_stop", Vec3Constants<float>::fVec3_Zero, Vec3Constants<float>::fVec3_OneY, FLAG_SOUND_DEFAULT_3D, eSoundSemantic_Vehicle);
}

void CFlowTrainNode::StopAllSounds()
{
	for (int i=0; i<m_coaches.size(); ++i)
	{
		if(m_coaches[i].m_pEntitySoundsProxy)
			m_coaches[i].m_pEntitySoundsProxy->StopAllSounds();
		m_coaches[i].m_runSoundID=INVALID_SOUNDID;
		m_coaches[i].m_breakSoundID=INVALID_SOUNDID;
	}
	m_hornSoundID=INVALID_SOUNDID;	
	m_engineStartSoundID=INVALID_SOUNDID;
}

void CFlowTrainNode::GetMemoryStatistics(ICrySizer * s)
{
	s->Add(*this);
}

std::vector<CFlowTrainNode *> CFlowTrainNode::gFlowTrainNodes;

REGISTER_FLOW_NODE("Crysis:Train", CFlowTrainNode);
