////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2002.
// -------------------------------------------------------------------------
//  File name:   CoarseShadowsMgr.cpp
//  Version:     v1.00
//  Created:     12/4/2010 by Tiago Sousa
//  Compilers:   Visual Studio.NET
//  Description: Coarse shadows query manager implementation
// -------------------------------------------------------------------------
//  Todo:
//		- caching scheme needs updates (eg:give higher update priority for nearby queries)
//
////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "CoarseShadowsMgr.h"

std::vector< SCoarseShadowQueryCache > CCoarseShadowManager::m_pQueriesCache;
uint32 CCoarseShadowManager::m_nCurrCacheID;

/////////////////////////////////////////////////////////////////////////

int16 CCoarseShadowManager::CreateQuery(void)
{
	if( !m_pQueries.empty() )
	{
		uint32 nSize = m_pQueries.size();
		for( uint32 q= 0; q < nSize; ++q )
		{
			// free slot available
			if( m_pQueries[q].nQueryFlags &CVQF_UNREGISTERED )
			{
				m_pQueries[q].nQueryFlags &= ~CVQF_UNREGISTERED;
				return q + 1;
			}
		}

		// Limit maximum amount of queries possible
		if( nSize >= m_nDefaultQueriesCount )
		{
			assert(0);
			return 0;
		}
	}

	// Need new slot
	SCoarseShadowQuery pNewQuery;
	m_pQueries.push_back( pNewQuery );

	return m_pQueries.size();
}

/////////////////////////////////////////////////////////////////////////

void CCoarseShadowManager::UnregisterQuery(uint16 nQueryID)
{
	assert( nQueryID >0 && nQueryID <= m_pQueries.size() );
	if( nQueryID >0 && nQueryID <= m_pQueries.size() )
		m_pQueries[nQueryID-1].nQueryFlags = CVQF_UNREGISTERED;
}

/////////////////////////////////////////////////////////////////////////

void CCoarseShadowManager::UnregisterQueries(uint16 *pQueryID, uint32 nQueryCount)
{
	// fixme
	assert( pQueryID );
	for(uint32 q = 0; q < nQueryCount; ++q)
	{
		//m_pQueries[ pQueryID[q] ].nQueryFlags |= CVQF_UNREGISTERED;
	}
}

/////////////////////////////////////////////////////////////////////////

float CCoarseShadowManager::GetShadowRatio(uint16 nQueryID) const
{
	assert( nQueryID >0 && nQueryID <= m_pQueries.size() );

	if( nQueryID >0 && nQueryID <= m_pQueries.size())
		return m_pQueries[nQueryID-1].fShadowRatio;

	return 1.0f;
}

//////////////////////////////////////////////////////////////////////////

void CCoarseShadowManager::SetQueryParams(uint16 nQueryID, const Vec3& vPos, const Vec3& vDir, float fRadius, bool bEnabled)
{
	// prevent bad usage
	assert( nQueryID >0 && nQueryID <= m_pQueries.size() );

	if( nQueryID >0 && nQueryID <= m_pQueries.size() )
	{
		SCoarseShadowQuery& pQuery = m_pQueries[nQueryID-1];
		pQuery.nQueryFlags |= CVQF_UPDATE;
		pQuery.vPos = vPos;
		pQuery.vDir = vDir;
		pQuery.fRadius = fRadius;
	}
}

/////////////////////////////////////////////////////////////////////////

void CCoarseShadowManager::Update()
{
	if( m_pQueries.empty() || !GetCVars()->e_CoarseShadowMask)
		return;

	FRAME_PROFILER( "CCoarseShadowManager::Update", GetSystem(), PROFILE_3DENGINE );

	uint32 nRecursionLevel = m_nRecursionLevel = (int)GetRenderer()->EF_Query(EFQ_RecurseLevel);
	uint32 nFrameID = GetRenderer()->GetFrameID(false);

	IPhysicalWorld *pPhysicalWorld = gEnv->pPhysicalWorld;
	bool bUseAsyncQueries = GetCVars()->e_CoarseShadowMask == 1;

	if( !m_bPhysCallbackInitialized )
	{
		pPhysicalWorld->AddEventClient(EventPhysRWIResult::id, OnRwiResult, 1);
		m_bPhysCallbackInitialized = true;
	}

	uint32 nQueriesCount = m_pQueries.size();
	for(uint32 q = 0; q < nQueriesCount; ++q)
	{
		SCoarseShadowQuery& pQuery = m_pQueries[q];
		if( (pQuery.nQueryFlags & CVQF_UPDATE) == 0 || (bUseAsyncQueries && (pQuery.nQueryFlags & CVQF_PHYSASYNCQUERY_WAITING_RESULT)) )
			continue;

		if( (nFrameID - pQuery.nLastFrameID) <= m_nDefaultFrameUpdateCount)
		{
			// First hit - return close result existing in cache
			uint32 nCacheQueriesCount = m_pQueriesCache.size();
			for( uint32 qc = 0; qc < nCacheQueriesCount; ++qc) 
			{
				SCoarseShadowQueryCache& pQueryCache = m_pQueriesCache[qc];
				if( pQueryCache.nLastFrameID && (nFrameID - pQueryCache.nLastFrameID) < m_nCacheFramesThreshold && pQuery.vPos.IsEquivalent(pQueryCache.vPos, 1.0f) ) 
				{
					pQuery.fShadowRatio = pQueryCache.fShadowRatio;
					break;
				}
			}

			continue;
		}

		Vec3 vRandDir = Vec3(cry_frand()*2.0f-1.0f, cry_frand()*2.0f-1.0f, cry_frand()*2.0f-1.0f) ;
		Vec3 vPos = pQuery.vPos + vRandDir * pQuery.fRadius;//  * 2.0f; 

		// Always check for cache hits whenever possible, since physics RayWorldIsec is extremely slow
		pQuery.nQueryFlags &= ~CVQF_CACHEHIT;
		float fCacheSampleAcc = 0.0f;
		uint32 nCacheNumSamplesAcc = 0;
		uint32 nCacheQueriesCount = m_pQueriesCache.size();
		for( uint32 qc = 0; qc < nCacheQueriesCount && nCacheNumSamplesAcc < 16; ++qc) 
		{
			SCoarseShadowQueryCache& pQueryCache = m_pQueriesCache[qc];
			if( pQueryCache.nLastFrameID && (nFrameID - pQueryCache.nLastFrameID) < m_nCacheFramesThreshold && vPos.IsEquivalent(pQueryCache.vPos, 1.0f) ) 
			{
				fCacheSampleAcc += (!pQueryCache.bHitTraced)? 1.0f : 0.0f;
				//pQuery.bHitTraced = pQueryCache.bHitTraced;
				pQuery.nQueryFlags |= CVQF_CACHEHIT;
				nCacheNumSamplesAcc++;
				//break;
			}
		}
		
		// normalize result
		if( nCacheNumSamplesAcc	)
			fCacheSampleAcc /= (float) nCacheNumSamplesAcc;

		if( !( pQuery.nQueryFlags & CVQF_CACHEHIT ))
		{
			// Cache miss - need to raycast again
			pQuery.bHitTraced = pPhysicalWorld->RayWorldIntersection(	vPos, pQuery.vDir, 
																										m_nObjectTypes|rwi_ignore_back_faces, m_nEntQueryFlags,
																										0, 1, 0, 0, 
																										bUseAsyncQueries ? (void*)&pQuery : 0, 
																										bUseAsyncQueries ? m_nPhysForeignID : 0) != 0;

			pQuery.nLastFrameID = nFrameID;
			pQuery.nQueryFlags |= CVQF_PHYSASYNCQUERY_WAITING_RESULT;
		}

		if( bUseAsyncQueries == false || pQuery.nQueryFlags & CVQF_CACHEHIT )
			pQuery.fShadowRatio += (fCacheSampleAcc - pQuery.fShadowRatio) * GetTimer()->GetFrameTime();

		pQuery.nQueryFlags &= ~CVQF_UPDATE;

		// Store in cache
		if( !( pQuery.nQueryFlags & CVQF_CACHEHIT ) && bUseAsyncQueries == false)
		{
			pQuery.nQueryFlags &= ~CVQF_PHYSASYNCQUERY_WAITING_RESULT;
			m_nCurrCacheID = (m_nCurrCacheID + 1)% m_nCacheSize;
			m_pQueriesCache[ m_nCurrCacheID ] = pQuery;
		}
	}

	// Invalidate cache after certain threshold
	for( uint32 qc = 0; qc < m_nCacheSize; ++qc) 
	{
		SCoarseShadowQueryCache& pQuery = m_pQueriesCache[qc];

		if( (nFrameID - pQuery.nLastFrameID) <= m_nCacheFramesThreshold )
			continue;

		pQuery.bHitTraced = 0;
		pQuery.nLastFrameID = 0;
	}
}

//////////////////////////////////////////////////////////////////////////

int CCoarseShadowManager::OnRwiResult(const EventPhys *pEvent)
{
	EventPhysRWIResult *pRWIResult = (EventPhysRWIResult*)pEvent;
	if( !pRWIResult )
		return 1;

	if (pRWIResult->iForeignData != m_nPhysForeignID)
		return 1;

	SCoarseShadowQuery* pQuery = (SCoarseShadowQuery*)pRWIResult->pForeignData;
	if( !pQuery )
		return 1;

	pQuery->bHitTraced = pRWIResult->nHits!=0;

	// Accumulate samples
	const float fRecipQueriesSampleCount = 1.0f / (float) m_nQueriesSampleCount;
	pQuery->fSampleAcc += (((!pQuery->bHitTraced)? 1.0f : 0.0f) - pQuery->fSampleAcc) * fRecipQueriesSampleCount; 
	
	// Smooth interpolate results
	pQuery->fShadowRatio += (pQuery->fSampleAcc  - pQuery->fShadowRatio) * GetTimer()->GetFrameTime() ;   

	pQuery->nQueryFlags &= ~CVQF_PHYSASYNCQUERY_WAITING_RESULT;

	// store in cache
	m_nCurrCacheID = (m_nCurrCacheID + 1)% m_nCacheSize; 
	m_pQueriesCache[ m_nCurrCacheID ] = *pQuery;

	return 1;
}

//////////////////////////////////////////////////////////////////////////

void CCoarseShadowManager::DrawDebug(uint32 nMode)
{
	IRenderer *pRenderer = GetRenderer();
	float fYLine=8.0f, fYStep=20.0f;

	float cColor[4]= {1,1,1,1};

	GetRenderer()->Draw2dLabel(8.0f,fYLine+=fYStep,2.0f,cColor,false,"Coarse shadow queries info:");
	GetRenderer()->Draw2dLabel(8.0f,fYLine+=fYStep,2.0f,cColor,false,"Count: %d", m_pQueries.size());

	switch( nMode )
	{
	case 1:
	case 2:
		{
			

			uint32 nSize = m_pQueries.size();
			uint32 nUpdated = 0;
			uint32 nCacheMisses = 0;
			for(uint32 q = 0; q < nSize; ++q)
			{
				SCoarseShadowQuery& pQuery = m_pQueries[q]; 
				if( (pQuery.nQueryFlags & CVQF_UPDATE) == 0 )
					continue;
				
				// display colored cache hits/misses 
				if( pQuery.nQueryFlags & CVQF_CACHEHIT)
				{
					cColor[0] = 0.0f;
					cColor[1] = 1.0f;
					cColor[2] = 0.0f;
				}
				else
				{
					cColor[0] = 1.0f;
					cColor[1] = 0.0f;
					cColor[2] = 0.0f;
					nCacheMisses++;
				}
				
				if( nMode == 1 )
				{
					// lite output
					pRenderer->DrawLabelEx( pQuery.vPos, 1.3f,cColor,true,true,"shadow ratio=%0.1f\nquery radius= %.3f",pQuery.fShadowRatio, pQuery.fRadius);
				}
				else
				{
					// extended output
					pRenderer->DrawLabelEx( pQuery.vPos, 1.3f,cColor,true,true,"shadow ratio=%0.1f\npos= %.3f %.3f %.3f\ndir= %.3f %.3f %.3f\nrad= %.3f",pQuery.fShadowRatio, 
					pQuery.vPos.x, pQuery.vPos.y, pQuery.vPos.z,
					pQuery.vDir.x, pQuery.vDir.y, pQuery.vDir.z,
					pQuery.fRadius);
				}

				nUpdated++;
			}
			
			cColor[0] = cColor[1] = cColor[2] = 1.0f;
			GetRenderer()->Draw2dLabel(8.0f,fYLine+=fYStep,2.0f,cColor,false,"Updated %d", nUpdated);
			GetRenderer()->Draw2dLabel(8.0f,fYLine+=fYStep,2.0f,cColor,false,"Cache misses %d", nCacheMisses);
		}
		break;
	case 3:
		{
			uint32 nSize = m_pQueries.size();
			for(uint32 q = 0; q < nSize; ++q)
			{
				SCoarseShadowQuery& pQuery = m_pQueries[q]; 
				if( (pQuery.nQueryFlags &CVQF_UPDATE) )
				{
					// valid
					cColor[0] = 0.0f;
					cColor[1] = 1.0f;
					cColor[2] = 1.0f;
					cColor[3] = 1.0f;
				}
				else
				{
					// invalidated cache
					cColor[0] = 0.5f;
					cColor[1] = 0.5f;
					cColor[2] = 0.5f;
					cColor[3] = 0.5f;
				}

				pRenderer->DrawLabelEx( pQuery.vPos, 1.3f,cColor,true,true,"shadow ratio=%0.1f\npos= %.3f %.3f %.3f\ndir= %.3f %.3f %.3f\nrad= %.3f",pQuery.fShadowRatio, 
					pQuery.vPos.x, pQuery.vPos.y, pQuery.vPos.z,
					pQuery.vDir.x, pQuery.vDir.y, pQuery.vDir.z,
					pQuery.fRadius);
			}
		}

		break;
	case 4:
	case 5:
		{
			uint32 nSize = m_pQueriesCache.size();
			for(uint32 q = 0; q < nSize; ++q)
			{
				SCoarseShadowQueryCache& pQueryCache = m_pQueriesCache[q];
				if( pQueryCache.nLastFrameID )
				{
					// valid
					cColor[0] = 1.0f;
					cColor[1] = 1.0f;
					cColor[2] = 0.0f;
					cColor[3] = 1.0f;
				}
				else
				{
					// invalidated cache
					cColor[0] = 0.5f;
					cColor[1] = 0.5f;
					cColor[2] = 0.5f;
					cColor[3] = 0.5f;
				}

				if( nMode == 4)
				{
					// lite
					pRenderer->DrawLabelEx( pQueryCache.vPos, 1.3f,cColor,true,true,"shadow hit= %d",pQueryCache.bHitTraced);
				}
				else
				if( nMode == 5)
				{
					// extended
					pRenderer->DrawLabelEx( pQueryCache.vPos, 1.3f,cColor,true,true,"shadow hit= %d pos= %.3f %.3f %.3f",pQueryCache.bHitTraced, pQueryCache.vPos.x, pQueryCache.vPos.y, pQueryCache.vPos.z);
				}
			}
		}

		break;

	default:
		break;
	}
}