////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2002.
// -------------------------------------------------------------------------
//  File name:   CoarseShadowsMgr.h
//  Version:     v1.00
//  Created:     12/4/2010 by Tiago Sousa
//  Compilers:   Visual Studio.NET
//  Description: Coarse shadows query manager
//	- mostly for alpha blended geometry
//	- checks for visibility on a spherical distribution along given direction
//	- workload is distributed over frames - N samples => N frames
//	- re-uses cached result from RayWorldIsec whenever possible - only updates cache entry every N frames
//
//	Todo: support for recursive rendering
//
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#ifndef COARSESHADOWSMGR_H
#define COARSESHADOWSMGR_H


enum eCoarseShadowQueryFlags
{
	CVQF_UPDATE = (1<<0),
	CVQF_PHYSASYNCQUERY_WAITING_RESULT = (1<<1),
	CVQF_UNREGISTERED = (1<<2),
	CVQF_CACHEHIT = (1<<3),
};

struct SCoarseShadowQueryCache;

//todo: taking 44 bytes per instance - decrease
struct SCoarseShadowQuery
{
public:

	SCoarseShadowQuery() : nQueryFlags(0), bHitTraced(0),nLastFrameID(0), fRadius(1.0f), fShadowRatio(0.5f), fSampleAcc(1.0f)
	{
		vPos = Vec3(ZERO);
		vDir = Vec3(ZERO);
	}

	Vec3 vPos;
	Vec3 vDir;

	float fRadius;
	float fShadowRatio;
	float fSampleAcc;

	uint32 nLastFrameID;

	uint8 nQueryFlags : 4;
	uint8 bHitTraced : 1;
};

struct SCoarseShadowQueryCache
{
public:

	SCoarseShadowQueryCache() : bHitTraced(0),nLastFrameID(0), fShadowRatio(0.5f)
	{
		vPos = Vec3(ZERO);
	}

	void operator =(const SCoarseShadowQuery &pQuery)
	{
		vPos = pQuery.vPos;
		fShadowRatio = pQuery.fShadowRatio;
		nLastFrameID = pQuery.nLastFrameID;
		bHitTraced = pQuery.bHitTraced;
	}

	Vec3 vPos;
	float fShadowRatio;
	uint32 nLastFrameID;
	uint8 bHitTraced : 1;
};

class CCoarseShadowManager: public Cry3DEngineBase
{
public:

	CCoarseShadowManager(): m_bPhysCallbackInitialized(false)
	{
		m_pQueries.reserve(m_nDefaultQueriesCount);
		m_pQueriesCache.resize( m_nCacheSize );

		m_nCurrCacheID = 0;
	}

	~CCoarseShadowManager()
	{

	}

	// Per-frame update
	void Update();

	// Creates/Registers a new visibility query - returns query ID
	int16 CreateQuery(void);

	// Set shadow query parameters
	void SetQueryParams(uint16 nQueryID, const Vec3& vPos, const Vec3& vDir, float fRadius, bool bEnabled = true);

	// Unregisters shadow query 
	void UnregisterQuery(uint16 nQueryID);

	// Unregisters list of shadow queries
	void UnregisterQueries(uint16 *pQueryID, uint32 nQueryCount);

	// Return shadow radio for a query ID
	float GetShadowRatio(uint16 nQueryID) const;

	// Draw debug information 
	void DrawDebug(uint32 nMode);

private:

	// Reset queries list
	void Reset()
	{
		m_pQueries.resize( 0 );
	}

	// Check asynchronous physics queries
	static int OnRwiResult(const EventPhys *pEvent);

private:

	static const uint16 m_nDefaultQueriesCount = 512;	
	static const uint8 m_nDefaultFrameUpdateCount = 2;
	static const uint8 m_nQueriesSampleCount = 9;

	static const uint16 m_nCacheSize = 256;
	static const uint32 m_nCacheFramesThreshold = 64;	// keep cache for 128 frames (todo: test using time instead)

	static const uint32 m_nPhysForeignID = PHYS_FOREIGN_ID_USER + 4; // phys ID (we should maintain a list somewhere regarding physics ids)
	static const uint32 m_nEntQueryFlags = rwi_any_hit|rwi_stop_at_pierceable|rwi_queue;
	static const uint32 m_nObjectTypes = ent_terrain|ent_static;

	std::vector< SCoarseShadowQuery > m_pQueries;

	static std::vector< SCoarseShadowQueryCache > m_pQueriesCache;
	static uint32 m_nCurrCacheID;
	uint32 m_nRecursionLevel;

	bool m_bPhysCallbackInitialized;

};

#endif 
