////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2002.
// -------------------------------------------------------------------------
//  File name:   statobjman.cpp
//  Version:     v1.00
//  Created:     28/5/2001 by Vladimir Kajalin
//  Compilers:   Visual Studio.NET
//  Description: Loading trees, buildings, ragister/unregister entities for rendering
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"

#include "StatObj.h"
#include "ObjMan.h"
#include "VisAreas.h"
#include "terrain_sector.h"
#include "CullBuffer.h"
#include "3dEngine.h"
#include "IndexedMesh.h"
#include "WaterVolumes.h"
#include "Brush.h"
#include "Vegetation.h"
#include "terrain.h"
#include "ObjectsTree.h"
#include "VoxMan.h"
#include "Material.h"
#include <IResourceManager.h>

#define BRUSH_LIST_FILE "brushlist.txt"
#define CGF_LEVEL_CACHE_PAK "cgf.pak"
#define DEFAULT_CGF_NAME "Shaders\\EngineAssets\\Objects\\Default.cgf"


//////////////////////////////////////////////////////////////////////////
IStatObj * CObjManager::GetStaticObjectByTypeID(int nTypeID)
{
  if(nTypeID>=0 && nTypeID<(int)m_lstStaticTypes.size())
    return m_lstStaticTypes[nTypeID].pStatObj;

  return 0;
}

IStatObj * CObjManager::FindStaticObjectByFilename(const char * filename)
{
	return stl::find_in_map(m_nameToObjectMap,CONST_TEMP_STRING(filename),NULL);
}

/*
void CObjManager::UnregisterCGFFromTypeTables(CStatObj * pStatObj)
{
	for(int i=0; i<m_lstStaticTypes.size(); i++)
	{
		if( m_lstStaticTypes[i].GetStatObj() == pStatObj )
    {
      m_lstStaticTypes[i].pStatObj = NULL;
			memset(&m_lstStaticTypes[i], 0, sizeof(m_lstStaticTypes[i]));
    }
	}
}
*/

void CObjManager::UnloadVegetationModels(bool bDeleteAll)
{
  //for(int i=0; i<m_lstStaticTypes.size(); i++)
  //{
  //	if( m_lstStaticTypes[i].GetStatObj() )
  //	{
  //		m_lstStaticTypes[i].GetStatObj()->Release();
  //		memset(&m_lstStaticTypes[i], 0, sizeof(m_lstStaticTypes[i]));
  //	}
  //}

	if (bDeleteAll)
	{
		m_lstStaticTypes.clear();
	}
}

//////////////////////////////////////////////////////////////////////////
void CObjManager::UnloadObjects(bool bDeleteAll)
{
  UnloadVegetationModels(bDeleteAll);

	CleanStreamingData();

	if (m_REFarTreeSprites)
		m_REFarTreeSprites->Release(true);
	m_REFarTreeSprites = 0;

	if(m_pRMBox)
	{
		GetRenderer()->DeleteRenderMesh(m_pRMBox);
		m_pRMBox = 0;
	}
	m_decalsToPrecreate.resize( 0 );


	// Clear all objects that are in the garbage collector.
	ClearStatObjGarbage();

	m_bGarbageCollectionEnabled = false;

	if (bDeleteAll)
	{
		m_lockedObjects.clear(); // Lock/Unlock resources will not work with this.

		// Release default stat obj.
		m_pDefaultCGF = 0;

		m_nameToObjectMap.clear();
		m_lstLoadedObjects.clear();

		int nNumLeaks = 0;
		std::vector<CStatObj*> garbage;
		for (CStatObj *pStatObj = CStatObj::get_intrusive_list_root(); pStatObj; pStatObj = pStatObj->m_next_intrusive)
		{
			garbage.push_back(pStatObj);

#if !defined(_RELEASE) && !defined(PS3) && !defined(XENON)
			if(!pStatObj->IsDefaultObject())
			{
				nNumLeaks++;
				Warning("StatObj not deleted: %s (%s)  RefCount: %d", pStatObj->m_szFileName.c_str(), pStatObj->m_szGeomName.c_str(),pStatObj->m_nUsers ) ;
			}
#endif //_RELEASE
		}

#ifndef _RELEASE
		// deleting leaked objects
		if (nNumLeaks > 0)
		{
			Warning("CObjManager::CheckObjectLeaks: %d object(s) found in memory", nNumLeaks );
		}
#endif //_RELEASE

		for (int i = 0,num = (int)garbage.size(); i < num; i++)
		{
			CStatObj *pStatObj = garbage[i];
			pStatObj->ShutDown();
		}
		for (int i = 0,num = (int)garbage.size(); i < num; i++)
		{
			CStatObj *pStatObj = garbage[i];
			delete pStatObj;
		}
	}
	m_bGarbageCollectionEnabled = true;
}

//////////////////////////////////////////////////////////////////////////
void CObjManager::CleanStreamingData()
{
	m_arrUpdateStreamingPrioriryStack.clear();
	m_arrStreamableObjects.clear();

	m_arrStreamableToRelease.clear();
	m_arrStreamableToLoad.clear();
	m_arrStreamableToDelete.clear();
}

//////////////////////////////////////////////////////////////////////////
// class for asyncronous preloading of level CGF's
//////////////////////////////////////////////////////////////////////////
struct CLevelStatObjLoader : public IStreamCallback, public Cry3DEngineBase
{
  int m_nTasksNum;

  CLevelStatObjLoader()
  {
    m_nTasksNum = 0;
  }

  void StartStreaming( const char * pFileName )
  {
    m_nTasksNum++;

    // request the file
    StreamReadParams params;
    params.dwUserData = 0;
    params.nSize = 0;
    params.pBuffer = NULL;
    params.nLoadTime = 0;
    params.nMaxLoadTime = 0;
    GetSystem()->GetStreamEngine()->StartRead(eStreamTaskTypeGeometry, pFileName, this, &params);
  }

  virtual void StreamOnComplete (IReadStream* pStream, unsigned nError)
  {
    if(!nError)
    {
      string szName = pStream->GetName();
      // remove game folder from path
      const char * szInGameName = strstr(szName,"\\");
      // load CGF from memory
      GetObjManager()->LoadStatObj( szInGameName+1, NULL, NULL, true, 0, pStream->GetBuffer(), pStream->GetSize() );
    }

    m_nTasksNum--;
  }
};

//////////////////////////////////////////////////////////////////////////
// Preload in efficient way all CGF's used in level
//////////////////////////////////////////////////////////////////////////
void CObjManager::PreloadLevelObjects()
{
  LOADING_TIME_PROFILE_SECTION;

  PrintMessage("Starting loading level CGF's");

  float fStartTime = GetCurAsyncTimeSec();

	bool bCgfCacheExist = GetISystem()->GetIResourceManager()->LoadLevelCachePak( CGF_LEVEL_CACHE_PAK,"" );
	IResourceList *pResList = GetISystem()->GetIResourceManager()->GetLevelResourceList();

  // Construct streamer object
  CLevelStatObjLoader cgfStreamer;

  int nCgfCounter=0;
	int nInLevelCacheCount = 0;

	bool bVerboseLogging = GetCVars()->e_StatObjPreload > 1;

  // Request objects loading from Streaming System.
	CryPathString cgfFilename;
  const char* pCgfName = pResList->GetFirst();
  while(pCgfName)
  {
    if(strstr(pCgfName,".cgf"))
    {
			const char *sLodName = strstr(pCgfName,"_lod");
			if (sLodName && (sLodName[4] >= '0' && sLodName[4] <= '9'))
			{
				// Ignore Lod files.
				pCgfName = pResList->GetNext();
				continue;
			}

			cgfFilename = pCgfName;
			bool bFromLevelCache = false;

			/*
			if (bCgfCacheExist)
			{
				CryPathString cacheFilename = "level_cache/cgf/";
				cacheFilename += pCgfName;
				if (gEnv->pCryPak->IsFileExist( cacheFilename.c_str(),ICryPak::eFileLocation_InPak ))
				{
					cgfFilename = cacheFilename;
					nInLevelCacheCount++;
					bFromLevelCache = true;
				}
			}
			*/

			if (bVerboseLogging)
			{
				CryLog( "%s",cgfFilename.c_str() );
			}
			CStatObj *pStatObj = GetObjManager()->LoadStatObj(cgfFilename.c_str(),NULL,0,true,0);
			if (pStatObj)
			{
				if (pStatObj->m_bMeshStrippedCGF)
				{
					nInLevelCacheCount++;
				}
			}
      //cgfStreamer.StartStreaming(cgfFilename.c_str());
      nCgfCounter++;
    }

    pCgfName = pResList->GetNext();
  }

	//////////////////////////////////////////////////////////////////////////
	// Enumerate all .CGF inside level from the "brushlist.txt" file.
	{
		string brushListFilename = Get3DEngine()->GetLevelFilePath( BRUSH_LIST_FILE );
		CCryFile file;
		if (file.Open( brushListFilename.c_str(),"rb" ) && file.GetLength() > 0)
		{
			int nFileLength = file.GetLength();
			char *buf = new char[nFileLength+1];
			buf[nFileLength] = 0; // Null terminate
			file.ReadRaw( buf,nFileLength );

			// Parse file, every line in a file represents a resource filename.
			char seps[] = "\r\n";
			char *token = strtok( buf, seps );
			while (token != NULL)
			{
				int nAliasLen = sizeof("%level%")-1;
				if (strncmp(token,"%level%",nAliasLen) == 0)
				{
					cgfFilename = Get3DEngine()->GetLevelFilePath(token+nAliasLen);
				}
				else
				{
					cgfFilename = token;
				}

				if (bVerboseLogging)
				{
					CryLog( "%s",cgfFilename.c_str() );
				}
				// Do not use streaming for the Brushes from level.pak.
				GetObjManager()->LoadStatObj(cgfFilename.c_str(),NULL,0,false,0);
				//cgfStreamer.StartStreaming(cgfFilename.c_str());
				nCgfCounter++;
				
				token = strtok( NULL, seps );
			}
			delete []buf;
		}
	}
	//////////////////////////////////////////////////////////////////////////


  PrintMessage("Finished requesting level CGF's: %d objects in %.1f sec", nCgfCounter, GetCurAsyncTimeSec()-fStartTime);

  // Continue updating streaming system until all CGF's are loaded
  while(cgfStreamer.m_nTasksNum>0)
  {
    {
      LOADING_TIME_PROFILE_SECTION_NAMED("CObjManager::PreloadLevelObjects_StreamEngine_Wait");
      CrySleep(20);
    }

    {
      LOADING_TIME_PROFILE_SECTION_NAMED("CObjManager::PreloadLevelObjects_StreamEngine_Update");
      GetSystem()->GetStreamEngine()->SuspendCallbackTimeQuota();
      GetSystem()->GetStreamEngine()->Update();
      GetSystem()->GetStreamEngine()->ResumeCallbackTimeQuota();
    }
  }

	GetISystem()->GetIResourceManager()->UnloadLevelCachePak( CGF_LEVEL_CACHE_PAK );

	float dt = GetCurAsyncTimeSec()-fStartTime;
  PrintMessage("Finished loading level CGF's: %d objects loaded (%d from LevelCache) in %.1f sec", nCgfCounter,nInLevelCacheCount, dt);
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////
// Create / delete object
//////////////////////////////////////////////////////////////////////////////////////////////////////////

CStatObj * CObjManager::LoadStatObj( const char *__szFileName,const char *_szGeomName,IStatObj::SSubObject **ppSubObject,bool bUseStreaming,unsigned long nLoadingFlags, const void * pData, int nDataSize )
{	
	if (!m_pDefaultCGF && strcmp(__szFileName,DEFAULT_CGF_NAME) != 0)
	{
		// Load default object if not yet loaded.
		const char *sDefaulObjFilename = DEFAULT_CGF_NAME;
		// prepare default object
		m_pDefaultCGF = LoadStatObj(sDefaulObjFilename, NULL, NULL, false);
		if(!m_pDefaultCGF)
		{
			Error("CObjManager::LoadStatObj: Default object not found (%s)",sDefaulObjFilename);
			m_pDefaultCGF = new CStatObj();
		}
		m_pDefaultCGF->m_bDefaultObject = true;
	}
	
	LOADING_TIME_PROFILE_SECTION;
	MEMSTAT_CONTEXT(EMemStatContextTypes::MSC_Other, 0, "Static Geometry" );

	if (ppSubObject)
		*ppSubObject = NULL;

	if(!strcmp(__szFileName,"NOFILE"))
	{ // make empty object to be filled from outside
		CStatObj * pObject = new CStatObj();
		m_lstLoadedObjects.insert(pObject);
		return pObject;
	}

  // Normalize file name
	char sFilename[_MAX_PATH];

	//////////////////////////////////////////////////////////////////////////
	// Remap %level% alias if needed an unify filename
	{
		int nAliasNameLen = sizeof("%level%")-1;
		if (strncmp(__szFileName,"%level%",nAliasNameLen) == 0)
		{
			strcpy( sFilename,Get3DEngine()->GetLevelFilePath(__szFileName+nAliasNameLen) );
		}
		else
			strcpy( sFilename,__szFileName );
		std::replace( sFilename,sFilename+strlen(sFilename),'\\','/' ); // To Unix Path
	}
	//////////////////////////////////////////////////////////////////////////

	bool bForceBreakable = strstr(sFilename,"breakable")!=0;
	if (_szGeomName && !strcmp(_szGeomName,"#ForceBreakable"))
	{
		bForceBreakable = true;
		_szGeomName = 0;
	}

	// Try to find already loaded object
	CStatObj *pObject = 0;

	int flagCloth = 0;
	if (_szGeomName && !strcmp(_szGeomName,"cloth"))
		_szGeomName=0, flagCloth=STATIC_OBJECT_DYNAMIC;
	//else
	{
		pObject = stl::find_in_map( m_nameToObjectMap,CONST_TEMP_STRING(sFilename),NULL );
		if (pObject)
		{
      assert(!pData);
      if(!pObject->m_bLodsLoaded && !pData)
      {
        pObject->LoadLowLODs(bUseStreaming,nLoadingFlags);
        pObject->m_bLodsLoaded = true;
      }

			if(_szGeomName && _szGeomName[0])
			{
				// Return SubObject.
				CStatObj::SSubObject *pSubObject = pObject->FindSubObject( _szGeomName );
				if (!pSubObject || !pSubObject->pStatObj)
					return 0;
				if (pSubObject->pStatObj)
				{
					if (ppSubObject)
						*ppSubObject = pSubObject;
					return (CStatObj*)pSubObject->pStatObj;
				}
			}
			return pObject;
		}
	}

  { // loading screen update
/*    static float fLastTime = 0;

    float fTime = GetTimer()->GetAsyncCurTime();

    if( fabs(fTime - fLastTime) > 0.5f )
    {
      GetLog()->UpdateLoadingScreen(0);
      fLastTime = fTime;
    }*/
  }

	// Load new CGF
	pObject = new CStatObj();
	pObject->m_nFlags |= flagCloth;

  bUseStreaming &= (GetCVars()->e_StreamCgf != 0);

  if(bUseStreaming)
    pObject->m_bUseStreaming = true;
	if (bForceBreakable)
		nLoadingFlags |= IStatObj::ELoadingFlagsForceBreakable;

  if (!pObject->LoadCGF( sFilename, strstr(sFilename, "_lod") != NULL, nLoadingFlags, pData, nDataSize ))
  { 
    // object not found
    // if geom name is specified - just return 0
    if(_szGeomName && _szGeomName[0]) 
    {
      delete pObject; 
      return 0;
    }

    delete pObject; 

    return m_pDefaultCGF;
  }

  // now try to load lods
  if(!pData)
  {
    pObject->LoadLowLODs(bUseStreaming,nLoadingFlags);
    pObject->m_bLodsLoaded = true;
  }

  while(CStatObj::m_arrStatObjForRenderMeshDelete.Count())
  {
    GetRenderer()->DeleteRenderMesh(CStatObj::m_arrStatObjForRenderMeshDelete[0]->m_pRenderMesh);
    CStatObj::m_arrStatObjForRenderMeshDelete[0]->m_pRenderMesh = NULL;
    CStatObj::m_arrStatObjForRenderMeshDelete.Delete((int)0);
  }

	m_lstLoadedObjects.insert(pObject);
	m_nameToObjectMap[pObject->m_szFileName] = pObject;

	if(_szGeomName && _szGeomName[0])
	{
		// Return SubObject.
		CStatObj::SSubObject *pSubObject = pObject->FindSubObject( _szGeomName );
		if (!pSubObject || !pSubObject->pStatObj)
			return 0;
		if (pSubObject->pStatObj)
		{
			if (ppSubObject)
				*ppSubObject = pSubObject;
			return (CStatObj*)pSubObject->pStatObj;
		}
	}

	return pObject;
}

//////////////////////////////////////////////////////////////////////////
bool CObjManager::InternalDeleteObject( CStatObj *pObject )
{
	assert(pObject);

	if (!m_bLockCGFResources && !IsResourceLocked(pObject->m_szFileName))
	{
		LoadedObjects::iterator it = m_lstLoadedObjects.find(pObject);
		if (it != m_lstLoadedObjects.end())
		{
			m_lstLoadedObjects.erase(it);
			m_nameToObjectMap.erase( pObject->m_szFileName );
		}
		else
		{
			//Warning( "CObjManager::ReleaseObject called on object not loaded in ObjectManager %s",pObject->m_szFileName.c_str() );
			//return false;
		}

		if (!pObject->m_bForInternalUse && pObject->m_szFileName[0])
		{
			if (!pObject->GetCloneSourceObject() && !pObject->GetParentObject())
			{
				//CryLog( "Unloading Object: %s  %s Ptr=0x%x",pObject->m_szFileName.c_str(), pObject->m_szGeomName.c_str(), (uint32)(UINT_PTR)pObject );
			}
		}

		delete pObject;
		return true;
	}
	else if (m_bLockCGFResources)
	{
		// Put them to locked stat obj list.
		stl::push_back_unique( m_lockedObjects,pObject );
	}

	return false;
}

CObjManager::CObjManager():
	m_pDefaultCGF (NULL),
	m_decalsToPrecreate(),
	m_bNeedProcessObjectsStreaming_Finish(false)
{
	m_pObjManager = this;
  m_fZoomFactor=1;

  m_vSkyColor.Set(0,0,0);
	m_fSunSkyRel = 0;
  m_vSunColor.Set(0,0,0);
	m_fILMul = 1.0f;
	m_fSkyBrightMul = 0.3f;
  m_fSSAOAmount = 1.f;
	m_fGIAmount = 1.f;
  
	m_fMaxViewDistanceScale=1.f;
	m_fGSMMaxDistance = 0;
	m_bLockCGFResources = false;

	if( GetRenderer()->GetFeatures() & RFT_OCCLUSIONTEST )
		m_pShaderOcclusionQuery = GetRenderer()->EF_LoadShader("OcclusionTest");
	else
		m_pShaderOcclusionQuery = 0;

	m_pRMBox = NULL;

	m_decalsToPrecreate.reserve( 128 );
	m_fOcclTimeRatio = 1.f;
	m_REFarTreeSprites = 0;
}

// make unit box for occlusion test
void CObjManager::MakeUnitCube()
{
	if (m_pRMBox)
		return;

	SVF_P3F_C4B_T2F arrVerts[8];
	arrVerts[0].xyz = Vec3(0,0,0);
	arrVerts[1].xyz = Vec3(1,0,0);
	arrVerts[2].xyz = Vec3(0,0,1);
	arrVerts[3].xyz = Vec3(1,0,1);
	arrVerts[4].xyz = Vec3(0,1,0);
	arrVerts[5].xyz = Vec3(1,1,0);
	arrVerts[6].xyz = Vec3(0,1,1);
	arrVerts[7].xyz = Vec3(1,1,1);

	//		6-------7
	//	 /		   /|
	//	2-------3	|
	//	|	 			|	|
	//	|	4			| 5
	//	|				|/
	//	0-------1

	uint16 arrIndices[36];
	typedef Vec3s_tpl<uint16> Vec3us;
	int i=0;
	// front + back
	memcpy(&arrIndices[i], &Vec3us(1,0,2)[0], sizeof(Vec3us)); i+=3;
	memcpy(&arrIndices[i], &Vec3us(2,3,1)[0], sizeof(Vec3us)); i+=3;
	memcpy(&arrIndices[i], &Vec3us(5,6,4)[0], sizeof(Vec3us)); i+=3;
	memcpy(&arrIndices[i], &Vec3us(5,7,6)[0], sizeof(Vec3us)); i+=3;
	
	// left + right
	memcpy(&arrIndices[i], &Vec3us(0,6,2)[0], sizeof(Vec3us)); i+=3;
	memcpy(&arrIndices[i], &Vec3us(0,4,6)[0], sizeof(Vec3us)); i+=3;
	memcpy(&arrIndices[i], &Vec3us(1,3,7)[0], sizeof(Vec3us)); i+=3;
	memcpy(&arrIndices[i], &Vec3us(1,7,5)[0], sizeof(Vec3us)); i+=3;

	// top + bottom
	memcpy(&arrIndices[i], &Vec3us(3,2,6)[0], sizeof(Vec3us)); i+=3;
	memcpy(&arrIndices[i], &Vec3us(6,7,3)[0], sizeof(Vec3us)); i+=3;
	memcpy(&arrIndices[i], &Vec3us(1,4,0)[0], sizeof(Vec3us)); i+=3;
	memcpy(&arrIndices[i], &Vec3us(1,5,4)[0], sizeof(Vec3us)); i+=3;

	m_pRMBox = GetRenderer()->CreateRenderMeshInitialized(arrVerts, 
                                                        8, 
                                                        eVF_P3F_C4B_T2F, 
                                                        arrIndices, 
                                                        i, 
                                                        R_PRIMV_TRIANGLES, 
                                                        "OcclusionQueryCube","OcclusionQueryCube", 
                                                        eRMT_Static);

	m_pRMBox->SetChunk(NULL, 0, 8, 0, i, 1.0f, 0, true);

	m_bGarbageCollectionEnabled = true;
}

CObjManager::~CObjManager()
{
	// free default object
	m_pDefaultCGF=0;

	// free brushes
/*  assert(!m_lstBrushContainer.Count());
	for(int i=0; i<m_lstBrushContainer.Count(); i++)
	{
    if(m_lstBrushContainer[i]->GetEntityStatObj())
      ReleaseObject((CStatObj*)m_lstBrushContainer[i]->GetEntityStatObj());
		delete m_lstBrushContainer[i];
	}
	m_lstBrushContainer.Reset();
*/
  UnloadObjects(true);
}


// mostly xy size
float CObjManager::GetXYRadius(int type)
{
  if(((int)m_lstStaticTypes.size()<=type || !m_lstStaticTypes[type].pStatObj))
    return 0;

  Vec3 vSize = m_lstStaticTypes[type].pStatObj->GetBoxMax() - m_lstStaticTypes[type].pStatObj->GetBoxMin();
  vSize.z *= 0.5f;

  float fRadius = m_lstStaticTypes[type].pStatObj->GetRadius();
  float fXYRadius = vSize.GetLength()*0.5f;

  return fXYRadius;
}

bool CObjManager::GetStaticObjectBBox(int nType, Vec3 & vBoxMin, Vec3 & vBoxMax)
{
  if(((int)m_lstStaticTypes.size()<=nType || !m_lstStaticTypes[nType].pStatObj))
    return 0;

  vBoxMin = m_lstStaticTypes[nType].pStatObj->GetBoxMin();
  vBoxMax = m_lstStaticTypes[nType].pStatObj->GetBoxMax();
  
  return true;
}


void CObjManager::AddDecalToRenderer( float fDistance,
																			IMaterial* pMat,
																			const int nDynLMask,
																			const uint8 sortPrio,
																			Vec3 right,
																			Vec3 up,
																			const UCol& ucResCol,
																			const EParticleBlendType eBlendType,
																			const Vec3& vAmbientColor,
																			Vec3 vPos,
																			const int nAfterWater,
																			CVegetation* pVegetation )
{
	FUNCTION_PROFILER_3DENGINE;

  Vec2 vFinalBending = (pVegetation && pVegetation->m_pRNTmpData) ? pVegetation->m_pRNTmpData->userData.m_Bending.m_vBending : Vec2(0,0);

	if(pVegetation && !!vFinalBending)
	{ // transfer decal into object space
		Matrix34A objMat;
		IStatObj * pEntObject = pVegetation->GetEntityStatObj(0, 0, &objMat);
		assert(pEntObject);
		if(pEntObject)
		{
			objMat.Invert();
			vPos = objMat.TransformPoint(vPos);
			right = objMat.TransformVector(right);
			up = objMat.TransformVector(up);
		}
	}

	// calculate render state
	uint32 nRenderState=0;
	switch(eBlendType)
	{
	case ParticleBlendType_AlphaBased:
		nRenderState = OS_ALPHA_BLEND;
		break;
	case ParticleBlendType_ColorBased:
		nRenderState = OS_COLOR_BLEND;
		break;
	case ParticleBlendType_Additive:
		nRenderState = OS_ADD_BLEND;
		break;
	}

	// repeated objects are free imedeately in renderer
	CRenderObject* pOb( GetIdentityCRenderObject() );		
  if (!pOb)
    return;

	if (pVegetation && !!vFinalBending)
	{
		pVegetation->GetEntityStatObj(0, 0, &pOb->m_II.m_Matrix);
		pOb->m_ObjFlags	|= FOB_TRANS_MASK;
		CStatObj * pBody = m_lstStaticTypes[pVegetation->m_nObjectTypeID].GetStatObj();
		assert(pBody);
		if(pVegetation && pBody && !!vFinalBending)
			pBody->SetupBending(pOb, vFinalBending, pBody->GetRadiusVert(), pBody->GetRadiusHors());
	}

	// prepare render object
	pOb->m_fDistance = fDistance;
	pOb->m_DynLMMask[m_nRenderThreadListID] = nDynLMask;
	pOb->m_nTextureID = -1;
	//pOb->m_nTextureID1 = -1;
	//pOb->m_RenderObjectColor = ColorF( (float) ucResCol.bcolor[0] / 255.f, (float) ucResCol.bcolor[1] / 255.f, (float) ucResCol.bcolor[2] / 255.f, (float) ucResCol.bcolor[3] / 255.f );
  pOb->m_fAlpha = (float)ucResCol.bcolor[3] / 255.f;
	pOb->m_II.m_AmbColor = vAmbientColor;
	pOb->m_RState = nRenderState;
	pOb->m_fSort = 0; 
	pOb->m_ObjFlags |= FOB_DECAL | FOB_NO_Z_PASS | FOB_INSHADOW;
	//float radius( (right + up ).GetLength() );
	//pOb->m_pShadowCasters = GetShadowFrustumsList( 0, AABB( vPos - Vec3( radius, radius, radius ), vPos + Vec3( radius, radius, radius ) ), 0.0f );
	pOb->m_nSort = sortPrio;

	SVF_P3F_C4B_T2F pVerts[4];
	SPipTangents pTangents[4];
	uint16 pIndices[6];

	// TODO: determine whether this is a decal on opaque or transparent geometry 
	// (put it in the respective renderlist for correct shadowing)
	// fill general vertex data
	pVerts[0].xyz = (-right-up) + vPos;
	pVerts[0].st = Vec2(0, 1); 
  pVerts[0].color.dcolor = ~0;

	pVerts[1].xyz = ( right-up) + vPos;
	pVerts[1].st = Vec2(1, 1); 
  pVerts[1].color.dcolor = ~0;

  pVerts[2].xyz = ( right+up) + vPos;
	pVerts[2].st = Vec2(1, 0); 
  pVerts[2].color.dcolor = ~0;

  pVerts[3].xyz = (-right+up) + vPos;
	pVerts[3].st = Vec2(0, 0); 
  pVerts[3].color.dcolor = ~0;

	// prepare tangent space (tangent, binormal) and fill it in
	Vec3 rightUnit( right.GetNormalized() );
	Vec3 upUnit( up.GetNormalized() );
	Vec4sf binormal( tPackF2B( -upUnit.x ), tPackF2B( -upUnit.y ), tPackF2B( -upUnit.z ), tPackF2B( 1 ) );
	Vec4sf tangent( tPackF2B( rightUnit.x ), tPackF2B( rightUnit.y ), tPackF2B( rightUnit.z ), tPackF2B( -1 ) );				
	pTangents[0].Tangent = tangent;
	pTangents[0].Binormal = binormal;
	pTangents[1].Tangent = tangent;
	pTangents[1].Binormal = binormal;
	pTangents[2].Tangent = tangent;
	pTangents[2].Binormal = binormal;
	pTangents[3].Tangent = tangent;
	pTangents[3].Binormal = binormal;

	// fill decals topology (two triangles)
	pIndices[ 0 ] = 0;
	pIndices[ 1 ] = 1;
	pIndices[ 2 ] = 2;	

	pIndices[ 3 ] = 0;
	pIndices[ 4 ] = 2;
	pIndices[ 5 ] = 3;

  GetRenderer()->EF_AddPolygonToScene( pMat->GetShaderItem(), 4, pVerts, pTangents, pOb, pIndices, 6, nAfterWater );
}


void CObjManager::GetMemoryUsage(class ICrySizer * pSizer) const
{	
  {
    SIZER_COMPONENT_NAME(pSizer, "Self");
	  pSizer->AddObject(this, sizeof(*this));
  }
  
  {
    SIZER_COMPONENT_NAME(pSizer, "StaticTypes");
	  pSizer->AddObject(m_lstStaticTypes);
  }

  for(int i=0; i<MAX_RECURSION_LEVELS; i++)
	{
    SIZER_COMPONENT_NAME(pSizer, "VegetationSprites");
    for(int t=0; t<nThreadsNum; t++)
		{
			pSizer->AddObject(m_arrVegetationSprites[i][t]);      
		}
	}  	
  
	{
		SIZER_COMPONENT_NAME(pSizer, "CMesh");
		pSizer->AddObject(CMesh::GetCounter());
	}      
  
	{
		SIZER_COMPONENT_NAME(pSizer, "VoxelObject");
		pSizer->AddObject(CVoxelObject::GetCounter());
	}
  		
	{
		SIZER_COMPONENT_NAME(pSizer, "StatObj");
		for (CStatObj *pStatObj = CStatObj::get_intrusive_list_root(); pStatObj; pStatObj = pStatObj->m_next_intrusive)
		{
			pStatObj->GetMemoryUsage(pSizer);
		}
	}

  {
    SIZER_COMPONENT_NAME(pSizer, "EmptyNodes");
    pSizer->AddObject(COctreeNode::m_arrEmptyNodes);
  }
}

void CObjManager::ReregisterEntitiesInArea(Vec3 vBoxMin, Vec3 vBoxMax)
{
	PodArray<SRNInfo> lstEntitiesInArea;

  AABB vBoxAABB(vBoxMin, vBoxMax);

	Get3DEngine()->MoveObjectsIntoListGlobal(&lstEntitiesInArea, &vBoxAABB, true);
  
  if(GetVisAreaManager())
    GetVisAreaManager()->MoveObjectsIntoList(&lstEntitiesInArea, vBoxAABB, true);
  
	int nChanged=0;
	for(int i=0; i<lstEntitiesInArea.Count(); i++)
	{
		IVisArea * pPrevArea = lstEntitiesInArea[i].pNode->GetEntityVisArea();
		Get3DEngine()->UnRegisterEntity(lstEntitiesInArea[i].pNode); 
		Get3DEngine()->RegisterEntity(lstEntitiesInArea[i].pNode);
		if(pPrevArea != lstEntitiesInArea[i].pNode->GetEntityVisArea())
			nChanged++;
	}
}

/*
int CObjManager::CountPhysGeomUsage(CStatObj * pStatObjToFind)
{
  int nRes=0;
	for(int i=0; i<m_lstBrushContainer.Count(); i++)
	{
		CBrush * pBrush = m_lstBrushContainer[i];
		IStatObj * pStatObj = pBrush->GetEntityStatObj();
//    assert(((CStatObj*)pStatObj)->m_bStreamable);
		if(pStatObjToFind == pStatObj)
    {
      if(pBrush->GetPhysGeomId(0)>=0 || pBrush->GetPhysGeomId(1)>=0)
        nRes++;
    }
	}

  return nRes;
}*/

void CObjManager::FreeNotUsedCGFs()
{
	//assert(!m_bLockCGFResources);
	m_lockedObjects.clear();

	if (!m_bLockCGFResources)
	{
		//Timur, You MUST use next here, or with erase you invalidating
		LoadedObjects::iterator next;
		for (LoadedObjects::iterator it = m_lstLoadedObjects.begin(); it != m_lstLoadedObjects.end(); it = next)
		{
			next = it; next++;
			CStatObj* p = (CStatObj*)(*it);
			if(p->m_nUsers<=0)
			{
				CheckForGarbage(p);
			}
		}
	}

	ClearStatObjGarbage();
}

//////////////////////////////////////////////////////////////////////////
void CObjManager::GetLoadedStatObjArray( IStatObj** pObjectsArray,int &nCount )
{
	if (!pObjectsArray)
	{
		nCount = m_lstLoadedObjects.size();
		return;
	}

	CObjManager::LoadedObjects::iterator it = m_lstLoadedObjects.begin();
	for (int i = 0; i < nCount && it != m_lstLoadedObjects.end(); i++,it++)
	{
		pObjectsArray[i] = *it;
	}
}

void StatInstGroup::Update(CVars * pCVars, int nGeomDetailScreenRes)
{
	m_dwRndFlags = 0;
	if(bCastShadow)
		m_dwRndFlags |= ERF_CASTSHADOWMAPS;
	if(bPrecShadow)
		m_dwRndFlags |= ERF_CASTSHADOWINTORAMMAP;
	if(bHideability)
		m_dwRndFlags |= ERF_HIDABLE;
	if(bHideabilitySecondary)
		m_dwRndFlags |= ERF_HIDABLE_SECONDARY;
	if(bPickable)
		m_dwRndFlags |= ERF_PICKABLE;
  if(bUseTerrainColor)
    m_dwRndFlags |= ERF_USE_TERRAIN_COLOR;
  if(bAffectedByVoxels)
    m_dwRndFlags |= ERF_AFFECTED_BY_VOXELS;

	uint32 nSpec = (uint32)minConfigSpec;
	if (nSpec != 0)
	{
		m_dwRndFlags &= ~ERF_SPEC_BITS_MASK;
		m_dwRndFlags |= (nSpec << ERF_SPEC_BITS_SHIFT) & ERF_SPEC_BITS_MASK;
	}

	if(GetStatObj())
	{
    fVegRadiusVert = GetStatObj()->GetRadiusVert();
    fVegRadiusHor = GetStatObj()->GetRadiusHors();
    fVegRadius = max(fVegRadiusVert,fVegRadiusHor);
  }
  else
  {
    fVegRadiusHor = fVegRadius = fVegRadiusVert = 0;
  }

  if(bUseSprites && fVegRadius>0)
  {
	  m_fSpriteSwitchDist = 18.f * fVegRadius * fSize *
      max(pCVars->e_VegetationSpritesDistanceCustomRatioMin, fSpriteDistRatio);
	  m_fSpriteSwitchDist *= pCVars->e_VegetationSpritesDistanceRatio;
    m_fSpriteSwitchDist *= max(1.f, (float)nGeomDetailScreenRes / 1024.f);
	  if(m_fSpriteSwitchDist < pCVars->e_VegetationSpritesMinDistance)
		  m_fSpriteSwitchDist = pCVars->e_VegetationSpritesMinDistance;
  }
  else
    m_fSpriteSwitchDist = 1000000.f;

  for(int j=0;j<FAR_TEX_COUNT;++j)
  {
    SVegetationSpriteLightInfo& rLightInfo = m_arrSSpriteLightInfo[j];
    if(rLightInfo.m_pDynTexture)
			rLightInfo.m_pDynTexture->SetFlags(rLightInfo.m_pDynTexture->GetFlags() | IDynTexture::fNeedRegenerate);
  }
}

bool CObjManager::SphereRenderMeshIntersection(IRenderMesh * pRenderMesh, const Vec3 & vInPos, const float fRadius,IMaterial*pMat)
{
  FUNCTION_PROFILER_3DENGINE;

  // get position offset and stride
  int nPosStride = 0;
  byte * pPos  = pRenderMesh->GetPosPtr(nPosStride, FSL_READ);

  // get indices
  uint16 *pInds = pRenderMesh->GetIndexPtr(FSL_READ);
  int nInds = pRenderMesh->GetIndicesCount();
  assert(nInds%3 == 0);

  // test tris
  PodArray<CRenderChunk>& Chunks = pRenderMesh->GetChunks();
  for(int nChunkId=0; nChunkId<Chunks.Count(); nChunkId++)
  {
    CRenderChunk * pChunk = &Chunks[nChunkId];
    if(pChunk->m_nMatFlags & MTL_FLAG_NODRAW || !pChunk->pRE)
      continue;

    // skip transparent and alpha test
    if(pMat)
    {
      const SShaderItem &shaderItem = pMat->GetShaderItem(pChunk->m_nMatID);
      if (!shaderItem.m_pShader || shaderItem.m_pShader->GetFlags2() & EF2_NODRAW)
        continue;
    }

    int nLastIndexId = pChunk->nFirstIndexId + pChunk->nNumIndices;
    for(int i=pChunk->nFirstIndexId; i<nLastIndexId; i+=3)
    {
      assert(pInds[i+0]<pRenderMesh->GetVerticesCount());
      assert(pInds[i+1]<pRenderMesh->GetVerticesCount());
      assert(pInds[i+2]<pRenderMesh->GetVerticesCount());

      // get triangle vertices
      Vec3 v0 = (*(Vec3*)&pPos[nPosStride*pInds[i+0]]);
      Vec3 v1 = (*(Vec3*)&pPos[nPosStride*pInds[i+1]]);
      Vec3 v2 = (*(Vec3*)&pPos[nPosStride*pInds[i+2]]);

      AABB triBox;
      triBox.min = v0;
      triBox.max = v0;
      triBox.Add(v1);
      triBox.Add(v2);

      if(	Overlap::Sphere_AABB(Sphere(vInPos,fRadius), triBox) )
        return true;
    }
  }

  return false;
}

//////////////////////////////////////////////////////////////////////////
void CObjManager::ClearStatObjGarbage()
{
	FUNCTION_PROFILER_3DENGINE;

	std::vector<CStatObj*> garbage;

	while (!m_checkForGarbage.empty())
	{
		garbage.resize(0);

		// Make sure all stat objects inside this array are unique.
		if (!m_checkForGarbage.empty())
		{
			// Only check explicitly added objects.
			// First ShutDown object clearing all pointers.
			for (int i = 0,num = (int)m_checkForGarbage.size(); i < num; i++)
			{
				CStatObj *pStatObj = m_checkForGarbage[i];
				// Check if it must be released.
				int nChildRefs = pStatObj->CountChildReferences();
				if (pStatObj->m_nUsers <= 0 && nChildRefs <= 0)
				{
					garbage.push_back(pStatObj);
				}
				else
				{
					pStatObj->m_bCheckGarbage = false;
				}
			}

			m_checkForGarbage.resize(0);
		}

		// First ShutDown object clearing all pointers.
		for (int i = 0,num = (int)garbage.size(); i < num; i++)
		{
			CStatObj *pStatObj = garbage[i];
			pStatObj->ShutDown();
		}

		// Then delete all garbage objects.
		for (int i = 0,num = (int)garbage.size(); i < num; i++)
		{
			CStatObj *pStatObj = garbage[i];
			InternalDeleteObject(pStatObj);
		}

	}
}

//////////////////////////////////////////////////////////////////////////
IRenderMesh * CObjManager::GetRenderMeshBox()
{
	if (!m_pRMBox)
	{
		MakeUnitCube();
	}
	return m_pRMBox;
}

//////////////////////////////////////////////////////////////////////////
void CObjManager::CheckForGarbage( CStatObj *pObject )
{
	if (m_bGarbageCollectionEnabled && 
			!pObject->m_bCheckGarbage)
	{
		pObject->m_bCheckGarbage = true;
		m_checkForGarbage.push_back(pObject);
	}
}
