////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2002.
// -------------------------------------------------------------------------
//  File name:   objmancullqueue.h
//  Version:     v1.00
//  Created:     2/12/2009 by Michael Glueck
//  Compilers:   Visual Studio.NET
//  Description: Declaration and entry point for asynchronous obj-culling queue
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#ifndef CObjManCullQueue_H
#define CObjManCullQueue_H

#include <platform.h>
#include <IEntityRenderState.h>
#include <Cry_Camera.h>

//forces usage of z-buffer based occlusion, enables queue processing of occlusion queries, results have 1 frame delay
#ifdef PS3
	#include <IJobManSPU.h>
	#define USE_CULL_QUEUE
#endif
#ifdef XENON
	#include <IThreadTask.h>
	#define USE_CULL_QUEUE
#endif

#include <Cry_Geo.h>

class CZBufferCuller;
class CCullBuffer;

namespace NCullQueue
{
	class CCullTask;
#ifdef USE_CULL_QUEUE
	#ifdef XENON
		enum {MAX_CULL_QUEUE_ITEM_COUNT = 2*4096};//2*128 KB
	#else
		#ifdef __SPU__
			enum {MAX_CULL_QUEUE_ITEM_COUNT = 2048};//64 KB, maximum to be handled
		#else
			enum {MAX_CULL_QUEUE_ITEM_COUNT = 4096};//128 KB, 2 loops for SPU, make flexible if increasing this val
		#endif
	#endif
#else
	enum {MAX_CULL_QUEUE_ITEM_COUNT = 1};
#endif

	struct SCullItem
	{
		AABB objBox;
		uint32 BufferID;
		OcclusionTestClient *pOcclTestVars;
	} _ALIGN(16);

	class SCullQueue
	{
	private:
		uint32 curIndex;
		Vec3 sunDir;
#ifdef XENON
		NCullQueue::CCullTask *m_pCullTask;
#endif
		SCullItem cullItemBuf[MAX_CULL_QUEUE_ITEM_COUNT+1] _ALIGN(16);

	public:
		void ProcessInternal(uint32 mainFrameID, CZBufferCuller *const pCullBuffer,const CCamera* const pCam);

		SCullQueue();
		~SCullQueue();

		uint32 Size(){return curIndex;}

		ILINE void AddItem(const AABB& objBox, const Vec3& lightDir, OcclusionTestClient *pOcclTestVars, uint32 mainFrameID)
		{
      // Quick quiet-nan check 
      assert(objBox.max.x == objBox.max.x);
      assert(objBox.max.y == objBox.max.y);
      assert(objBox.max.z == objBox.max.z);
      assert(objBox.min.x == objBox.min.x);
      assert(objBox.min.y == objBox.min.y);
      assert(objBox.min.z == objBox.min.z);

      if(curIndex < MAX_CULL_QUEUE_ITEM_COUNT-1)
			{
				SCullItem& RESTRICT_REFERENCE rItem = cullItemBuf[curIndex];
				rItem.objBox				= objBox;
				sunDir							=	lightDir;
				rItem.BufferID			= 14;//1<<1 1<<2 1<<3  shadows
				rItem.pOcclTestVars = pOcclTestVars;
				++curIndex;//store here to make it more likely SCullItem has been written
				return;
			}
			pOcclTestVars->nLastVisibleMainFrameID = mainFrameID;//not enough space to hold item
		}

		ILINE void AddItem(const AABB& objBox, float fDistance, OcclusionTestClient *pOcclTestVars, uint32 mainFrameID)
		{
      assert(objBox.max.x == objBox.max.x);
      assert(objBox.max.y == objBox.max.y);
      assert(objBox.max.z == objBox.max.z);
      assert(objBox.min.x == objBox.min.x);
      assert(objBox.min.y == objBox.min.y);
      assert(objBox.min.z == objBox.min.z);

			if(curIndex < MAX_CULL_QUEUE_ITEM_COUNT-1)
			{
				SCullItem& RESTRICT_REFERENCE rItem = cullItemBuf[curIndex];
				rItem.objBox				= objBox;
				rItem.BufferID			= 1;	//1<<0 zbuffer
				rItem.pOcclTestVars = pOcclTestVars;
				++curIndex;//store here to make it more likely SCullItem has been written
				return;
			}
			pOcclTestVars->nLastVisibleMainFrameID = mainFrameID;//not enough space to hold item
		}
		void Process(uint32 mainFrameID, CCullBuffer *const pCullBuffer,const CCamera* const pCam);
		void Wait();
	};

#ifdef XENON
	class CCullTask : public IThreadTask 
	{
	private:
		SCullQueue* m_pQueue;
		uint32 m_MainFrameID;
		CZBufferCuller * m_pCullBuffer;
		const CCamera* m_pCam;
		bool m_Quit;
		volatile bool m_IsProcessing;
		CryMutex							m_Lock;
		CryConditionVariable	m_ProcessCondition;
		void Signal()
		{ 
			m_Lock.Lock();
			m_ProcessCondition.Notify();
			m_Lock.Unlock();
		}

	public:
		CCullTask(SCullQueue* pQueue) : m_pQueue(pQueue), m_Quit(false), m_IsProcessing(false){}
		virtual void OnUpdate();
		bool IsProcessing(){return m_IsProcessing;}
		void AddCullTask(uint32 mainFrameID, CZBufferCuller *const pCullBuffer,const CCamera* const pCam)
		{
			m_MainFrameID = mainFrameID;
			m_pCullBuffer = pCullBuffer;
			m_pCam				= pCam;
			Signal();
		}
		virtual void Stop() 
		{ 
			m_Lock.Lock();
			m_Quit = true;
			m_ProcessCondition.Notify();
			m_Lock.Unlock();
		}
		virtual SThreadTaskInfo* GetTaskInfo() { return &m_TaskInfo; }
		SThreadTaskInfo m_TaskInfo;
	};
#endif
};


#endif // CObjManCullQueue_H
