////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2002.
// -------------------------------------------------------------------------
//  File name:   ParticleEmitter.cpp
//  Created:     18/7/2003 by Timur.
//  Modified:    17/3/2005 by Scott Peter
//  Compilers:   Visual Studio.NET
//  Description: 
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "ParticleEmitter.h"
#include "ParticleContainer.h"
#include "ICryAnimation.h"
#include "Particle.h"
#include "partman.h"
#include "FogVolumeRenderNode.h"
#include "CoarseShadowsMgr.h"
#include "ISound.h"
#include "IThreadTask.h"

#include <SpuUtils.h>

#ifdef WIN64
	#pragma warning(disable: 4244)
#endif // WIN64

#if defined(PS3)&& !defined(__SPU__) && !defined(__CRYCG__)
  #define USE_SPU
#endif // PS3 && !__SPU__ && !__CRYCG__

#define fSOUND_RANGE_START_BUFFER								float(2.f)		// Distance to max sound range in order to start sound
#define fVELOCITY_SMOOTHING_TIME								float(0.125f)	// Averaging interval to smooth computed entity velocity

// global holder class for SPU deferred Release Objects
SpuDeferredReleaseObjects gSPUDeferredReleaseObjects;

/*
	Scheme for Emitter updating & bounding volume computation.

	Whenever possible, we update emitter particles only on viewing.
	However, the bounding volume for all particles must be precomputed, 
	and passed to the renderer,	for visibility. Thus, whenever possible, 
	we precompute a Static Bounding Box, loosely estimating the possible
	travel of all particles, without actually updating them.

	Currently, Dynamic BBs are computed for emitters that perform physics,
	or are affected by non-uniform physical forces.
*/

#if defined(PS3)
#if !defined(CRYCG_CM)
SPU_ENTRY(UpdateParticles)
#endif
void UpdateParticlesBatch( int seed, int nRendereFrame, class CDevBufferMan *pDevBufferMan, UpdateParticlesEmitterBatch batch )
{
#if defined(__SPU__)
	InitTimer();
	SetSpuRNGSeed( seed );
	gRendererFrameID = nRendereFrame;
	g_pDevBufMan = pDevBufferMan;	
#endif // __SPU__
	
	for( uint32 i = 0 ; i < UpdateParticlesEmitterBatch::MAX_BATCHES && batch.batch[i] != NULL ; ++i )
	{
		SPU_MAIN_PTR(batch.batch[i])->UpdateAllParticleContainerSpu();
	}
	
	__spu_flush_cache(); // flush cache before marking job as finished
	
	for( uint32 i = 0 ; i < UpdateParticlesEmitterBatch::MAX_BATCHES && batch.batch[i] != NULL ; ++i )
	{
		SPU_MAIN_PTR(batch.batch[i])->SetSPUStopped();
	}
}

void CParticleEmitter::UpdateAllParticleContainerSpu()
{
	for_all_ptrs (CParticleContainer, pCont, m_Containers)
	{
		if (pCont->NeedSpuUpdate())
		{			
			pCont->SetNeedSpuUpdate(false);
			SPU_MAIN_PTR( static_cast<CParticleContainer*>(pCont) )->UpdateParticles();
		}
	}
}

void CParticleEmitter::SyncSPUS()
{
#if defined(USE_SPU)
	if (InvokeJobOnSPU("UpdateParticles"))
	{	
		FUNCTION_PROFILER_SYS(PARTICLE);
		// cast dummy member structur to real spu driver structure to make use of all sync features
		GetIJobManSPU()->WaitSPUJob( *(NSPU::NDriver::SExtJobState*)&m_UpdateParticlesSPUState, 10/*ms timeout*/);
	} 
#endif // USE_SPU
}

void CParticleContainer::PreLoadPhysicAreasForSPU()
{
	// get envs to check if physareas are needed		 
	uint32 nEnv = GetMain().GetUniformPhysEnv().m_nNonUniformFlags & GetEnvironmentFlags();
	if ( NeedSpuUpdate() && CanThread() && nEnv )			
		m_SPUPhysicEnv.GetPhysAreas( &GetMain(), m_bbWorld, nEnv, &m_aNonUniformAreas );
}
#endif // PS3

//////////////////////////////////////////////////////////////////////////
// CParticleLocEmitter implementation.
//////////////////////////////////////////////////////////////////////////

SPU_NO_INLINE CParticleLocEmitter::CParticleLocEmitter(float fAge)
	: m_vVel(ZERO)
	, m_fStopAge(fHUGE)
	, m_fCollideAge(fHUGE)
	, m_eMaxState(eEmitter_Active)
	, m_qpLoc(IDENTITY)
{
	m_timeCreated = GetTimer()->GetFrameStartTime() - CTimeValue(fAge);
}

SPU_NO_INLINE void CParticleLocEmitter::Destroy()
{
	assert(GetRefCount() == 0);
	if (m_EmitGeom.m_pStatObj)
	{
#if defined(__SPU__)
		gSPUDeferredReleaseObjects.AddDeferredStatObjForRelease( m_EmitGeom.m_pStatObj );		
#else
		m_EmitGeom.m_pStatObj->Release();
#endif
	}
	if (m_EmitGeom.m_pChar)
		m_EmitGeom.m_pChar->Release();
	if (m_EmitGeom.m_pPhysEnt)
		m_EmitGeom.m_pPhysEnt->Release();
}

SPU_NO_INLINE CParticleSubEmitter* CParticleLocEmitter::AddEmitter( CParticleSubEmitter* pParentEmitter, CParticleContainer* pContainer )
{
	CParticleSubEmitter* pEmitter = pContainer->AddEmitter( this, pParentEmitter );
	pEmitter->Activate(eAct_Activate, GetAge());
	return pEmitter;
}

//////////////////////////////////////////////////////////////////////////
SPU_NO_INLINE bool CParticleLocEmitter::SetEmitGeom( GeomRef geom )
{
	if (geom.m_pStatObj == m_EmitGeom.m_pStatObj
	 && geom.m_pChar == m_EmitGeom.m_pChar
	 && geom.m_pPhysEnt == m_EmitGeom.m_pPhysEnt)
	 return false;

	if (m_EmitGeom.m_pStatObj)
	{
#if defined(__SPU__)
		gSPUDeferredReleaseObjects.AddDeferredStatObjForRelease( m_EmitGeom.m_pStatObj );		
#else
		m_EmitGeom.m_pStatObj->Release();
#endif
	}
	if (m_EmitGeom.m_pChar)
		m_EmitGeom.m_pChar->Release();
	if (m_EmitGeom.m_pPhysEnt)
		m_EmitGeom.m_pPhysEnt->Release();

	m_EmitGeom = geom;

	if (m_EmitGeom.m_pStatObj)
		m_EmitGeom.m_pStatObj->AddRef();
	if (m_EmitGeom.m_pChar)
		m_EmitGeom.m_pChar->AddRef();
	if (m_EmitGeom.m_pPhysEnt)
		m_EmitGeom.m_pPhysEnt->AddRef();

	return true;
}

EEmitterState CParticleEmitter::GetState() const
{
	if (m_LocEmitter.GetMaxState() == eEmitter_Dead)
	{
		return eEmitter_Dead;
	}
	EEmitterState eState = eEmitter_Dead;
	for_all_ptrs (const CParticleContainer, c, m_Containers)
	{
		if (const CParticleSubEmitter* e = c->GetDirectEmitter())
		{
			eState = (EEmitterState)max(eState, e->GetState());
		}
	}
	return (EEmitterState) min(eState, m_LocEmitter.GetMaxState());
}

SPU_NO_INLINE void CParticleEmitter::Activate( EActivateMode mode, float fPast )
{
	switch (mode)
	{
		case eAct_Activate:
			m_LocEmitter.SetStopAge(fHUGE);
			if (m_LocEmitter.IsActive())
				return;
			m_LocEmitter.SetMaxState(eEmitter_Active);
			break;
		case eAct_Resume:
			m_LocEmitter.SetStopAge(fHUGE);
			m_LocEmitter.SetMaxState(eEmitter_Active);
			break;
		case eAct_Deactivate:
			if (m_LocEmitter.GetMaxState() > eEmitter_Particles)
				m_LocEmitter.SetMaxState(eEmitter_Particles);
			// continue
		case eAct_Pause:
			m_LocEmitter.SetStopAge(m_LocEmitter.GetAge() - fPast);
			break;
		case eAct_Reactivate:
			Reset();
	}

	for_all_ptrs (CParticleContainer, c, m_Containers)
	{
		if (CParticleSubEmitter* pE = c->GetDirectEmitter())
		{
			pE->Activate(mode, fPast);
		}
	}
}

void CParticleEmitter::Prime()
{
	float fEqTime = 0.f;
	for_all_ptrs (CParticleContainer, c, m_Containers)
	{
		if (!c->GetParent() && c->HasEquilibrium())
		{
			// Continuous immortal effect.
			fEqTime = max(fEqTime, c->GetEquilibriumAge());
		}
	}

	CTimeValue timePrime = GetTimer()->GetFrameStartTime() - CTimeValue(fEqTime);
	if (timePrime < m_LocEmitter.GetTimeCreated())
	{
		m_LocEmitter.SetTimeCreated(timePrime);
	}
}

float CParticleEmitter::GetEmitCountScale() const
{
	float fCount = m_SpawnParams.fCountScale;
	if (m_SpawnParams.bCountPerUnit)
	{
		// Count multiplied by geometry extent.
		fCount *= GetExtent(m_LocEmitter.m_GeomQuery, m_LocEmitter.m_EmitGeom, m_SpawnParams.eAttachForm);
		fCount *= ScaleExtent(m_SpawnParams.eAttachForm, m_LocEmitter.m_qpLoc.s);
	}
	fCount *= max(0.75f, GetCVars()->e_ParticlesLod);		// clamped to low spec value to prevent cheating
	return fCount;
}

void CParticleEmitter::ClearUnused()
{
	// Clear in reverse order, children before parents.
	for_rev_all_ptrs (CParticleContainer, c, m_Containers)
	{
		if (!c->IsUsed())
			m_Containers.erase(c);
	}
}

CParticleContainer* CParticleEmitter::AddContainer(CParticleContainer* pParentContainer, const CParticleEffect* pEffect, const ParticleParams* pParams)
{
	CParticleContainer* pContainer = new(m_Containers.push_back_new()) CParticleContainer(pParentContainer, this, pEffect, pParams);
	return pContainer;
}

//////////////////////////////////////////////////////////////////////////
// CParticleEmitter implementation.
////////////////////////////////////////////////////////////////////////

CParticleEmitter::CParticleEmitter( bool bIndependent )
	: m_bIndependent(bIndependent)
	, m_BindToCamera(false)
{
	m_nEntityId = 0;
	m_nEntitySlot = 0;
	m_bSelected = false;
	m_nEnvFlags = 0;
	m_WSBBox.Reset();
	m_bbWorld.Reset();
	m_bbWorldDyn.Reset();
	m_bbWorldEnv.Reset();
	m_vPrevPos.zero();
	m_pCoarseShadowQueries[0] = m_pCoarseShadowQueries[1] = m_pCoarseShadowQueries[2] = m_pCoarseShadowQueries[3] = 0;

#if defined(PS3)
	memset( &m_UpdateParticlesSPUState, 0, sizeof(m_UpdateParticlesSPUState) );
#endif
  GetInstCount(GetRenderNodeType())++;
}

//////////////////////////////////////////////////////////////////////////
CParticleEmitter::~CParticleEmitter() 
{
	// Clear all particles & indirect emitters in reverse order first, to eliminate pointer dependencies.
	Reset();
  Get3DEngine()->FreeRenderNodeState(this); // Also does unregister entity.
  GetInstCount(GetRenderNodeType())--;
}

//////////////////////////////////////////////////////////////////////////
void CParticleEmitter::Register(bool b)
{
	if (!b)
	{
		if (!m_WSBBox.IsReset())
		{
			Get3DEngine()->UnRegisterEntity(this);
			m_WSBBox.Reset();
		}
	}
	else
	{
		// Register top-level render node if applicable.
		if (m_SpawnParams.bIgnoreLocation)
			Get3DEngine()->UnRegisterEntity(this);
		else if (!IsEquivalent(m_WSBBox, m_bbWorld))
		{
			if (!m_WSBBox.IsReset())
				Get3DEngine()->UnRegisterEntity(this);
			m_WSBBox = m_bbWorld;
			if (!m_WSBBox.IsReset())
			{
				// Register node.
				// Normally, use emitter's designated position for visibility.
				// However, if all bounds are computed dynamically, they might not contain the origin, so use bounds for visibility.
				if (BoundsTypes() & AlphaBit('s'))
					SetRndFlags(ERF_REGISTER_BY_POSITION, true);
				else
					SetRndFlags(ERF_REGISTER_BY_POSITION, false);
				Get3DEngine()->RegisterEntity(this);
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////////
string CParticleEmitter::GetDebugString(char type) const
{
	string s = GetName();
	if (type == 's')
	{
		// Serialization debugging.
		IEntity* pEntity = gEnv->pEntitySystem->GetEntity(m_nEntityId);
		if (pEntity)
			s += string().Format(" entity=%s slot=%d", pEntity->GetName(), m_nEntitySlot);
		if (m_bIndependent)
			s += " indep";
	}
	else
	{
		SParticleCounts counts;
		GetCounts(counts, false);
		s += string().Format(" E=%.0f P=%.0f R=%0.f", counts.EmittersActive, counts.ParticlesActive, counts.ParticlesRendered);
	}
	
	switch (GetState())
	{
		case eEmitter_Dead:
			s += " dead";
			break;
		case eEmitter_Dormant:
			s += " dormant";
			break;
		case eEmitter_Particles:
			s += " inactive";
			break;
	}
	s += string().Format(" age=%.3f", m_LocEmitter.GetAge());
	return s;
}

//////////////////////////////////////////////////////////////////////////
void CParticleEmitter::AddEffect( CParticleContainer* pParentContainer, CParticleSubEmitter* pParentEmitter, CParticleEffect const* pEffect, ParticleParams const* pParams, bool bUpdate )
{
	if (pEffect)
		pParams = &pEffect->GetParams();
	else if (!pParams)
		return;

	CParticleContainer* pContainer = pParentContainer;

	// All descendents of an indirect effect are also indirect effects, with the same parent.
	if (!pParams->bSecondGeneration && pParentContainer && pParentContainer->GetParent())
	{
		pParentContainer = pParentContainer->GetParent();
	}

	// Add if playable in current config.
	if (CPartManager::GetManager()->IsActive(*pParams))
	{
		if (m_bIndependent && static_cast<const ResourceParticleParams*>(pParams)->IsImmortal())
		{
			// Do not allow immortal effects on independent emitters.
			static std::set<const ParticleParams*> s_setWarnings;
			if (s_setWarnings.insert(pParams).second)
			{
				Warning("Ignoring spawning of immortal independent particle sub-effect %s",
								(pEffect ? pEffect->GetName() : "[programmatic]"));
			}
		}
		else
		{
			if (bUpdate)
			{
				// Look for existing container. 
				// Note: Inactive/disabled effects will not be removed, so they can be disabled/re-enabled when editing.
				// They will be skipped on playback.
				for_all_ptrs (CParticleContainer, c, m_Containers)
				{
					if (c && !c->IsUsed() && c->Is( pEffect, pParentContainer ))
					{
						pContainer = c;
						pContainer->SetUsed(true);
						break;
					}
				}
			}
			if (pContainer == pParentContainer)
			{
				// Add new container
				pContainer = AddContainer(pParentContainer, pEffect, pParams);
				if (pContainer && !pContainer->GetParent())
				{
					// this is not a second-gen container, so it will need only a single SubEmitter - we can
					// reserve the memory for that SubEmitter here
					const bool emitterAvailable = pContainer->ReserveSubEmitters(1);
					assert(emitterAvailable);
					if(emitterAvailable)
					{
						pParentEmitter = m_LocEmitter.AddEmitter( pParentEmitter, pContainer );
					}
				}
			}
		}
	}

	// Recurse effect tree.
	if (pEffect)
	{
		for (int i = 0, n = pEffect->GetChildCount(); i < n; i++)
		{
			CParticleEffect const* pChild = static_cast<const CParticleEffect*>(pEffect->GetChild(i));
			if(pChild)
			{
				if (!pChild->GetParams().bSecondGeneration)
				{
					AddEffect( pParentContainer, pParentEmitter, pChild, 0, bUpdate );
				}
				else if (pContainer)
				{
					AddEffect( pContainer, 0, pChild, 0, bUpdate );
				}
			}
		}
	}
}

void CParticleEmitter::SetEffect( CParticleEffect const* pEffect, ParticleParams const* pParams )
{
  FUNCTION_PROFILER_SYS(PARTICLE);

	if (m_pTopEffect.get() != pEffect)
	{
		// Clear any existing emitters only if changing effect.
		Reset();
		m_Containers.clear();
		m_pTopEffect = &non_const(*pEffect);
	}

	// Create or update emitters to reflect effect tree. Works correctly in game or from editor changes.
	bool bUpdate = !m_Containers.empty();
	for_all_ptrs (CParticleContainer, c, m_Containers)
	{
		c->SetUsed(false);
	}
	AddEffect(0, 0, pEffect, pParams, bUpdate);
	if(!m_Containers.empty() && m_Containers.front().GetParams().bBindEmitterToCamera)
	{
		// cache the first sub-emitter's "bind to camera" variable to speed up access in UpdateEmitter
		m_BindToCamera = m_Containers.front().GetParams().bBindEmitterToCamera;
	}

	if (bUpdate)
	{
		ClearUnused();
	}

	m_nEnvFlags = ENV_WATER;
	for_all_ptrs (CParticleContainer, c, m_Containers)
	{
		m_nEnvFlags |= c->GetEnvironmentFlags();
		c->ComputeReserveCount();
		c->InvalidateStaticBounds();
	}

	m_bbWorldEnv.Reset();
	m_VisEnviron.Invalidate();
}

SPU_NO_INLINE CParticleLocEmitter* CParticleEmitter::CreateIndirectEmitter( CParticleSubEmitter* pParentEmitter, QuatTS const& loc, float fPast )
{
	CParticleLocEmitter* pLoc = 0;
	CParticleSubEmitter* pEmitter = 0;
	CParticleContainer & rParentContainer = pParentEmitter->GetContainer();

	if (rParentContainer.GetChildFlags() & REN_ANY)
	{
		// First we work out if we need to reserve capacity in any of the containers, and how many matches there are
		// The matching containers count is used to determine if we will be able to allocate enough emitters from the
		// pool. "needToReserve" should normally only be true for the very first 2nd-gen particle emitted. After that,
		// we will have reserved enough memory for the emitters. When "needToReserve" is false we don't do any 
		// synchronization (locking) or calls to the reserve functions.
		uint32 minCapacity = rParentContainer.GetLocEmitterCapacity();
		uint32 matchingContainers = 0;
		for_all_ptrs (CParticleContainer, currentContainer, m_Containers)
		{
			if (currentContainer->GetParent() == &rParentContainer)
			{
				minCapacity = min(minCapacity, currentContainer->GetSubEmitterCapacity());
				matchingContainers++;
			}
		}

		const int maxParticleCount = max(rParentContainer.GetMaxParticles(), rParentContainer.GetNumActiveParticles());

		const bool needToReserve = (int)minCapacity < maxParticleCount;
		// the number of emitters we might need is 1 for the LocEmitter, and matchingContainers number of SubEmitters, per particle in the indirect parent
		const int maxEmitterCount = (matchingContainers + 1) * maxParticleCount;
		if(maxParticleCount > 0)
		{
			if(needToReserve)
			{
				ParticleMemory::g_emitterPool.Lock();
			}
			if(!needToReserve || (ParticleMemory::g_emitterPool.UnsynchronisedGetNumFreeBlocks() >= maxEmitterCount)) // only need to check this if we don't already have enough emitters
			{
				for_all_ptrs (CParticleContainer, currentContainer, m_Containers)
				{
					if (currentContainer->GetParent() == &rParentContainer)
					{
						// Found an indirect container for this emitter.
						if (!pLoc)
						{
							const bool reservedSomeMemory = !needToReserve || rParentContainer.ReserveLocEmittersUnsynchronized(maxParticleCount);
							if(reservedSomeMemory)
							{
								pLoc = rParentContainer.AddLocEmitter(fPast);
								if(pLoc)
								{
									pLoc->SetLoc(loc);
								}
								else
								{
									// can't allocate any more emitters yet so stop trying
									break;
								}
							}
						}
						assert(pLoc);

						if (currentContainer->GetEffect())
						{
							if (pEmitter && currentContainer->GetEffect()->GetParent() == pEmitter->GetContainer().GetEffect())
							{
								pParentEmitter = pEmitter;
							}
						}

						const bool reservedSomeMemory = !needToReserve || currentContainer->ReserveSubEmittersUnsynchronized(maxParticleCount);
						if(reservedSomeMemory)
						{
							pEmitter = pLoc->AddEmitter(pParentEmitter, currentContainer);
						}
					}
				}
			}
			if(needToReserve)
			{
				ParticleMemory::g_emitterPool.Unlock();
			}
		}
	}
	return pLoc;
}

void CParticleEmitter::SetLoc( QuatTS const& qp )
{
	if (!m_LocEmitter.m_qpLoc.IsEquivalent(qp, 1e-5f))
	{
		InvalidateStaticBounds();
		m_VisEnviron.Invalidate();
		if (m_LocEmitter.GetAge() == 0.f)
			m_vPrevPos = qp.t;
	}

	m_LocEmitter.SetLoc(qp);
}

void CParticleEmitter::SetSpawnParams( SpawnParams const& spawnParams, GeomRef geom )
{
	m_SpawnParams = spawnParams;
	if (m_SpawnParams.fPulsePeriod > 0.f && m_SpawnParams.fPulsePeriod < 0.1f)
		Warning("Particle emitter (effect %s) PulsePeriod %g too low to be useful", 
			GetName(), m_SpawnParams.fPulsePeriod);
	NormalizeEmitGeom(geom, m_SpawnParams.eAttachType);
	if (m_LocEmitter.SetEmitGeom(geom))
		InvalidateStaticBounds();
}

bool GetEntityVelocity( Vec3& vVel, IEntity* pEnt )
{
	IPhysicalEntity* pPhysEnt = pEnt->GetPhysics();
	if (pPhysEnt)
	{
		pe_status_dynamics dyn;
		if (pPhysEnt->GetStatus(&dyn))
		{
			if (vVel != dyn.v)
				vVel = dyn.v;
			return true;
		}
	}
	return false;
}

void CParticleEmitter::UpdateFromEntity() 
{
  FUNCTION_PROFILER_SYS(PARTICLE);

	m_bSelected = false;

	// Get emitter entity.
	IEntity* pEntity = gEnv->pEntitySystem->GetEntity(m_nEntityId);

	// Set external target.
	if (!m_Target.bPriority)
	{
		ParticleTarget target;
		if (pEntity)
		{
			for (IEntityLink* pLink = pEntity->GetEntityLinks(); pLink; pLink = pLink->next)
			{
				if (!stricmp(pLink->name, "Target") || !strnicmp(pLink->name, "Target-", 7))
				{
					IEntity* pTarget = gEnv->pEntitySystem->GetEntity(pLink->entityId);
					if (pTarget)
					{
						target.bTarget = true;
						target.vTarget = pTarget->GetPos();
						if (!GetEntityVelocity(target.vVelocity, pTarget))
							target.vVelocity.zero();
						AABB bb;
						pTarget->GetLocalBounds(bb);
						target.fRadius = max(bb.min.len(), bb.max.len());
						break;
					}
				}
			}
		}

		if (target.bTarget != m_Target.bTarget || target.vTarget != m_Target.vTarget)
			InvalidateStaticBounds();
		m_Target = target;
	}

	bool bShadows = (m_nEnvFlags & REN_CAST_SHADOWS) != 0;

	// Get entity of attached parent.
	if (pEntity)
	{
		if (!(pEntity->GetFlags() & ENTITY_FLAG_CASTSHADOW))
			bShadows = false;

		if (pEntity->GetParent())
			pEntity = pEntity->GetParent();

		const SpawnParams& spawn = GetSpawnParams();
		if (spawn.eAttachType != GeomType_None)
		{
			// If entity attached, find attached physics and geometry on parent.
			GeomRef geom;

			// Set physical entity as well.
			if (spawn.eAttachType == GeomType_Physics)
				geom.m_pPhysEnt = pEntity->GetPhysics();

			if (!geom.m_pPhysEnt)
			{
				int nStart = spawn.nAttachSlot < 0 ? 0 : spawn.nAttachSlot;
				int nEnd = spawn.nAttachSlot < 0 ? 	pEntity->GetSlotCount() : spawn.nAttachSlot+1;
				for (int nSlot = nStart; nSlot < nEnd; nSlot++)
				{
					SEntitySlotInfo slotInfo;
					if (pEntity->GetSlotInfo( nSlot, slotInfo ))
					{
						geom.m_pStatObj = slotInfo.pStatObj;
						geom.m_pChar = slotInfo.pCharacter;
						NormalizeEmitGeom(geom, spawn.eAttachType);

						if (geom)
						{
							if (slotInfo.pWorldTM)
								SetMatrix(*slotInfo.pWorldTM);
							break;
						}
					}
				}
			}

			SetSpawnParams(spawn, geom);
		}

		// Get velocity from entity.
		if (!GetEntityVelocity(m_LocEmitter.GetWorldVelocity(), pEntity))
		{
			// Compute from motion.
			float fStep = gEnv->pTimer->GetFrameTime();
			if (fStep > 0.f)
			{
				Vec3 vNewVel = (m_LocEmitter.m_qpLoc.t - m_vPrevPos) / fStep;
				float fDecay = expf(- fStep / fVELOCITY_SMOOTHING_TIME);
				m_LocEmitter.GetWorldVelocity() = (m_LocEmitter.GetWorldVelocity() * fDecay + vNewVel * (1.f - fDecay));
			}
		}
		m_vPrevPos = m_LocEmitter.m_qpLoc.t;

		// Flag whether selected.
		if (GetSystem()->IsEditorMode())
		{
			IEntityRenderProxy *pRenderProxy = (IEntityRenderProxy*)pEntity->GetProxy(ENTITY_PROXY_RENDER);
			if (pRenderProxy)
			{
				IRenderNode *pRenderNode = pRenderProxy->GetRenderNode();
				if (pRenderNode)
					m_bSelected = (pRenderNode->GetRndFlags() & ERF_SELECTED) != 0;
			}
		}
	}

	SetRndFlags(ERF_CASTSHADOWMAPS, bShadows);
}

void CParticleEmitter::GetLocalBounds( AABB& bbox )
{
	if (m_bbWorld.IsReset())
	{
		bbox.min = bbox.max = Vec3(0);
	}
	else
	{
		bbox.min = m_bbWorld.min - m_LocEmitter.m_qpLoc.t;
		bbox.max = m_bbWorld.max - m_LocEmitter.m_qpLoc.t;
	}
}

float CParticleEmitter::GetMinDrawPixels() const
{
	return max(GetCVars()->e_ParticlesMinDrawPixels, 0.125f) / max(GetViewDistRatioNormilized(), 0.01f);
}

float CParticleEmitter::GetMaxViewDist()
{
	return GetMaxParticleSize(true) * (GetRenderer()->GetHeight() / GetRenderer()->GetCamera().GetFov()) / GetMinDrawPixels();
}

EEmitterState CParticleEmitter::UpdateEmitter()
{
  FUNCTION_PROFILER_SYS(PARTICLE);

	EEmitterState returnEmitterState = eEmitter_Dead;

	const uint32 cvarsParticlesDebug = (GetCVars()->e_ParticlesDebug & AlphaBit('z'));
	const uint32 cvarsParticlesActive = (GetCVars()->e_Particles);	
	const bool maxEmitterStateDead = (m_LocEmitter.GetMaxState() == eEmitter_Dead);
	const uint32 renderFlagsHidden = GetRndFlags() & ERF_HIDDEN;

	if(!maxEmitterStateDead)
	{
		if(!cvarsParticlesDebug)
		{
			UpdateFromEntity();

			// Check render flags for emitter
			if(!renderFlagsHidden)
			{
				if(m_BindToCamera)
				{
					SetMatrix(GetRenderer()->GetCamera().GetMatrix());
				}

				static const float fENV_BOX_INITIAL = 20.f, fENV_BOX_EXPAND = 10.f;
				if (!m_SpawnParams.bIgnoreLocation && m_bbWorldEnv.IsReset())
				{
					// For first-computed bounds, query environment in an area around origin.
					m_bbWorldEnv.Add(m_LocEmitter.m_qpLoc.t, fENV_BOX_INITIAL);
					m_PhysEnviron.GetPhysAreas( this, m_bbWorldEnv, m_nEnvFlags );
				}

				m_bbWorld.Reset();
				m_bbWorldDyn.Reset();
				m_uBoundsTypes = 0;

				EEmitterState eSubState = eEmitter_Dead;

				// Update containers, and generate emitter bounding box(es).
				for_all_ptrs(CParticleContainer, c, m_Containers)
				{
					DEBUG_MODIFYLOCK(c->GetLock());

					// Update subemitters.
					CParticleSubEmitter* pEmitter = c->GetDirectEmitter();
					if (pEmitter)
					{
						pEmitter->Update();
						eSubState = pEmitter->GetState();
						eSubState = (EEmitterState) min(eSubState, m_LocEmitter.GetMaxState());
						returnEmitterState = (EEmitterState) max(returnEmitterState, eSubState);
					}

					c->Update();
					if(eSubState >= eEmitter_Particles)
					{
						m_uBoundsTypes |= c->UpdateBounds(m_bbWorld, m_bbWorldDyn);
					}

					c->UpdateSound();
				}

				if (!m_SpawnParams.bIgnoreLocation)
				{
					m_VisEnviron.Update(m_LocEmitter.m_qpLoc.t, m_bbWorld);
					if (!m_bbWorldEnv.ContainsBox(m_bbWorld) || CPartManager::GetManager()->GetUpdatedAreaBB().IsIntersectBox(m_bbWorld))
					{
						// When emitter leaves valid range, or areas have been updated, requery environment.
						m_bbWorldEnv = m_bbWorld;
						m_bbWorldEnv.Expand( Vec3(fENV_BOX_EXPAND) );
						m_PhysEnviron.GetPhysAreas( this, m_bbWorldEnv, m_nEnvFlags );
					}
				}

				if(!cvarsParticlesActive)
				{
					Reset();
					returnEmitterState = (EEmitterState) min(returnEmitterState, eEmitter_Dormant);
				}
			}
			else
			{
				Reset();
				returnEmitterState = eEmitter_Dormant;
			}
		}
		else
		{
			returnEmitterState = eEmitter_Active;
		}
	}

	return returnEmitterState;
}

void CParticleEmitter::UpdateForce()
{
	if (m_nEnvFlags & ENV_FORCE)
	{
		for_all_ptrs (CParticleContainer, c, m_Containers)
			c->UpdateForce();
	}
}

void CParticleEmitter::EmitParticle( IStatObj* pStatObj, IPhysicalEntity* pPhysEnt, QuatTS* pLocation, Vec3* pVel )
{
	if (!m_Containers.empty())
	{
		if (CParticleSubEmitter* pEmitter = m_Containers.front().GetDirectEmitter())
		{
			// Ensure any child containers have enough particle space (1 extra on first container).
			int extraReserve = 1;
			for_all_ptrs (CParticleContainer, c, m_Containers)
			{
				c->AllocParticles(extraReserve);
				extraReserve = 0;
			}
			SParticleUpdateContext context;
			pEmitter->EmitParticle( context, 0.f, pStatObj, pPhysEnt, pLocation, pVel );
		}
	}
}

void CParticleEmitter::SetEntity( IEntity* pEntity, int nSlot )
{
	m_nEntityId = pEntity ? pEntity->GetId() : 0;
	m_nEntitySlot = nSlot;
	UpdateFromEntity();
	if (m_bIndependent)
		m_nEntityId = 0;

	for_all_ptrs (CParticleContainer, c, m_Containers) 
		if (CParticleSubEmitter* e = c->GetDirectEmitter())
			e->ResetLoc();
}

void CParticleEmitter::UnregisterCoarseShadowsQueries()
{
	if(m_pCoarseShadowQueries[0])
	GetCoarseShadowManager()->UnregisterQuery( m_pCoarseShadowQueries[0] );
}


void CParticleEmitter::TraceCoarseShadowMask( SPartRenderParams &PRParams )
{
	*((uint32*)PRParams.m_nCoarseShadowMask) = ~0;

	if( GetCVars()->e_ParticlesCoarseShadowMask == 0)
		return;

	Vec3 pPos = m_bbWorld.GetCenter();
	I3DEngine *pEng = gEnv->p3DEngine;  
	CCoarseShadowManager *pCoarseVisManager = GetCoarseShadowManager();
	

	if( m_pCoarseShadowQueries[0] == 0 )
		m_pCoarseShadowQueries[0] = pCoarseVisManager->CreateQuery();

	// Only tracking sun for now
	Vec3 v = pEng->GetSunDirNormalized()*10000.0f;
	pCoarseVisManager->SetQueryParams(m_pCoarseShadowQueries[0], pPos, v, 1.0f, true);
	PRParams.m_nCoarseShadowMask[0] = int_round( pCoarseVisManager->GetShadowRatio( m_pCoarseShadowQueries[0] ) * 255.0f ) & 0xFF;

	PRParams.m_nCoarseShadowMask[1] = PRParams.m_nCoarseShadowMask[2] = PRParams.m_nCoarseShadowMask[3] = 0;
}

void CParticleEmitter::Render( SRendParams const& RenParams )
{
	if (!GetCVars()->e_Particles)
		return;

  FUNCTION_PROFILER_SYS(PARTICLE);

	if (m_nRenderStackLevel > 0 && !m_Containers.empty() && m_Containers.front().GetParams().bBindEmitterToCamera)
		return; 

	// Calculate contribution of fog volumes in scene
	m_timeLastRendered = GetTimer()->GetFrameStartTime();

	SPartRenderParams PRParams;
	PRParams.m_nEmitterOrder = 0;
	PRParams.m_vCamPos = GetRenderer()->GetCamera().GetPosition();
	PRParams.m_fCamDistance = m_bbWorld.GetDistance(PRParams.m_vCamPos);
	PRParams.m_fAngularRes = GetRenderer()->GetHeight() / GetRenderer()->GetCamera().GetFov();
	PRParams.m_fHDRDynamicMultiplier = GetRenderer()->EF_Query(EFQ_HDRModeEnabled) ? Get3DEngine()->GetHDRDynamicMultiplier() : 1.f;
	
	TraceCoarseShadowMask( PRParams );

	ColorF fogVolumeContrib;
	CFogVolumeRenderNode::TraceFogVolumes( m_LocEmitter.m_qpLoc.t, fogVolumeContrib );
	PRParams.m_nFogVolumeContribIdx = GetRenderer()->PushFogVolumeContribution( fogVolumeContrib );

	uint32 nRenFlags = REN_SPRITE | REN_GEOMETRY | REN_DECAL | REN_TAKE_SHADOWS;
	if (!m_SpawnParams.bIgnoreLocation)
	{
		if (Get3DEngine()->_GetRenderIntoShadowmap())
		{
			// Only geometry in shadow maps.
			nRenFlags &= REN_GEOMETRY;
			nRenFlags |= REN_CAST_SHADOWS;
		}
		else if (GetCVars()->e_ParticlesLights && !m_nRenderStackLevel)
			nRenFlags |= REN_LIGHTS;
	}

	// Render all containers.
	nRenFlags &= m_nEnvFlags;
	PRParams.m_nRenFlags = nRenFlags;
	if (PRParams.m_nRenFlags)
		for_all_ptrs (CParticleContainer, c, m_Containers)
			c->Render( RenParams, PRParams );
}

void CParticleEmitter::RenderDebugInfo()
{
	if (TimeNotRendered() < 0.25f && (m_bSelected || (GetCVars()->e_ParticlesDebug & AlphaBit('b'))))
	{
		// Draw bounding box.
		CCamera const& cam = GetRenderer()->GetCamera();
		ColorF color(1,1,1,1);

		// Compute label position, in bb clipped to camera.
		AABB const& bb = GetBBox();
		Vec3 vLabel = m_LocEmitter.GetLocation().t;

		// Clip to cam frustum.
		float fBorder = 1.f; 
		for (int i = 0; i < FRUSTUM_PLANES; i++)
		{
			Plane const& pl = *cam.GetFrustumPlane(i);
			float f = pl.DistFromPlane(vLabel) + fBorder;
			if (f > 0.f)
				vLabel -= pl.n * f;
		}

		Vec3 vDist = vLabel - cam.GetPosition();
		vLabel += (cam.GetViewdir() * (vDist * cam.GetViewdir()) - vDist) * 0.1f;
		vLabel.CheckMax(bb.min);
		vLabel.CheckMin(bb.max);

		SParticleCounts counts;
		GetCounts(counts, false);
		float fPixToScreen = 1.f / ((float)GetRenderer()->GetWidth() * (float)GetRenderer()->GetHeight());

		if (!m_bSelected)
		{
			// Randomize hue by entity.
			uint32 uRand = (uint32)(this);
			if (BoundsTypes() & AlphaBit('d'))
			{
				color.r = 1.f;
				color.g = ((uRand>>8)&0x3F) * (0.5f/63.f) + 0.25f;
				color.b = ((uRand>>14)&0x3F) * (0.5f/63.f) + 0.25f;
			}
			else
			{
				color.r = ((uRand>>2)&0x3F) * (0.5f/63.f) + 0.25f;
				color.g = ((uRand>>8)&0x3F) * (0.75f/63.f) + 0.25f;
				color.b = ((uRand>>14)&0x3F) * (0.75f/63.f) + 0.25f;
			}

			// Adjust by view angle, count, and fill.
			float fDist = vDist.NormalizeSafe();
			float fDistWt = div_min(20.f, fDist, 1.f);

			float fViewCos = cam.GetViewdir().GetNormalized() * vDist;
			float fAngleWt = sqr(sqr(sqr(fViewCos)));

			float fCountWt = clamp_tpl( sqrt_tpl(counts.ParticlesRendered * counts.PixelsRendered * fPixToScreen), 0.25f, 1.f );

			color.a = fDistWt * fAngleWt * fCountWt;
		}

		IRenderAuxGeom* pRenAux = GetRenderer()->GetIRenderAuxGeom();
		pRenAux->SetRenderFlags(SAuxGeomRenderFlags());

		char sLabel[256];
		sprintf(sLabel, "P=%.0f F=%.3f %s",
			counts.ParticlesRendered, // counts.ParticlesActive, 
			counts.PixelsRendered * fPixToScreen, // counts.PixelsProcessed * fPixToScreen,
			GetName());
		if (counts.ParticlesCollideTerrain)
			sprintf(sLabel+strlen(sLabel), " ColTr=%.0f", counts.ParticlesCollideTerrain);
		if (counts.ParticlesCollideObjects)
			sprintf(sLabel+strlen(sLabel), " ColOb=%.0f", counts.ParticlesCollideObjects);
		if (counts.ParticlesClip)
			sprintf(sLabel+strlen(sLabel), " Clip=%.0f", counts.ParticlesClip);
		if (BoundsTypes() & AlphaBit('d'))
		{
			strcat(sLabel, " Dyn");
			if (BoundsTypes() & AlphaBit('s'))
				strcat(sLabel, "+Stat");
		}
		GetRenderer()->DrawLabelEx( vLabel, 1.5f, (float*)&color, true, false, sLabel );

		color.a *= 0.4f;

		// Compare static and dynamic boxes.
		if (!m_bbWorldDyn.IsReset())
		{
			if (m_bbWorldDyn.min != bb.min || m_bbWorldDyn.max != bb.max)
			{
				// Separate dynamic BB.
				// Color outlying points bright red, draw connecting lines.
				ColorF colorGood = color * 0.6f;
				ColorF colorBad = Col_Red;
				Vec3 vStat[8], vDyn[8];
				ColorB clrDyn[8];
				for (int i = 0; i < 8; i++)
				{
					vStat[i] = Vec3( i&1 ? bb.max.x : bb.min.x, i&2 ? bb.max.y : bb.min.y, i&4 ? bb.max.z : bb.min.z );
					vDyn[i] = Vec3( i&1 ? m_bbWorldDyn.max.x : m_bbWorldDyn.min.x, i&2 ? m_bbWorldDyn.max.y : m_bbWorldDyn.min.y, i&4 ? m_bbWorldDyn.max.z : m_bbWorldDyn.min.z );
					clrDyn[i] = bb.IsContainPoint(vDyn[i]) ? colorGood : colorBad;
					pRenAux->DrawLine(vStat[i], color, vDyn[i], clrDyn[i]);
				}
				
				// Draw dyn bb.
				if (bb.ContainsBox(m_bbWorldDyn))
				{
					pRenAux->DrawAABB(m_bbWorldDyn, false, colorGood, eBBD_Faceted);
				}
				else
				{
					pRenAux->DrawLine(vDyn[0], clrDyn[0], vDyn[1], clrDyn[1]);
					pRenAux->DrawLine(vDyn[0], clrDyn[0], vDyn[2], clrDyn[2]);
					pRenAux->DrawLine(vDyn[0], clrDyn[0], vDyn[4], clrDyn[4]);

					pRenAux->DrawLine(vDyn[1], clrDyn[1], vDyn[3], clrDyn[3]);
					pRenAux->DrawLine(vDyn[1], clrDyn[1], vDyn[5], clrDyn[5]);

					pRenAux->DrawLine(vDyn[2], clrDyn[2], vDyn[3], clrDyn[3]);
					pRenAux->DrawLine(vDyn[2], clrDyn[2], vDyn[6], clrDyn[6]);

					pRenAux->DrawLine(vDyn[3], clrDyn[3], vDyn[7], clrDyn[7]);

					pRenAux->DrawLine(vDyn[4], clrDyn[4], vDyn[5], clrDyn[5]);
					pRenAux->DrawLine(vDyn[4], clrDyn[4], vDyn[6], clrDyn[6]);

					pRenAux->DrawLine(vDyn[5], clrDyn[5], vDyn[7], clrDyn[7]);

					pRenAux->DrawLine(vDyn[6], clrDyn[6], vDyn[7], clrDyn[7]);
				}
			}
			else
			{
				// Identical static & dynamic, presumably dynamic, render in brighter color.
				color = Col_Yellow;
			}
		}

		pRenAux->DrawAABB(bb, false, color, eBBD_Faceted);
	}
}

void CParticleEmitter::Serialize(TSerialize ser)
{
	ser.BeginGroup("Emitter");

	// Effect.
	string sEffect = GetEffect() ? GetEffect()->GetName() : "";
	ser.Value("Effect", sEffect);

	// Time value.
	CTimeValue timeCreated = m_LocEmitter.GetTimeCreated();
	bool bActive = m_LocEmitter.IsActive();
	ser.Value("CreationTime", timeCreated);
	float stopAge = m_LocEmitter.GetStopAge();
	ser.Value("StopAge", stopAge);
	ser.Value("Active", bActive);

	// Location.
	ser.Value("Pos", m_LocEmitter.m_qpLoc.t);
	ser.Value("Rot", m_LocEmitter.m_qpLoc.q);
	ser.Value("Scale", m_LocEmitter.m_qpLoc.s);

	// Spawn params.
	ser.Value("SizeScale", m_SpawnParams.fSizeScale);
	ser.Value("SpeedScale", m_SpawnParams.fSpeedScale);
	ser.Value("CountScale", m_SpawnParams.fCountScale);
	ser.Value("CountPerUnit", m_SpawnParams.bCountPerUnit);
	ser.Value("PulsePeriod", m_SpawnParams.fPulsePeriod);

	if (ser.IsReading())
	{
		SetEffect(m_pPartManager->FindEffect(sEffect));
		m_LocEmitter.SetStopAge(stopAge);
		m_LocEmitter.SetTimeCreated(timeCreated);
		m_LocEmitter.SetMaxState(bActive ? eEmitter_Active : eEmitter_Particles);
	}

	ser.EndGroup();
}

void CParticleEmitter::GetMemoryUsage( ICrySizer* pSizer ) const
{
	m_Containers.GetMemoryUsage(pSizer);
}

bool CParticleEmitter::UpdateStreamableComponents(float fImportance, Matrix34A & objMatrix, IRenderNode * pRenderNode, float fEntDistance)
{
  FUNCTION_PROFILER_3DENGINE;

	for_all_ptrs (const CParticleContainer, c, m_Containers)
  {
    ResourceParticleParams const& params = c->GetParams();
    if(params.pStatObj)
		{
      static_cast<CStatObj*>(params.pStatObj.get())->UpdateStreamableComponents(fImportance, objMatrix, pRenderNode, fEntDistance);
		}

		if(params.nTexId > 0)
		{
			ITexture* pTexture = GetRenderer()->EF_GetTextureByID((int)params.nTexId);
			if(pTexture)
			{
				const float minMipFactor = fEntDistance * fEntDistance * c->GetTexelAreaDensity();
				GetRenderer()->EF_PrecacheResource(pTexture, minMipFactor, 0.f, 0, GetObjManager()->m_nUpdateStreamingPrioriryRoundId);
			}
		}

		if(params.pMaterial)
		{
			Get3DEngine()->m_pObjManager->PrecacheMaterial(params.pMaterial, fEntDistance, NULL);
		}
  }

  return true;
}

void SpuDeferredReleaseObjects::ReleaseAll()
{
	for (int i = 0 ; i < NUM_SPUS ; ++i)
	{
		while (!m_deferredReleaseCalls[i].m_arrStatObj.empty())
		{
			IStatObj *pStatObj = m_deferredReleaseCalls[i].m_arrStatObj.pop();
			pStatObj->Release();
		}

		while (!m_deferredReleaseCalls[i].m_arrSound.empty())
		{
			ISound *pSound = m_deferredReleaseCalls[i].m_arrSound.pop();
			pSound->Stop();
			pSound->Release();
		}


	}
}
void CParticleEmitter::CollectSpuUsage( VecSpuUsageT &vecSpuUsage ) const
{
	for_all_ptrs (const CParticleContainer, c, m_Containers)
		c->CollectSpuUsage( vecSpuUsage );
}

#include UNIQUE_VIRTUAL_WRAPPER(IParticleEmitter)
#undef USE_SPU


