////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2010.
// -------------------------------------------------------------------------
//  File name:   ParticleFixedList.h
//  Version:     v1.00
//  Created:     23/04/2010 by Corey
//  Compilers:   Visual Studio.NET
//  Description: A simple collection class with a fixed memory size. Memory is
//               only physically allocated with Reserve and Clear calls. The
//               GetNextFree and Erase calls only calls the memory to be moved 
//               between the internal active and free lists.
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#ifndef _CRY_PARTICLE_FIXED_LIST_H_
#define _CRY_PARTICLE_FIXED_LIST_H_
#pragma once

#if defined(_PS3) || defined(__SPU__) || !defined(_DEBUG)
#define VALIDATE_PARTICLE_FIXED_LIST 0
#else // _PS3 || __SPU__ || !_DEBUG
#define VALIDATE_PARTICLE_FIXED_LIST 0
#endif // _PS3 || __SPU__ || !_DEBUG

#if VALIDATE_PARTICLE_FIXED_LIST
#include "platform.h"
#endif // VALIDATE_PARTICLE_FIXED_LIST

namespace ParticleFixedListNS
{
	ILINE bool TestPtrValidity(void* pointerToTest)
	{
		assert(pointerToTest != NULL);
		return true;
	}
}

#define for_all_ptrs_pfl(Type, p, cont) \
	for (Type::ForwardIterator p(cont); p && ParticleFixedListNS::TestPtrValidity((void*)p); ++p)

#define for_rev_all_ptrs_pfl(Type, p, cont) \
	for (Type::ReverseIterator p(cont, false); p && ParticleFixedListNS::TestPtrValidity((void*)p); --p)

template<class T>
class CParticleFixedList
{
public:
	struct Node : T
	{
		Node* m_pNext;
		Node* m_pPrev;
	};

private:
	struct Iterator
	{
		Iterator(Node* a_pCurrentActiveNode)
			: m_pCurrentActiveNode(a_pCurrentActiveNode)
		{
		}
		~Iterator() {}

		ILINE operator bool() const
		{
			return (m_pCurrentActiveNode != NULL);
		}

		ILINE T* operator->() const
		{
			assert(m_pCurrentActiveNode);
			T* currentInstance = static_cast<T*>(m_pCurrentActiveNode);
			return currentInstance;
		}

		ILINE operator T*() const
		{
			assert(m_pCurrentActiveNode);
			T* currentInstance = static_cast<T*>(m_pCurrentActiveNode);
			return currentInstance;
		}
	protected:
		Node* m_pCurrentActiveNode;
	};

public:

	struct ForwardIterator : public Iterator
	{
		using Iterator::m_pCurrentActiveNode;

		ForwardIterator(const CParticleFixedList& a_fixedList) : Iterator(a_fixedList.m_pActiveHead) {}
		~ForwardIterator() {}

		ILINE void operator++()
		{
			assert(m_pCurrentActiveNode);
			m_pCurrentActiveNode = m_pCurrentActiveNode->m_pNext;
		}

		ILINE Node* Erase()
		{
			Node* nodeToErase = m_pCurrentActiveNode;
			m_pCurrentActiveNode = m_pCurrentActiveNode->m_pNext;
			return nodeToErase;
		}
	};

	struct ReverseIterator : Iterator
	{
		using Iterator::m_pCurrentActiveNode;

		ReverseIterator(const CParticleFixedList& a_fixedList) : Iterator(a_fixedList.m_pActiveTail) {}
		~ReverseIterator() {}

		ILINE void operator--()
		{
			assert(m_pCurrentActiveNode);
			m_pCurrentActiveNode = m_pCurrentActiveNode->m_pPrev;
		}

		ILINE Node* Erase()
		{
			Node* nodeToErase = m_pCurrentActiveNode;
			m_pCurrentActiveNode = m_pCurrentActiveNode->m_pPrev;
			return nodeToErase;
		}
	};

	CParticleFixedList()
	: m_pActiveHead(NULL)
	, m_pActiveTail(NULL)
	, m_numActive(0)
	, m_pFreeHead(NULL)
	, m_numFree(0)
#if VALIDATE_PARTICLE_FIXED_LIST
	, m_ownerThreadId(0)
#endif // VALIDATE_PARTICLE_FIXED_LIST
	{
	}

	~CParticleFixedList()
	{
		assert(m_numActive == 0);
	}

	// Returns true if some memory could be reserved, even if not the full amount.
	template <class LockingAllocator>
	bool Reserve(const uint32 a_reserveCount, LockingAllocator* a_lockingAllocator)
	{
		bool success = false;
		if(a_reserveCount > Capacity())
		{
			const uint32 growCount = a_reserveCount - Capacity();
			// allocate the particle memory in advance so the SPUs won't need to do much allocating, 
			// and the locking on the allocator is minimised
#if !defined(XENON) // assert fails incorrectly on X360
			assert(a_lockingAllocator->IsLocked());
#endif // !XENON
			{
				const uint32 numFreeBlocks = a_lockingAllocator->UnsynchronisedGetNumFreeBlocks();
				const uint32 reserveCount = min(growCount, numFreeBlocks);
				for(uint32 i=0; i< reserveCount; ++i)
				{
					T* newNode = (T*)a_lockingAllocator->UnsynchronisedAllocateElement();
					assert(newNode);
					AddToFreeList(GetNode(newNode));
				}
				success = (reserveCount > 0);
			}
		}
		else
		{
			// we already have the required number of elements in total capacity
			success = true;
		}
		return success;
	}

	template <class LockingAllocator>
	void Clear(LockingAllocator* a_lockingAllocator)
	{
		AllocatorLocker<LockingAllocator> allocatorLock(a_lockingAllocator);
		{
			uint32 numFreed = 0;
			for(Node* currentNode = m_pActiveHead; currentNode != NULL; )
			{
				Node* nextNode = currentNode->m_pNext;
				T* currentInstance = GetInstance(currentNode);
				currentInstance->~T();
				a_lockingAllocator->UnsynchronisedDeallocateElement(currentNode);
				numFreed++;
				currentNode = nextNode;
			}
			for(Node* currentNode = m_pFreeHead; currentNode != NULL; )
			{
				Node* nextNode = currentNode->m_pNext;
				a_lockingAllocator->UnsynchronisedDeallocateElement(currentNode);
				numFreed++;
				currentNode = nextNode;
			}
			assert(numFreed == Capacity());
		}
		Reset();
	}

	// Returns the next free node and adds it to the active list so it can be iterated
	// over. Note that this is deliberately not threadsafe. This method should only be
	// called from within a threadsafe atomic object.
	T* GetNextFromPool()
	{
		// first remove the node from the current free list
		Node* pNextFreeNode = m_pFreeHead;
		T* pNextInstance = NULL;
		if(m_pFreeHead != NULL)
		{
			m_numFree--;
			m_pFreeHead = m_pFreeHead->m_pNext;

			// and then add it to the active list - all new items get tacked
			// on to the end of the list so that ForwardIteration = order in
			// which items were added
			pNextFreeNode->m_pPrev = m_pActiveTail;
			pNextFreeNode->m_pNext = NULL;
			if(m_pActiveTail != NULL)
			{
				m_pActiveTail->m_pNext = pNextFreeNode;
			}
			m_pActiveTail = pNextFreeNode;
			if(m_pActiveHead == NULL)
			{
				m_pActiveHead = pNextFreeNode;
			}
			m_numActive++;

			pNextInstance = GetInstance(pNextFreeNode);
		}
		return pNextInstance;
	}

	// Returns the item back to the free list and removes it from the active list.
	// Note that this is deliberately not threadsafe. This method should only be
	// called from within a threadsafe atomic object.
	void ReturnToPool(T* a_instance)
	{
		Node* pNode = GetNode(a_instance);

		// remove from the active list
		if((pNode == m_pActiveHead) && (pNode == m_pActiveTail))
		{
			// only node in list
			m_pActiveHead = NULL;
			m_pActiveTail = NULL;
		}
		else if(pNode == m_pActiveHead)
		{
			// node is head node
			m_pActiveHead = pNode->m_pNext;
			m_pActiveHead->m_pPrev = NULL;
		}
		else if(pNode == m_pActiveTail)
		{
			// node is tail node
			m_pActiveTail = pNode->m_pPrev;
			m_pActiveTail->m_pNext = NULL;
		}
		else
		{
			// somewhere in the middle of the list
			pNode->m_pPrev->m_pNext = pNode->m_pNext;
			pNode->m_pNext->m_pPrev= pNode->m_pPrev;
		}
		m_numActive--;

		T* pInstance = GetInstance(pNode);
		pInstance->~T();

		// and add it back to the free pool
		AddToFreeList(pNode);
	}

	ILINE bool IsEmpty() const
	{
		return (m_numActive == 0);
	}

	ILINE T* Front() const
	{
		Node* frontNode = m_pActiveHead;
		return GetInstance(frontNode);
	}

	ILINE uint32 GetNumActive() const
	{
		return m_numActive;
	}

	ILINE uint32 GetNumFree() const
	{
		return m_numFree;
	}

	ILINE uint32 Capacity() const
	{
		return GetNumActive() + GetNumFree();
	}

	void GetMemoryUsage(ICrySizer* pSizer) const
	{
		pSizer->AddObject(this, Capacity() * sizeof(T));
	}

	void ResetPerFrameDebugInfo()
	{
#if VALIDATE_PARTICLE_FIXED_LIST
		m_ownerThreadId = 0;
#endif // VALIDATE_PARTICLE_FIXED_LIST
	}

private:

	friend struct ForwardIterator;
	friend struct ReverseIterator;

	template<class LockingAllocator>
	struct AllocatorLocker
	{
		AllocatorLocker(LockingAllocator* a_lockingAllocator)
		: m_lockingAllocator(a_lockingAllocator)
		{
			m_lockingAllocator->Lock();
		}

		~AllocatorLocker()
		{
			m_lockingAllocator->Unlock();
		}
	private:
		LockingAllocator* m_lockingAllocator;
	};

	Node* m_pActiveHead;
	Node* m_pActiveTail;
	uint32 m_numActive;

	Node* m_pFreeHead;
	uint32 m_numFree;

#if VALIDATE_PARTICLE_FIXED_LIST
	uint32 m_ownerThreadId; // used to validate that this is being used single-threaded only
#endif // VALIDATE_PARTICLE_FIXED_LIST

	void AddToFreeList(Node* a_newNode)
	{
		assert(a_newNode);
		a_newNode->m_pNext = m_pFreeHead;
		m_pFreeHead = a_newNode;
		m_numFree++;
	}

	void Reset()
	{
		m_pActiveHead = NULL;
		m_pActiveTail = NULL;
		m_numActive = 0;
		m_pFreeHead = NULL;
		m_numFree = 0;
#if VALIDATE_PARTICLE_FIXED_LIST
		m_ownerThreadId = 0;
#endif // VALIDATE_PARTICLE_FIXED_LIST
	}

	Node* GetNode(T* a_pInstance) const
	{
		return static_cast<Node*>(a_pInstance);
	}

	T* GetInstance(Node* a_pNode) const
	{
		return static_cast<T*>(a_pNode);
	}
};

#endif // _CRY_PARTICLE_FIXED_LIST_H_
