////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2010.
// -------------------------------------------------------------------------
//  File name:   ParticleFixedSizeElementPool.cpp
//  Version:     v1.00
//  Created:     15/03/2010 by Corey.
//  Compilers:   Visual Studio.NET
//  Description: A fragmentation-free allocator from a fixed-size pool and which
//							 only allocates elements of the same size.
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "ParticleFixedSizeElementPool.h"


void ParticleFixedSizeElementPool::Init(void* a_memoryPool, const int a_numElements, const int a_elementSize)
{
	m_memoryPoolBase = a_memoryPool;
#if CHECK_FIXED_SIZE_ELEMENT_POOL_CONSISTENCY
	m_maxElements = a_numElements;
	m_elementSize = a_elementSize;
	m_memoryPoolEnd = (((char*)a_memoryPool + (a_numElements * a_elementSize)));
#endif // CHECK_FIXED_SIZE_ELEMENT_POOL_CONSISTENCY

	assert(a_elementSize >= sizeof(ParticleFixedSizeElementPool::ListNode));

#if TRACK_FIXED_SIZE_ELEMENT_POOL_IN_USE_PTRS
	for(int i=0; i < MAX_TRACKABLE_PTRS; ++i)
	{
		m_trackedPtrs[i] = NULL;
	}
	m_currentLastTracked = 0;
	for(int i=0; i < MAX_TRACKABLE_FREED_PTRS; ++i)
	{
		m_recentlyFreedPtrs[i] = NULL;
	}
	m_currentLastRecentlyFreed = 0;
#endif // TRACK_FIXED_SIZE_ELEMENT_POOL_IN_USE_PTRS

	char* pElementMemory = (char*)m_memoryPoolBase;
	for(int i=a_numElements-1; i>=0; --i)
	{
		// Build up the linked list of free nodes to begin with - do it in
		// reverse order so that consecutive allocations will be given chunks
		// in a cache-friendly order
		ListNode* pNewNode = (ListNode*)(pElementMemory + (i * a_elementSize));
		WriteDebugPattern((void*)pNewNode, 0xbeef1337);
		pNewNode->m_next = m_pFreeListHead;
		m_pFreeListHead = pNewNode;
	}
	m_numFreeBlocks = a_numElements;
}

void* ParticleFixedSizeElementPool::Allocate()
{
	CheckPointerValid(m_pFreeListHead);
	ListNode* pOldHead = m_pFreeListHead;
	if(m_pFreeListHead != NULL)
	{
		m_numFreeBlocks--;
		m_pFreeListHead = m_pFreeListHead->m_next;
		TrackPointerAlloc(pOldHead);
	}

	// Verify that the list is either empty or it seems to have valid items in it
	CheckPointerValid(m_pFreeListHead);
	assert((m_numFreeBlocks && m_pFreeListHead) || (!m_numFreeBlocks && !m_pFreeListHead));
	WriteDebugPattern(pOldHead, 0x1337c0de);
	return (void*)pOldHead;
}

void ParticleFixedSizeElementPool::Deallocate(void* a_memoryToDeallocate)
{
	assert(a_memoryToDeallocate);
	CheckPointerValid(a_memoryToDeallocate);
	TrackPointerFree(a_memoryToDeallocate);
	WriteDebugPattern(a_memoryToDeallocate, 0xdeadbeef);

	ListNode* pNewHead = (ListNode*)a_memoryToDeallocate;
	pNewHead->m_next = m_pFreeListHead;
	m_pFreeListHead = pNewHead;
	m_numFreeBlocks++;
}

void ParticleFixedSizeElementPool::WriteDebugPattern(void* a_memoryChunk, const int a_debugPattern)
{
#if CHECK_FIXED_SIZE_ELEMENT_POOL_CONSISTENCY
	// Write the code pattern 0xc0debeef to deallocated memory
	// to help with debugging
	const int numPatterns = m_elementSize/sizeof(uint32);
	const int numLeftover = m_elementSize - (numPatterns*m_elementSize);

	uint32* memoryAsUint = (uint32*)a_memoryChunk;
	for(int i=0; i < numPatterns; ++i)
	{
		memoryAsUint[i] = a_debugPattern;
	}

	char* memoryAsChar = (char*)&memoryAsUint[numPatterns];
	for(int i=0; i < numLeftover; ++i)
	{
		memoryAsChar[i] = ((char*)&a_debugPattern)[i];
	}
#endif // CHECK_FIXED_SIZE_ELEMENT_POOL_CONSISTENCY
}

void ParticleFixedSizeElementPool::CheckPointerValid(void* a_memoryToCheck)
{
#if CHECK_FIXED_SIZE_ELEMENT_POOL_CONSISTENCY
	if(a_memoryToCheck != NULL)
	{
		assert(m_numFreeBlocks <= m_maxElements);
		assert((size_t)a_memoryToCheck < (size_t)m_memoryPoolEnd);
		assert((size_t)a_memoryToCheck >= (size_t)m_memoryPoolBase);
		assert(((size_t)a_memoryToCheck % m_elementSize) == 0);
	}
#endif // CHECK_FIXED_SIZE_ELEMENT_POOL_CONSISTENCY
}

void ParticleFixedSizeElementPool::TrackPointerAlloc(void* a_allocatedMemory)
{
#if TRACK_FIXED_SIZE_ELEMENT_POOL_IN_USE_PTRS
		m_trackedPtrs[m_currentLastTracked] = a_allocatedMemory;
		m_currentLastTracked = (m_currentLastTracked + 1) % MAX_TRACKABLE_PTRS;
#endif // TRACK_FIXED_SIZE_ELEMENT_POOL_IN_USE_PTRS
}

void ParticleFixedSizeElementPool::TrackPointerFree(void* a_memoryToDeallocate)
{
#if TRACK_FIXED_SIZE_ELEMENT_POOL_IN_USE_PTRS
	// add to cyclic "recently freed" list to help debug recent releases which might be
	// held on to by lying clients
	m_recentlyFreedPtrs[m_currentLastRecentlyFreed] = a_memoryToDeallocate;
	m_currentLastRecentlyFreed = (m_currentLastRecentlyFreed + 1) % MAX_TRACKABLE_FREED_PTRS;
#endif // TRACK_FIXED_SIZE_ELEMENT_POOL_IN_USE_PTRS
}
