////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2010.
// -------------------------------------------------------------------------
//  File name:   ParticleFixedSizeElementPool.h
//  Version:     v1.00
//  Created:     15/03/2010 by Corey.
//  Compilers:   Visual Studio.NET
//  Description: A fragmentation-free allocator from a fixed-size pool and which
//							 only allocates elements of the same size.
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////


#ifndef __particlefixedsizeelementpool_h__
#define __particlefixedsizeelementpool_h__
#pragma once

#if defined(_PS3) || defined(__SPU__) || !defined(_DEBUG)
  #define CHECK_FIXED_SIZE_ELEMENT_POOL_CONSISTENCY 0
  #define TRACK_FIXED_SIZE_ELEMENT_POOL_IN_USE_PTRS 0
#elif defined(_DEBUG)
  #define CHECK_FIXED_SIZE_ELEMENT_POOL_CONSISTENCY 1
  #define TRACK_FIXED_SIZE_ELEMENT_POOL_IN_USE_PTRS 1
#endif // _PS3 || __SPU__ || !_DEBUG


class ParticleFixedSizeElementPool
{
public:
	ParticleFixedSizeElementPool()
	: m_memoryPoolBase(NULL)
	, m_pFreeListHead(NULL)
	, m_numFreeBlocks(0)
	{
	}

	~ParticleFixedSizeElementPool() {}

	void Init(void* a_memoryPool, const int a_numElements, const int a_elementSize);
	void* Allocate();
	void Deallocate(void* a_memoryToDeallocate);

	int GetNumFreeBlocks()
	{
		return m_numFreeBlocks;
	}

	void FlushCache()
	{
#if defined(__SPU__)
		__spu_flush_cache_range(((uint32)&m_pFreeListHead)&~127, 128);
#endif // __SPU__
	}

private:
	struct ListNode
	{
		ListNode* m_next;
	};

	void* m_memoryPoolBase;
	ListNode* m_pFreeListHead;
	int m_numFreeBlocks;

#if TRACK_FIXED_SIZE_ELEMENT_POOL_IN_USE_PTRS
	static const uint32 MAX_TRACKABLE_PTRS = 256;
	void* m_trackedPtrs[MAX_TRACKABLE_PTRS];
	int m_currentLastTracked;
	static const uint32 MAX_TRACKABLE_FREED_PTRS = 128;
	void* m_recentlyFreedPtrs[MAX_TRACKABLE_FREED_PTRS];
	int m_currentLastRecentlyFreed;
#endif // TRACK_FIXED_SIZE_ELEMENT_POOL_IN_USE_PTRS

#if CHECK_FIXED_SIZE_ELEMENT_POOL_CONSISTENCY
	int m_maxElements;
	int m_elementSize;
	char* m_memoryPoolEnd;
#endif // CHECK_FIXED_SIZE_ELEMENT_POOL_CONSISTENCY

	void WriteDebugPattern(void* a_memoryChunk, const int a_debugPattern);
	void CheckPointerValid(void* a_memoryToCheck);
	void TrackPointerAlloc(void* a_allocatedMemory);
	void TrackPointerFree(void* a_memoryToDeallocate);
};

class ParticleFixedSizeElementPoolSynchronized
{
public:

	class AutoLock
	{
	public:
		AutoLock(ParticleFixedSizeElementPoolSynchronized* a_pool)
		: m_pool(a_pool)
		{
			m_pool->Lock();
		}

		~AutoLock()
		{
			m_pool->Unlock();
		}
	private:
		ParticleFixedSizeElementPoolSynchronized* m_pool;
	};

	void Init(void* a_memoryPool, const int a_numElements, const int a_elementSize)
	{
		m_fixedSizeElementPool.Init(a_memoryPool, a_numElements, a_elementSize);
	}

	void Lock()
	{
#if !defined(XENON) // assert fails incorrectly on X360
		assert(!m_lock.IsLocked());
#endif // !XENON
		m_lock.Lock();
		m_fixedSizeElementPool.FlushCache();
	}

	void* UnsynchronisedAllocateElement()
	{
#if !defined(XENON) // assert fails incorrectly on X360
		assert(m_lock.IsLocked());
#endif // !XENON
		return m_fixedSizeElementPool.Allocate();
	}

	void UnsynchronisedDeallocateElement(void* a_memoryToDeallocate)
	{
#if !defined(XENON) // assert fails incorrectly on X360
		assert(m_lock.IsLocked());
#endif // !XENON
		m_fixedSizeElementPool.Deallocate(a_memoryToDeallocate);
	}

	SPU_NO_INLINE int UnsynchronisedGetNumFreeBlocks()
	{
#if !defined(XENON) // assert fails incorrectly on X360
		assert(m_lock.IsLocked());
#endif // !XENON
		return m_fixedSizeElementPool.GetNumFreeBlocks();
	}

	void Unlock()
	{
#if !defined(XENON) // assert fails incorrectly on X360
		assert(m_lock.IsLocked());
#endif // !XENON
		m_fixedSizeElementPool.FlushCache();
		m_lock.Unlock();
	}

	bool IsLocked()
	{
		return m_lock.IsLocked();
	}

private:
	CryCriticalSectionNonRecursive m_lock;
	ParticleFixedSizeElementPool m_fixedSizeElementPool;
} _ALIGN(128);
#endif // __fixedsizeelementpool_h__