////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2002.
// -------------------------------------------------------------------------
//  File name:   ParticleSubEmitter.cpp
//  Created:     20/04/2010 by Corey
//  Description: Split out from ParticleEmitter.cpp
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "ParticleSubEmitter.h"
#include "ParticleEmitter.h"
#include "ParticleContainer.h"
#include "ICryAnimation.h"
#include "Particle.h"
#include "partman.h"
#include "FogVolumeRenderNode.h"
#include "ISound.h"
#include "IThreadTask.h"

#include <SpuUtils.h>

#ifdef WIN64
	#pragma warning(disable: 4244)
#endif // WIN64

#if defined(PS3)&& !defined(__SPU__) && !defined(__CRYCG__)
  #define USE_SPU
#endif // PS3 && !__SPU__ && !__CRYCG__

#define fSOUND_RANGE_START_BUFFER								float(2.f)		// Distance to max sound range in order to start sound
#define fVELOCITY_SMOOTHING_TIME								float(0.125f)	// Averaging interval to smooth computed entity velocity

/*
	Scheme for Emitter updating & bounding volume computation.

	Whenever possible, we update emitter particles only on viewing.
	However, the bounding volume for all particles must be precomputed, 
	and passed to the renderer,	for visibility. Thus, whenever possible, 
	we precompute a Static Bounding Box, loosely estimating the possible
	travel of all particles, without actually updating them.

	Currently, Dynamic BBs are computed for emitters that perform physics,
	or are affected by non-uniform physical forces.
*/

//////////////////////////////////////////////////////////////////////////
// Misc functions.

inline float round_up(float f, float r)
{
	if (f >= 0.f && r > 0.f)
	{
		float m = fmod(f, r);
		f += r-m;
	}
	return f;
}

SPU_NO_INLINE void Interp( QuatTS& out, QuatTS const& a, QuatTS const& b, float t )
{
	out.q.SetNlerp( a.q, b.q, t );
	out.s = a.s * (1.f-t) + b.s * t;
	out.t.SetLerp( a.t, b.t, t );
}

//////////////////////////////////////////////////////////////////////////
// CParticleSubEmitter implementation.
//////////////////////////////////////////////////////////////////////////

SPU_NO_INLINE CParticleSubEmitter::CParticleSubEmitter( CParticleSubEmitter* pParent, CParticleLocEmitter* pLoc, CParticleContainer* pCont )
	: m_ChaosKey(0U)
	, m_nEmitIndex(0)
	, m_fSoundRange(0.f)
	, m_pParams(pCont->GetParamsAddr())
	, m_pContainer(pCont)
	, m_pLocEmitter(pLoc)
	, m_pParentEmitter(pParent)
{
	m_qpLastLoc = pLoc->GetLocation();
	m_fStartAge = m_fEndAge = m_fRepeatAge = fHUGE;
	m_fStrength = 0.f;
	m_fToEmit = 0.f;
	m_pForce = NULL;
	m_bSoundPlayed = false;
}

SPU_NO_INLINE void CParticleSubEmitter::Destroy()
{
	assert(GetRefCount() == 0);
	Deactivate();
}

//////////////////////////////////////////////////////////////////////////
SPU_NO_INLINE void CParticleSubEmitter::Initialize( float fPast )
{
	m_fToEmit = 0.f;

	// Reseed randomness.
	m_ChaosKey = CChaosKey(cry_rand32());

	// Compute lifetime params.
	m_fStartAge = m_fEndAge = GetAge() - fPast + GetCurValue(m_pParams->fSpawnDelay);
	if (m_pParams->bContinuous)
	{
		if (m_pParams->fEmitterLifeTime)
		{
			m_fEndAge += GetCurValue(m_pParams->fEmitterLifeTime);
		}
		else
		{
			m_fEndAge = fHUGE;
		}
	}

	if (m_pParams->eSoundControlTime != SoundControlTime_EmitterPulsePeriod)
	{
		m_bSoundPlayed = false;
	}

	// Compute next repeat age.
	m_fRepeatAge = fHUGE;
	if (m_pParams->fPulsePeriod.GetMaxValue() > 0.f)
	{
		float fRepeat = GetCurValue(m_pParams->fPulsePeriod);
		m_fRepeatAge = GetAge() - fPast + fRepeat;
		if (m_fRepeatAge < GetAge())
		{
			m_fRepeatAge += round_up( GetAge()-m_fRepeatAge, fRepeat );
		}
	}
	else if (m_pParams->fPulsePeriod.GetMaxValue() < 0.f && m_pParentEmitter)
	{
		// Inherit from parent.
		m_fRepeatAge = m_pParentEmitter->m_fRepeatAge;
	}
}

//////////////////////////////////////////////////////////////////////////
SPU_NO_INLINE void CParticleSubEmitter::Deactivate()
{
	if (m_pSound != 0)
	{
#if defined(__SPU__)	
		gSPUDeferredReleaseObjects.AddDeferredSoundForRelease( ReleaseOwnership(m_pSound) );
#else
		m_pSound->Stop();
		m_pSound = 0;
#endif
	}
	IF (m_pForce, false)
	{
		GetPhysicalWorld()->DestroyPhysicalEntity(m_pForce);
		m_pForce = 0;
	}
}

//////////////////////////////////////////////////////////////////////////
SPU_NO_INLINE void CParticleSubEmitter::Activate( EActivateMode mode, float fPast )
{
	float fAge = GetAge() - fPast;
	switch (mode)
	{
		case eAct_Pause:
		case eAct_Deactivate:
			Deactivate();
			break;
		case eAct_Restart:
			Deactivate();
			// continue;
		case eAct_Activate:
			if (!GetParams().bSpawnOnParentCollision && fAge < m_fStartAge)
				Initialize(fPast);
			break;
		case eAct_Resume:
			m_fToEmit = 0.f;
			break;
		case eAct_Reactivate:
			m_fToEmit = 0.f;
			if (fAge < m_fStartAge)
			{
				Deactivate();
				Initialize();
			}
			break;
	}
}

float CParticleSubEmitter::GetStrength( float fAgeAdjust /* = 0.f */, ESoundControlTime const eControl /* = SoundControlTime_EmitterLifeTime */ ) const
{
	float fStrength = GetMain().GetSpawnParams().fStrength;
	if (fStrength < 0.f)
		return GetRelativeAge(fAgeAdjust, eControl);
	else
		return min(fStrength, 1.f);
}

EEmitterState CParticleSubEmitter::GetState() const
{
	CParticleContainer&		rContainer = GetContainer();
	CParticleLocEmitter&	rLocEmitter = GetLoc();
	const ResourceParticleParams& params = GetParams();

	float fAge = GetAge();
	if (fAge >= m_fStartAge)
	{
		float fEndAge = GetEndAge();
		if (fAge <= fEndAge)
			// Emitter still active.
			return eEmitter_Active;

		if (params.fCount || rContainer.HasParticles())
		{
			// Has particles.
			if (rContainer.HasExtendedLifetime())
				return eEmitter_Particles;

			float fParticleLife = rContainer.GetMaxParticleLife();
			if (fAge <= fEndAge + fParticleLife + GetTimer()->GetFrameTime()*2)
				return eEmitter_Particles;
		}
	}

	// No particles, see if emitter is dead.
	if (fAge > rLocEmitter.GetStopAge())
	{
		if (rLocEmitter.IsActive())
			// Paused but still alive.
			return eEmitter_Dormant;
	}
	else
	{
		if (fAge < m_fStartAge)
			return eEmitter_Dormant;
		if (GetMain().GetSpawnParams().fPulsePeriod || params.fPulsePeriod)
			return eEmitter_Dormant;
	}

	return eEmitter_Dead;
}

SPU_NO_INLINE void CParticleSubEmitter::EmitParticles( SParticleUpdateContext const& context )
{
	// Emit new particles only for enabled effects.
	float fAge = GetAge();
	if (fAge >= m_fStartAge)
	{
		UpdateStrength();

		// Target total particles in system.
		float fEmitRate = GetEmitRate();
		if (fEmitRate > 0.f)
		{
			const float fUpdateTime = GetTimeToUpdate();
			float fAge0 = fAge - fUpdateTime;
			fAge0 = max(fAge0, m_fStartAge);
			float fLife = GetContainer().GetMaxParticleLife();

			if (m_pParams->bContinuous)
			{
				// Determine time window to update.
				float fAge1 = min(fAge, GetEndAge());

				m_fToEmit += (fAge1-fAge0) * fEmitRate;
				if (m_fToEmit > 0.f)
				{
					// Skip time before emitted particles would still be alive.
					// To to opt: avoid even this much updating for steady-state emitters (freeze).
					float fAgeIncrement = 1.f / fEmitRate;
					fAge0 = fAge1 - m_fToEmit * fAgeIncrement;
					float fSkip = round_up(fAge-fLife-fAge0, fAgeIncrement);
					if (fSkip > 0)
					{
						fAge0 += fSkip;
					}
					float fPartAge = fAge-fAge0;
					for (; fPartAge > fAge-fAge1; fPartAge -= fAgeIncrement)
					{
						if (!EmitParticle(context, fPartAge))
						{
							GetContainer().GetCounts().ParticlesReject += fPartAge * fEmitRate;
							break;
						}
					}
					fAge0 = fAge-fPartAge;
					m_fToEmit = (fAge1-fAge0) * fEmitRate;
				}
			}
			else if (m_fToEmit == 0.f)
			{
				// Emit only once, if still valid.
				// Always emit first frame, even if lifetime < frame time.				
				if (fAge <= m_fStartAge + fLife + GetTimer()->GetFrameTime()*2)
				{
					for (int nEmit = int_round(fEmitRate); nEmit > 0; nEmit--)
					{
						if (!EmitParticle(context, fAge - fAge0))
						{
							GetContainer().GetCounts().ParticlesReject += nEmit;
#if defined(PS3)
							GetContainer().SetNeedSpuUpdate(true);
#endif
							break;
						}
						else 
						{
							// Emit only once, if still valid. 
							// was moved here because of spu allocation which can cause the memory for the needed particles only be avaible in the next frame
							m_fToEmit = -1.f;
						}
					}
				}
			}
		}
	}
}

SPU_NO_INLINE void CParticleSubEmitter::MoveRelative( Vec3& vPos, Quat& qRot, Vec3& vVel )
{
	const QuatTS& qpCur = GetLoc().GetLocation();
	if (m_qpLastLoc.s > 0.f || qpCur.s == m_qpLastLoc.s)
	{
		Quat qMove = qpCur.q * !m_qpLastLoc.q;
		float fMoveScale = qpCur.s == m_qpLastLoc.s ? 1.f : qpCur.s / m_qpLastLoc.s;

		qRot = qMove * qRot;
		vVel = fMoveScale * (qMove * vVel);
		vPos = qpCur.t + fMoveScale * (qMove * (vPos - m_qpLastLoc.t));
	}
}

bool CParticleSubEmitter::GetLocalTarget( ParticleTarget& target ) const
{
	if (GetContainer().HasLocalTarget())
	{
		// Local target from parent emitter.
		for (CParticleSubEmitter* pParent = m_pParentEmitter; pParent; pParent = pParent->m_pParentEmitter)
		{
			if (pParent->GetParams().eForceGeneration == ParticleForce_Target)
			{
				if (pParent->GetState() >= eEmitter_Active)
				{
					target.vTarget = pParent->GetEmitPos();
					target.fRadius = pParent->GetParams().fPosRandomOffset;
					target.vVelocity = pParent->GetLoc().GetVel();
					return true;
				}
			}
		}
	}
	return false;
}

void CParticleSubEmitter::UpdateForce()
{
	if (m_pParams->eForceGeneration == ParticleForce_None
	|| m_pParams->eForceGeneration == ParticleForce_Target)
		return;

  FUNCTION_PROFILER_SYS(PARTICLE);

	UpdateStrength();

	// Set or clear physical force.
	if (GetState() >= eEmitter_Particles)
	{
		struct SForceGeom
		{
			QuatTS	qpLoc;							// world placement of force
			AABB		bbOuter, bbInner;		// local boundaries of force.
			Vec3		vForce3;
			float		fForceW;
		} force;

		//
		// Compute force geom.
		//

		SPhysEnviron const& PhysEnv = GetMain().GetUniformPhysEnv();
		float fAge = GetAge();

		// Location.
		force.qpLoc = GetLoc().m_qpLoc;
		force.qpLoc.s *= GetMain().GetParticleScale();
		force.qpLoc.t = force.qpLoc * m_pParams->vPositionOffset;

		// Direction.
		Vec3 vFocus = force.qpLoc.q.GetColumn1();
		if (m_pParams->bFocusGravityDir)
			vFocus = (-PhysEnv.m_vUniformGravity).GetNormalizedSafe(vFocus);

		float fFocusAngle = GetCurValue(m_pParams->fFocusAngle);
		if (fFocusAngle != 0.f)
		{
			float fAzimuth = GetCurValue(m_pParams->fFocusAzimuth, 360.f);

			// Rotate focus about X.
			Vec3 vRot(1,0,0);
			vFocus = Quat::CreateRotationAA(DEG2RAD(fAzimuth), Vec3(0,1,0)) 
						 * Quat::CreateRotationAA(DEG2RAD(fFocusAngle), vRot) * vFocus;
		}
		force.qpLoc.q = Quat::CreateRotationV0V1(Vec3(0,1,0),vFocus);

		// Set inner box from spawn geometry.
		Quat qToLocal = force.qpLoc.q.GetInverted() * m_pLocEmitter->m_qpLoc.q;
		Vec3 vOffset = qToLocal * Vec3(m_pParams->vRandomOffset);
		force.bbInner.Reset();
		force.bbInner.Add(vOffset, m_pParams->fPosRandomOffset);
		force.bbInner.Add(-vOffset, m_pParams->fPosRandomOffset);

		// Emission directions.
		float fPhiMax = DEG2RAD(GetCurValue(1.f, m_pParams->fEmitAngle)), 
					fPhiMin = DEG2RAD(GetCurValue(0.f, m_pParams->fEmitAngle));

		AABB bbTrav;
		bbTrav.max.y = cosf(fPhiMin);
		bbTrav.min.y = cosf(fPhiMax);
		float fCosAvg = (bbTrav.max.y + bbTrav.min.y) * 0.5f;
		bbTrav.max.x = bbTrav.max.z = (bbTrav.min.y * bbTrav.max.y < 0.f ? 1.f : sin(fPhiMax));
		bbTrav.min.x = bbTrav.min.z = -bbTrav.max.x;
		bbTrav.Add(Vec3(ZERO));

		// Force magnitude: speed times relative particle density.
		float fSpeed = GetCurValue(1.f, m_pParams->fSpeed) * GetMain().GetSpawnParams().fSpeedScale;
		float fForce = fSpeed * GetCurValue(1.f, m_pParams->fAlpha) * force.qpLoc.s;

		float fPLife = GetCurValue(1.f, m_pParams->fParticleLifeTime);
		float fTime = fAge-m_fStartAge;
		if (m_pParams->bContinuous && fPLife > 0.f)
		{
			// Ramp up/down over particle life.
			float fEndAge = GetEndAge();
			if (fTime < fPLife)
				fForce *= fTime/fPLife;
			else if (fTime > fEndAge)
				fForce *= 1.f - (fTime-fEndAge) / fPLife;
		}

		// Force direction.
		force.vForce3.zero();
		force.vForce3.y = fCosAvg * fForce;
		force.fForceW = sqrtf(1.f - square(fCosAvg)) * fForce;

		// Travel distance.
		float fDist = TravelDistance( abs(fSpeed), GetCurValue(1.f, m_pParams->fAirResistance), min(fTime,fPLife) );
		bbTrav.min *= fDist;
		bbTrav.max *= fDist;

		// Set outer box.
		force.bbOuter = force.bbInner;
		force.bbOuter.Augment(bbTrav);

		// Expand by size.
		float fSize = GetCurValue(1.f, m_pParams->fSize);
		force.bbOuter.Expand( Vec3(fSize) );

		// Scale: Normalise box size, so we can handle some geom changes through scaling.
		Vec3 vSize = force.bbOuter.GetSize()*0.5f;
		float fRadius = max(max(vSize.x, vSize.y), vSize.z);

		if (fForce * fRadius == 0.f)
		{
			// No force.
			if (m_pForce)
			{
				GetPhysicalWorld()->DestroyPhysicalEntity(m_pForce);
				m_pForce = NULL;
			}
			return;
		}

		force.qpLoc.s *= fRadius;
		float fIRadius = 1.f / fRadius;
		force.bbOuter.min *= fIRadius;
		force.bbOuter.max *= fIRadius;
		force.bbInner.min *= fIRadius;
		force.bbInner.max *= fIRadius;

		//
		// Create physical area for force.
		//

		primitives::box geomBox;
		geomBox.Basis.SetIdentity();
		geomBox.bOriented = 0;
		geomBox.center = force.bbOuter.GetCenter();
		geomBox.size = force.bbOuter.GetSize() * 0.5f;

		pe_status_pos spos;
		if (m_pForce)
		{
			// Check whether shape changed.
			m_pForce->GetStatus(&spos);
			if (spos.pGeom)
			{
				primitives::box curBox;
				spos.pGeom->GetBBox(&curBox);
				if (!curBox.center.IsEquivalent(geomBox.center, 0.001f)
				 || !curBox.size.IsEquivalent(geomBox.size, 0.001f))
				 spos.pGeom = NULL;
			}
			if (!spos.pGeom)
			{
				GetPhysicalWorld()->DestroyPhysicalEntity(m_pForce);
				m_pForce = NULL;
			}
		}

		if (!m_pForce)
		{
			IGeometry *pGeom = m_pPhysicalWorld->GetGeomManager()->CreatePrimitive( primitives::box::type, &geomBox );
			m_pForce = m_pPhysicalWorld->AddArea( pGeom, force.qpLoc.t, force.qpLoc.q, force.qpLoc.s );
			if (!m_pForce)
				return;

			// Tag area with this emitter, so we can ignore it in the emitter family.
			pe_params_foreign_data fd;
			fd.pForeignData = (void*)&GetMain();
			fd.iForeignData = fd.iForeignFlags = 0;
			m_pForce->SetParams(&fd);
		}
		else
		{
			// Update position & box size as needed.
			if (!spos.pos.IsEquivalent(force.qpLoc.t, 0.01f)
				|| !spos.q.IsEquivalent(force.qpLoc.q)
				|| spos.scale != force.qpLoc.s)
			{
				pe_params_pos pos;
				pos.pos = force.qpLoc.t;
				pos.q = force.qpLoc.q;
				pos.scale = force.qpLoc.s;
				m_pForce->SetParams(&pos);
			}
		}

		// To do: 4D flow
		pe_params_area area;
		float fVMagSqr = force.vForce3.GetLengthSquared(),
					fWMagSqr = square(force.fForceW);
		float fMag = sqrtf(fVMagSqr + fWMagSqr);
		area.bUniform = (fVMagSqr > fWMagSqr) * 2;
		if (area.bUniform)
		{
			force.vForce3 *= fMag * isqrt_tpl(fVMagSqr);
		}
		else
		{
			force.vForce3.z = fMag * (force.fForceW < 0.f ? -1.f : 1.f);
			force.vForce3.x = force.vForce3.y = 0.f;
		}
		area.falloff0 = force.bbInner.GetRadius();
		area.size.x = max( abs(force.bbOuter.min.x), abs(force.bbOuter.max.x) );
		area.size.y = max( abs(force.bbOuter.min.y), abs(force.bbOuter.max.y) );
		area.size.z = max( abs(force.bbOuter.min.z), abs(force.bbOuter.max.z) );

		if (m_pParams->eForceGeneration == ParticleForce_Gravity)
			area.gravity = force.vForce3;
		m_pForce->SetParams(&area);

		if (m_pParams->eForceGeneration == ParticleForce_Wind)
		{
			pe_params_buoyancy buoy;
			buoy.iMedium = 1;
			buoy.waterDensity = buoy.waterResistance = 0;
			buoy.waterFlow = force.vForce3;
			buoy.waterPlane.n = PhysEnv.m_plUniformWater.n;
			buoy.waterPlane.origin = PhysEnv.m_plUniformWater.n * -PhysEnv.m_plUniformWater.d;
			m_pForce->SetParams(&buoy);
		}
	}
	else
	{
		if (m_pForce)
		{
			GetPhysicalWorld()->DestroyPhysicalEntity(m_pForce);
			m_pForce = NULL;
		}
	}
}

SPU_NO_INLINE void CParticleSubEmitter::Update()
{
	// Evolve emitter state.
	float fAge = GetAge();

	// Handle pulsing.
	// First of individual sub-effect repetition, or emitter entity repetition.
	if (GetMain().GetSpawnParams().fPulsePeriod > 0.f)
	{
		float fRepeat = round_up(fAge, GetMain().GetSpawnParams().fPulsePeriod);
		m_fRepeatAge = min(m_fRepeatAge, fRepeat);
	}

	if (fAge >= m_fRepeatAge)
		Initialize(fAge - m_fRepeatAge);

	if (GetParams().bSpawnOnParentCollision && fAge < m_fStartAge && fAge >= GetLoc().GetCollideAge())
		Initialize(fAge - GetLoc().GetCollideAge());
}

void CParticleSubEmitter::UpdateSound()
{
	if (!m_bSoundPlayed)
	{
		// Start sound if required.
		if (m_fSoundRange > 0.f)
		{
			if (!SoundInRange())
				return;
		}

		if (m_pParams->sSound.empty() || !GetCVars()->e_Particles)
			return;

		// Stop any existing looping sound. Do not stop existing one-shot sounds!!
		// If we hit a start delay fAge will be smaller than m_fStartAge
		// therefore make sure we always stop remaining sounds
		if (m_pSound != 0 && (m_pSound->GetFlags() & FLAG_SOUND_LOOP))
		{
			m_pSound->Stop();
			m_pSound = 0;
		}

		// Handle sound start/stop.
		float fAge = GetAge();
		if (fAge >= m_fStartAge)
		{
			float fEndAgeLoop = GetEndAge(),
						fEndAgeTransient = m_fStartAge + 0.25f;

			if (fAge < max(fEndAgeLoop, fEndAgeTransient))
			{
				int nSndFlags = FLAG_SOUND_DEFAULT_3D;
				if (m_pParams->bContinuous)
					// Will apply only to legacy .wav sounds, not events.
					nSndFlags |= FLAG_SOUND_LOOP;

				m_pSound = gEnv->pSoundSystem->CreateSound( m_pParams->sSound.c_str(), nSndFlags );
				if (m_pSound)
				{
					m_pSound->SetSemantic(eSoundSemantic_Particle);

					// Don't play sounds too late (this is only important if pulse time and emitter time are not set to 0 (infinite emitter))
					if (fAge >= fEndAgeTransient && GetCurValue(m_pParams->fPulsePeriod) != 0.0f && GetCurValue(m_pParams->fEmitterLifeTime) != 0.0f)
					{
						m_pSound->Stop();
						m_pSound = 0;
					}

					if (m_fSoundRange == 0.f && m_pSound && m_pSound->GetFlags() & FLAG_SOUND_RADIUS)
					{
						// Only start sounds within range.
						m_fSoundRange = m_pSound->GetMaxDistance() + fSOUND_RANGE_START_BUFFER;
						if (!SoundInRange())
						{
							int const bLooping = m_pSound->GetFlags() & FLAG_SOUND_LOOP;
							m_pSound->Stop();
							m_pSound = 0;
							if (bLooping)
								// Do not set bSoundPlayed, check range every frame.
								return;
						}
					}

					if (m_pSound)
					{
						// Start sound.
						m_pSound->SetPosition( GetEmitPos() );
						m_pSound->GetInterfaceExtended()->SetVelocity( GetLoc().GetVel() );
				
						if (m_pParams->fSoundFXParam.GetMinValue() < 1.f)
						{
								m_pSound->SetParam("particlefx", 1.f - m_pParams->fSoundFXParam.GetVarValue(m_ChaosKey, GetStrength(0.0f, m_pParams->eSoundControlTime), 0.0f), false);
						}

						m_pSound->Play();
						m_bSoundPlayed = true;
					}
				}
			}
		}
	}
	else if (m_pSound != 0)
	{
		// Stop a looping sound on its desired stop-time.
		bool bStopSound = false;
		if (m_pParams->eSoundControlTime != SoundControlTime_EmitterPulsePeriod && (m_pSound->GetFlags() & FLAG_SOUND_LOOP))
		{
			switch (GetState())
			{
			case eEmitter_Particles:
				{
					// If this state is set we just finished emitter life time,
					// therefore stop here if the control time is set to emitter life time.
					if (m_pParams->eSoundControlTime == SoundControlTime_EmitterLifeTime)
						bStopSound = true;
				}
				break;
			case eEmitter_Dormant:
				{
					// If this state is set we just finished emitter extended life time (last particle died),
					// therefore stop here if the control time is set to extended emitter life time.
					if (m_pParams->eSoundControlTime == SoundControlTime_EmitterExtendedLifeTime)
						bStopSound = true;
				}
				break;
			}
		}

		if (bStopSound)
		{
			m_pSound->Stop();
			m_pSound = 0;
		}
		else
		{
			// Update all sounds (looping + one-shot)
			m_pSound->SetPosition( GetEmitPos() );
			m_pSound->GetInterfaceExtended()->SetVelocity( GetLoc().GetVel() );

			if (!m_pParams->fSoundFXParam.IsConstant())
			{
				// Update SoundFX modifier
				m_pSound->SetParam("particlefx", 1.f - m_pParams->fSoundFXParam.GetVarValue(m_ChaosKey, GetStrength(0.0f, m_pParams->eSoundControlTime), 0.0f), false);
			}

		}
	}
}

SPU_NO_INLINE int CParticleSubEmitter::EmitParticle(SParticleUpdateContext const& context, float fAge, IStatObj* pStatObj, 
																										IPhysicalEntity* pPhysEnt, const QuatTS* pLocation, const Vec3* pVel)
{
	QuatTS qpLoc;
	if (pLocation)
	{
		// Emit at specified location.
		qpLoc = *pLocation;
	}
	else
	{
		// Use emitter location, interpolate if emitter moved.
		float fUpdateTime = GetTimeToUpdate();
		QuatTS rLocLocation = GetLoc().GetLocation();
		if (fAge*fUpdateTime > 0.f && m_qpLastLoc.s >= 0.f && !m_qpLastLoc.IsEquivalent(rLocLocation))
		{
			if (GetParams().bMoveRelEmitter)
			{
				qpLoc = m_qpLastLoc;
			}
			else
			{
				// Interpolate emission position.
				Interp(qpLoc, rLocLocation, m_qpLastLoc, fAge / fUpdateTime);
			}
		}
		else
		{
			qpLoc = rLocLocation;
		}
	}

	if (!pStatObj && !pPhysEnt && m_pParams->pStatObj != 0 && m_pParams->pStatObj->GetSubObjectCount())
	{
		// If bGeomInPieces, emit one particle per piece.
		// Else iterate in 2 passes to count pieces, and emit a random piece.
		int nPiece = -1;
		while (!pStatObj)
		{
			int nPieces = 0;
			for (int i = m_pParams->GetSubGeometryCount()-1; i >= 0; i--)
			{
				IStatObj::SSubObject* pSub = SPU_MAIN_PTR(m_pParams->GetSubGeometry(i));
				if (pSub)
				{
					if (m_pParams->bGeometryInPieces)
					{
						QuatTS qpSub = qpLoc * QuatTS(pSub->localTM);
						if (!EmitParticle(context, fAge, pSub->pStatObj, pPhysEnt, &qpSub, pVel))
							continue;
					}
					else if (nPieces == nPiece)
					{
						pStatObj = pSub->pStatObj;
						break;
					}
					nPieces++;
				}
			}
			if (m_pParams->bGeometryInPieces)
				return nPieces;
			if (nPieces <= 1)
				break;
			nPiece = Random(nPieces);
		}
	}

  // Allow particle growth if specified; otherwise, reject.
	CParticle* pPart = GetContainer().AddParticle(this);
	if (pPart)
	{
		// use a local buffer on SPU and then transfer the result to main memory
#if defined(__SPU__)
		char buffer[sizeof(CParticle)] _ALIGN(128) = {0};
#else
		char buffer = NULL;
#endif

		CParticle* pParticle = SPU_PTR_SELECT( pPart, (CParticle*)&buffer[0] );
	
		pParticle->Init( context, fAge, this, qpLoc, pLocation != 0, pVel, pStatObj, pPhysEnt );

#if defined(__SPU__)
		__spu_invalidate_cache_range( (uint32_t) pPart, sizeof(CParticle) );
		memtransfer_to_main( pPart, pParticle, sizeof(CParticle), 1 );
		memtransfer_sync(1);
#endif

		m_nEmitIndex++;
		return 1;
	}

	return 0;
}

//////////////////////////////////////////////////////////////////////////
bool CParticleSubEmitter::SoundInRange() const
{
	Vec3 vSoundPos = GetEmitPos();
	Vec3 vCamPos = gEnv->pRenderer->GetCamera().GetPosition();
	return (vCamPos - vSoundPos).GetLengthSquared() <= sqr(m_fSoundRange);
}

SPU_NO_INLINE float CParticleSubEmitter::GetEmitRate() const
{
	const ResourceParticleParams& params = GetParams();

	float fCount = GetCurValue(m_pParams->fCount);
	if (!GetContainer().GetParent())
		fCount *= GetMain().GetEmitCountScale();
	float fLife = GetContainer().GetMaxParticleLife(false);
	float fPulse = m_fRepeatAge - m_fStartAge;

	if (params.bContinuous)
	{
		if (fLife <= 0.f)
			return 0.f;

		float fEmitterLife = GetCurValue(params.fEmitterLifeTime);
		if (fEmitterLife > 0.f && fEmitterLife < fPulse)
		{
			// Actually pulsing.
			if (fEmitterLife < fLife)
				// Ensure enough particles emitted.
				fCount *= fLife / fEmitterLife;

			// Reduce emit rate for overlapping pulsing.
			if (fLife >= fPulse)
			{
				float fLdP = floor(fLife/fPulse);
				float fLmP = fLife - fLdP*fPulse;
				float fOverlap = (fLdP * fEmitterLife + min(fLmP,fEmitterLife)) / min(fLife,fEmitterLife);
				fLife *= fOverlap;
			}
		}

		// Compute continual emission rate which maintains fCount particles.
		return fCount / fLife;
	}
	else
	{
		if (fPulse < fLife)
		{
			// Reduce emit count for overlapping pulsing.
			fCount *= fPulse / fLife;
		}
		return fCount;
	}
}

Vec3 CParticleSubEmitter::GetEmitPos() const
{
	return GetLoc().GetWorldPosition(GetParams().vPositionOffset);
}

float CParticleSubEmitter::GetAge() const
{
	return GetLoc().GetAge();
}

float CParticleSubEmitter::GetEndAge() const
{
	return min(m_fEndAge, GetLoc().GetStopAge());
}

CParticleEmitter& CParticleSubEmitter::GetMain() const
{ 
	return GetContainer().GetMain(); 
}

void CParticleSubEmitter::SetLastLoc()
{
	m_qpLastLoc = GetLoc().GetLocation();
}

#undef USE_SPU


