////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2010.
// -------------------------------------------------------------------------
//  File name:   ParticleSubEmitter.h
//  Version:     v1.00
//  Created:     20/04/2010 by Corey.
//  Description: Split out from ParticleEmitter.h
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#ifndef __particlesubemitter_h__
#define __particlesubemitter_h__
#pragma once

#ifdef _WIN32
#pragma warning(disable: 4355)
#endif

#include "ParticleEffect.h"
#include "ParticleEnviron.h"
#include "ParticleUtils.h"
#include "ParticleContainer.h"
#include "GeomQuery.h"
#include "BitFiddling.h"
#include <SpuUtils.h>

#undef PlaySound

class CParticleContainer;
class CParticleLocEmitter;
class CParticleEmitter;
struct SParticleUpdateContext;

//////////////////////////////////////////////////////////////////////////
enum EActivateMode
{ 
	eAct_Activate,
	eAct_Deactivate,
	eAct_Reactivate,
	eAct_Restart,
	eAct_Pause,
	eAct_Resume,
};

enum EEmitterState
{
	eEmitter_Dead,				// No current or future particles.
	eEmitter_Dormant,			// No current particles, but may in future.
	eEmitter_Particles,		// Currently has particles.
	eEmitter_Active,			// Currently emitting particles
};

#if defined(PS3)
//////////////////////////////////////////////////////////////////////////
// struct to group a batch of emitters together for SPU update
struct UpdateParticlesEmitterBatch
{		
	// don't incrase this to much, since it is passed by value to the spu,
	// where a maximum of 80 bytes is allowed
	static const int MAX_BATCHES = 15;
	CParticleEmitter *batch[MAX_BATCHES];
};
#endif // PS3


class CParticleSubEmitter: public Cry3DEngineBase, public DumbRefCount
// Maintains an emitter source state, emits particles to a container
// Ref count increased only by emitted particles
{
public:

	CParticleSubEmitter( CParticleSubEmitter* pParent, CParticleLocEmitter* pLoc, CParticleContainer* pCont );
	ILINE ~CParticleSubEmitter()										{ Destroy(); }

	ResourceParticleParams const& GetParams() const	
	{
#if defined(__SPU__)
		return *( gSpuParticleParams.GetParams( (ResourceParticleParams *)m_pParams ) );
#else
		return *m_pParams; 
#endif
	}
	CParticleContainer& GetContainer() const				{ return *m_pContainer; }
	CParticleLocEmitter& GetLoc() const							{ return *m_pLocEmitter; }
	bool GetLocalTarget( ParticleTarget& target ) const;
	CParticleEmitter& GetMain() const;

	// State.
	EEmitterState GetState() const;
	void Activate( EActivateMode mode, float fPast = 0.f );

	// Timing.
	float GetAge() const;

	SPU_NO_INLINE float GetRelativeAge( float fAgeAdjust = 0.f, ESoundControlTime const eControl = SoundControlTime_EmitterLifeTime ) const
	{
		// Choose control time.
		float fEnd;
		switch (eControl)
		{
		default:
		case SoundControlTime_EmitterLifeTime:
			fEnd = min(GetEndAge(), m_fRepeatAge);
			break;
		case SoundControlTime_EmitterExtendedLifeTime:
			fEnd = min(GetEndAge() + GetContainer().GetMaxParticleLife(), m_fRepeatAge);
			break;
		case SoundControlTime_EmitterPulsePeriod:
			fEnd = m_fRepeatAge;
			break;
		}

		float const fAge = GetAge() + fAgeAdjust;
		if (min(fAge, fEnd) <= m_fStartAge)
			return 0.f;

		return div_min(fAge - m_fStartAge, fEnd - m_fStartAge, 1.f);
	}

	float GetStrength( float fAgeAdjust = 0.f, ESoundControlTime const eControl = SoundControlTime_EmitterLifeTime ) const;

	// Param evaluation.
	template<class T> inline
	T GetCurValue( TVarParam<T> const& param ) const
	{
		return param.GetVarValue( m_ChaosKey );
	}

	template<class T> inline
	T GetCurValue( TVarEParam<T> const& param, float fAbs = 0.f ) const
	{
		return param.GetVarValue( m_ChaosKey, m_fStrength, fAbs );
	}

	template<class T> inline
	T GetCurValue( float fMinMax, TVarEParam<T> const& param ) const
	{
		return param.GetVarValue( fMinMax, m_fStrength );
	}

	float GetEmitRate() const;
	Vec3 GetEmitPos() const;

	// Actions.
	void Update();
	void UpdateSound();
	void ResetLoc()
	{
		m_qpLastLoc.s = -1.f;
	}
	void SetLastLoc();
	void MoveRelative( Vec3& vPos, Quat& qRot, Vec3& vVel );
	void UpdateForce();
	int EmitParticle(SParticleUpdateContext const& context, const float fAge, IStatObj* pStatObj = NULL, IPhysicalEntity* pPhysEnt = NULL, 
									 const QuatTS* pLocation = SPU_LOCAL_PTR((QuatTS*)NULL), const Vec3* pVel = NULL);
	void EmitParticles( SParticleUpdateContext const& context );
	
	uint32 GetEmitIndex() const
	{
		return m_nEmitIndex;
	}

	void GetMemoryUsage( ICrySizer *pSizer ) const
	{
		if( pSizer->AddObject(this, sizeof(*this) + 16) == false )
			return;

		pSizer->AddObject(m_pParams);
		pSizer->AddObject(m_pContainer);
		pSizer->AddObject(m_pParentEmitter);
		pSizer->AddObject(m_pLocEmitter);
	}

private:

	// Associated structures.
	ResourceParticleParams const*		m_pParams;
	CParticleContainer*							m_pContainer;					// Direct or shared container to emit particles into.
	_smart_ptr<CParticleSubEmitter>	m_pParentEmitter;
	_smart_ptr<CParticleLocEmitter>	m_pLocEmitter;

	// State.
	float					m_fStartAge;			// Relative age when scheduled to start (default 0).
	float					m_fEndAge;				// Relative age when scheduled to end (fHUGE if never).
	float					m_fRepeatAge;			// Relative age when scheduled to repeat (fHUGE if never).

	float					m_fStrength;			// Emitter strength, set externally, or from relative age.

	CChaosKey			m_ChaosKey;				// Seed for randomising; inited every pulse.
	uint32				m_nEmitIndex;
	float					m_fToEmit;				// Fractional particles to emit.
	QuatTS				m_qpLastLoc;

	// External objects.
	bool								m_bSoundPlayed;		// Has been played since last Initialize.
	float								m_fSoundRange;		// Cached sound radius, to avoid playing far sounds.
	_smart_ptr<ISound>	m_pSound;
	IPhysicalEntity*		m_pForce;

	// Methods.
	void Initialize( float fPast = 0.f );
	void Deactivate();
	bool SoundInRange() const;

	float GetEndAge() const;

	ILINE float GetTimeToUpdate() const
	{
		//return min( GetContainer().GetTimeToUpdate(), GetAge() );
		const float containerUpdateTime = GetContainer().GetTimeToUpdate();
		const float age = GetAge();

		if(containerUpdateTime < age)
		{
			return containerUpdateTime;
		}
		else
		{
			return age;
		}
	}

	void UpdateStrength()
	{
		m_fStrength = GetStrength();
		assert(m_fStrength >= 0.f && m_fStrength <= 1.f);
	}

	void Destroy();

} _ALIGN(SUB_EMITTER_ALIGNMENT);

#endif // __particlesubemitter_h__
