////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2002.
// -------------------------------------------------------------------------
//  File name:   terrain_det_tex.cpp
//  Version:     v1.00
//  Created:     28/5/2001 by Vladimir Kajalin
//  Compilers:   Visual Studio.NET
//  Description: terrain detail textures
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "terrain.h"
#include "terrain_sector.h"
#include "3dEngine.h"
#include "VoxMan.h"
#include "IndexedMesh.h"
#include "MeshCompiler/MeshCompiler.h"
#include "VoxTerrain.h"

CVoxMeshNode::CVoxMeshNode(AABB aabb, CVoxMeshNode * pParent) 
{ 
  ZeroStruct(*this);
  m_nodeBox = aabb;
  m_pParent = pParent;
  m_fNodeSize = m_nodeBox.GetSize().z;
}

CVoxDataNode::CVoxDataNode(short nMatId) 
{ 
  ZeroStruct(*this);
  m_nMatId = nMatId;
}

void CVoxDataNode::ReleaseChilds()
{
  for(int nChildId=0; nChildId<8; nChildId++)
    SAFE_DELETE(m_arrChilds[nChildId]);
}

void CVoxMeshNode::ReleaseChilds()
{
  for(int nChildId=0; nChildId<8; nChildId++)
    SAFE_DELETE(m_arrChilds[nChildId]);
}

CVoxDataNode::~CVoxDataNode()
{
  ReleaseChilds();
}

CVoxMeshNode::~CVoxMeshNode()
{
  ReleaseChilds();
  SAFE_DELETE(m_pVoxObj);
  m_pVoxTerrain->m_rebuildLine.Delete(this);
}

Vec3 ColorB_To_Vec3(ColorB color)
{
  Vec3 vCol;
  vCol.x = 1.f/255.f*color.r;
  vCol.y = 1.f/255.f*color.g;
  vCol.z = 1.f/255.f*color.b;
  return vCol;
}

ColorB Vec3_To_ColorB(Vec3 vec)
{
  vec.CheckMin(Vec3(1,1,1));
  ColorB col = vec*255;
  return col;
}

void CVoxDataNode::UpdateStateFromChilds(int nMatId, EVoxelEditOperation eOperation)
{
  FUNCTION_PROFILER_3DENGINE;

  m_fIsoVal = 0;

  for(int nChildId=0; nChildId<8; nChildId++)
  {
    if(m_arrChilds[nChildId])
      m_fIsoVal += m_arrChilds[nChildId]->m_fIsoVal;
  }

  m_fIsoVal /= 8.f;

  assert(m_fIsoVal>=0.f && m_fIsoVal<=1.f);

  m_nMatId = nMatId;
 
  if(m_fIsoVal == 0)
  {
    ReleaseChilds();
  }
}

void CVoxMeshNode::ScheduleRebuild()
{
  CVoxMeshNode * pNode = this;
  while(pNode)
  {
    if(pNode->m_pVoxObj)
      pNode->m_pVoxObj->m_nScheduleRebuildFrameId = GetMainFrameID();
    pNode = pNode->m_pParent;
  }

  if(m_pVoxTerrain && m_pVoxTerrain->m_rebuildLine.Find(this)<0)
    m_pVoxTerrain->m_rebuildLine.Add(this);
}

void CVoxDataNode::Submit(Sphere sp, EVoxelBrushShape eShape, const AABB & nodeBox)
{
  bool bOverlap = false;

  if(eShape == evbsSphere)
  {
    bOverlap = Overlap::Sphere_AABB(sp,nodeBox);
  }
  else if(eShape == evbsBox)
  {
    Vec3 r(sp.radius,sp.radius,sp.radius);
    AABB brushBox(sp.center-r, sp.center+r);
    bOverlap = Overlap::AABB_AABB(brushBox,nodeBox);
  }

  if(!bOverlap)
    return;

  m_fIsoValPrev = m_fIsoVal;

  for(int nChildId=0; nChildId<8; nChildId++)
    if(m_arrChilds[nChildId])
      m_arrChilds[nChildId]->Submit(sp, eShape, GetChildBBox(nodeBox, nChildId));
}

void CVoxMeshNode::RequestUpdate(Sphere sp, EVoxelBrushShape eShape)
{
  bool bOverlap = false;

  if(eShape == evbsSphere)
  {
    bOverlap = Overlap::Sphere_AABB(sp,m_nodeBox);
  }
  else if(eShape == evbsBox)
  {
    Vec3 r(sp.radius,sp.radius,sp.radius);
    AABB brushBox(sp.center-r, sp.center+r);
    bOverlap = Overlap::AABB_AABB(brushBox,m_nodeBox);
  }

  if(!bOverlap)
    return;

  if(!HasChilds())
  {
    ScheduleRebuild();
  }
  else
  {
    for(int nChildId=0; nChildId<8; nChildId++)
      if(m_arrChilds[nChildId])
        m_arrChilds[nChildId]->RequestUpdate(sp, eShape);
  }
}

void CVoxMeshNode::CheckAllocate(Sphere sp, EVoxelBrushShape eShape, int nDepth, int nMaxDepth)
{
  nDepth++;

  bool bOverlap = false;

  if(eShape == evbsSphere)
  {
    bOverlap = Overlap::Sphere_AABB(sp,m_nodeBox);
  }
  else if(eShape == evbsBox)
  {
    Vec3 r(sp.radius,sp.radius,sp.radius);
    AABB brushBox(sp.center-r, sp.center+r);
    bOverlap = Overlap::AABB_AABB(brushBox,m_nodeBox);
  }

  if(!bOverlap)
    return;

  if(nDepth < nMaxDepth)
  {
    for(int nChildId=0; nChildId<8; nChildId++)
    {
      if(!m_arrChilds[nChildId])
        m_arrChilds[nChildId] = new CVoxMeshNode(GetChildBBox(nChildId), this);

      m_arrChilds[nChildId]->CheckAllocate(sp, eShape, nDepth, nMaxDepth);
    }
  }
  else
  {
//    ScheduleRebuild();
  }
}

void CVoxDataNode::DoVoxelShape(Sphere sp, EVoxelBrushShape eShape, EVoxelEditOperation eOperation, int nMatId, 
                                const AABB & nodeBox, int nDepth, int nMaxDepth)
{
  FUNCTION_PROFILER_3DENGINE;

  nDepth++;

  if(nDepth < nMaxDepth)
  {
    for(int nChildId=0; nChildId<8; nChildId++)
    {
      AABB aabb = GetChildBBox(nodeBox, nChildId);

      bool bOverlap = false;

      if(eOperation == eveoCopyTerrain && aabb.GetSize().z < 64.f)
      {
        sp = Sphere(aabb.GetCenter(), 2);
        sp.center.z = 2;
      }

      if(eShape == evbsSphere)
      {
        bOverlap = Overlap::Sphere_AABB(sp,aabb);
      }
      else if(eShape == evbsBox)
      {
        float r = sp.radius;
        AABB brushBox(sp.center-Vec3(r,r,r), sp.center+Vec3(r,r,r));
        bOverlap = Overlap::AABB_AABB(brushBox,aabb);
      }

      if(bOverlap)
      {
        if(!m_arrChilds[nChildId] && (eOperation==eveoCreate || eOperation==eveoCopyTerrain))
        {
          m_arrChilds[nChildId] = new CVoxDataNode(nMatId);
        }

        if(m_arrChilds[nChildId])
          m_arrChilds[nChildId]->DoVoxelShape(sp, eShape, eOperation, nMatId, aabb, nDepth, nMaxDepth);
      }
    }

    UpdateStateFromChilds(nMatId, eOperation);
  }
  else
  {
    float t = 1.f;

    if(eShape == evbsSphere) 
      t = sqrt(SATURATE(2.f - 2.f * nodeBox.GetCenter().GetDistance(sp.center) / sp.radius));

    if(eOperation == eveoCreate)
    {
      m_fIsoVal += t;
      if(t > 0.25f)
        m_nMatId = nMatId;
    }
    else if(eOperation == eveoSubstract)
    {
      m_fIsoVal -= t;
    }
    else if(eOperation == eveoMaterial)
    {
      if(t > 0.25f)
        m_nMatId = nMatId;
    }
    else if(eOperation == eveoCopyTerrain)
    {
      m_fIsoVal = 1.f;
    }
    else if(eOperation == eveoBlur)
    {
      Vec3 vPos = nodeBox.GetCenter();
      float fNewVal = 0;
      short nMatId = 0;
      float fCount = 0;

      for(int x=-1; x<=1; x++)
      {
        for(int y=-1; y<=1; y++)
        {
          for(int z=-1; z<=1; z++)
          {
            float fNodeSize = 2.f;
            fNewVal += m_pVoxTerrain->m_pRootData->GetVoxelValueF(vPos + Vec3(x,y,z)*fNodeSize, fNodeSize, &nMatId, true, m_pVoxTerrain->m_pRootMesh->GetNodeAABB());
            fCount += 1.f;
          }
        }
      }

      fNewVal = fNewVal / fCount;

      float t2 = t*t;

      m_fIsoVal = min(m_fIsoVal, t2*fNewVal + (1.f-t2)*m_fIsoVal);
    }

    m_fIsoVal = SATURATE(m_fIsoVal);
  }
}

AABB CVoxDataNode::GetChildBBox(const AABB & parentBox, int nChildId)
{
  int x = (nChildId/4);
  int y = (nChildId-x*4)/2;
  int z = (nChildId-x*4-y*2);
  Vec3 vSize = parentBox.GetSize()*0.5f;
  Vec3 vOffset = vSize;
  vOffset.x *= x;
  vOffset.y *= y;
  vOffset.z *= z;
  AABB childBox;
  childBox.min = parentBox.min + vOffset;
  childBox.max = childBox.min + vSize;
  return childBox;
}

AABB CVoxMeshNode::GetChildBBox(int nChildId)
{
  int x = (nChildId/4);
  int y = (nChildId-x*4)/2;
  int z = (nChildId-x*4-y*2);
  Vec3 vSize = m_nodeBox.GetSize()*0.5f;
  Vec3 vOffset = vSize;
  vOffset.x *= x;
  vOffset.y *= y;
  vOffset.z *= z;
  AABB childBox;
  childBox.min = m_nodeBox.min + vOffset;
  childBox.max = childBox.min + vSize;
  return childBox;
}

void CVoxDataNode::RenderDebug(const AABB & nodeBox)
{
  if(!GetCamera().IsAABBVisible_E(nodeBox))
    return;

  FUNCTION_PROFILER_3DENGINE;

  float fDistance = sqrtf(Distance::Point_AABBSq(GetCamera().GetPosition(), nodeBox));

  for(int nChildId=0; nChildId<8; nChildId++)
  {
    if(m_arrChilds[nChildId])
      m_arrChilds[nChildId]->RenderDebug(GetChildBBox(nodeBox, nChildId));
  }

  short nMatId=0;
  AABB aabb = nodeBox;
  aabb.min -= Vec3(1,1,1);
  aabb.max -= Vec3(1,1,1);
  DrawBBox(aabb, (m_fIsoVal>.5) ? Col_White : Col_Green);
}

void CVoxMeshNode::Render()
{
  if(!GetCamera().IsAABBVisible_E(m_nodeBox))
    return;

  FUNCTION_PROFILER_3DENGINE;

  float fDistance = sqrtf(Distance::Point_AABBSq(GetCamera().GetPosition(), m_nodeBox));

  if(GetCVars()->e_voxel_debug==1)
    DrawBBox(m_nodeBox, Col_Red);

  if(((fDistance > m_fNodeSize && (GetCVars()->e_voxel_lods_num && m_pVoxObj && m_pVoxObj->m_nScheduleRebuildFrameId < m_pVoxObj->m_nRebuildDoneFrameId)) || !HasChilds()))
  {
    SRendParams rParams;
    rParams.nDLightMask = 1;
    rParams.AmbientColor = Get3DEngine()->GetSkyColor();
    if(m_pVoxObj)
      m_pVoxObj->Render(rParams);
  }
  else
  {
    for(int nChildId=0; nChildId<8; nChildId++)
    {
      if(m_arrChilds[nChildId])
        m_arrChilds[nChildId]->Render();
    }
  }
}

void CVoxMeshNode::UpdateMesh()
{
  if(!HasChilds())
  {
    InitVoxelObject();
    m_pVoxObj->Compile(NULL);

    if(m_pParent && GetCVars()->e_voxel_lods_num)
      m_pParent->ScheduleRebuild();
  }
  else 
  {
    InitVoxelObjectFromChilds();

    if(m_pParent && GetCVars()->e_voxel_lods_num)
      m_pParent->ScheduleRebuild();
  }

  if(m_pVoxObj)
    m_pVoxObj->m_nRebuildDoneFrameId = GetMainFrameID();
}

void CVoxMeshNode::InitVoxelObject()
{
  uint nSize = 32;
  SVoxelChunkVer3 chunk;
  ZeroStruct(chunk);
  chunk.nChunkVersion = 3;
  chunk.vSize.x = chunk.vSize.y = chunk.vSize.z = nSize;
  SSurfaceType * pLayers = GetTerrain()->GetSurfaceTypes();
  for(int i=0; i<VOX_MAX_SURF_TYPES_NUM; i++)
    strcpy(chunk.m_arrSurfaceNames[i], pLayers[i].szName);
  if(!m_pVoxObj)
    m_pVoxObj = new CVoxelObject(m_nodeBox.min, 2.f, nSize, nSize, nSize);
  
  AABB nodeBoxPlus = m_nodeBox;
  nodeBoxPlus.min -= Vec3(2,2,2);
  nodeBoxPlus.max += Vec3(2,2,2);
  CVoxDataNode * pDataProvider = m_pVoxTerrain->m_pRootData;//->GetMinNodeContainingAABB(nodeBoxPlus, m_pVoxTerrain->m_pRootMesh->GetNodeAABB());
  m_pVoxObj->SetDataProvider(pDataProvider ? pDataProvider : m_pVoxTerrain->m_pRootData);

  m_pVoxObj->SetData(&chunk, 0);
  m_pVoxObj->SetBBox(m_nodeBox);  
  Matrix34 tm; tm.SetIdentity();
  tm.SetTranslation(m_nodeBox.min);
  m_pVoxObj->SetMatrix(tm);
}

void CVoxMeshNode::InitVoxelObjectFromChilds()
{
  PrintMessage("Building terrain node mesh, size = %d ...", (int)m_fNodeSize);

  CIndexedMesh * pMeshMerged = new CIndexedMesh();

  // collect meshes
  for(int i=0; i<8; i++)
  {
    if(m_arrChilds[i] && m_arrChilds[i]->m_pVoxObj)
    {
      if(CIndexedMesh * pChildMesh = m_arrChilds[i]->m_pVoxObj->GetIndexedMesh())
      {
        pChildMesh->CheckValid();

        if(!pChildMesh->GetFacesCount())
          pChildMesh->RestoreFacesFromIndices();
        pChildMesh->SetIndexCount(0);

        pChildMesh->CheckValid();

        int nOldVertsCount = pMeshMerged->GetVertexCount();
        pMeshMerged->Append(*pChildMesh->GetMesh());

        void* pStream; 
        int nElementSize;
        pMeshMerged->GetStreamInfo(CMesh::POSITIONS,pStream,nElementSize);

        // translate child vertices
        for(int v=nOldVertsCount; v<pMeshMerged->GetVertexCount(); v++)
        {
          ((Vec3*)pStream)[v] = ((Vec3*)pStream)[v] + m_arrChilds[i]->m_nodeBox.min - m_nodeBox.min;
        }
      }
    }
  }

  pMeshMerged->InitStreamSize();
  pMeshMerged->SetSubSetCount(1);
  pMeshMerged->CalcBBox();

  pMeshMerged->CheckValid();

  // simplify
  Vec3 vMin(VEC_EPSILON,VEC_EPSILON,VEC_EPSILON);
  Vec3 vMax = m_nodeBox.GetSize() - vMin;
  float fScale = m_fNodeSize / 64.f;
  pMeshMerged->SimplifyMesh(fScale * 3.f, AABB(vMin, vMax), fScale * 2.5f);

  pMeshMerged->InitStreamSize();
  pMeshMerged->SetSubSetCount(1);
  pMeshMerged->CalcBBox();

  SAFE_DELETE(m_pVoxObj);

  if(!pMeshMerged->GetFacesCount())
    return;

  InitVoxelObject();

  m_pVoxObj->m_pNodeTex = pMeshMerged->GenerateTexture(GetCVars()->e_voxel_terrain_tex_res, m_pVoxObj->m_pNodeTex, 
    m_nodeBox.GetSize().z, m_pVoxObj->GetPos(true), m_pVoxTerrain->m_pRootData);

  mesh_compiler::CMeshCompiler meshCompiler;
  meshCompiler.Compile( *pMeshMerged, 0 );
  if(!pMeshMerged->GetIndexCount())
    return;

  pMeshMerged->SetFacesCount(0);

  CIndexedMesh::MarkSharpVertices(0.1f, pMeshMerged);

  if(!m_pVoxObj->m_pVoxelMesh)
    m_pVoxObj->m_pVoxelMesh = new CVoxelMesh(&m_pVoxObj->m_arrSurfacesPalette[0]);

  m_pVoxObj->SerializeMesh( pMeshMerged );

  m_pVoxObj->m_pMesh = pMeshMerged;

  m_pVoxObj->m_pVoxelMesh->CreateRenderMeshFromIndexedMesh(pMeshMerged, m_pVoxObj, 0);

  Get3DEngine()->m_lstVoxelObjectsForUpdate.Delete(m_pVoxObj);

  PrintMessagePlus(" ok");
}

float CVoxDataNode::GetVoxelValueF(Vec3 vPos, float fGridSize, short * pMatId, bool bPrev, const AABB & nodeBox)
{
  if(pMatId) 
    *pMatId = -1;

  const float & fIsoVal = bPrev ? m_fIsoValPrev : m_fIsoVal;

  if(fIsoVal == 0.f)
    return 0;

  if(!HasChilds())
  {
    if(fIsoVal>0)
      if(pMatId) 
        *pMatId = m_nMatId;

    return fIsoVal;
  }
  else
  {
    Vec3 vCenter = nodeBox.GetCenter();

    int nFirst = 
      ((vPos.x > vCenter.x) ? 4 : 0) |
      ((vPos.y > vCenter.y) ? 2 : 0) |
      ((vPos.z > vCenter.z) ? 1 : 0);

    if(m_arrChilds[nFirst])
      return m_arrChilds[nFirst]->GetVoxelValueF(vPos, fGridSize, pMatId, bPrev, GetChildBBox(nodeBox, nFirst));
  }

  return 0;
}

int CVoxDataNode::GetTreeDepth(Vec3 vPos, int nDepth, const AABB & nodeBox)
{
  nDepth++;

  if(!HasChilds())
  {
    return nDepth;
  }
  else
  {
    Vec3 vCenter = nodeBox.GetCenter();

    int nFirst = 
      ((vPos.x > vCenter.x) ? 4 : 0) |
      ((vPos.y > vCenter.y) ? 2 : 0) |
      ((vPos.z > vCenter.z) ? 1 : 0);

    if(m_arrChilds[nFirst])
      return m_arrChilds[nFirst]->GetTreeDepth(vPos, nDepth, GetChildBBox(nodeBox, nFirst));
  }

  return 0;
}

int CVoxMeshNode::GetTreeDepth(Vec3 vPos, int nDepth)
{
  nDepth++;

  if(!HasChilds())
  {
    return nDepth;
  }
  else
  {
    Vec3 vCenter = m_nodeBox.GetCenter();

    int nFirst = 
      ((vPos.x > vCenter.x) ? 4 : 0) |
      ((vPos.y > vCenter.y) ? 2 : 0) |
      ((vPos.z > vCenter.z) ? 1 : 0);

    if(m_arrChilds[nFirst])
      return m_arrChilds[nFirst]->GetTreeDepth(vPos, nDepth);
  }

  return 0;
}

/*
ushort CVoxNode::GetVoxelValue(Vec3 vPos, float fGridSize, short * pMatId)
{
  if(pMatId) *pMatId = -1;

  if(m_fIsoVal == 0.f)
    return 0;

  if(!m_nodeBox.IsContainPoint(vPos))
    return 0;

  if(m_fNodeSize <= fGridSize || !m_bHasChilds)
  {
    if(m_fIsoVal>0)
    {
      if(pMatId) *pMatId = m_nMatId;
    }

    return ushort(m_fIsoVal*VOX_MAX_USHORT) & ~VOX_MAT_MASK;
  }
  else
  {
    for(int nChildId=0; nChildId<8; nChildId++)
    {
      if(m_arrChilds[nChildId])
        if(ushort usVal = m_arrChilds[nChildId]->GetVoxelValue(vPos, fGridSize, pMatId))
          return usVal;
    }
  }

  return 0;
}
*/
bool IsAABBInsideHull(SPlaneObject * pHullPlanes, int nPlanesNum, const AABB & aabbBox);
bool IsSphereInsideHull(SPlaneObject * pHullPlanes, int nPlanesNum, const Sphere & objSphere);

void CVoxMeshNode::FillShadowCastersList(CDLight * pLight, ShadowMapFrustum * pFr, PodArray<SPlaneObject> * pShadowHull, bool bUseFrustumTest)
{
  if(pFr->bForSubSurfScattering)
    return;

  FUNCTION_PROFILER_3DENGINE;

  if(bUseFrustumTest && !pFr->Intersect(m_nodeBox))
    return;

  bool bSun = (pLight->m_Flags&DLF_SUN) != 0;

  if(pShadowHull && !pFr->bUseAdditiveBlending && !IsAABBInsideHull(pShadowHull->GetElements(), pShadowHull->Count(), m_nodeBox))
    return;

  float fGSMBoxSize = 1.0f / (pFr->fFrustrumSize * GetCVars()->e_gsm_range);

  Vec3 vCamPos = GetCamera().GetPosition();

  assert(pFr->pLightOwner == pLight->m_pOwner);

  if(!HasChilds())
  {
    if(m_pVoxObj)
      pFr->pCastersList->Add(m_pVoxObj);
  }
  else
  {
    for(int i=0; i<8; i++) 
      if(m_arrChilds[i])
        m_arrChilds[i]->FillShadowCastersList(pLight, pFr, pShadowHull, bUseFrustumTest);
  }
}

float CVoxMeshNode::GetElevation(Vec3 vPos, float & fMaxZ)
{
  Vec3 vStart = vPos; vStart.z = 1024.f;
  Vec3 vEnd = vPos; vEnd.z = 0;
  Ray ray(vStart, vEnd-vStart);
  Vec3 vOut;

  if(!Intersect::Ray_AABB(ray,m_nodeBox,vOut))
    return 0;

  if(!HasChilds())
  {
    if(!m_pVoxObj)
      return 0;

    const AABB & objBox = m_pVoxObj->GetBBox();

    if(	vStart.x>objBox.max.x || vStart.y>objBox.max.y || vStart.x<objBox.min.x || vStart.y<objBox.min.y )
      return 0;

    Matrix34 objMatrix = m_pVoxObj->GetMatrix();
    IRenderMesh * pRenderMesh = m_pVoxObj->GetRenderMesh(0);
    if(!pRenderMesh)
      return 0;

    Matrix34 matInv = objMatrix.GetInverted();
    Vec3 vOSStart = matInv.TransformPoint(vStart);
    Vec3 vOSEnd = matInv.TransformPoint(vEnd);

    Vec3 vOSHitPoint(0,0,0), vOSHitNorm(0,0,0);
    if(CObjManager::RayRenderMeshIntersection(pRenderMesh, vOSStart, vOSEnd-vOSStart, vOSHitPoint, vOSHitNorm, false, NULL))
    {
      Vec3 vHitPoint = objMatrix.TransformPoint(vOSHitPoint);
      if(fMaxZ < vHitPoint.z)
        fMaxZ = vHitPoint.z;
    }
  }
  else
  {
    for(int nChildId=0; nChildId<8; nChildId++)
      if(m_arrChilds[nChildId])
        m_arrChilds[nChildId]->GetElevation(vPos, fMaxZ);
  }

  return fMaxZ;
}

ILINE char AABB_AABB_Inside_Ex( const AABB& aabb1, const AABB& aabb2 ) {
  if ( Overlap::AABB_AABB(aabb1,aabb2) ) {
    if  (aabb1.min.x<aabb2.min.x) return 1;
    if  (aabb1.min.y<aabb2.min.y) return 1; 
    if 	(aabb1.min.z<aabb2.min.z) return 1; 
    if  (aabb1.max.x>aabb2.max.x) return 1;
    if  (aabb1.max.y>aabb2.max.y) return 1;
    if  (aabb1.max.z>aabb2.max.z) return 1;
    //yes, its inside 
    return 2;
  }
  return 0;
}

CVoxDataNode * CVoxDataNode::GetMinNodeContainingAABB(const AABB & box, const AABB & nodeBox)
{
  if(AABB_AABB_Inside_Ex(box, nodeBox) == 2)
  {
    for(int nChildId=0; nChildId<8; nChildId++)
      if(m_arrChilds[nChildId])
        if(CVoxDataNode * pNode = m_arrChilds[nChildId]->GetMinNodeContainingAABB(box, GetChildBBox(nodeBox,nChildId)))
          return pNode;

    return this;
  }

  return NULL;
}

uint CVoxMeshNode::GetRebuildFrameId() 
{ 
  return m_pVoxObj ? m_pVoxObj->m_nRebuildDoneFrameId : 0; 
}

bool CVoxMeshNode::HasChilds()
{ 
  const int tmp[8] = {0,0,0,0,0,0,0,0}; 
  return memcmp(m_arrChilds, tmp, sizeof(tmp)) != 0;
}

bool CVoxDataNode::HasChilds()
{ 
  const int tmp[8] = {0,0,0,0,0,0,0,0}; 
  return memcmp(m_arrChilds, tmp, sizeof(tmp)) != 0;
}


bool CVoxMeshNode::CleanUpTree()
{
  bool bMeshFound = false;

  for(int nChildId=0; nChildId<8; nChildId++)
    if(m_arrChilds[nChildId])
      if(m_arrChilds[nChildId]->CleanUpTree())
        bMeshFound = true;

  if(m_pVoxObj && m_pVoxObj->GetRenderMesh(0) && m_pVoxObj->GetRenderMesh(0)->GetSysIndicesCount())
  {
    bMeshFound = true;
  }

  if(!bMeshFound)
    ReleaseChilds();    

  return bMeshFound;
}